<?php
/**
 * Media Controller
 *
 * @package AltAudit
 * @since 1.0.0
 */

// Prevent direct access.
if ( ! defined( 'ABSPATH' ) ) {
	exit;
}

/**
 * Media Controller class
 *
 * Handles media library integration including attachment fields,
 * bulk actions, and media-specific functionality.
 *
 * @since 1.0.0
 */
class Altaudit82ai_Media_Controller {



	/**
	 * Settings service
	 *
	 * @var Altaudit82ai_Settings
	 */
	private $settings;

	/**
	 * API client service
	 *
	 * @var Altaudit82ai_Api_Client
	 */
	private $api_client;

	/**
	 * Quality checker service
	 *
	 * @var Altaudit82ai_Quality_Service
	 */
	private $quality_service;

	/**
	 * Template service
	 *
	 * @var Altaudit82ai_Template_Service
	 */
	private $template_service;

	/**
	 * Constructor
	 *
	 * @param Altaudit82ai_Settings         $settings         Settings service.
	 * @param Altaudit82ai_Api_Client       $api_client       API client service.
	 * @param Altaudit82ai_Quality_Service  $quality_service  Quality service.
	 * @param Altaudit82ai_Template_Service $template_service Template service.
	 */
	public function __construct(
		?Altaudit82ai_Settings $settings = null,
		?Altaudit82ai_Api_Client $api_client = null,
		?Altaudit82ai_Quality_Service $quality_service = null,
		?Altaudit82ai_Template_Service $template_service = null
	) {
		$this->settings         = $settings;
		$this->api_client       = $api_client;
		$this->quality_service  = $quality_service;
		$this->template_service = $template_service;

		$this->init_hooks();
	}

	/**
	 * Initialize hooks
	 *
	 * @return void
	 */
	private function init_hooks() {
		// Media library hooks.
		add_action( 'admin_enqueue_scripts', array( $this, 'enqueue_media_scripts' ) );
		add_action( 'wp_ajax_altaudit82ai_media_generate_ai', array( $this, 'ajax_media_generate_ai' ) );
		add_action( 'wp_ajax_altaudit82ai_media_generate_rule', array( $this, 'ajax_media_generate_rule' ) );
		add_action( 'wp_ajax_altaudit82ai_media_audit_quality', array( $this, 'ajax_media_audit_quality' ) );
		add_action( 'wp_ajax_altaudit82ai_media_generate', array( $this, 'ajax_media_generate' ) );

		// Upload hooks.
		add_action( 'add_attachment', array( $this, 'handle_new_upload' ) );

		// Bulk action hooks.
		add_filter( 'bulk_actions-upload', array( $this, 'add_bulk_actions' ) );
		add_filter( 'handle_bulk_actions-upload', array( $this, 'handle_bulk_actions' ), 10, 3 );
		add_action( 'admin_notices', array( $this, 'bulk_action_notices' ) );

		// List table column hooks.
		add_filter( 'manage_media_columns', array( $this, 'add_media_columns' ) );
		add_action( 'manage_media_custom_column', array( $this, 'display_media_column' ), 10, 2 );

		// Attachment details modal hooks.
		add_filter( 'attachment_fields_to_edit', array( $this, 'add_attachment_fields' ), 10, 2 );
		add_filter( 'attachment_fields_to_save', array( $this, 'save_attachment_fields' ), 10, 2 );
	}

	/**
	 * Enqueue media library scripts
	 *
	 * @param string $hook_suffix Current admin page.
	 * @return void
	 */
	public function enqueue_media_scripts( $hook_suffix ) {
		// Only load on media library pages.
		if ( ! in_array( $hook_suffix, array( 'upload.php', 'post.php', 'post-new.php' ), true ) ) {
			return;
		}

		// Media library CSS.
		wp_enqueue_style(
			'altaudit82ai-media',
			ALTAUDIT82AI_ASSETS_URL . 'css/media.css',
			array(),
			ALTAUDIT82AI_VERSION
		);

		// Media library JS.
		wp_enqueue_script(
			'altaudit82ai-media',
			ALTAUDIT82AI_ASSETS_URL . 'js/media.js',
			array( 'jquery', 'media-editor', 'media-views' ),
			ALTAUDIT82AI_VERSION,
			true
		);

		// Localize script.
		wp_localize_script(
			'altaudit82ai-media',
			'altaudit82aiMedia',
			array(
				'ajaxUrl'       => admin_url( 'admin-ajax.php' ),
				'nonce'         => wp_create_nonce( 'altaudit82ai_nonce' ),
				'apiConfigured' => $this->settings->is_api_configured(),
				'autoGenerate'  => $this->settings->get( 'auto_generate', false ),
				'strings'       => array(
					'generateAlt'    => __( 'Generate Alt Text', 'alt-audit' ),
					'generating'     => __( 'Generating...', 'alt-audit' ),
					'checkQuality'   => __( 'Check Quality', 'alt-audit' ),
					'qualityScore'   => /* translators: Placeholder for dynamic content */ __( 'Quality Score: %s', 'alt-audit' ),
					'excellent'      => __( 'Excellent', 'alt-audit' ),
					'good'           => __( 'Good', 'alt-audit' ),
					'weak'           => __( 'Weak', 'alt-audit' ),
					'missing'        => __( 'Missing', 'alt-audit' ),
					'decorative'     => __( 'Decorative', 'alt-audit' ),
					'error'          => __( 'Error generating alt text', 'alt-audit' ),
					'success'        => __( 'Alt text generated successfully', 'alt-audit' ),
					'saved'          => __( 'Alt text saved', 'alt-audit' ),
					'apiKeyRequired' => __( 'API key required in settings', 'alt-audit' ),
				),
			)
		);
	}

	/**
	 * Add attachment fields to edit form
	 *
	 * @param array   $form_fields Array of form fields.
	 * @param WP_Post $post        Attachment post object.
	 * @return array Modified form fields.
	 */
	public function attachment_fields_to_edit( $form_fields, $post ) {
		// Only for images.
		if ( ! wp_attachment_is_image( $post->ID ) ) {
			return $form_fields;
		}

		$alt_text = get_post_meta( $post->ID, '_wp_attachment_image_alt', true );
		$quality  = null;

		// Check quality if alt text exists.
		if ( ! empty( $alt_text ) && $this->quality_service ) {
			$quality = $this->quality_service->assess_quality( $alt_text );
		}

		// Add Alt Audit section.
		$form_fields['altaudit82ai_section'] = array(
			'label' => __( 'Alt Audit', 'alt-audit' ),
			'input' => 'html',
			'html'  => $this->render_attachment_field_html( $post->ID, $alt_text, $quality ),
		);

		return $form_fields;
	}

	/**
	 * Save attachment fields
	 *
	 * @param array $post       Post data.
	 * @param array $attachment Attachment data.
	 * @return array Modified post data.
	 */
	public function attachment_fields_to_save( $post, $attachment ) {
		// This is handled by the standard WordPress alt text field.
		// Our custom functionality is handled via AJAX.
		return $post;
	}

	/**
	 * Add custom fields to attachment edit form
	 *
	 * @param array   $form_fields Array of form fields.
	 * @param WP_Post $post        Attachment post object.
	 * @return array Modified form fields.
	 */
	public function add_attachment_fields( $form_fields, $post ) {
		return $this->attachment_fields_to_edit( $form_fields, $post );
	}

	/**
	 * Save custom attachment fields
	 *
	 * @param array $post       Post data.
	 * @param array $attachment Attachment data.
	 * @return array Modified post data.
	 */
	public function save_attachment_fields( $post, $attachment ) {
		return $this->attachment_fields_to_save( $post, $attachment );
	}

	/**
	 * Handle new upload
	 *
	 * @param int $attachment_id Attachment ID.
	 * @return void
	 */
	public function handle_new_upload( $attachment_id ) {
		// Only process images.
		if ( ! wp_attachment_is_image( $attachment_id ) ) {
			return;
		}

		// Auto-generate if enabled.
		if ( $this->settings->get( 'auto_generate', false ) && $this->settings->is_api_configured() ) {
			$this->auto_generate_alt_text( $attachment_id );
		}
	}

	/**
	 * Add bulk actions to media library
	 *
	 * @param array $bulk_actions Existing bulk actions.
	 * @return array Modified bulk actions.
	 */
	public function add_bulk_actions( $bulk_actions ) {
		$bulk_actions['altaudit82ai_generate']       = __( 'Generate Alt Text', 'alt-audit' );
		$bulk_actions['altaudit82ai_assess_quality'] = __( 'Check Alt Text Quality', 'alt-audit' );

		return $bulk_actions;
	}

	/**
	 * Handle bulk actions
	 *
	 * @param string $redirect_to Redirect URL.
	 * @param string $doaction    Action being performed.
	 * @param array  $post_ids    Array of post IDs.
	 * @return string Modified redirect URL.
	 */
	public function handle_bulk_actions( $redirect_to, $doaction, $post_ids ) {
		if ( ! in_array( $doaction, array( 'altaudit82ai_generate', 'altaudit82ai_assess_quality' ), true ) ) {
			return $redirect_to;
		}

		// Create nonce for bulk action notices.
		$nonce = wp_create_nonce( 'altaudit82ai_bulk_notice' );

		// Filter to only images.
		$image_ids = array();
		foreach ( $post_ids as $post_id ) {
			if ( wp_attachment_is_image( $post_id ) ) {
				$image_ids[] = $post_id;
			}
		}

		if ( empty( $image_ids ) ) {
			return add_query_arg(
				array(
					'altaudit82ai_error' => 'no_images',
					'_wpnonce'           => $nonce,
				),
				$redirect_to
			);
		}

		switch ( $doaction ) {
			case 'altaudit82ai_generate':
				$result = $this->bulk_generate_alt_text( $image_ids );
				break;

			case 'altaudit82ai_assess_quality':
				$result = $this->bulk_assess_quality( $image_ids );
				break;

			default:
				return $redirect_to;
		}

		// Add nonce to result params.
		$result['_wpnonce'] = $nonce;

		return add_query_arg( $result, $redirect_to );
	}

	/**
	 * Display bulk action notices
	 *
	 * @return void
	 */
	public function bulk_action_notices() {
		// Verify nonce is present first - REQUIRED, not optional.
		if ( ! isset( $_GET['_wpnonce'] ) ) {
			return;
		}

		// Verify nonce is valid.
		if ( ! wp_verify_nonce( sanitize_text_field( wp_unslash( $_GET['_wpnonce'] ) ), 'altaudit82ai_bulk_notice' ) ) {
			return;
		}

		// Check if any notice params exist (after nonce verification).
		if ( ! isset( $_GET['altaudit82ai_generated'] ) && ! isset( $_GET['altaudit82ai_checked'] ) && ! isset( $_GET['altaudit82ai_error'] ) ) {
			return;
		}

		if ( isset( $_GET['altaudit82ai_generated'] ) ) {
			$count = intval( wp_unslash( $_GET['altaudit82ai_generated'] ) );
			printf(
				'<div class="notice notice-success is-dismissible"><p>%s</p></div>',
				/* translators: %d: Number of images processed */
				esc_html( sprintf( _n( 'Generated alt text for %d image.', 'Generated alt text for %d images.', $count, 'alt-audit' ), $count ) )
			);
		}

		if ( isset( $_GET['altaudit82ai_checked'] ) ) {
			$count = intval( wp_unslash( $_GET['altaudit82ai_checked'] ) );
			printf(
				'<div class="notice notice-success is-dismissible"><p>%s</p></div>',
				/* translators: %d: Number of images processed */
				esc_html( sprintf( _n( 'Checked quality for %d image.', 'Checked quality for %d images.', $count, 'alt-audit' ), $count ) )
			);
		}

		if ( isset( $_GET['altaudit82ai_error'] ) ) {
			$error   = sanitize_text_field( wp_unslash( $_GET['altaudit82ai_error'] ) );
			$message = __( 'An error occurred.', 'alt-audit' );

			// Validate error against known error types.
			$allowed_errors = array( 'no_images', 'api_error', 'no_api_key' );
			if ( in_array( $error, $allowed_errors, true ) ) {
				switch ( $error ) {
					case 'no_images':
						$message = __( 'No images selected.', 'alt-audit' );
						break;
					case 'api_error':
						$message = __( 'API error occurred.', 'alt-audit' );
						break;
					case 'no_api_key':
						$message = __( 'API key not configured.', 'alt-audit' );
						break;
				}
			}

			printf(
				'<div class="notice notice-error is-dismissible"><p>%s</p></div>',
				esc_html( $message )
			);
		}
	}

	/**
	 * Add custom columns to media library
	 *
	 * @param array $columns Existing columns.
	 * @return array Modified columns.
	 */
	public function add_media_columns( $columns ) {
		$columns['altaudit82ai_quality'] = __( 'Alt Text Quality', 'alt-audit' );
		return $columns;
	}

	/**
	 * Display custom media column content
	 *
	 * @param string $column_name Column name.
	 * @param int    $post_id     Post ID.
	 * @return void
	 */
	public function display_media_column( $column_name, $post_id ) {
		if ( 'altaudit82ai_quality' !== $column_name ) {
			return;
		}

		// Only for images.
		if ( ! wp_attachment_is_image( $post_id ) ) {
			echo '—';
			return;
		}

		$alt_text = get_post_meta( $post_id, '_wp_attachment_image_alt', true );

		if ( empty( $alt_text ) ) {
			echo '<span class="alt-audit-status missing">' . esc_html__( 'Missing', 'alt-audit' ) . '</span>';
			return;
		}

		if ( $this->quality_service ) {
			$quality      = $this->quality_service->assess_quality( $alt_text );
			$status_class = sanitize_html_class( $quality['status'] );
			$status_label = ucfirst( $quality['status'] );

			printf(
				'<span class="alt-audit-status %s" title="%s">%s (%d%%)</span>',
				esc_attr( $status_class ),
				esc_attr( $quality['message'] ?? '' ),
				esc_html( $status_label ),
				intval( $quality['score'] ?? 0 )
			);
		} else {
			echo '<span class="alt-audit-status present">' . esc_html__( 'Present', 'alt-audit' ) . '</span>';
		}
	}

	/**
	 * AJAX handler for media library alt text generation
	 *
	 * @return void
	 */
	public function ajax_media_generate() {
		// Verify nonce - using check_ajax_referer for clear static analysis.
		if ( ! check_ajax_referer( 'altaudit82ai_nonce', 'nonce', false ) ) {
			wp_send_json_error( __( 'Security check failed.', 'alt-audit' ) );
		}

		// Check user permissions.
		if ( ! current_user_can( 'upload_files' ) ) {
			wp_send_json_error( __( 'Insufficient permissions.', 'alt-audit' ) );
		}

		$attachment_id = isset( $_POST['attachment_id'] ) ? intval( wp_unslash( $_POST['attachment_id'] ) ) : 0;
		if ( ! $attachment_id || ! wp_attachment_is_image( $attachment_id ) ) {
			wp_send_json_error( __( 'Invalid image ID.', 'alt-audit' ) );
		}

		// Validate style against allowed values.
		$allowed_styles = array( 'descriptive', 'technical', 'creative', 'brief' );
		$style          = isset( $_POST['style'] ) ? sanitize_text_field( wp_unslash( $_POST['style'] ) ) : 'descriptive';
		$style          = in_array( $style, $allowed_styles, true ) ? $style : 'descriptive';

		$options = array(
			'style'   => $style,
			'context' => isset( $_POST['context'] ) ? sanitize_textarea_field( wp_unslash( $_POST['context'] ) ) : '',
		);

		$result = $this->api_client->generate_alt_text( $attachment_id, $options );

		if ( ! $result['success'] ) {
			wp_send_json_error( $result['message'] );
		}

		// Save alt text.
		update_post_meta( $attachment_id, '_wp_attachment_image_alt', $result['alt_text'] );

		// Check quality.
		$quality = null;
		if ( $this->quality_service ) {
			$quality = $this->quality_service->assess_quality( $result['alt_text'] );
		}

		// Logging removed - not needed.

		wp_send_json_success(
			array(
				'alt_text' => $result['alt_text'],
				'quality'  => $quality,
				'message'  => __( 'Alt text generated successfully.', 'alt-audit' ),
			)
		);
	}

	/**
	 * Auto-generate alt text for new upload
	 *
	 * @param int $attachment_id Attachment ID.
	 * @return void
	 */
	private function auto_generate_alt_text( $attachment_id ) {
		$result = $this->api_client->generate_alt_text( $attachment_id );

		if ( $result['success'] ) {
			update_post_meta( $attachment_id, '_wp_attachment_image_alt', $result['alt_text'] );

			// Logging removed - not needed.
		}
	}

	/**
	 * Bulk generate alt text
	 *
	 * @param array $image_ids Array of image IDs.
	 * @return array Result array for redirect.
	 */
	private function bulk_generate_alt_text( $image_ids ) {
		if ( ! $this->settings->is_api_configured() ) {
			return array( 'altaudit82ai_error' => 'no_api_key' );
		}

		$generated     = 0;
		$skip_existing = $this->settings->get( 'skip_existing', true );

		foreach ( $image_ids as $image_id ) {
			// Skip if already has alt text and skip_existing is enabled.
			if ( $skip_existing ) {
				$existing_alt = get_post_meta( $image_id, '_wp_attachment_image_alt', true );
				if ( ! empty( $existing_alt ) ) {
					continue;
				}
			}

			$result = $this->api_client->generate_alt_text( $image_id );

			if ( $result['success'] ) {
				update_post_meta( $image_id, '_wp_attachment_image_alt', $result['alt_text'] );
				++$generated;

				// Logging removed - not needed.
			}
		}

		return array( 'altaudit82ai_generated' => $generated );
	}

	/**
	 * Bulk check quality
	 *
	 * @param array $image_ids Array of image IDs.
	 * @return array Result array for redirect.
	 */
	private function bulk_assess_quality( $image_ids ) {
		if ( ! $this->quality_service ) {
			return array( 'altaudit82ai_error' => 'no_quality_service' );
		}

		$checked = 0;

		foreach ( $image_ids as $image_id ) {
			$alt_text = get_post_meta( $image_id, '_wp_attachment_image_alt', true );

			if ( ! empty( $alt_text ) ) {
				$this->quality_service->assess_quality( $alt_text );
				++$checked;
			}
		}

		return array( 'altaudit82ai_checked' => $checked );
	}

	/**
	 * Render attachment field HTML
	 *
	 * @param int    $attachment_id Attachment ID.
	 * @param string $alt_text      Current alt text.
	 * @param array  $quality       Quality check result.
	 * @return string HTML output.
	 */
	private function render_attachment_field_html( $attachment_id, $alt_text, $quality ) {
		ob_start();
		include ALTAUDIT82AI_TEMPLATES_DIR . 'admin/attachment-field.php';
		return ob_get_clean();
	}

	/**
	 * AJAX handler for AI alt text generation
	 *
	 * @return void
	 */
	public function ajax_media_generate_ai() {
		check_ajax_referer( 'altaudit82ai_nonce', 'nonce' );

		if ( ! current_user_can( 'upload_files' ) ) {
			wp_send_json_error( __( 'Insufficient permissions.', 'alt-audit' ) );
		}

		$attachment_id = isset( $_POST['attachment_id'] ) ? intval( wp_unslash( $_POST['attachment_id'] ) ) : 0;
		if ( ! $attachment_id || ! wp_attachment_is_image( $attachment_id ) ) {
			wp_send_json_error( __( 'Invalid image ID.', 'alt-audit' ) );
		}

		$result = $this->api_client->generate_alt_text( $attachment_id, array( 'style' => 'descriptive' ) );

		if ( ! $result['success'] ) {
			wp_send_json_error( $result['message'] );
		}

		// Don't save to database - let user click "Update" to save.

		$quality = null;
		if ( $this->quality_service ) {
			$quality = $this->quality_service->assess_quality( $result['alt_text'] );
		}

		wp_send_json_success(
			array(
				'alt_text' => $result['alt_text'],
				'quality'  => $quality,
				'message'  => __( 'AI-powered alt text generated. Click "Update" to save.', 'alt-audit' ),
			)
		);
	}

	/**
	 * AJAX handler for rule-based alt text generation
	 *
	 * @return void
	 */
	public function ajax_media_generate_rule() {
		check_ajax_referer( 'altaudit82ai_nonce', 'nonce' );

		if ( ! current_user_can( 'upload_files' ) ) {
			wp_send_json_error( __( 'Insufficient permissions.', 'alt-audit' ) );
		}

		$attachment_id = isset( $_POST['attachment_id'] ) ? intval( wp_unslash( $_POST['attachment_id'] ) ) : 0;
		if ( ! $attachment_id || ! wp_attachment_is_image( $attachment_id ) ) {
			wp_send_json_error( __( 'Invalid image ID.', 'alt-audit' ) );
		}

		// Use template service if available.
		if ( $this->template_service ) {
			$rule_based_alt = $this->template_service->generate_from_template( $attachment_id );
		} else {
			// Fallback if service not available (shouldn't happen).
			$attachment     = get_post( $attachment_id );
			$filename       = basename( get_attached_file( $attachment_id ) );
			$rule_based_alt = ucwords( str_replace( array( '-', '_', '.' ), ' ', pathinfo( $filename, PATHINFO_FILENAME ) ) );
		}

		// Don't save to database - let user click "Update" to save.

		$quality = null;
		if ( $this->quality_service ) {
			$quality = $this->quality_service->assess_quality( $rule_based_alt );
		}

		wp_send_json_success(
			array(
				'alt_text' => $rule_based_alt,
				'quality'  => $quality,
				'message'  => __( 'Rule-based alt text generated. Click "Update" to save.', 'alt-audit' ),
			)
		);
	}

	/**
	 * AJAX handler for quality audit
	 *
	 * @return void
	 */
	public function ajax_media_audit_quality() {
		check_ajax_referer( 'altaudit82ai_nonce', 'nonce' );

		if ( ! current_user_can( 'upload_files' ) ) {
			wp_send_json_error( __( 'Insufficient permissions.', 'alt-audit' ) );
		}

		$attachment_id = isset( $_POST['attachment_id'] ) ? intval( wp_unslash( $_POST['attachment_id'] ) ) : 0;
		if ( ! $attachment_id || ! wp_attachment_is_image( $attachment_id ) ) {
			wp_send_json_error( __( 'Invalid image ID.', 'alt-audit' ) );
		}

		$alt_text = get_post_meta( $attachment_id, '_wp_attachment_image_alt', true );

		if ( empty( $alt_text ) ) {
			wp_send_json_error( __( 'No alt text to audit.', 'alt-audit' ) );
		}

		if ( ! $this->quality_service ) {
			wp_send_json_error( __( 'Quality service not available.', 'alt-audit' ) );
		}

		$quality = $this->quality_service->assess_quality( $alt_text );

		wp_send_json_success(
			array(
				'quality' => $quality,
				'message' => __( 'Quality audit completed.', 'alt-audit' ),
			)
		);
	}
}
