<?php
/**
 * Images List Table
 *
 * @package AltAudit
 * @since 1.0.0
 */

// Prevent direct access.
if ( ! defined( 'ABSPATH' ) ) {
	exit;
}

// Include WordPress WP_List_Table class if not already loaded.
if ( ! class_exists( 'WP_List_Table' ) ) {
	require_once ABSPATH . 'wp-admin/includes/class-wp-list-table.php';
}

/**
 * Images List Table class
 *
 * Extends WP_List_Table to display images with alt text and quality scores.
 * Supports pagination, sorting, filtering, bulk actions, and inline editing.
 *
 * @since 1.0.0
 */
class Altaudit82ai_Images_List_Table extends WP_List_Table {




	/**
	 * Quality service
	 *
	 * @var Altaudit82ai_Quality_Service
	 */
	private $quality_service;

	/**
	 * Settings service
	 *
	 * @var Altaudit82ai_Settings
	 */
	private $settings;

	/**
	 * Statistics service
	 *
	 * @var Altaudit82ai_Statistics_Service
	 */
	private $statistics_service;

	/**
	 * Total items count
	 *
	 * @var int
	 */
	private $total_items = 0;

	/**
	 * Custom parameters passed to list table (used for AJAX requests).
	 *
	 * @var array
	 */
	private $custom_params = array();

	/**
	 * Constructor
	 *
	 * @param Altaudit82ai_Quality_Service    $quality_service    Quality service instance.
	 * @param Altaudit82ai_Settings           $settings           Settings service instance.
	 * @param array                           $custom_params      Optional custom parameters for AJAX context.
	 * @param Altaudit82ai_Statistics_Service $statistics_service Statistics service instance.
	 */
	public function __construct( $quality_service = null, $settings = null, $custom_params = array(), $statistics_service = null ) {
		$this->quality_service    = $quality_service;
		$this->settings           = $settings;
		$this->custom_params      = $custom_params;
		$this->statistics_service = $statistics_service ?? new Altaudit82ai_Statistics_Service();

		parent::__construct(
			array(
				'singular' => 'image',
				'plural'   => 'images',
				'ajax'     => true,
			)
		);
	}

	/**
	 * Get table columns
	 *
	 * @return array Table columns.
	 */
	public function get_columns() {
		return array(
			'cb'            => '<input type="checkbox" />',
			'thumbnail'     => __( 'Image', 'alt-audit' ),
			'filename'      => __( 'File Name', 'alt-audit' ),
			'alt_text'      => __( 'Alt Text', 'alt-audit' ),
			'status'        => __( 'Status', 'alt-audit' ),
			'quality_score' => __( 'Quality Score', 'alt-audit' ),
			'explanation'   => __( 'Explanation', 'alt-audit' ),
		);
	}

	/**
	 * Get sortable columns
	 *
	 * @return array Sortable columns.
	 */
	public function get_sortable_columns() {
		return array(
			'status'        => array( 'status', false ),
			'quality_score' => array( 'quality_score', false ),
		);
	}

	/**
	 * Get bulk actions
	 *
	 * @return array Bulk actions.
	 */
	public function get_bulk_actions() {
		$actions = array(
			'generate_rule_based' => __( '🆓 Generate Alt Text (Rule-Based)', 'alt-audit' ),
			'generate_ai'         => __( '🤖 Generate Alt Text (AI)', 'alt-audit' ),
		);

		// Add delete action for users who can delete attachments (currently disabled).
		// phpcs:ignore Generic.CodeAnalysis.EmptyStatement.DetectedIf -- Placeholder for future feature.
		if ( current_user_can( 'delete_posts' ) ) {
			// Reserved for delete action.
		}

		return $actions;
	}

	/**
	 * Get a parameter value, checking custom_params first, then $_GET.
	 *
	 * Security: $_GET parameters require valid nonce verification.
	 * Custom params (from AJAX) are trusted as they're set internally.
	 *
	 * @param string $key     Parameter key.
	 * @param mixed  $default Default value if not found.
	 * @return mixed Parameter value.
	 */
	private function get_param( $key, $default = '' ) {
		// Custom params are set internally (e.g., from AJAX handlers that already verified nonce).
		if ( isset( $this->custom_params[ $key ] ) ) {
			return sanitize_text_field( $this->custom_params[ $key ] );
		}

		// For $_GET parameters, REQUIRE nonce verification - never bypass.
		// If nonce is missing or invalid, return default value for security.
		if ( ! isset( $_GET['_altaudit82ai_nonce'] ) ) {
			return $default;
		}

		if ( ! wp_verify_nonce( sanitize_text_field( wp_unslash( $_GET['_altaudit82ai_nonce'] ) ), 'altaudit82ai_list_table' ) ) {
			return $default;
		}

		// Nonce verified - safe to access $_GET parameter.
		if ( isset( $_GET[ $key ] ) ) {
			return sanitize_text_field( wp_unslash( $_GET[ $key ] ) );
		}

		return $default;
	}

	/**
	 * Override get_pagenum to check custom_params first.
	 *
	 * This ensures pagination works correctly during AJAX requests
	 * without modifying $_GET superglobal.
	 *
	 * @return int Current page number.
	 */
	public function get_pagenum() {
		$paged = $this->get_param( 'paged', 0 );
		if ( $paged ) {
			return absint( $paged );
		}

		return parent::get_pagenum();
	}

	/**
	 * Prepare items for display
	 *
	 * @return void
	 */
	public function prepare_items() {
		// Get current page (uses overridden get_pagenum that checks custom_params).
		$current_page = $this->get_pagenum();
		$per_page     = $this->get_items_per_page( 'altaudit82ai_images_per_page', 20 );

		// Prepare column headers.
		$columns  = $this->get_columns();
		$hidden   = array();
		$sortable = $this->get_sortable_columns();

		$this->_column_headers = array( $columns, $hidden, $sortable );

		// Get data using get_param helper (checks custom_params first, then $_GET).
		$data = $this->get_images_data(
			array(
				'offset'   => ( $current_page - 1 ) * $per_page,
				'per_page' => $per_page,
				'orderby'  => $this->get_param( 'orderby', 'post_date' ),
				'order'    => $this->get_param( 'order', 'DESC' ),
			)
		);

		$this->items       = $data['items'];
		$this->total_items = $data['total'];

		// Set pagination.
		$this->set_pagination_args(
			array(
				'total_items' => $this->total_items,
				'per_page'    => $per_page,
				'total_pages' => ceil( $this->total_items / $per_page ),
			)
		);
	}

	/**
	 * Get images data using WP_Query
	 *
	 * Uses WordPress's standard WP_Query API for automatic safety and simplicity.
	 *
	 * @param array $args Query arguments.
	 * @return array Images data with items and total count.
	 */
	private function get_images_data( $args = array() ) {
		$defaults = array(
			'offset'   => 0,
			'per_page' => 20,
			'orderby'  => 'post_date',
			'order'    => 'DESC',
			'search'   => '',
			'status'   => '',
		);

		$args = wp_parse_args( $args, $defaults );

		// Handle search using get_param helper.
		$search = $this->get_param( 's' );
		if ( ! empty( $search ) ) {
			$args['search'] = $search;
		}

		// Handle status filter using get_param helper.
		$status_filter    = $this->get_param( 'status_filter' );
		$allowed_statuses = array( 'missing', 'weak', 'good', 'excellent', 'decorative' );
		if ( ! empty( $status_filter ) && in_array( $status_filter, $allowed_statuses, true ) ) {
			$args['status'] = $status_filter;
		}

		// For status sorting, we need to fetch all items first, then sort and paginate.
		// For other sorting, use normal WP_Query pagination.
		$is_status_sort = ( 'status' === sanitize_key( $args['orderby'] ) );

		// Build WP_Query arguments with proper sanitization.
		$query_args = array(
			'post_type'      => 'attachment',
			'post_status'    => 'inherit',
			'post_mime_type' => Altaudit82ai::get_supported_mime_types(),
			'posts_per_page' => $is_status_sort ? -1 : absint( $args['per_page'] ),
			'offset'         => $is_status_sort ? 0 : absint( $args['offset'] ),
			'orderby'        => $this->map_orderby( sanitize_key( $args['orderby'] ) ),
			'order'          => ( 'ASC' === strtoupper( sanitize_key( $args['order'] ) ) ) ? 'ASC' : 'DESC',
		);

		// Add meta_key when sorting by quality_score.
		$orderby = sanitize_key( $args['orderby'] );
		if ( 'quality_score' === $orderby ) {
			// phpcs:ignore WordPress.DB.SlowDBQuery.slow_db_query_meta_key -- Required for sorting by quality_score meta field.
			$query_args['meta_key'] = '_altaudit82ai_quality_score';
			// Note: Not using meta_query to avoid JOIN complexity that might interfere with ORDER BY.
			// Posts without quality_score will naturally appear last (with empty/NULL values).
		}

		// Add search with sanitization.
		if ( ! empty( $args['search'] ) ) {
			$query_args['s'] = sanitize_text_field( $args['search'] );
		}

		// Add status filter via tax query (uses indexed taxonomy tables).
		if ( ! empty( $args['status'] ) ) {
			// phpcs:ignore WordPress.DB.SlowDBQuery.slow_db_query_tax_query -- Taxonomy approach is faster than meta_query; uses indexed term_relationships table.
			$query_args['tax_query'] = $this->build_status_tax_query( $args['status'] );
		}

		// Execute query.
		$query = new WP_Query( $query_args );

		// Process each image.
		$processed_items = array();
		foreach ( $query->posts as $post ) {
			$image             = $this->convert_post_to_image_object( $post );
			$item              = $this->process_image_item( $image );
			$processed_items[] = $item;
		}

		// Handle status sorting (taxonomy doesn't support WP_Query sorting).
		if ( $is_status_sort ) {
			$status_order = array(
				'missing'    => 0,
				'weak'       => 1,
				'good'       => 2,
				'excellent'  => 3,
				'decorative' => 4,
			);
			$order_dir    = strtoupper( sanitize_key( $args['order'] ) );
			usort(
				$processed_items,
				function ( $a, $b ) use ( $status_order, $order_dir ) {
					$a_val = isset( $status_order[ $a['status'] ] ) ? $status_order[ $a['status'] ] : 999;
					$b_val = isset( $status_order[ $b['status'] ] ) ? $status_order[ $b['status'] ] : 999;
					return ( 'DESC' === $order_dir ) ? ( $b_val <=> $a_val ) : ( $a_val <=> $b_val );
				}
			);

			// Apply pagination after sorting.
			$total_items     = count( $processed_items );
			$processed_items = array_slice( $processed_items, absint( $args['offset'] ), absint( $args['per_page'] ) );

			return array(
				'items' => $processed_items,
				'total' => $total_items,
			);
		}

		return array(
			'items' => $processed_items,
			'total' => $query->found_posts,
		);
	}

	/**
	 * Map orderby parameter to WP_Query format
	 *
	 * @param string $orderby Order by column.
	 * @return string WP_Query orderby value.
	 */
	private function map_orderby( $orderby ) {
		$map = array(
			'post_title'    => 'title',
			'post_date'     => 'date',
			'status'        => 'date',
			'quality_score' => 'meta_value_num',
		);

		return isset( $map[ $orderby ] ) ? $map[ $orderby ] : 'date';
	}

	/**
	 * Build tax query for status filter
	 *
	 * Uses custom taxonomy for efficient filtering instead of meta_query.
	 *
	 * @param string $status Status to filter by.
	 * @return array Tax query array.
	 */
	private function build_status_tax_query( $status ) {
		// Sanitize status to ensure it's a safe string.
		$status = sanitize_key( $status );

		return Altaudit82ai_Taxonomy::build_status_tax_query( $status );
	}

	/**
	 * Convert WP_Post object to image data object
	 *
	 * @param WP_Post $post Post object.
	 * @return object Image data object matching expected format.
	 */
	private function convert_post_to_image_object( $post ) {
		$alt_text          = sanitize_text_field( get_post_meta( $post->ID, '_wp_attachment_image_alt', true ) );
		$is_decorative     = sanitize_text_field( get_post_meta( $post->ID, '_altaudit82ai_decorative', true ) );
		$is_auto_generated = sanitize_text_field( get_post_meta( $post->ID, '_altaudit82ai_auto_generated', true ) );

		return (object) array(
			'ID'                => absint( $post->ID ),
			'post_title'        => sanitize_text_field( $post->post_title ),
			'post_date'         => sanitize_text_field( $post->post_date ),
			'guid'              => esc_url_raw( $post->guid ),
			'alt_text'          => ! empty( $alt_text ) ? $alt_text : '',
			'is_decorative'     => ! empty( $is_decorative ) ? $is_decorative : '0',
			'is_auto_generated' => ! empty( $is_auto_generated ) ? $is_auto_generated : '0',
		);
	}

	/**
	 * Process individual image item
	 *
	 * @param object $image Raw image data from database.
	 * @return array Processed image item.
	 */
	private function process_image_item( $image ) {
		$attachment_id = (int) $image->ID;
		$alt_text      = $image->alt_text;
		$is_decorative = '1' === $image->is_decorative;

		// Get thumbnail.
		$thumbnail_url = wp_get_attachment_image_src( $attachment_id, array( 60, 60 ) );
		$thumbnail_url = $thumbnail_url ? $thumbnail_url[0] : '';

		// Determine status and quality.
		// IMPORTANT: If alt text exists, it takes precedence over decorative flag (which may be stale).
		$explanation = '';
		if ( ! empty( $alt_text ) ) {
			// Has alt text - assess quality.
			if ( $this->quality_service ) {
				$quality_result = $this->quality_service->assess_quality( $alt_text );
				// IMPORTANT: Use 'score' (final combined score) for consistency with media edit page.
				$quality_score = $quality_result['score'] ?? 0;
				// IMPORTANT: Recalculate status based on FINAL score to ensure consistency.
				$status      = $this->get_status_from_score( $quality_score );
				$explanation = $this->generate_explanation( $quality_result, $status, $quality_score );
			} else {
				// Fallback quality assessment.
				$length = strlen( $alt_text );
				if ( $length < 10 ) {
					$status        = 'weak';
					$quality_score = 30;
					$explanation   = __( 'Alt text is too short to be descriptive.', 'alt-audit' );
				} elseif ( $length < 50 ) {
					$status        = 'good';
					$quality_score = 70;
					$explanation   = __( 'Alt text has good length but could be more descriptive.', 'alt-audit' );
				} else {
					$status        = 'excellent';
					$quality_score = 90;
					$explanation   = __( 'Alt text is descriptive and meets accessibility standards.', 'alt-audit' );
				}
			}
			// Clear decorative flag since we have alt text.
			$is_decorative = false;
		} elseif ( $is_decorative ) {
			// Empty alt text + decorative flag = properly marked decorative.
			$status        = 'decorative';
			$quality_score = 100;
			$explanation   = __( 'Image is properly marked as decorative with empty alt text.', 'alt-audit' );
		} else {
			// Empty alt text + no decorative flag = missing.
			$status        = 'missing';
			$quality_score = 0;
			$explanation   = __( 'Alt text is missing. This image needs descriptive alt text for accessibility.', 'alt-audit' );
		}

		// Store quality status as taxonomy term for efficient filtering.
		// Only update if different from stored value to avoid unnecessary DB writes.
		$stored_status = Altaudit82ai_Taxonomy::get_status( $attachment_id );
		if ( $stored_status !== $status ) {
			Altaudit82ai_Taxonomy::set_status( $attachment_id, $status );
		}

		// Store quality score in post meta.
		$stored_score = absint( get_post_meta( $attachment_id, '_altaudit82ai_quality_score', true ) );
		if ( $stored_score !== $quality_score ) {
			update_post_meta( $attachment_id, '_altaudit82ai_quality_score', absint( $quality_score ) );
		}

		return array(
			'ID'                => $attachment_id,
			'post_title'        => sanitize_text_field( $image->post_title ),
			'post_date'         => sanitize_text_field( $image->post_date ),
			'filename'          => sanitize_file_name( basename( $image->guid ) ),
			'alt_text'          => sanitize_text_field( $alt_text ),
			'thumbnail_url'     => esc_url( $thumbnail_url ),
			'edit_url'          => esc_url( admin_url( 'post.php?post=' . $attachment_id . '&action=edit' ) ),
			'is_decorative'     => $is_decorative,
			'is_auto_generated' => '1' === $image->is_auto_generated,
			'status'            => sanitize_key( $status ),
			'quality_status'    => sanitize_key( $status ),
			'quality_score'     => absint( $quality_score ),
			'explanation'       => sanitize_text_field( $explanation ),
		);
	}

	/**
	 * Generate explanation for quality score
	 *
	 * @param array  $quality_result Quality assessment result from quality service.
	 * @param string $status         Quality status.
	 * @param int    $quality_score  Quality score.
	 * @return string Explanation text.
	 */
	private function generate_explanation( $quality_result, $status, $quality_score ) {
		$explanation_parts = array();

		// Get detailed breakdown.
		$breakdown = $quality_result['score_breakdown'] ?? array();

		// Build detailed explanation based on score components.
		$positives = array();
		$issues    = array();

		// Analyze length.
		if ( ! empty( $breakdown['length'] ) ) {
			$length_score = $breakdown['length']['score'] ?? 0;
			$length_max   = $breakdown['length']['max'] ?? 25;

			if ( $length_score >= $length_max * 0.8 ) {
				$positives[] = __( 'good length', 'alt-audit' );
			} elseif ( $length_score < $length_max * 0.5 ) {
				$issues[] = __( 'length could be improved', 'alt-audit' );
			}
		}

		// Analyze word count.
		if ( ! empty( $breakdown['word_count'] ) ) {
			$word_score = $breakdown['word_count']['score'] ?? 0;
			$word_max   = $breakdown['word_count']['max'] ?? 20;

			if ( $word_score >= $word_max * 0.8 ) {
				$positives[] = __( 'appropriate word count', 'alt-audit' );
			} elseif ( $word_score < $word_max * 0.5 ) {
				$issues[] = __( 'needs more descriptive words', 'alt-audit' );
			}
		}

		// Analyze descriptiveness.
		if ( ! empty( $breakdown['descriptiveness'] ) ) {
			$desc_score = $breakdown['descriptiveness']['score'] ?? 0;
			$desc_max   = $breakdown['descriptiveness']['max'] ?? 30;

			if ( $desc_score >= $desc_max * 0.8 ) {
				$positives[] = __( 'highly descriptive', 'alt-audit' );
			} elseif ( $desc_score < $desc_max * 0.5 ) {
				$issues[] = __( 'lacks descriptive detail', 'alt-audit' );
			}
		}

		// Analyze structure.
		if ( ! empty( $breakdown['structure'] ) ) {
			$struct_score = $breakdown['structure']['score'] ?? 0;
			$struct_max   = $breakdown['structure']['max'] ?? 25;

			if ( $struct_score >= $struct_max * 0.8 ) {
				$positives[] = __( 'well-structured', 'alt-audit' );
			} elseif ( $struct_score < $struct_max * 0.5 ) {
				$issues[] = __( 'structure needs improvement', 'alt-audit' );
			}
		}

		// Build explanation based on status and score.
		if ( 'excellent' === $status ) {
			if ( ! empty( $positives ) ) {
				/* translators: %s: comma-separated list of positive qualities */
				$explanation_parts[] = sprintf( __( 'Excellent alt text: %s', 'alt-audit' ), implode( ', ', $positives ) );
			} else {
				$explanation_parts[] = __( 'Excellent alt text that meets all accessibility standards', 'alt-audit' );
			}
		} elseif ( 'good' === $status ) {
			if ( ! empty( $positives ) ) {
				/* translators: %s: comma-separated list of positive qualities */
				$explanation_parts[] = sprintf( __( 'Good alt text with %s', 'alt-audit' ), implode( ', ', $positives ) );
			} else {
				$explanation_parts[] = __( 'Good alt text that meets most accessibility standards', 'alt-audit' );
			}
			if ( ! empty( $issues ) ) {
				/* translators: %s: comma-separated list of issues */
				$explanation_parts[] = sprintf( __( 'Consider: %s', 'alt-audit' ), implode( ', ', $issues ) );
			}
		} elseif ( 'weak' === $status ) {
			$explanation_parts[] = __( 'Weak alt text', 'alt-audit' );
			if ( ! empty( $issues ) ) {
				/* translators: %s: comma-separated list of issues */
				$explanation_parts[] = sprintf( __( 'Issues: %s', 'alt-audit' ), implode( ', ', $issues ) );
			}
			if ( ! empty( $positives ) ) {
				/* translators: %s: comma-separated list of positive qualities */
				$explanation_parts[] = sprintf( __( 'Has: %s', 'alt-audit' ), implode( ', ', $positives ) );
			}
		}

		// Add specific suggestions.
		if ( ! empty( $quality_result['suggestions'] ) && is_array( $quality_result['suggestions'] ) ) {
			$suggestion = reset( $quality_result['suggestions'] );
			if ( ! empty( $suggestion ) && 'excellent' !== $status ) {
				$explanation_parts[] = $suggestion;
			}
		}

		// Add score for context.
		if ( 'excellent' !== $status && 'decorative' !== $status && 'missing' !== $status ) {
			/* translators: %d: quality score out of 100 */
			$explanation_parts[] = sprintf( __( 'Score: %d/100', 'alt-audit' ), $quality_score );
		}

		// Join all parts.
		$explanation = implode( '. ', $explanation_parts );

		// Ensure proper punctuation.
		if ( ! empty( $explanation ) && ! in_array( substr( $explanation, -1 ), array( '.', '!', '?' ), true ) ) {
			$explanation .= '.';
		}

		return $explanation;
	}

	/**
	 * Get status from quality score
	 *
	 * Ensures status matches the actual score displayed to user.
	 *
	 * @param int $score Quality score (0-100).
	 * @return string Status (weak, good, excellent).
	 */
	private function get_status_from_score( $score ) {
		if ( $score >= 86 ) {
			return 'excellent';
		} elseif ( $score >= 61 ) {
			return 'good';
		} elseif ( $score > 0 ) {
			return 'weak';
		} else {
			return 'missing';
		}
	}

	/**
	 * Display extra tablenav
	 *
	 * @param string $which Position of tablenav (top or bottom).
	 * @return void
	 */
	public function extra_tablenav( $which ) {
		if ( 'top' === $which ) {
			echo '<div class="alignleft actions alt-audit-actions">';

			// Status filter dropdown.
			$status_options = array(
				''          => __( 'All Statuses', 'alt-audit' ),
				'missing'   => __( 'Missing Alt Text', 'alt-audit' ),
				'weak'      => __( 'Weak Alt Text', 'alt-audit' ),
				'good'      => __( 'Good Alt Text', 'alt-audit' ),
				'excellent' => __( 'Excellent Alt Text', 'alt-audit' ),
			);

			// Only process status filter if nonce is present and valid.
			$current_status = '';
			if ( isset( $_GET['_altaudit82ai_nonce'] ) ) {
				if ( wp_verify_nonce( sanitize_text_field( wp_unslash( $_GET['_altaudit82ai_nonce'] ) ), 'altaudit82ai_list_table' ) ) {
					// Nonce verified - safe to read status_filter parameter.
					$current_status   = isset( $_GET['status_filter'] ) ? sanitize_text_field( wp_unslash( $_GET['status_filter'] ) ) : '';
					$allowed_statuses = array( '', 'missing', 'weak', 'good', 'excellent', 'decorative' );
					$current_status   = in_array( $current_status, $allowed_statuses, true ) ? $current_status : '';
				}
			}

			echo '<select name="status_filter" id="status_filter" class="alt-audit-select alt-audit-status-filter">';
			foreach ( $status_options as $value => $label ) {
				printf(
					'<option value="%s"%s>%s</option>',
					esc_attr( $value ),
					selected( $current_status, $value, false ),
					esc_html( $label )
				);
			}
			echo '</select>';

			submit_button( __( 'Filter', 'alt-audit' ), 'altaudit82ai-button alt-audit-button-secondary', 'filter_action', false );

			echo '</div>';
		}
	}

	/**
	 * Display the table navigation and filter controls
	 *
	 * Overrides parent to always display controls even when no items,
	 * allowing users to clear filters/search when table is empty.
	 *
	 * @param string $which Position of the controls (top or bottom).
	 * @return void
	 */
	protected function display_tablenav( $which ) {
		// IMPORTANT: Always display tablenav, even with no items.
		// This allows users to clear filters/search when table is empty.
		?>
		<div class="tablenav <?php echo esc_attr( $which ); ?>">

			<?php if ( 'top' === $which ) : ?>
				<div class="alignleft actions bulkactions">
					<?php $this->bulk_actions( $which ); ?>
				</div>
			<?php endif; ?>

			<?php
			// Display filters and extra controls.
			$this->extra_tablenav( $which );

			// Display pagination.
			$this->pagination( $which );
			?>

			<br class="clear" />
		</div>
		<?php
	}

	/**
	 * Generate bulk actions markup with custom classes
	 *
	 * @param string $which Position of bulk actions (top or bottom).
	 * @return void
	 */
	protected function bulk_actions( $which = '' ) {
		if ( is_null( $this->_actions ) ) {
			$this->_actions = $this->get_bulk_actions();

			$two_part = array();
			foreach ( $this->_actions as $name => $title ) {
				if ( is_array( $title ) ) {
					$two_part[] = $title;
				}
			}

			if ( ! empty( $two_part ) ) {
				$this->_actions = array_diff_key( $this->_actions, $two_part );
			}
		}

		if ( empty( $this->_actions ) ) {
			return;
		}

		echo '<label for="bulk-action-selector-' . esc_attr( $which ) . '" class="screen-reader-text">' . esc_html__( 'Select bulk action', 'alt-audit' ) . '</label>';
		echo '<select name="action' . esc_attr( 'top' === $which ? '' : '2' ) . '" id="bulk-action-selector-' . esc_attr( $which ) . '" class="alt-audit-select alt-audit-bulk-select">';
		echo '<option value="-1">' . esc_html__( 'Bulk actions', 'alt-audit' ) . '</option>';

		foreach ( $this->_actions as $name => $title ) {
			$class = 'edit' === $name ? ' class="hide-if-no-js"' : '';
			echo "\t" . '<option value="' . esc_attr( $name ) . '"' . esc_attr( $class ) . '>' . esc_html( $title ) . "</option>\n";
		}

		echo "</select>\n";

		submit_button( __( 'Apply', 'alt-audit' ), 'altaudit82ai-button alt-audit-button-secondary', '', false, array( 'id' => 'doaction' . ( 'top' === $which ? '' : '2' ) ) );
		echo "\n";
	}

	/**
	 * Display search box with custom classes
	 *
	 * Overrides parent to always show search box even when no items,
	 * allowing users to clear search when no results found.
	 *
	 * @param string $text     Button text.
	 * @param string $input_id Input ID.
	 * @return void
	 */
	public function search_box( $text, $input_id ) {
		// IMPORTANT: Always show search box, even when no items.
		// Parent class hides it when empty, but we want it visible so users can clear search.

		$input_id = $input_id . '-search-input';

		// Only preserve query params if nonce is present and valid.
		if ( isset( $_REQUEST['_altaudit82ai_nonce'] ) ) {
			if ( wp_verify_nonce( sanitize_text_field( wp_unslash( $_REQUEST['_altaudit82ai_nonce'] ) ), 'altaudit82ai_list_table' ) ) {
				// Nonce verified - safe to preserve query parameters.
				// Validate and preserve orderby parameter.
				if ( ! empty( $_REQUEST['orderby'] ) ) {
					$orderby         = sanitize_text_field( wp_unslash( $_REQUEST['orderby'] ) );
					$allowed_orderby = array( 'post_title', 'post_date', 'status', 'quality_score' );
					if ( in_array( $orderby, $allowed_orderby, true ) ) {
						echo '<input type="hidden" name="orderby" value="' . esc_attr( $orderby ) . '" />';
					}
				}
				// Validate and preserve order parameter.
				if ( ! empty( $_REQUEST['order'] ) ) {
					$order         = strtoupper( sanitize_text_field( wp_unslash( $_REQUEST['order'] ) ) );
					$allowed_order = array( 'ASC', 'DESC' );
					if ( in_array( $order, $allowed_order, true ) ) {
						echo '<input type="hidden" name="order" value="' . esc_attr( $order ) . '" />';
					}
				}
				// Preserve post_mime_type parameter (validated by WordPress).
				if ( ! empty( $_REQUEST['post_mime_type'] ) ) {
					echo '<input type="hidden" name="post_mime_type" value="' . esc_attr( sanitize_text_field( wp_unslash( $_REQUEST['post_mime_type'] ) ) ) . '" />';
				}
				// Validate and preserve detached parameter.
				if ( ! empty( $_REQUEST['detached'] ) ) {
					$detached = sanitize_text_field( wp_unslash( $_REQUEST['detached'] ) );
					if ( in_array( $detached, array( '0', '1' ), true ) ) {
						echo '<input type="hidden" name="detached" value="' . esc_attr( $detached ) . '" />';
					}
				}
			}
		}
		?>
		<!-- Search box -->
		<p class="search-box">
			<label class="screen-reader-text"
				for="<?php echo esc_attr( $input_id ); ?>"><?php echo esc_html( $text ); ?>:</label>
			<input type="search" id="<?php echo esc_attr( $input_id ); ?>" name="s" value="<?php _admin_search_query(); ?>"
				class="alt-audit-search-input"
				placeholder="<?php esc_attr_e( 'Search by filename or alt text...', 'alt-audit' ); ?>" />
			<?php submit_button( $text, 'altaudit82ai-button alt-audit-button-primary', '', false, array( 'id' => 'search-submit' ) ); ?>
		</p>
		<?php
	}

	/**
	 * Column checkbox
	 *
	 * @param array $item Item data.
	 * @return string Checkbox HTML.
	 */
	public function column_cb( $item ) {
		return sprintf(
			'<input type="checkbox" name="attachment_id[]" value="%s" />',
			esc_attr( $item['ID'] )
		);
	}

	/**
	 * Column thumbnail
	 *
	 * @param array $item Item data.
	 * @return string Thumbnail HTML.
	 */
	public function column_thumbnail( $item ) {
		$thumbnail = wp_get_attachment_image(
			$item['ID'],
			array( 80, 60 ),
			false,
			array(
				'class' => 'attachment-80x60',
			)
		);

		if ( empty( $thumbnail ) ) {
			return '<div class="attachment-preview"><span class="dashicons dashicons-format-image"></span></div>';
		}

		return wp_kses_post(
			sprintf(
				'<div class="attachment-preview">
				%s
			</div>',
				$thumbnail
			)
		);
	}

	/**
	 * Column filename
	 *
	 * @param array $item Item data.
	 * @return string Filename HTML with row actions.
	 */
	public function column_filename( $item ) {
		$actions = array();

		// Edit link.
		$actions['edit'] = sprintf(
			'<a href="%s">%s</a>',
			esc_url( $item['edit_url'] ),
			__( 'Edit', 'alt-audit' )
		);

		// Rule-based generation (free).
		$actions['generate_rule'] = sprintf(
			'<a href="#" class="alt-audit-generate-rule" data-attachment-id="%d">%s</a>',
			esc_attr( $item['ID'] ),
			__( '🆓 Generate (Free)', 'alt-audit' )
		);

		// AI generation (paid).
		$actions['generate_ai'] = sprintf(
			'<a href="#" class="alt-audit-generate-ai" data-attachment-id="%d">%s</a>',
			esc_attr( $item['ID'] ),
			__( '🤖 Generate (AI)', 'alt-audit' )
		);

		// View attachment.
		$actions['view'] = sprintf(
			'<a href="%s" target="_blank">%s</a>',
			esc_url( wp_get_attachment_url( $item['ID'] ) ),
			__( 'View', 'alt-audit' )
		);

		$filename = ! empty( $item['post_title'] ) ? $item['post_title'] : $item['filename'];

		return wp_kses_post(
			sprintf(
				'<strong><a href="%s">%s</a></strong>%s',
				esc_url( $item['edit_url'] ),
				esc_html( $filename ),
				$this->row_actions( $actions )
			)
		);
	}

	/**
	 * Column alt text
	 *
	 * @param array $item Item data.
	 * @return string Alt text HTML.
	 */
	public function column_alt_text( $item ) {
		// Display priority: Alt text takes precedence over decorative flag.
		if ( ! empty( $item['alt_text'] ) ) {
			$display_text = strlen( $item['alt_text'] ) > 100
				? substr( $item['alt_text'], 0, 100 ) . '...'
				: $item['alt_text'];

			$auto_generated_indicator = $item['is_auto_generated']
				? ' <span class="alt-audit-auto-generated" title="' . esc_attr__( 'Auto-generated', 'alt-audit' ) . '">🤖</span>'
				: '';

			return wp_kses_post(
				sprintf(
					'<div class="alt-audit-alt-text" data-attachment-id="%d" data-full-alt-text="%s">
					<span class="alt-text-display">%s</span>%s
					<button type="button" class="button-link edit-alt-text" title="%s">
						<span class="dashicons dashicons-edit"></span>
					</button>
				</div>',
					esc_attr( $item['ID'] ),
					esc_attr( $item['alt_text'] ),
					esc_html( $display_text ),
					$auto_generated_indicator,
					esc_attr__( 'Edit alt text', 'alt-audit' )
				)
			);
		}

		// Empty alt text - check if decorative.
		if ( $item['is_decorative'] ) {
			return wp_kses_post( '<em>' . __( '(Decorative)', 'alt-audit' ) . '</em>' );
		}

		// Empty alt text + not decorative = missing.
		return wp_kses_post( '<span class="alt-audit-missing">' . __( 'Missing', 'alt-audit' ) . '</span>' );
	}

	/**
	 * Column status
	 *
	 * @param array $item Item data.
	 * @return string Status HTML.
	 */
	public function column_status( $item ) {
		$status_labels = array(
			'missing'    => __( 'Missing', 'alt-audit' ),
			'weak'       => __( 'Weak', 'alt-audit' ),
			'good'       => __( 'Good', 'alt-audit' ),
			'excellent'  => __( 'Excellent', 'alt-audit' ),
			'decorative' => __( 'Decorative', 'alt-audit' ),
		);

		$status = $item['status'];
		$label  = $status_labels[ $status ] ?? ucfirst( $status );

		return wp_kses_post(
			sprintf(
				'<span class="alt-audit-status alt-audit-status-%s">%s</span>',
				esc_attr( $status ),
				esc_html( $label )
			)
		);
	}

	/**
	 * Column quality score
	 *
	 * @param array $item Item data.
	 * @return string Quality score HTML.
	 */
	public function column_quality_score( $item ) {
		$score  = $item['quality_score'];
		$status = $item['status'];

		if ( 'decorative' === $status ) {
			return wp_kses_post( '<span class="alt-audit-score-decorative">—</span>' );
		}

		$score_class = 'altaudit82ai-score';
		if ( $score >= 80 ) {
			$score_class .= ' score-excellent';
		} elseif ( $score >= 60 ) {
			$score_class .= ' score-good';
		} elseif ( $score > 0 ) {
			$score_class .= ' score-weak';
		} else {
			$score_class .= ' score-missing';
		}

		return wp_kses_post(
			sprintf(
				'<div class="%s">
				<span class="score-number">%d</span>
				<span class="score-bar">
					<span class="score-fill" style="width: %d%%"></span>
				</span>
			</div>',
				esc_attr( $score_class ),
				esc_html( $score ),
				esc_attr( $score )
			)
		);
	}

	/**
	 * Column explanation
	 *
	 * @param array $item Item data.
	 * @return string Explanation HTML.
	 */
	public function column_explanation( $item ) {
		$explanation = $item['explanation'] ?? '';

		if ( empty( $explanation ) ) {
			return '<span class="alt-audit-no-explanation">—</span>';
		}

		// Display full explanation text without truncation.
		return sprintf(
			'<span class="alt-audit-explanation">%s</span>',
			esc_html( $explanation )
		);
	}

	/**
	 * Default column handler
	 *
	 * @param array  $item        Item data.
	 * @param string $column_name Column name.
	 * @return string Column content.
	 */
	public function column_default( $item, $column_name ) {
		return isset( $item[ $column_name ] ) ? esc_html( $item[ $column_name ] ) : '';
	}

	/**
	 * Display when no items found
	 *
	 * @return void
	 */
	public function no_items() {
		esc_html_e( 'No images found.', 'alt-audit' );
	}

	/**
	 * Get single row HTML for AJAX updates
	 *
	 * Renders a single table row for seamless AJAX updates without full page refresh.
	 *
	 * @since  1.0.0
	 * @param  int $attachment_id Attachment ID.
	 * @return string|false Row HTML or false on failure.
	 */
	public function get_single_row_html( $attachment_id ) {
		// Sanitize the attachment ID.
		$attachment_id = absint( $attachment_id );

		// Get post using WordPress API.
		$post = get_post( $attachment_id );

		// Verify this is an image attachment.
		if ( ! $post || 'attachment' !== $post->post_type || ! wp_attachment_is_image( $attachment_id ) ) {
			return false;
		}

		// Convert post to image object.
		$image = $this->convert_post_to_image_object( $post );

		// Process the image item.
		$item = $this->process_image_item( $image );

		// Capture single row output.
		ob_start();
		$this->single_row( $item );
		$row_html = ob_get_clean();

		return $row_html;
	}

	/**
	 * Generate custom row classes and attributes
	 *
	 * @since  1.0.0
	 * @param  array $item Item data.
	 * @return void
	 */
	protected function single_row_attributes( $item ) {
		printf(
			'data-attachment-id="%s" id="post-%s"',
			esc_attr( $item['ID'] ),
			esc_attr( $item['ID'] )
		);
	}

	/**
	 * Overwrite parent single_row to add custom attributes
	 *
	 * @since  1.0.0
	 * @param  array $item Item data.
	 * @return void
	 */
	public function single_row( $item ) {
		echo '<tr ';
		$this->single_row_attributes( $item );
		echo '>';
		$this->single_row_columns( $item );
		echo '</tr>';
	}

	/**
	 * Get overall accessibility statistics
	 *
	 * @return array Statistics array.
	 */
	/**
	 * Get statistics for all images
	 *
	 * Uses centralized Statistics Service for consistency across all pages.
	 *
	 * @since  1.0.0
	 * @return array Image statistics.
	 */
	public function get_statistics() {
		return $this->statistics_service->get_audit_stats();
	}
}
