<?php
/**
 * Gutenberg Controller
 *
 * @package AltAudit
 * @since 1.0.0
 */

// Prevent direct access.
if ( ! defined( 'ABSPATH' ) ) {
	exit;
}

/**
 * Gutenberg Controller class
 *
 * Handles Gutenberg block editor integration including
 * block registration, editor enhancements, and alt text tools.
 *
 * @since 1.0.0
 */
class Altaudit82ai_Gutenberg_Controller {

	/**
	 * Settings service
	 *
	 * @var Altaudit82ai_Settings
	 */
	private $settings;

	/**
	 * API client service
	 *
	 * @var Altaudit82ai_Api_Client
	 */
	private $api_client;

	/**
	 * Quality checker service
	 *
	 * @var Altaudit82ai_Quality_Service
	 */
	private $quality_service;

	/**
	 * Constructor
	 *
	 * @param Altaudit82ai_Settings|null        $settings        Settings service.
	 * @param Altaudit82ai_Api_Client|null      $api_client      API client service.
	 * @param Altaudit82ai_Quality_Service|null $quality_service Quality service.
	 */
	public function __construct(
		?Altaudit82ai_Settings $settings = null,
		?Altaudit82ai_Api_Client $api_client = null,
		?Altaudit82ai_Quality_Service $quality_service = null
	) {
		$this->settings        = $settings;
		$this->api_client      = $api_client;
		$this->quality_service = $quality_service;

		$this->init_hooks();
	}

	/**
	 * Initialize hooks
	 *
	 * @return void
	 */
	private function init_hooks() {
		add_action( 'init', array( $this, 'register_blocks' ) );
		add_filter( 'block_editor_settings_all', array( $this, 'modify_editor_settings' ) );
		add_action( 'wp_ajax_altaudit82ai_block_generate', array( $this, 'ajax_block_generate' ) );
	}

	/**
	 * Enqueue block editor assets
	 *
	 * @return void
	 */
	public function enqueue_block_editor_assets() {
		// Only load if user can upload files.
		if ( ! current_user_can( 'upload_files' ) ) {
			return;
		}

		// Block editor CSS.
		wp_enqueue_style(
			'altaudit82ai-blocks',
			ALTAUDIT82AI_ASSETS_URL . 'css/blocks.css',
			array( 'wp-edit-blocks' ),
			ALTAUDIT82AI_VERSION
		);

		// Block editor JS.
		wp_enqueue_script(
			'altaudit82ai-blocks',
			ALTAUDIT82AI_ASSETS_URL . 'js/blocks.js',
			array(
				'wp-blocks',
				'wp-editor',
				'wp-components',
				'wp-element',
				'wp-compose',
				'wp-data',
				'wp-hooks',
				'wp-i18n',
				'wp-api-fetch',
			),
			ALTAUDIT82AI_VERSION,
			true
		);

		// Localize script.
		wp_localize_script(
			'altaudit82ai-blocks',
			'altaudit82aiBlocks',
			array(
				'ajaxUrl'          => admin_url( 'admin-ajax.php' ),
				'nonce'            => wp_create_nonce( 'altaudit82ai_nonce' ),
				'apiConfigured'    => $this->settings->is_api_configured(),
				'autoGenerate'     => $this->settings->get( 'auto_generate', false ),
				'qualityThreshold' => $this->settings->get( 'quality_threshold', 70 ),
				'maxLength'        => $this->settings->get( 'max_length', 125 ),
				'defaultStyle'     => $this->settings->get( 'style', 'descriptive' ),
				'styleOptions'     => $this->settings->get_style_options(),
				'strings'          => array(
					'generateAlt'     => __( 'Generate Alt Text', 'alt-audit' ),
					'generating'      => __( 'Generating...', 'alt-audit' ),
					'checkQuality'    => __( 'Check Quality', 'alt-audit' ),
					'qualityScore'    => __( 'Quality Score', 'alt-audit' ),
					'excellent'       => __( 'Excellent', 'alt-audit' ),
					'good'            => __( 'Good', 'alt-audit' ),
					'weak'            => __( 'Weak', 'alt-audit' ),
					'missing'         => __( 'Missing', 'alt-audit' ),
					'decorative'      => __( 'Decorative', 'alt-audit' ),
					'improvementTips' => __( 'Improvement Tips', 'alt-audit' ),
					'error'           => __( 'Error generating alt text', 'alt-audit' ),
					'apiKeyRequired'  => __( 'API key required', 'alt-audit' ),
					'invalidImage'    => __( 'Invalid image', 'alt-audit' ),
				),
			)
		);

		// Set script translations.
		wp_set_script_translations( 'altaudit82ai-blocks', 'alt-audit' );
	}

	/**
	 * Register custom blocks
	 *
	 * @return void
	 */
	public function register_blocks() {
		// Register Alt Audit Inspector block (for image blocks).
		register_block_type(
			'alt-audit/inspector',
			array(
				'editor_script'   => 'altaudit82ai-blocks',
				'editor_style'    => 'altaudit82ai-blocks',
				'render_callback' => '__return_empty_string', // This is an editor-only block.
			)
		);

		// Register Alt Audit Quality Indicator block.
		register_block_type(
			'alt-audit/quality-indicator',
			array(
				'editor_script'   => 'altaudit82ai-blocks',
				'editor_style'    => 'altaudit82ai-blocks',
				'render_callback' => '__return_empty_string', // This is an editor-only block.
			)
		);
	}

	/**
	 * Modify block editor settings
	 *
	 * @param array $settings Current editor settings.
	 * @return array Modified settings.
	 */
	public function modify_editor_settings( $settings ) {
		// Add Alt Audit settings to editor.
		$settings['altAudit'] = array(
			'apiConfigured'    => $this->settings->is_api_configured(),
			'autoGenerate'     => $this->settings->get( 'auto_generate', false ),
			'qualityThreshold' => $this->settings->get( 'quality_threshold', 70 ),
			'contextAware'     => $this->settings->get( 'context_aware', true ),
		);

		return $settings;
	}

	/**
	 * AJAX handler for block-specific alt text generation
	 *
	 * @return void
	 */
	public function ajax_block_generate() {
		// Verify nonce.
		if ( ! check_ajax_referer( 'altaudit82ai_nonce', 'nonce', false ) ) {
			wp_send_json_error( __( 'Security check failed.', 'alt-audit' ) );
		}

		// Check user permissions.
		if ( ! current_user_can( 'upload_files' ) ) {
			wp_send_json_error( __( 'Insufficient permissions.', 'alt-audit' ) );
		}

		$image_id      = isset( $_POST['image_id'] ) ? intval( wp_unslash( $_POST['image_id'] ) ) : 0;
		$context       = isset( $_POST['context'] ) ? sanitize_textarea_field( wp_unslash( $_POST['context'] ) ) : '';
		$default_style = $this->settings->get( 'style', 'descriptive' );

		// Validate style against allowed values.
		$allowed_styles = array( 'descriptive', 'technical', 'creative', 'brief' );
		$style          = isset( $_POST['style'] ) ? sanitize_text_field( wp_unslash( $_POST['style'] ) ) : $default_style;
		$style          = in_array( $style, $allowed_styles, true ) ? $style : $default_style;

		if ( ! $image_id ) {
			wp_send_json_error( __( 'Invalid image ID.', 'alt-audit' ) );
		}

		// Check if it's actually an image.
		if ( ! wp_attachment_is_image( $image_id ) ) {
			wp_send_json_error( __( 'Attachment is not an image.', 'alt-audit' ) );
		}

		// Get surrounding content for context if context-aware is enabled.
		if ( $this->settings->get( 'context_aware', true ) && empty( $context ) ) {
			$post_content = isset( $_POST['post_content'] ) ? wp_kses_post( wp_unslash( $_POST['post_content'] ) ) : '';
			$context      = $this->extract_content_context( $post_content, $image_id );
		}

		$options = array(
			'style'   => $style,
			'context' => $context,
		);

		$result = $this->api_client->generate_alt_text( $image_id, $options );

		if ( ! $result['success'] ) {
			wp_send_json_error( $result['message'] );
		}

		// Check quality.
		$quality = null;
		if ( $this->quality_service ) {
			$quality = $this->quality_service->assess_quality( $result['alt_text'] );
		}

		wp_send_json_success(
			array(
				'alt_text'      => $result['alt_text'],
				'quality_score' => $result['quality_score'] ?? 0,
				'quality'       => $quality,
				'suggestions'   => $this->get_improvement_suggestions( $result['alt_text'] ),
			)
		);
	}

	/**
	 * Extract context from post content
	 *
	 * @param string $content Post content.
	 * @param int    $image_id Image ID.
	 * @return string Extracted context.
	 */
	private function extract_content_context( $content, $image_id ) {
		if ( empty( $content ) ) {
			return '';
		}

		// Parse blocks to find image context.
		$blocks  = parse_blocks( $content );
		$context = $this->find_image_context_in_blocks( $blocks, $image_id );

		return $context;
	}

	/**
	 * Find image context in blocks
	 *
	 * @param array $blocks Parsed blocks.
	 * @param int   $image_id Image ID.
	 * @return string Context text.
	 */
	private function find_image_context_in_blocks( $blocks, $image_id ) {
		foreach ( $blocks as $index => $block ) {
			// Check if this block contains our image.
			if ( $this->block_contains_image( $block, $image_id ) ) {
				// Get context from surrounding blocks.
				$context_blocks = array();

				// Previous block.
				if ( $index > 0 ) {
					$context_blocks[] = $blocks[ $index - 1 ];
				}

				// Current block.
				$context_blocks[] = $block;

				// Next block.
				if ( $index < count( $blocks ) - 1 ) {
					$context_blocks[] = $blocks[ $index + 1 ];
				}

				return $this->extract_text_from_blocks( $context_blocks );
			}

			// Check inner blocks.
			if ( ! empty( $block['innerBlocks'] ) ) {
				$context = $this->find_image_context_in_blocks( $block['innerBlocks'], $image_id );
				if ( ! empty( $context ) ) {
					return $context;
				}
			}
		}

		return '';
	}

	/**
	 * Check if block contains specific image
	 *
	 * @param array $block Block data.
	 * @param int   $image_id Image ID.
	 * @return bool True if block contains image.
	 */
	private function block_contains_image( $block, $image_id ) {
		// Check image block.
		if ( 'core/image' === $block['blockName'] ) {
			return isset( $block['attrs']['id'] ) && (int) $block['attrs']['id'] === $image_id;
		}

		// Check gallery block.
		if ( 'core/gallery' === $block['blockName'] ) {
			$images = $block['attrs']['images'] ?? array();
			foreach ( $images as $image ) {
				if ( isset( $image['id'] ) && (int) $image['id'] === $image_id ) {
					return true;
				}
			}
		}

		// Check media & text block.
		if ( 'core/media-text' === $block['blockName'] ) {
			return isset( $block['attrs']['mediaId'] ) && (int) $block['attrs']['mediaId'] === $image_id;
		}

		// Check cover block.
		if ( 'core/cover' === $block['blockName'] ) {
			return isset( $block['attrs']['id'] ) && (int) $block['attrs']['id'] === $image_id;
		}

		return false;
	}

	/**
	 * Extract text content from blocks
	 *
	 * @param array $blocks Array of blocks.
	 * @return string Extracted text.
	 */
	private function extract_text_from_blocks( $blocks ) {
		$text = '';

		foreach ( $blocks as $block ) {
			// Extract text based on block type.
			switch ( $block['blockName'] ) {
				case 'core/paragraph':
				case 'core/heading':
				case 'core/quote':
					$text .= wp_strip_all_tags( $block['innerHTML'] ?? '' ) . ' ';
					break;

				case 'core/list':
					$text .= wp_strip_all_tags( $block['innerHTML'] ?? '' ) . ' ';
					break;

				case 'core/image':
					// Get caption.
					if ( ! empty( $block['attrs']['caption'] ) ) {
						$text .= wp_strip_all_tags( $block['attrs']['caption'] ) . ' ';
					}
					break;
			}

			// Process inner blocks.
			if ( ! empty( $block['innerBlocks'] ) ) {
				$text .= $this->extract_text_from_blocks( $block['innerBlocks'] );
			}
		}

		// Clean up text.
		$text = preg_replace( '/\s+/', ' ', $text );
		$text = trim( $text );

		// Limit context length.
		if ( strlen( $text ) > 500 ) {
			$text = substr( $text, 0, 500 ) . '...';
		}

		return $text;
	}

	/**
	 * Get improvement suggestions for alt text
	 *
	 * @param string $alt_text Alt text to analyze.
	 * @return array Suggestions.
	 */
	private function get_improvement_suggestions( $alt_text ) {
		$suggestions = array();

		if ( empty( $alt_text ) ) {
			return $suggestions;
		}

		// Length suggestions.
		$length = strlen( $alt_text );
		if ( $length > 125 ) {
			$suggestions[] = __( 'Consider shortening the alt text to under 125 characters.', 'alt-audit' );
		} elseif ( $length < 10 ) {
			$suggestions[] = __( 'Consider adding more descriptive details.', 'alt-audit' );
		}

		// Common issues.
		if ( preg_match( '/^(image|picture|photo|graphic) (of|showing)/i', $alt_text ) ) {
			$suggestions[] = __( 'Avoid starting with "image of" or "picture of".', 'alt-audit' );
		}

		if ( preg_match( '/\.(jpg|jpeg|png|gif|webp)$/i', $alt_text ) ) {
			$suggestions[] = __( 'Remove file extensions from alt text.', 'alt-audit' );
		}

		// Quality suggestions from quality checker.
		if ( $this->quality_service ) {
			$quality = $this->quality_service->assess_quality( $alt_text );
			if ( ! empty( $quality['suggestions'] ) ) {
				$suggestions = array_merge( $suggestions, $quality['suggestions'] );
			}
		}

		return $suggestions;
	}

	/**
	 * Check if Gutenberg is active
	 *
	 * @return bool True if Gutenberg is active.
	 */
	private function is_gutenberg_active() {
		return function_exists( 'register_block_type' );
	}
}
