<?php
/**
 * API Controller
 *
 * @package AltAudit
 * @since 1.0.0
 */

// Prevent direct access.
if ( ! defined( 'ABSPATH' ) ) {
	exit;
}

/**
 * API Controller class
 *
 * Handles AJAX requests and API interactions between
 * the frontend/admin and the Alt Audit service.
 *
 * @since 1.0.0
 */
class Altaudit82ai_Api_Controller {


	/**
	 * API client service
	 *
	 * @var Altaudit82ai_Api_Client
	 */
	private $api_client;

	/**
	 * Settings service
	 *
	 * @var Altaudit82ai_Settings
	 */
	private $settings;

	/**
	 * Quality checker service
	 *
	 * @var Altaudit82ai_Quality_Service
	 */
	private $quality_service;

	/**
	 * Template service
	 *
	 * @var Altaudit82ai_Template_Service
	 */
	private $template_service;

	/**
	 * Constructor
	 *
	 * @param Altaudit82ai_Api_Client       $api_client       API client service.
	 * @param Altaudit82ai_Settings         $settings         Settings service.
	 * @param Altaudit82ai_Quality_Service  $quality_service  Quality service.
	 * @param Altaudit82ai_Template_Service $template_service Template service.
	 */
	public function __construct(
		?Altaudit82ai_Api_Client $api_client = null,
		?Altaudit82ai_Settings $settings = null,
		?Altaudit82ai_Quality_Service $quality_service = null,
		?Altaudit82ai_Template_Service $template_service = null
	) {
		$this->api_client       = $api_client;
		$this->settings         = $settings;
		$this->quality_service  = $quality_service;
		$this->template_service = $template_service;

		$this->init_hooks();
	}

	/**
	 * Initialize hooks
	 *
	 * @return void
	 */
	private function init_hooks() {
		// AJAX hooks for logged-in users.
		add_action( 'wp_ajax_altaudit82ai_generate', array( $this, 'ajax_generate_alt_text' ) );
		add_action( 'wp_ajax_altaudit82ai_generate_rule_based', array( $this, 'ajax_generate_rule_based' ) );
		add_action( 'wp_ajax_altaudit82ai_test_connection', array( $this, 'ajax_test_connection' ) );
		add_action( 'wp_ajax_altaudit82ai_bulk_process', array( $this, 'ajax_bulk_process' ) );
		add_action( 'wp_ajax_altaudit82ai_save_alt_text', array( $this, 'ajax_save_alt_text' ) );
		add_action( 'wp_ajax_altaudit82ai_assess_quality', array( $this, 'ajax_assess_quality' ) );
		add_action( 'wp_ajax_altaudit82ai_get_user_info', array( $this, 'ajax_get_user_info' ) );
	}

	/**
	 * AJAX handler for generating alt text
	 *
	 * @return void
	 */
	public function ajax_generate_alt_text() {
		// Verify nonce.
		if ( ! check_ajax_referer( 'altaudit82ai_nonce', 'nonce', false ) ) {
			wp_send_json_error( __( 'Security check failed.', 'alt-audit' ) );
		}

		// Check user permissions.
		if ( ! current_user_can( 'upload_files' ) ) {
			wp_send_json_error( __( 'Insufficient permissions.', 'alt-audit' ) );
		}

		$attachment_id = isset( $_POST['attachment_id'] ) ? intval( wp_unslash( $_POST['attachment_id'] ) ) : 0;
		if ( ! $attachment_id ) {
			wp_send_json_error( __( 'Invalid attachment ID.', 'alt-audit' ) );
		}

		// Get and validate generation options.
		$allowed_styles = array( 'descriptive', 'technical', 'creative', 'brief' );
		$style          = isset( $_POST['style'] ) ? sanitize_text_field( wp_unslash( $_POST['style'] ) ) : 'descriptive';
		$style          = in_array( $style, $allowed_styles, true ) ? $style : 'descriptive';

		$options = array(
			'style'   => $style,
			'context' => isset( $_POST['context'] ) ? sanitize_textarea_field( wp_unslash( $_POST['context'] ) ) : '',
		);

		// Generate alt text.
		$result = $this->api_client->generate_alt_text( $attachment_id, $options );

		if ( ! $result['success'] ) {
			wp_send_json_error( $result['message'] );
		}

		// Save the generated alt text.
		$alt_text = $result['alt_text'];
		update_post_meta( $attachment_id, '_wp_attachment_image_alt', $alt_text );

		// Logging removed - not needed.

		// Check quality.
		$quality_result = null;
		$status         = 'unknown';
		if ( $this->quality_service ) {
			$quality_result = $this->quality_service->assess_quality( $alt_text );
			$status         = $quality_result['status'] ?? 'unknown';
		}

		wp_send_json_success(
			array(
				'alt_text'      => $alt_text,
				'quality_score' => $quality_result['accessibility_score'] ?? 0,
				'status'        => $status,
				'quality'       => array(
					'score'  => $quality_result['accessibility_score'] ?? 0,
					'status' => $status,
				),
				'message'       => __( 'Alt text generated successfully.', 'alt-audit' ),
			)
		);
	}

	/**
	 * AJAX handler for rule-based alt text generation
	 *
	 * @return void
	 */
	public function ajax_generate_rule_based() {
		// Verify nonce.
		if ( ! check_ajax_referer( 'altaudit82ai_nonce', 'nonce', false ) ) {
			wp_send_json_error( __( 'Security check failed.', 'alt-audit' ) );
		}

		// Check user permissions.
		if ( ! current_user_can( 'upload_files' ) ) {
			wp_send_json_error( __( 'Insufficient permissions.', 'alt-audit' ) );
		}

		$attachment_id = isset( $_POST['attachment_id'] ) ? intval( wp_unslash( $_POST['attachment_id'] ) ) : 0;
		if ( ! $attachment_id ) {
			wp_send_json_error( __( 'Invalid attachment ID.', 'alt-audit' ) );
		}

		// Get template service.
		if ( ! $this->template_service ) {
			wp_send_json_error( __( 'Template service not available.', 'alt-audit' ) );
		}

		// Generate alt text using template-based approach.
		$alt_text = $this->template_service->generate_from_template( $attachment_id );

		if ( empty( $alt_text ) ) {
			wp_send_json_error( __( 'Failed to generate alt text using rule-based method.', 'alt-audit' ) );
		}

		// Save the generated alt text.
		update_post_meta( $attachment_id, '_wp_attachment_image_alt', $alt_text );

		// Assess quality of generated text.
		$quality_result = null;
		$status         = 'unknown';
		if ( $this->quality_service ) {
			$quality_result = $this->quality_service->assess_quality( $alt_text );
			$status         = $quality_result['status'] ?? 'unknown';
		}

		wp_send_json_success(
			array(
				'alt_text'      => $alt_text,
				'quality_score' => $quality_result['accessibility_score'] ?? 0,
				'status'        => $status,
				'quality'       => array(
					'score'  => $quality_result['accessibility_score'] ?? 0,
					'status' => $status,
				),
				'method'        => 'rule_based',
				'message'       => __( 'Alt text generated successfully using rule-based method.', 'alt-audit' ),
			)
		);
	}

	/**
	 * AJAX handler for testing API connection
	 *
	 * @return void
	 */
	public function ajax_test_connection() {
		// Verify nonce.
		if ( ! check_ajax_referer( 'altaudit82ai_nonce', 'nonce', false ) ) {
			wp_send_json_error( __( 'Security check failed.', 'alt-audit' ) );
		}

		// Check user permissions.
		if ( ! current_user_can( 'manage_options' ) ) {
			wp_send_json_error( __( 'Insufficient permissions.', 'alt-audit' ) );
		}

		$result = $this->api_client->test_connection();

		if ( $result['success'] ) {
			wp_send_json_success( $result );
		} else {
			wp_send_json_error( $result['message'] );
		}
	}

	/**
	 * AJAX handler for bulk processing
	 *
	 * @return void
	 */
	public function ajax_bulk_process() {
		// Verify nonce.
		if ( ! check_ajax_referer( 'altaudit82ai_nonce', 'nonce', false ) ) {
			wp_send_json_error( __( 'Security check failed.', 'alt-audit' ) );
		}

		// Check user permissions.
		if ( ! current_user_can( 'upload_files' ) ) {
			wp_send_json_error( __( 'Insufficient permissions.', 'alt-audit' ) );
		}

		$bulk_action = sanitize_text_field( wp_unslash( $_POST['bulk_action'] ?? '' ) );
		// phpcs:ignore WordPress.Security.ValidatedSanitizedInput.InputNotSanitized -- Sanitized via intval in array_map.
		$attachment_ids = isset( $_POST['attachment_ids'] ) ? array_map( 'intval', wp_unslash( $_POST['attachment_ids'] ) ) : array();

		if ( empty( $attachment_ids ) ) {
			wp_send_json_error( __( 'No images selected.', 'alt-audit' ) );
		}

		if ( empty( $bulk_action ) ) {
			wp_send_json_error( __( 'No action specified.', 'alt-audit' ) );
		}

		// Extract optional parameters here (nonce already verified above).
		$allowed_styles = array( 'descriptive', 'technical', 'creative', 'brief' );
		$style          = isset( $_POST['style'] ) ? sanitize_text_field( wp_unslash( $_POST['style'] ) ) : 'descriptive';
		$style          = in_array( $style, $allowed_styles, true ) ? $style : 'descriptive';
		$template       = isset( $_POST['template'] ) ? sanitize_text_field( wp_unslash( $_POST['template'] ) ) : '';

		// Handle different bulk actions.
		switch ( $bulk_action ) {
			case 'generate_rule_based':
				$this->process_bulk_generate_rule_based( $attachment_ids, $template );
				break;

			case 'generate_ai':
			case 'generate_alt_text':
				$this->process_bulk_generate_alt_text( $attachment_ids, $style );
				break;

			case 'mark_decorative':
				$this->process_bulk_mark_decorative( $attachment_ids );
				break;

			case 'delete':
				$this->process_bulk_delete( $attachment_ids );
				break;

			default:
				wp_send_json_error( __( 'Invalid bulk action.', 'alt-audit' ) );
		}
	}

	/**
	 * Process bulk generate alt text action
	 *
	 * @param array  $attachment_ids Array of attachment IDs.
	 * @param string $style          Alt text generation style.
	 * @return void
	 */
	private function process_bulk_generate_alt_text( $attachment_ids, $style = 'descriptive' ) {
		$options = array(
			'style' => $style,
		);

		$batch_size = $this->settings->get( 'batch_size', 5 );
		$batch      = array_slice( $attachment_ids, 0, $batch_size );

		$results = array();
		$errors  = array();

		foreach ( $batch as $attachment_id ) {
			// Skip if already has alt text and skip_existing is enabled.
			if ( $this->settings->get( 'skip_existing', true ) ) {
				$existing_alt = get_post_meta( $attachment_id, '_wp_attachment_image_alt', true );
				if ( ! empty( $existing_alt ) ) {
					continue;
				}
			}

			$result = $this->api_client->generate_alt_text( $attachment_id, $options );

			if ( $result['success'] ) {
				// Save alt text.
				update_post_meta( $attachment_id, '_wp_attachment_image_alt', $result['alt_text'] );

				// Logging removed - not needed.

				$results[] = array(
					'attachment_id' => $attachment_id,
					'alt_text'      => $result['alt_text'],
					'quality_score' => $result['quality_score'] ?? 0,
				);
			} else {
				$errors[] = array(
					'attachment_id' => $attachment_id,
					'message'       => $result['message'],
				);
			}
		}

		wp_send_json_success(
			array(
				'processed'     => count( $results ),
				'errors'        => count( $errors ),
				'results'       => $results,
				'error_details' => $errors,
				'remaining'     => count( $attachment_ids ) - $batch_size,
			)
		);
	}

	/**
	 * Process bulk rule-based generation action
	 *
	 * @param array  $attachment_ids Array of attachment IDs.
	 * @param string $template       Template for rule-based generation.
	 * @return void
	 */
	private function process_bulk_generate_rule_based( $attachment_ids, $template = '' ) {
		// Get or create rule generator.
		$rule_generator = $this->get_rule_generator();

		if ( ! $rule_generator ) {
			wp_send_json_error( __( 'Rule-based generator not available.', 'alt-audit' ) );
			return;
		}

		$results = $rule_generator->bulk_generate( $attachment_ids, $template );

		// Logging removed - not needed.

		// Format response to match expected structure.
		wp_send_json_success(
			array(
				'success' => $results['success'],
				'errors'  => $results['errors'],
				'data'    => array(
					'success' => $results['success'],
					'errors'  => $results['errors'],
					'details' => $results['details'],
				),
				'method'  => 'rule_based',
				'message' => sprintf(
					/* translators: %1$d: number of successful generations, %2$d: number of errors */
					__( 'Rule-based generation completed. Success: %1$d, Errors: %2$d', 'alt-audit' ),
					$results['success'],
					$results['errors']
				),
			)
		);
	}

	/**
	 * Get rule generator instance
	 *
	 * @return Altaudit82ai_Rule_Generator|null
	 */
	private function get_rule_generator() {
		// Try to get from container if available.
		if ( method_exists( $this, 'container' ) && $this->container ) {
			return $this->container->get( 'rule_generator' );
		}

		// Create new instance.
		require_once ALTAUDIT82AI_PLUGIN_DIR . 'includes/Services/class-alt-audit-rule-generator.php';
		return new Altaudit82ai_Rule_Generator( $this->settings );
	}

	/**
	 * Process bulk mark decorative action
	 *
	 * @param array $attachment_ids Array of attachment IDs.
	 * @return void
	 */
	private function process_bulk_mark_decorative( $attachment_ids ) {
		$processed = 0;
		$errors    = 0;

		foreach ( $attachment_ids as $attachment_id ) {
			// Mark as decorative.
			$result = update_post_meta( $attachment_id, '_altaudit82ai_decorative', '1' );

			if ( false !== $result ) {
				// Clear existing alt text for decorative images.
				update_post_meta( $attachment_id, '_wp_attachment_image_alt', '' );

				// Logging removed - not needed.

				++$processed;
			} else {
				++$errors;
			}
		}

		wp_send_json_success(
			array(
				'processed' => $processed,
				'errors'    => $errors,
				'remaining' => 0,
			)
		);
	}

	/**
	 * Process bulk delete action
	 *
	 * @param array $attachment_ids Array of attachment IDs.
	 * @return void
	 */
	private function process_bulk_delete( $attachment_ids ) {
		// Check delete permissions.
		if ( ! current_user_can( 'delete_posts' ) ) {
			wp_send_json_error( __( 'Insufficient permissions to delete images.', 'alt-audit' ) );
		}

		$processed = 0;
		$errors    = 0;

		foreach ( $attachment_ids as $attachment_id ) {
			// Logging removed - not needed.

			$result = wp_delete_attachment( $attachment_id, true );

			if ( false !== $result ) {
				++$processed;
			} else {
				++$errors;
			}
		}

		wp_send_json_success(
			array(
				'processed' => $processed,
				'errors'    => $errors,
				'remaining' => 0,
			)
		);
	}

	/**
	 * AJAX handler for saving alt text
	 *
	 * @return void
	 */
	public function ajax_save_alt_text() {
		// Verify nonce.
		if ( ! check_ajax_referer( 'altaudit82ai_nonce', 'nonce', false ) ) {
			wp_send_json_error( __( 'Security check failed.', 'alt-audit' ) );
		}

		// Check user permissions.
		if ( ! current_user_can( 'upload_files' ) ) {
			wp_send_json_error( __( 'Insufficient permissions.', 'alt-audit' ) );
		}

		$attachment_id = isset( $_POST['attachment_id'] ) ? intval( wp_unslash( $_POST['attachment_id'] ) ) : 0;
		$alt_text      = isset( $_POST['alt_text'] ) ? sanitize_textarea_field( wp_unslash( $_POST['alt_text'] ) ) : '';

		if ( ! $attachment_id ) {
			wp_send_json_error( __( 'Invalid attachment ID.', 'alt-audit' ) );
		}

		// Update alt text.
		update_post_meta( $attachment_id, '_wp_attachment_image_alt', $alt_text );

		// Check quality.
		$quality = null;
		if ( $this->quality_service ) {
			$quality = $this->quality_service->assess_quality( $alt_text );
		}

		// Logging removed - not needed.

		wp_send_json_success(
			array(
				'alt_text' => $alt_text,
				'quality'  => $quality,
				'message'  => __( 'Alt text saved successfully.', 'alt-audit' ),
			)
		);
	}

	/**
	 * AJAX handler for checking alt text quality
	 *
	 * @return void
	 */
	public function ajax_assess_quality() {
		// Verify nonce.
		if ( ! check_ajax_referer( 'altaudit82ai_nonce', 'nonce', false ) ) {
			wp_send_json_error( __( 'Security check failed.', 'alt-audit' ) );
		}

		// Check user permissions.
		if ( ! current_user_can( 'upload_files' ) ) {
			wp_send_json_error( __( 'Insufficient permissions.', 'alt-audit' ) );
		}

		$alt_text = isset( $_POST['alt_text'] ) ? sanitize_textarea_field( wp_unslash( $_POST['alt_text'] ) ) : '';

		if ( empty( $alt_text ) ) {
			wp_send_json_error( __( 'Alt text is required.', 'alt-audit' ) );
		}

		if ( ! $this->quality_service ) {
			wp_send_json_error( __( 'Quality checker not available.', 'alt-audit' ) );
		}

		$quality = $this->quality_service->assess_quality( $alt_text );

		wp_send_json_success( $quality );
	}

	/**
	 * AJAX handler for getting user account info
	 *
	 * @return void
	 */
	public function ajax_get_user_info() {
		// Verify nonce.
		if ( ! check_ajax_referer( 'altaudit82ai_nonce', 'nonce', false ) ) {
			wp_send_json_error( __( 'Security check failed.', 'alt-audit' ) );
		}

		// Check user permissions.
		if ( ! current_user_can( 'manage_options' ) ) {
			wp_send_json_error( __( 'Insufficient permissions.', 'alt-audit' ) );
		}

		$result = $this->api_client->get_user_info();

		if ( $result['success'] ) {
			wp_send_json_success( $result['data'] );
		} else {
			wp_send_json_error( $result['message'] );
		}
	}

	/**
	 * Rate limit check
	 *
	 * @param string $action Action being performed.
	 * @return bool True if within rate limit.
	 */
	private function check_rate_limit( $action = 'default' ) {
		$rate_limit = $this->settings->get( 'rate_limit', 10 );
		$cache_key  = 'altaudit82ai_rate_limit_' . get_current_user_id() . '_' . $action;

		$current_count = get_transient( $cache_key );

		if ( false === $current_count ) {
			set_transient( $cache_key, 1, MINUTE_IN_SECONDS );
			return true;
		}

		if ( $current_count >= $rate_limit ) {
			return false;
		}

		set_transient( $cache_key, $current_count + 1, MINUTE_IN_SECONDS );
		return true;
	}

	/**
	 * Validate attachment ID
	 *
	 * @param int $attachment_id Attachment ID.
	 * @return bool True if valid.
	 */
	private function validate_attachment_id( $attachment_id ) {
		$post = get_post( $attachment_id );
		return $post && 'attachment' === $post->post_type && wp_attachment_is_image( $attachment_id );
	}
}
