<?php
/**
 * Admin Controller
 *
 * @package AltAudit
 * @since 1.0.0
 */

// Prevent direct access.
if ( ! defined( 'ABSPATH' ) ) {
	exit;
}

/**
 * Admin Controller class
 *
 * Handles all admin-related functionality including menu creation,
 * settings pages, and admin-specific features.
 *
 * @since 1.0.0
 */
class Altaudit82ai_Admin_Controller {













	/**
	 * Settings service
	 *
	 * @var Altaudit82ai_Settings
	 */
	private $settings;

	/**
	 * API client service
	 *
	 * @var Altaudit82ai_Api_Client
	 */
	private $api_client;

	/**
	 * Quality service
	 *
	 * @var Altaudit82ai_Quality_Service
	 */
	private $quality_service;

	/**
	 * Statistics service
	 *
	 * @var Altaudit82ai_Statistics_Service
	 */
	private $statistics_service;

	/**
	 * Images list table
	 *
	 * @var Altaudit82ai_Images_List_Table
	 */
	private $images_list_table;

	/**
	 * Constructor
	 *
	 * @param Altaudit82ai_Settings           $settings           Settings service.
	 * @param Altaudit82ai_Api_Client         $api_client         API client service.
	 * @param Altaudit82ai_Quality_Service    $quality_service    Quality service.
	 * @param Altaudit82ai_Statistics_Service $statistics_service Statistics service.
	 */
	public function __construct(
		?Altaudit82ai_Settings $settings = null,
		?Altaudit82ai_Api_Client $api_client = null,
		?Altaudit82ai_Quality_Service $quality_service = null,
		?Altaudit82ai_Statistics_Service $statistics_service = null
	) {
		$this->settings           = $settings;
		$this->api_client         = $api_client;
		$this->quality_service    = $quality_service;
		$this->statistics_service = $statistics_service ?? new Altaudit82ai_Statistics_Service();

		$this->init_hooks();
	}

	/**
	 * Initialize hooks
	 *
	 * @return void
	 */
	private function init_hooks() {
		add_action( 'admin_notices', array( $this, 'admin_notices' ) );
		add_action( 'admin_post_altaudit82ai_save_settings', array( $this, 'save_settings' ) );
		add_action( 'admin_post_altaudit82ai_test_api', array( $this, 'test_api_connection' ) );
		add_action( 'admin_post_altaudit82ai_bulk_scan', array( $this, 'bulk_scan' ) );

		// AJAX handlers for audit page.
		add_action( 'wp_ajax_altaudit82ai_inline_edit_alt_text', array( $this, 'ajax_inline_edit_alt_text' ) );
		add_action( 'wp_ajax_altaudit82ai_bulk_action', array( $this, 'ajax_bulk_action' ) );
		add_action( 'wp_ajax_altaudit82ai_refresh_quality_score', array( $this, 'ajax_refresh_quality_score' ) );
		add_action( 'wp_ajax_altaudit82ai_get_statistics', array( $this, 'ajax_get_statistics' ) );
		add_action( 'wp_ajax_altaudit82ai_single_bulk_action', array( $this, 'ajax_single_bulk_action' ) );
		add_action( 'wp_ajax_altaudit82ai_refresh_table', array( $this, 'ajax_refresh_table' ) );
		add_action( 'wp_ajax_altaudit82ai_scan_all_images', array( $this, 'ajax_scan_all_images' ) );
		add_action( 'wp_ajax_altaudit82ai_dismiss_notice', array( $this, 'ajax_dismiss_notice' ) );

		// AJAX handlers for bulk tools page.
		add_action( 'wp_ajax_altaudit82ai_count_images', array( $this, 'ajax_count_images' ) );
		add_action( 'wp_ajax_altaudit82ai_bulk_generate_rule', array( $this, 'ajax_bulk_generate_rule' ) );
		add_action( 'wp_ajax_altaudit82ai_bulk_generate_ai', array( $this, 'ajax_bulk_generate_ai' ) );
		add_action( 'wp_ajax_altaudit82ai_get_user_credits', array( $this, 'ajax_get_user_credits' ) );
		add_action( 'wp_ajax_altaudit82ai_get_queue_status', array( $this, 'ajax_get_queue_status' ) );
		add_action( 'wp_ajax_altaudit82ai_export_data', array( $this, 'ajax_export_data' ) );
	}

	/**
	 * Initialize admin area
	 *
	 * @return void
	 */
	public function admin_init() {
		// Register settings.
		$this->register_settings();

		// Add meta boxes.
		$this->add_meta_boxes();
	}

	/**
	 * Add admin menu
	 *
	 * @return void
	 */
	public function admin_menu() {
		// Main menu page.
		add_menu_page(
			__( 'Accessibility Audit Dashboard', 'alt-audit' ),
			__( 'Alt Audit', 'alt-audit' ),
			'manage_options',
			'alt-audit',
			array( $this, 'dashboard_page' ),
			'dashicons-visibility',
			30
		);

		// Dashboard submenu.
		add_submenu_page(
			'alt-audit',
			__( 'Accessibility Audit Dashboard', 'alt-audit' ),
			__( 'Dashboard', 'alt-audit' ),
			'manage_options',
			'alt-audit',
			array( $this, 'dashboard_page' )
		);

		// Settings submenu.
		add_submenu_page(
			'alt-audit',
			__( 'Settings', 'alt-audit' ),
			__( 'Settings', 'alt-audit' ),
			'manage_options',
			'alt-audit-settings',
			array( $this, 'settings_page' )
		);

		// Audit submenu.
		add_submenu_page(
			'alt-audit',
			__( 'Accessibility Audit', 'alt-audit' ),
			__( 'Accessibility Audit', 'alt-audit' ),
			'upload_files',
			'alt-audit-audit',
			array( $this, 'audit_page' )
		);

		// Bulk Tools submenu.
		add_submenu_page(
			'alt-audit',
			__( 'Bulk Tools', 'alt-audit' ),
			__( 'Bulk Tools', 'alt-audit' ),
			'manage_options',
			'alt-audit-bulk',
			array( $this, 'bulk_tools_page' )
		);

		// Help submenu.
		add_submenu_page(
			'alt-audit',
			__( 'Help & Support', 'alt-audit' ),
			__( 'Help', 'alt-audit' ),
			'manage_options',
			'alt-audit-help',
			array( $this, 'help_page' )
		);
	}

	/**
	 * Enqueue admin scripts and styles
	 *
	 * @param string $hook_suffix Current admin page.
	 * @return void
	 */
	public function enqueue_scripts( $hook_suffix ) {
		// Only load on Alt Audit pages.
		if ( false === strpos( $hook_suffix, 'alt-audit' ) ) {
			return;
		}

		// Admin CSS.
		wp_enqueue_style(
			'altaudit82ai-admin',
			ALTAUDIT82AI_ASSETS_URL . 'css/admin.css',
			array(),
			ALTAUDIT82AI_VERSION
		);

		// Admin JS.
		wp_enqueue_script(
			'altaudit82ai-admin',
			ALTAUDIT82AI_ASSETS_URL . 'js/admin.min.js',
			array( 'jquery', 'wp-util' ),
			ALTAUDIT82AI_VERSION,
			true
		);

		// Localize script.
		wp_localize_script(
			'altaudit82ai-admin',
			'altaudit82aiAdmin',
			array(
				'ajaxUrl' => admin_url( 'admin-ajax.php' ),
				'nonce'   => wp_create_nonce( 'altaudit82ai_nonce' ),
				'strings' => array(
					'processing'    => __( 'Processing...', 'alt-audit' ),
					'generating'    => __( 'Generating...', 'alt-audit' ),
					'error'         => __( 'An error occurred.', 'alt-audit' ),
					'success'       => __( 'Success!', 'alt-audit' ),
					'confirm'       => __( 'Are you sure?', 'alt-audit' ),
					'noImagesFound' => __( 'No images found.', 'alt-audit' ),
				),
			)
		);

		// Enqueue WordPress media library if needed.
		if ( 'alt-audit_page_alt-audit-bulk' === $hook_suffix ) {
			wp_enqueue_media();
		}
	}

	/**
	 * Dashboard page
	 *
	 * @return void
	 */
	public function dashboard_page() {
		// Get dashboard stats.
		$stats = $this->get_dashboard_stats();

		include ALTAUDIT82AI_VIEWS_DIR . 'Admin/dashboard.php';
	}

	/**
	 * Settings page
	 *
	 * @return void
	 */
	public function settings_page() {
		// Default to 'general' tab.
		$active_tab = 'general';

		// Check for tab parameter in GET.
		// phpcs:ignore WordPress.Security.NonceVerification.Recommended -- Tab is display-only, validated against whitelist below.
		if ( isset( $_GET['tab'] ) ) {
			// phpcs:ignore WordPress.Security.NonceVerification.Recommended
			$tab = sanitize_text_field( wp_unslash( $_GET['tab'] ) );
			// Validate against allowed tabs (strict whitelist - this is the security).
			$allowed_tabs = array( 'general', 'advanced', 'templates', 'quality' );
			if ( in_array( $tab, $allowed_tabs, true ) ) {
				$active_tab = $tab;
			}
		}

		include ALTAUDIT82AI_VIEWS_DIR . 'Admin/settings.php';
	}

	/**
	 * Audit page
	 *
	 * @return void
	 */
	public function audit_page() {
		// Initialize the list table.
		$this->prepare_images_list_table();

		// Handle list table actions.
		$this->handle_list_table_actions();

		// Prepare the list table data.
		$this->images_list_table->prepare_items();

		include ALTAUDIT82AI_VIEWS_DIR . 'Admin/audit.php';
	}

	/**
	 * Bulk tools page
	 *
	 * @return void
	 */
	public function bulk_tools_page() {
		include ALTAUDIT82AI_VIEWS_DIR . 'Admin/bulk-tools.php';
	}

	/**
	 * Help page
	 *
	 * @return void
	 */
	public function help_page() {
		include ALTAUDIT82AI_VIEWS_DIR . 'Admin/help.php';
	}

	/**
	 * Display admin notices
	 *
	 * @return void
	 */
	public function admin_notices() {
		// Only show notices on Alt Audit pages.
		$screen = get_current_screen();
		if ( ! $screen || false === strpos( $screen->id, 'alt-audit' ) ) {
			return;
		}

		// API key missing notice.
		if ( ! $this->settings->is_api_configured() ) {
			$this->show_notice(
				sprintf(
					/* translators: %s: Settings page URL */
					__( 'Alt Audit requires an API key to function. Please <a href="%s">configure your settings</a>.', 'alt-audit' ),
					admin_url( 'admin.php?page=alt-audit-settings' )
				),
				'warning'
			);
		}

		// Initial scan reminder notice.
		$this->show_scan_reminder_notice();
	}

	/**
	 * Save settings
	 *
	 * @return void
	 */
	public function save_settings() {
		// Verify nonce is present - REQUIRED, not optional.
		if ( ! isset( $_POST['_wpnonce'] ) ) {
			wp_die( esc_html__( 'Security check failed.', 'alt-audit' ) );
		}

		// Verify nonce is valid.
		if ( ! wp_verify_nonce( sanitize_text_field( wp_unslash( $_POST['_wpnonce'] ) ), 'altaudit82ai_settings' ) ) {
			wp_die( esc_html__( 'Security check failed.', 'alt-audit' ) );
		}

		// Check permissions.
		if ( ! current_user_can( 'manage_options' ) ) {
			wp_die( esc_html__( 'You do not have permission to perform this action.', 'alt-audit' ) );
		}

		$settings_to_save = array();
		$tab              = isset( $_POST['tab'] ) ? sanitize_text_field( wp_unslash( $_POST['tab'] ) ) : 'general';

		// Define allowed values for validation.
		$allowed_styles              = array( 'descriptive', 'technical', 'creative', 'brief' );
		$allowed_notification_levels = array( 'none', 'errors', 'all' );

		// Process settings based on tab.
		switch ( $tab ) {
			case 'general':
				// Sanitize and validate style.
				$style = isset( $_POST['style'] ) ? sanitize_text_field( wp_unslash( $_POST['style'] ) ) : 'descriptive';
				$style = in_array( $style, $allowed_styles, true ) ? $style : 'descriptive';

				// Sanitize and validate quality_threshold (0-100).
				$quality_threshold = isset( $_POST['quality_threshold'] ) ? intval( wp_unslash( $_POST['quality_threshold'] ) ) : 70;
				$quality_threshold = max( 0, min( 100, $quality_threshold ) );

				// Sanitize and validate max_length (1-500).
				$max_length = isset( $_POST['max_length'] ) ? intval( wp_unslash( $_POST['max_length'] ) ) : 125;
				$max_length = max( 1, min( 500, $max_length ) );

				$settings_to_save = array(
					'api_key'           => isset( $_POST['api_key'] ) ? sanitize_text_field( wp_unslash( $_POST['api_key'] ) ) : '',
					'auto_generate'     => isset( $_POST['auto_generate'] ),
					'quality_threshold' => $quality_threshold,
					'max_length'        => $max_length,
					'style'             => $style,
				);
				break;

			case 'advanced':
				$enabled_post_types = isset( $_POST['enabled_post_types'] ) ? array_map( 'sanitize_text_field', wp_unslash( $_POST['enabled_post_types'] ) ) : array();

				// Sanitize and validate notification_level.
				$notification_level = isset( $_POST['notification_level'] ) ? sanitize_text_field( wp_unslash( $_POST['notification_level'] ) ) : 'errors';
				$notification_level = in_array( $notification_level, $allowed_notification_levels, true ) ? $notification_level : 'errors';

				$settings_to_save = array(
					'context_aware'      => isset( $_POST['context_aware'] ),
					'bulk_processing'    => isset( $_POST['bulk_processing'] ),
					'cache_results'      => isset( $_POST['cache_results'] ),
					'debug_mode'         => isset( $_POST['debug_mode'] ),
					'enabled_post_types' => $enabled_post_types,
					'skip_existing'      => isset( $_POST['skip_existing'] ),
					'notification_level' => $notification_level,
				);
				break;

			case 'templates':
				// Process Alt Text Builder pattern arrays.
				// phpcs:disable Generic.Files.LineLength.MaxExceeded -- Long default arrays.
				$home_images_alt    = isset( $_POST['home_images_alt'] )
					? array_map( 'sanitize_text_field', wp_unslash( $_POST['home_images_alt'] ) )
					: array( 'Site Name', '|', 'Page Title' );
				$pages_images_alt   = isset( $_POST['pages_images_alt'] )
					? array_map( 'sanitize_text_field', wp_unslash( $_POST['pages_images_alt'] ) )
					: array( 'Site Name', '|', 'Page Title', 'Site Description' );
				$post_images_alt    = isset( $_POST['post_images_alt'] )
					? array_map( 'sanitize_text_field', wp_unslash( $_POST['post_images_alt'] ) )
					: array( 'Site Name', '|', 'Post Title' );
				$product_images_alt = isset( $_POST['product_images_alt'] )
					? array_map( 'sanitize_text_field', wp_unslash( $_POST['product_images_alt'] ) )
					: array( 'Site Name', '|', 'Product Title' );
				$cpt_images_alt     = isset( $_POST['cpt_images_alt'] )
					? array_map( 'sanitize_text_field', wp_unslash( $_POST['cpt_images_alt'] ) )
					: array( 'Site Name', '|', 'Post Title' );
				// phpcs:enable Generic.Files.LineLength.MaxExceeded

				$settings_to_save = array(
					'template_type'               => isset( $_POST['template_type'] ) ? sanitize_text_field( wp_unslash( $_POST['template_type'] ) ) : 'basic',
					'custom_template'             => isset( $_POST['custom_template'] ) ? sanitize_text_field( wp_unslash( $_POST['custom_template'] ) ) : '',
					'template_skip_existing'      => isset( $_POST['template_skip_existing'] ),
					'template_override_weak'      => isset( $_POST['template_override_weak'] ),
					'template_remove_hyphens'     => isset( $_POST['template_remove_hyphens'] ),
					'template_remove_underscores' => isset( $_POST['template_remove_underscores'] ),
					'template_remove_dots'        => isset( $_POST['template_remove_dots'] ),
					'template_remove_commas'      => isset( $_POST['template_remove_commas'] ),
					'template_remove_numbers'     => isset( $_POST['template_remove_numbers'] ),
					'template_case_format'        => isset( $_POST['template_case_format'] ) ? sanitize_text_field( wp_unslash( $_POST['template_case_format'] ) ) : 'sentence',
					// Alt Text Builder pattern arrays.
					'home_images_alt'             => $home_images_alt,
					'pages_images_alt'            => $pages_images_alt,
					'post_images_alt'             => $post_images_alt,
					'product_images_alt'          => $product_images_alt,
					'cpt_images_alt'              => $cpt_images_alt,
				);
				break;
		}

		// Save settings.
		$success = $this->settings->set_multiple( $settings_to_save );

		// Redirect with message.
		$redirect_url = admin_url( 'admin.php?page=alt-audit-settings&tab=' . $tab );

		if ( $success ) {
			$redirect_url = add_query_arg(
				array(
					'message' => rawurlencode( __( 'Settings saved successfully.', 'alt-audit' ) ),
					'type'    => 'success',
				),
				$redirect_url
			);
		} else {
			$redirect_url = add_query_arg(
				array(
					'message' => rawurlencode( __( 'Failed to save settings.', 'alt-audit' ) ),
					'type'    => 'error',
				),
				$redirect_url
			);
		}

		wp_safe_redirect( $redirect_url );
		exit;
	}

	/**
	 * Test API connection
	 *
	 * @return void
	 */
	public function test_api_connection() {
		// Verify nonce.
		$nonce = isset( $_POST['_wpnonce'] ) ? sanitize_text_field( wp_unslash( $_POST['_wpnonce'] ) ) : '';
		if ( ! wp_verify_nonce( $nonce, 'altaudit82ai_test_api' ) ) {
			wp_die( esc_html__( 'Security check failed.', 'alt-audit' ) );
		}

		// Check permissions.
		if ( ! current_user_can( 'manage_options' ) ) {
			wp_die( esc_html__( 'You do not have permission to perform this action.', 'alt-audit' ) );
		}

		$result = $this->api_client->test_connection();

		$redirect_url = admin_url( 'admin.php?page=alt-audit-settings' );
		$redirect_url = add_query_arg(
			array(
				'message' => rawurlencode( $result['message'] ),
				'type'    => $result['success'] ? 'success' : 'error',
			),
			$redirect_url
		);

		wp_safe_redirect( $redirect_url );
		exit;
	}

	/**
	 * Bulk scan images
	 *
	 * @return void
	 */
	public function bulk_scan() {
		// Verify nonce is present - REQUIRED, not optional.
		if ( ! isset( $_POST['_wpnonce'] ) ) {
			wp_die( esc_html__( 'Security check failed.', 'alt-audit' ) );
		}

		// Verify nonce is valid.
		if ( ! wp_verify_nonce( sanitize_text_field( wp_unslash( $_POST['_wpnonce'] ) ), 'altaudit82ai_bulk_scan' ) ) {
			wp_die( esc_html__( 'Security check failed.', 'alt-audit' ) );
		}

		// Check permissions.
		if ( ! current_user_can( 'upload_files' ) ) {
			wp_die( esc_html__( 'You do not have permission to perform this action.', 'alt-audit' ) );
		}

		// Process in background or redirect to bulk tools page.
		wp_safe_redirect( admin_url( 'admin.php?page=alt-audit-bulk&action=scan' ) );
		exit;
	}

	/**
	 * Register settings
	 *
	 * @return void
	 */
	private function register_settings() {
		// Settings are handled by the Settings service.
		// This method can be used for additional WordPress settings registration if needed.
	}

	/**
	 * Add meta boxes
	 *
	 * @return void
	 */
	private function add_meta_boxes() {
		// Add meta box to post edit screens.
		$enabled_post_types = $this->settings->get( 'enabled_post_types', array( 'post', 'page' ) );

		foreach ( $enabled_post_types as $post_type ) {
			add_meta_box(
				'altaudit82ai_meta_box',
				__( 'Alt Audit', 'alt-audit' ),
				array( $this, 'meta_box_callback' ),
				$post_type,
				'side',
				'default'
			);
		}
	}

	/**
	 * Meta box callback
	 *
	 * @param WP_Post $post Current post object.
	 * @return void
	 */
	public function meta_box_callback( $post ) {
		// Get images in post.
		$images = $this->get_post_images( $post->ID );

		include ALTAUDIT82AI_TEMPLATES_DIR . 'admin/meta-box.php';
	}

	/**
	 * Get dashboard statistics
	 *
	 * Uses centralized Statistics Service for consistency across all pages.
	 *
	 * @since  1.0.0
	 * @return array Dashboard stats.
	 */
	private function get_dashboard_stats() {
		return $this->statistics_service->get_dashboard_stats();
	}

	/**
	 * Get site audit results
	 *
	 * @return array Audit results.
	 */
	private function get_site_audit_results() {
		// This would contain the site audit logic.
		return array(
			'pages_scanned' => 0,
			'images_found'  => 0,
			'issues_found'  => array(),
		);
	}

	/**
	 * Get images in post
	 *
	 * @param int $post_id Post ID.
	 * @return array Images in post.
	 */
	private function get_post_images( $post_id ) {
		// Get attached images.
		$images = get_attached_media( 'image', $post_id );

		// Get images in content.
		$content = get_post_field( 'post_content', $post_id );
		preg_match_all( '/<img[^>]+>/i', $content, $img_tags );

		return array(
			'attached'   => $images,
			'in_content' => $img_tags[0],
		);
	}

	/**
	 * Show admin notice
	 *
	 * @param string $message Notice message.
	 * @param string $type    Notice type (success, error, warning, info).
	 * @return void
	 */
	private function show_notice( $message, $type = 'info' ) {
		printf(
			'<div class="notice notice-%s is-dismissible"><p>%s</p></div>',
			esc_attr( $type ),
			wp_kses_post( $message )
		);
	}

	/**
	 * Prepare images list table
	 *
	 * @param array $custom_params Optional custom parameters for AJAX context.
	 * @return void
	 */
	private function prepare_images_list_table( $custom_params = array() ) {
		if ( ! class_exists( 'Altaudit82ai_Images_List_Table' ) ) {
			require_once ALTAUDIT82AI_CONTROLLERS_DIR . 'class-altaudit82ai-images-list-table.php';
		}

		$this->images_list_table = new Altaudit82ai_Images_List_Table(
			$this->quality_service,
			$this->settings,
			$custom_params
		);
	}

	/**
	 * Handle list table actions
	 *
	 * @return void
	 */
	private function handle_list_table_actions() {
		if ( ! isset( $_POST['_wpnonce'] ) ) {
			return;
		}

		$action = $this->images_list_table->current_action();
		$nonce  = sanitize_text_field( wp_unslash( $_POST['_wpnonce'] ) );

		if ( ! wp_verify_nonce( $nonce, 'altaudit82ai_bulk_action' ) ) {
			return;
		}

		if ( ! current_user_can( 'upload_files' ) ) {
			wp_die( esc_html__( 'You do not have permission to perform this action.', 'alt-audit' ) );
		}

		$attachment_ids = isset( $_POST['attachment_id'] ) ? array_map( 'intval', (array) wp_unslash( $_POST['attachment_id'] ) ) : array();

		if ( empty( $attachment_ids ) ) {
			return;
		}

		switch ( $action ) {
			case 'generate_alt_text':
				$this->handle_bulk_generate_alt_text( $attachment_ids );
				break;

			case 'mark_decorative':
				$this->handle_bulk_mark_decorative( $attachment_ids );
				break;

			case 'delete':
				$this->handle_bulk_delete( $attachment_ids );
				break;
		}
	}

	/**
	 * Handle bulk generate alt text action
	 *
	 * @param array $attachment_ids Array of attachment IDs.
	 * @return void
	 */
	private function handle_bulk_generate_alt_text( $attachment_ids ) {
		$generated = 0;
		$errors    = 0;

		foreach ( $attachment_ids as $attachment_id ) {
			// Generate alt text using API.
			$result = $this->api_client->generate_alt_text( $attachment_id );

			if ( ! empty( $result['alt_text'] ) ) {
				update_post_meta( $attachment_id, '_wp_attachment_image_alt', $result['alt_text'] );
				update_post_meta( $attachment_id, '_altaudit82ai_auto_generated', true );
				++$generated;
			} else {
				++$errors;
			}
		}

		$message = sprintf(
			/* translators: %1$d: number generated, %2$d: number of errors */
			_n(
				'Generated alt text for %1$d image. %2$d errors.',
				'Generated alt text for %1$d images. %2$d errors.',
				$generated,
				'alt-audit'
			),
			$generated,
			$errors
		);

		$this->add_admin_notice( $message, $errors > 0 ? 'warning' : 'success' );
	}

	/**
	 * Handle bulk mark decorative action
	 *
	 * @param array $attachment_ids Array of attachment IDs.
	 * @return void
	 */
	private function handle_bulk_mark_decorative( $attachment_ids ) {
		$marked = 0;

		foreach ( $attachment_ids as $attachment_id ) {
			update_post_meta( $attachment_id, '_wp_attachment_image_alt', '' );
			update_post_meta( $attachment_id, '_altaudit82ai_decorative', true );
			++$marked;
		}

		$message = sprintf(
			/* translators: %d: number of images marked */
			_n(
				'Marked %d image as decorative.',
				'Marked %d images as decorative.',
				$marked,
				'alt-audit'
			),
			$marked
		);

		$this->add_admin_notice( $message, 'success' );
	}

	/**
	 * Handle bulk delete action
	 *
	 * @param array $attachment_ids Array of attachment IDs.
	 * @return void
	 */
	private function handle_bulk_delete( $attachment_ids ) {
		$deleted = 0;

		foreach ( $attachment_ids as $attachment_id ) {
			if ( wp_delete_attachment( $attachment_id, true ) ) {
				++$deleted;
			}
		}

		$message = sprintf(
			/* translators: %d: number of images deleted */
			_n(
				'Deleted %d image.',
				'Deleted %d images.',
				$deleted,
				'alt-audit'
			),
			$deleted
		);

		$this->add_admin_notice( $message, 'success' );
	}

	/**
	 * AJAX handler for inline edit alt text
	 *
	 * @return void
	 */
	public function ajax_inline_edit_alt_text() {
		check_ajax_referer( 'altaudit82ai_nonce', 'nonce' );

		if ( ! current_user_can( 'upload_files' ) ) {
			wp_send_json_error( __( 'You do not have permission to perform this action.', 'alt-audit' ) );
		}

		$attachment_id = isset( $_POST['attachment_id'] ) ? intval( wp_unslash( $_POST['attachment_id'] ) ) : 0;
		$alt_text      = isset( $_POST['alt_text'] ) ? sanitize_textarea_field( wp_unslash( $_POST['alt_text'] ) ) : '';

		if ( ! $attachment_id || ! get_post( $attachment_id ) ) {
			wp_send_json_error( __( 'Invalid attachment ID.', 'alt-audit' ) );
		}

		// Update alt text.
		update_post_meta( $attachment_id, '_wp_attachment_image_alt', $alt_text );

		// Remove auto-generated flag since it's manually edited.
		delete_post_meta( $attachment_id, '_altaudit82ai_auto_generated' );

		// Always remove existing decorative flag first.
		delete_post_meta( $attachment_id, '_altaudit82ai_decorative' );

		// Get updated quality assessment.
		$quality_result = $this->quality_service->assess_quality( $alt_text );
		$status         = $quality_result['status'] ?? 'unknown';

		// Update decorative flag based on new assessment.
		if ( 'decorative' === $status ) {
			update_post_meta( $attachment_id, '_altaudit82ai_decorative', '1' );
		}

		// Get updated row HTML for table refresh.
		$row_html = $this->get_single_row_html( $attachment_id );

		wp_send_json_success(
			array(
				'alt_text'      => $alt_text,
				'quality_score' => $quality_result['score'] ?? 0,
				'status'        => $status,
				'quality'       => array(
					'score'  => $quality_result['score'] ?? 0,
					'status' => $status,
				),
				'message'       => __( 'Alt text updated successfully.', 'alt-audit' ),
				'row_html'      => $row_html,
				'attachment_id' => $attachment_id,
			)
		);
	}

	/**
	 * AJAX handler for bulk actions
	 *
	 * @return void
	 */
	public function ajax_bulk_action() {
		check_ajax_referer( 'altaudit82ai_nonce', 'nonce' );

		if ( ! current_user_can( 'upload_files' ) ) {
			wp_send_json_error( __( 'You do not have permission to perform this action.', 'alt-audit' ) );
		}

		$action         = isset( $_POST['bulk_action'] ) ? sanitize_text_field( wp_unslash( $_POST['bulk_action'] ) ) : '';
		$attachment_ids = isset( $_POST['attachment_ids'] ) ? array_map( 'intval', (array) wp_unslash( $_POST['attachment_ids'] ) ) : array();

		if ( empty( $action ) || empty( $attachment_ids ) ) {
			wp_send_json_error( __( 'Invalid action or no images selected.', 'alt-audit' ) );
		}

		$results = array();

		switch ( $action ) {
			case 'generate_alt_text':
				$results = $this->ajax_bulk_generate_alt_text( $attachment_ids );
				break;

			case 'mark_decorative':
				$results = $this->ajax_bulk_mark_decorative( $attachment_ids );
				break;

			case 'delete':
				$results = $this->ajax_bulk_delete( $attachment_ids );
				break;

			default:
				wp_send_json_error( __( 'Invalid bulk action.', 'alt-audit' ) );
		}

		wp_send_json_success( $results );
	}

	/**
	 * AJAX bulk generate alt text
	 *
	 * @param array $attachment_ids Array of attachment IDs.
	 * @return array Results array.
	 */
	private function ajax_bulk_generate_alt_text( $attachment_ids ) {
		$results = array(
			'success' => 0,
			'errors'  => 0,
			'details' => array(),
		);

		foreach ( $attachment_ids as $attachment_id ) {
			$result = $this->api_client->generate_alt_text( $attachment_id );

			if ( ! empty( $result['alt_text'] ) ) {
				update_post_meta( $attachment_id, '_wp_attachment_image_alt', $result['alt_text'] );
				update_post_meta( $attachment_id, '_altaudit82ai_auto_generated', true );

				$quality_result = $this->quality_service->assess_quality( $result['alt_text'] );
				$quality_score  = $quality_result['score'] ?? 0;

				$results['details'][ $attachment_id ] = array(
					'status'         => 'success',
					'alt_text'       => $result['alt_text'],
					'quality_score'  => $quality_score,
					'quality_status' => Altaudit82ai_Taxonomy::get_status_from_score( $quality_score, false, true ),
				);

				++$results['success'];
			} else {
				$results['details'][ $attachment_id ] = array(
					'status' => 'error',
					'error'  => $result['error'] ?? __( 'Failed to generate alt text.', 'alt-audit' ),
				);

				++$results['errors'];
			}
		}

		return $results;
	}

	/**
	 * AJAX bulk mark decorative
	 *
	 * @param array $attachment_ids Array of attachment IDs.
	 * @return array Results array.
	 */
	private function ajax_bulk_mark_decorative( $attachment_ids ) {
		$results = array(
			'success' => 0,
			'details' => array(),
		);

		foreach ( $attachment_ids as $attachment_id ) {
			update_post_meta( $attachment_id, '_wp_attachment_image_alt', '' );
			update_post_meta( $attachment_id, '_altaudit82ai_decorative', true );

			$results['details'][ $attachment_id ] = array(
				'status'         => 'success',
				'alt_text'       => '',
				'quality_score'  => 100,
				'quality_status' => 'decorative',
			);

			++$results['success'];
		}

		return $results;
	}

	/**
	 * AJAX bulk delete
	 *
	 * @param array $attachment_ids Array of attachment IDs.
	 * @return array Results array.
	 */
	private function ajax_bulk_delete( $attachment_ids ) {
		$results = array(
			'success' => 0,
			'errors'  => 0,
			'details' => array(),
		);

		foreach ( $attachment_ids as $attachment_id ) {
			if ( wp_delete_attachment( $attachment_id, true ) ) {
				$results['details'][ $attachment_id ] = array( 'status' => 'deleted' );
				++$results['success'];
			} else {
				$results['details'][ $attachment_id ] = array(
					'status' => 'error',
					'error'  => __( 'Failed to delete attachment.', 'alt-audit' ),
				);
				++$results['errors'];
			}
		}

		return $results;
	}

	/**
	 * AJAX handler for refreshing quality score
	 *
	 * @return void
	 */
	public function ajax_refresh_quality_score() {
		check_ajax_referer( 'altaudit82ai_nonce', 'nonce' );

		if ( ! current_user_can( 'upload_files' ) ) {
			wp_send_json_error( __( 'You do not have permission to perform this action.', 'alt-audit' ) );
		}

		$attachment_id = isset( $_POST['attachment_id'] ) ? intval( wp_unslash( $_POST['attachment_id'] ) ) : 0;

		if ( ! $attachment_id || ! get_post( $attachment_id ) ) {
			wp_send_json_error( __( 'Invalid attachment ID.', 'alt-audit' ) );
		}

		$alt_text       = get_post_meta( $attachment_id, '_wp_attachment_image_alt', true );
		$quality_result = $this->quality_service->assess_quality( $alt_text );

		wp_send_json_success(
			array(
				'quality_score' => $quality_result['score'] ?? 0,
				'status'        => $quality_result['status'] ?? 'unknown',
				'suggestions'   => $quality_result['suggestions'] ?? array(),
			)
		);
	}

	/**
	 * AJAX handler for getting updated statistics
	 *
	 * @return void
	 */
	public function ajax_get_statistics() {
		check_ajax_referer( 'altaudit82ai_nonce', 'nonce' );

		if ( ! current_user_can( 'upload_files' ) ) {
			wp_send_json_error( __( 'You do not have permission to perform this action.', 'alt-audit' ) );
		}

		// Check if images_list_table is available.
		if ( ! isset( $this->images_list_table ) || ! $this->images_list_table ) {
			// Create a temporary instance.
			require_once ALTAUDIT82AI_PLUGIN_DIR . 'includes/Controllers/class-altaudit82ai-images-list-table.php';
			$this->images_list_table = new Altaudit82ai_Images_List_Table( $this->quality_service );
		}

		// Get fresh statistics.
		$stats = $this->images_list_table->get_statistics();

		wp_send_json_success( $stats );
	}

	/**
	 * AJAX handler for getting user credits
	 *
	 * @return void
	 */
	public function ajax_get_user_credits() {
		check_ajax_referer( 'altaudit82ai_nonce', 'nonce' );

		if ( ! current_user_can( 'upload_files' ) ) {
			wp_send_json_error( __( 'You do not have permission to perform this action.', 'alt-audit' ) );
		}

		// Get user info from API.
		if ( ! $this->api_client ) {
			wp_send_json_error( __( 'API client not available.', 'alt-audit' ) );
		}

		$user_info = $this->api_client->get_user_info();

		if ( ! $user_info['success'] ) {
			// Return 0 credits if API call fails.
			wp_send_json_success( array( 'credits' => 0 ) );
			return;
		}

		// Extract credits from user info. API returns: { success: true, data: { data: { credits: 391, ... } } }.
		$credits = 0;
		if ( isset( $user_info['data']['data']['credits'] ) ) {
			$credits = intval( $user_info['data']['data']['credits'] );
		} elseif ( isset( $user_info['data']['credits'] ) ) {
			$credits = intval( $user_info['data']['credits'] );
		}

		wp_send_json_success( array( 'credits' => $credits ) );
	}

	/**
	 * AJAX handler for getting queue status (Phase 1B)
	 *
	 * @return void
	 */
	public function ajax_get_queue_status() {
		check_ajax_referer( 'altaudit82ai_nonce', 'nonce' );

		if ( ! current_user_can( 'upload_files' ) ) {
			wp_send_json_error( __( 'You do not have permission to perform this action.', 'alt-audit' ) );
		}

		if ( ! $this->api_client ) {
			wp_send_json_error( __( 'API client not available.', 'alt-audit' ) );
		}

		$status = $this->api_client->get_queue_status();

		if ( ! $status['success'] ) {
			wp_send_json_error( $status['message'] ?? __( 'Failed to get queue status.', 'alt-audit' ) );
		}

		wp_send_json_success( $status['data'] );
	}

	/**
	 * AJAX handler for exporting alt text data
	 *
	 * @return void
	 */
	public function ajax_export_data() {
		check_ajax_referer( 'altaudit82ai_nonce', 'nonce' );

		if ( ! current_user_can( 'upload_files' ) ) {
			wp_send_json_error( __( 'You do not have permission to perform this action.', 'alt-audit' ) );
		}

		$format = isset( $_POST['format'] ) ? sanitize_text_field( wp_unslash( $_POST['format'] ) ) : 'csv';

				// Get all images using WP_Query.
				$images_query = new WP_Query(
					array(
						'post_type'      => 'attachment',
						'post_status'    => 'inherit',
						'post_mime_type' => Altaudit82ai::get_supported_mime_types(),
						'posts_per_page' => -1,
						'orderby'        => 'ID',
						'order'          => 'ASC',
					)
				);
		$results              = array();
		foreach ( $images_query->posts as $image ) {
			$results[] = array(
				'image_id'          => $image->ID,
				'title'             => $image->post_title,
				'url'               => $image->guid,
				'alt_text'          => get_post_meta( $image->ID, '_wp_attachment_image_alt', true ),
				'quality_score'     => get_post_meta( $image->ID, '_altaudit82ai_quality_score', true ),
				'generation_method' => get_post_meta( $image->ID, '_altaudit82ai_generation_method', true ),
				'is_auto_generated' => get_post_meta( $image->ID, '_altaudit82ai_auto_generated', true ),
				'created_date'      => $image->post_date,
			);
		}

		if ( empty( $results ) ) {
			wp_send_json_error( __( 'No image data found to export.', 'alt-audit' ) );
		}

		// Generate export based on format.
		switch ( $format ) {
			case 'json':
				$content   = wp_json_encode( $results, JSON_PRETTY_PRINT );
				$mime_type = 'application/json';
				$extension = 'json';
				break;

			case 'xml':
				$content   = $this->generate_xml_export( $results );
				$mime_type = 'application/xml';
				$extension = 'xml';
				break;

			case 'csv':
			default:
				$content   = $this->generate_csv_export( $results );
				$mime_type = 'text/csv';
				$extension = 'csv';
				break;
		}

		$filename = 'altaudit82ai-export-' . gmdate( 'Y-m-d-His' ) . '.' . $extension;

		wp_send_json_success(
			array(
				'content'   => $content,
				'filename'  => $filename,
				'mime_type' => $mime_type,
				'count'     => count( $results ),
			)
		);
	}

	/**
	 * Generate CSV export from results
	 *
	 * @param array $results Query results.
	 * @return string CSV content.
	 */
	private function generate_csv_export( $results ) {
		// phpcs:ignore WordPress.WP.AlternativeFunctions.file_system_operations_fopen -- Using php://temp stream for in-memory CSV generation.
		$output = fopen( 'php://temp', 'r+' );

		// Add BOM for Excel UTF-8 support.
		// phpcs:ignore WordPress.WP.AlternativeFunctions.file_system_operations_fputs -- Using php://temp stream.
		fputs( $output, "\xEF\xBB\xBF" );

		// Header row.
		// phpcs:ignore WordPress.WP.AlternativeFunctions.file_system_operations_fputcsv -- Using php://temp stream.
		fputcsv( $output, array( 'Image ID', 'Title', 'URL', 'Alt Text', 'Quality Score', 'Generation Method', 'Auto Generated', 'Created Date' ) );

		// Data rows.
		foreach ( $results as $row ) {
			// phpcs:ignore WordPress.WP.AlternativeFunctions.file_system_operations_fputcsv -- Using php://temp stream.
			fputcsv(
				$output,
				array(
					$row['image_id'],
					$row['title'],
					$row['url'],
					$row['alt_text'],
					$row['quality_score'],
					$row['generation_method'],
					$row['is_auto_generated'] ? 'Yes' : 'No',
					$row['created_date'],
				)
			);
		}

		rewind( $output );
		$csv = stream_get_contents( $output );
		// phpcs:ignore WordPress.WP.AlternativeFunctions.file_system_operations_fclose -- Using php://temp stream.
		fclose( $output );

		return $csv;
	}

	/**
	 * Generate XML export from results
	 *
	 * @param array $results Query results.
	 * @return string XML content.
	 */
	private function generate_xml_export( $results ) {
		$xml  = '<?xml version="1.0" encoding="UTF-8"?>' . "\n";
		$xml .= '<altaudit82ai_export date="' . gmdate( 'c' ) . '">' . "\n";

		foreach ( $results as $row ) {
			$xml .= "\t<image>\n";
			$xml .= "\t\t<id>" . esc_xml( $row['image_id'] ) . "</id>\n";
			$xml .= "\t\t<title>" . esc_xml( $row['title'] ) . "</title>\n";
			$xml .= "\t\t<url>" . esc_xml( $row['url'] ) . "</url>\n";
			$xml .= "\t\t<alt_text>" . esc_xml( $row['alt_text'] ) . "</alt_text>\n";
			$xml .= "\t\t<quality_score>" . esc_xml( $row['quality_score'] ) . "</quality_score>\n";
			$xml .= "\t\t<generation_method>" . esc_xml( $row['generation_method'] ) . "</generation_method>\n";
			$xml .= "\t\t<is_auto_generated>" . ( $row['is_auto_generated'] ? 'true' : 'false' ) . "</is_auto_generated>\n";
			$xml .= "\t\t<created_date>" . esc_xml( $row['created_date'] ) . "</created_date>\n";
			$xml .= "\t</image>\n";
		}

		$xml .= '</altaudit82ai_export>';

		return $xml;
	}

	/**
	 * AJAX handler for counting images in media library for bulk operations
	 *
	 * @return void
	 */
	public function ajax_count_images() {
		check_ajax_referer( 'altaudit82ai_nonce', 'nonce' );

		if ( ! current_user_can( 'upload_files' ) ) {
			wp_send_json_error( __( 'You do not have permission to perform this action.', 'alt-audit' ) );
		}

		// Get filter options - sanitize specific parameters.
		// phpcs:ignore WordPress.Security.ValidatedSanitizedInput.InputNotSanitized -- Strict comparison to 'true' string.
		$skip_existing = isset( $_POST['skip_existing'] ) && 'true' === sanitize_text_field( wp_unslash( $_POST['skip_existing'] ) );
		// phpcs:ignore WordPress.Security.ValidatedSanitizedInput.InputNotSanitized -- Strict comparison to 'true' string.
		$overwrite_weak = isset( $_POST['overwrite_weak'] ) && 'true' === sanitize_text_field( wp_unslash( $_POST['overwrite_weak'] ) );

		$count = $this->count_images_for_bulk_processing( $skip_existing, $overwrite_weak );

		wp_send_json_success( array( 'count' => $count ) );
	}

	/**
	 * Count images for bulk processing using WP_Query
	 *
	 * @param bool $skip_existing  Whether to skip images with existing alt text.
	 * @param bool $overwrite_weak Whether to include images with weak quality alt text.
	 * @return int Total count.
	 */
	private function count_images_for_bulk_processing( $skip_existing, $overwrite_weak ) {
		// Supported MIME types for AI alt text generation.
		// SVG files are excluded as AI vision APIs cannot analyze vector graphics.
		$supported_mime_types = Altaudit82ai::get_supported_mime_types();

		// Get all image attachments using WP_Query.
		$images_query = new WP_Query(
			array(
				'post_type'      => 'attachment',
				'post_status'    => 'inherit',
				'post_mime_type' => $supported_mime_types,
				'posts_per_page' => -1,
				'fields'         => 'ids',
			)
		);

		// If both options are unchecked, process ALL images (overwrite everything).
		if ( ! $skip_existing && ! $overwrite_weak ) {
			return $images_query->found_posts;
		}

		$count = 0;

		// Filter images based on criteria.
		foreach ( $images_query->posts as $image_id ) {
			// Use taxonomy status to match audit page behavior.
			$quality_status = Altaudit82ai_Taxonomy::get_status( $image_id );

			// If both options are checked: skip existing good ones, but process weak/missing.
			// This means only process missing since we skip existing (which includes weak with alt text).
			if ( $skip_existing && $overwrite_weak ) {
				// Only count images with "missing" taxonomy status (no alt text).
				if ( 'missing' === $quality_status || null === $quality_status ) {
					++$count;
				}
			} elseif ( $overwrite_weak ) {
				// Count images with "weak" OR "missing" taxonomy status.
				// These images need alt text improvement (weak quality or no alt text).
				if ( 'weak' === $quality_status || 'missing' === $quality_status || null === $quality_status ) {
					++$count;
				}
			} elseif ( $skip_existing ) {
				// Count images with "missing" taxonomy status (no alt text).
				if ( 'missing' === $quality_status || null === $quality_status ) {
					++$count;
				}
			}
		}

		return $count;
	}

	/**
	 * AJAX handler for bulk rule-based alt text generation
	 *
	 * Processes images in batches to generate alt text using rule-based templates.
	 * Supports resumable batch processing via WordPress transients.
	 *
	 * @return void
	 */
	public function ajax_bulk_generate_rule() {
		check_ajax_referer( 'altaudit82ai_nonce', 'nonce' );

		if ( ! current_user_can( 'upload_files' ) ) {
			wp_send_json_error( __( 'You do not have permission to perform this action.', 'alt-audit' ) );
		}

		// Get operation parameters - sanitize specific parameters.
		$batch_size = 10; // Process 10 images per batch.
		// phpcs:ignore WordPress.Security.ValidatedSanitizedInput.InputNotSanitized -- Strict comparison to 'true' string.
		$skip_existing = isset( $_POST['skip_existing'] ) && 'true' === sanitize_text_field( wp_unslash( $_POST['skip_existing'] ) );
		// phpcs:ignore WordPress.Security.ValidatedSanitizedInput.InputNotSanitized -- Strict comparison to 'true' string.
		$overwrite_weak = isset( $_POST['overwrite_weak'] ) && 'true' === sanitize_text_field( wp_unslash( $_POST['overwrite_weak'] ) );
		$batch_offset   = isset( $_POST['batch_offset'] ) ? intval( wp_unslash( $_POST['batch_offset'] ) ) : 0;
		$error_count    = isset( $_POST['error_count'] ) ? intval( wp_unslash( $_POST['error_count'] ) ) : 0;

		// Calculate correct offset for query.
		if ( $skip_existing || $overwrite_weak ) {
			$query_offset = $error_count;
		} else {
			$query_offset = $batch_offset;
		}

		// Get list of images to process.
		$image_ids = $this->get_images_for_bulk_processing( $skip_existing, $overwrite_weak, $query_offset, $batch_size );

		if ( empty( $image_ids ) ) {
			// No more images to process.
			wp_send_json_success(
				array(
					'batch_complete' => true,
					'all_complete'   => true,
					'processed'      => $batch_offset,
					'total'          => $batch_offset,
					'success_count'  => 0,
					'error_count'    => 0,
					'percentage'     => 100,
					'status_message' => __( 'All images processed successfully!', 'alt-audit' ),
					'errors'         => array(),
				)
			);
		}

		// Get total count for progress tracking.
		// Use cached total from frontend if provided (prevents count from changing as images are processed).
		$total_count = isset( $_POST['total_images'] ) ? intval( wp_unslash( $_POST['total_images'] ) ) : $this->get_total_images_count( $skip_existing, $overwrite_weak );

		// Initialize template service for rule-based generation.
		$template_service = altaudit82ai()->get_service( 'template_service' );

		// Process batch.
		$results = array(
			'success_count'    => 0,
			'error_count'      => 0,
			'errors'           => array(),
			'processed_images' => array(),
		);

		foreach ( $image_ids as $attachment_id ) {
			// Skip logic removed - process all images regardless of existing alt text or quality score.

			// Generate alt text using Template Service with configured patterns.
			$generated_alt = $template_service->generate_from_template( $attachment_id );

			if ( ! empty( $generated_alt ) ) {
				update_post_meta( $attachment_id, '_wp_attachment_image_alt', $generated_alt );
				update_post_meta( $attachment_id, '_altaudit82ai_auto_generated', true );
				update_post_meta( $attachment_id, '_altaudit82ai_generation_method', 'rule_based' );

				// Assess quality.
				$quality_result = $this->quality_service->assess_quality( $generated_alt );

				// Update taxonomy status to reflect the new alt text quality.
				// This is critical for filtering in subsequent batches.
				$quality_score  = $quality_result['score'] ?? 0;
				$quality_status = Altaudit82ai_Taxonomy::get_status_from_score( $quality_score, false, true );
				Altaudit82ai_Taxonomy::set_status( $attachment_id, sanitize_key( $quality_status ) );

				// Collect detailed image data for table view.
				$results['processed_images'][] = $this->get_image_result_data( $attachment_id, $generated_alt, $quality_result, true );

				++$results['success_count'];
			} else {
				++$results['error_count'];
				$results['errors'][] = array(
					'attachment_id' => $attachment_id,
					'message'       => __( 'Failed to generate alt text', 'alt-audit' ),
				);

				// Collect failed image data.
				$results['processed_images'][] = $this->get_image_result_data( $attachment_id, '', array(), false );
			}
		}

		// Calculate progress.
		$processed  = $batch_offset + count( $image_ids );
		$percentage = $total_count > 0 ? round( ( $processed / $total_count ) * 100 ) : 100;
		$remaining  = $total_count - $processed;

		$response = array(
			'batch_complete'    => true,
			'all_complete'      => $remaining <= 0,
			'processed'         => $processed,
			'total'             => $total_count,
			'success_count'     => $results['success_count'],
			'error_count'       => $results['error_count'],
			'percentage'        => $percentage,
			'status_message'    => sprintf(
				/* translators: %1$d: processed count, %2$d: total count */
				__( 'Processing image %1$d of %2$d...', 'alt-audit' ),
				$processed,
				$total_count
			),
			'errors'            => $results['errors'],
			'processed_images'  => $results['processed_images'],
			'next_batch_offset' => $processed,
		);

		wp_send_json_success( $response );
	}

	/**
	 * AJAX handler for bulk AI-powered alt text generation
	 *
	 * Processes images in batches to generate alt text using AI vision API.
	 * Supports resumable batch processing and credit tracking.
	 *
	 * @return void
	 */
	public function ajax_bulk_generate_ai() {
		check_ajax_referer( 'altaudit82ai_nonce', 'nonce' );

		if ( ! current_user_can( 'upload_files' ) ) {
			wp_send_json_error( __( 'You do not have permission to perform this action.', 'alt-audit' ) );
		}

		// Verify required services are available.
		if ( ! $this->api_client ) {
			wp_send_json_error( __( 'API client not available. Please check plugin configuration.', 'alt-audit' ) );
		}

		if ( ! $this->quality_service ) {
			wp_send_json_error( __( 'Quality service not available. Please check plugin configuration.', 'alt-audit' ) );
		}

		// Get operation parameters - sanitize specific parameters.
		// Reduced to 3 images per batch to prevent timeout (each image takes ~6 seconds).
		$batch_size = 3;
		// phpcs:ignore WordPress.Security.ValidatedSanitizedInput.InputNotSanitized -- Strict comparison to 'true' string.
		$skip_existing = isset( $_POST['skip_existing'] ) && 'true' === sanitize_text_field( wp_unslash( $_POST['skip_existing'] ) );
		// phpcs:ignore WordPress.Security.ValidatedSanitizedInput.InputNotSanitized -- Strict comparison to 'true' string.
		$overwrite_weak = isset( $_POST['overwrite_weak'] ) && 'true' === sanitize_text_field( wp_unslash( $_POST['overwrite_weak'] ) );
		$batch_offset   = isset( $_POST['batch_offset'] ) ? intval( wp_unslash( $_POST['batch_offset'] ) ) : 0;
		$error_count    = isset( $_POST['error_count'] ) ? intval( wp_unslash( $_POST['error_count'] ) ) : 0;

		// Calculate correct offset for query.
		// If we are skipping existing or overwriting weak, successful images are removed from the query results.
		// So we only need to offset by the number of errors (images that remained in the list).
		if ( $skip_existing || $overwrite_weak ) {
			$query_offset = $error_count;
		} else {
			$query_offset = $batch_offset;
		}

		// Get list of images to process.
		$image_ids = $this->get_images_for_bulk_processing( $skip_existing, $overwrite_weak, $query_offset, $batch_size );

		if ( empty( $image_ids ) ) {
			// No more images to process.
			wp_send_json_success(
				array(
					'batch_complete' => true,
					'all_complete'   => true,
					'processed'      => $batch_offset,
					'total'          => $batch_offset,
					'success_count'  => 0,
					'error_count'    => 0,
					'percentage'     => 100,
					'status_message' => __( 'All images processed successfully!', 'alt-audit' ),
					'errors'         => array(),
				)
			);
		}

		// Get total count for progress tracking.
		// Use cached total from frontend if provided (prevents count from changing as images are processed).
		$total_count = isset( $_POST['total_images'] ) ? intval( wp_unslash( $_POST['total_images'] ) ) : $this->get_total_images_count( $skip_existing, $overwrite_weak );

		// Process batch.
		$results = array(
			'success_count'    => 0,
			'error_count'      => 0,
			'skipped_count'    => 0,
			'errors'           => array(),
			'processed_images' => array(),
			'log_messages'     => array(), // Frontend log messages.
		);

		// Track consecutive critical errors to prevent infinite loops.
		$consecutive_critical_errors = 0;
		$max_consecutive_errors      = 3;

		foreach ( $image_ids as $attachment_id ) {
			// Log image processing start.
			$start_log = $this->log_image_processing_start( $attachment_id );

			// Build frontend log message with image details.
			$log_msg = sprintf(
				/* translators: %1$s: filename, %2$s: file size, %3$s: dimensions */
				__( 'Processing: %1$s (%2$s, %3$s)', 'alt-audit' ),
				esc_html( $start_log['filename'] ),
				isset( $start_log['file_size'] ) ? esc_html( $start_log['file_size'] ) : __( 'unknown size', 'alt-audit' ),
				isset( $start_log['dimensions'] ) ? esc_html( $start_log['dimensions'] ) : __( 'unknown dimensions', 'alt-audit' )
			);
			$results['log_messages'][] = array(
				'message' => $log_msg,
				'type'    => 'info',
				'data'    => $start_log,
			);

			// Check if we should skip this image.
			// Skip logic removed - process all images regardless of existing alt text or quality score.
			// This ensures the user gets what they expect when running bulk generation.

			// Generate alt text using AI.
			$result = $this->api_client->generate_alt_text( $attachment_id );

			// Check for critical errors - but continue processing other images.
			if ( isset( $result['code'] ) && in_array( $result['code'], array( 'rate_limit', 'server_error', 'auth_error' ), true ) ) {
				++$consecutive_critical_errors;

				// Log the critical error.
				$error_msg = isset( $result['message'] ) ? $result['message'] : __( 'Unknown error', 'alt-audit' );
				$error_log = $this->log_image_error( $attachment_id, $result['code'], $error_msg );

				$results['log_messages'][] = array(
					'message' => sprintf(
						/* translators: %1$s: filename, %2$s: error code, %3$s: error message */
						__( 'Error: %1$s - %2$s: %3$s', 'alt-audit' ),
						esc_html( $error_log['filename'] ),
						esc_html( strtoupper( $result['code'] ) ),
						esc_html( $error_msg )
					),
					'type'    => 'error',
					'data'    => $error_log,
				);

				// Add to errors list.
				++$results['error_count'];
				$results['errors'][] = array(
					'attachment_id' => $attachment_id,
					'message'       => isset( $result['message'] ) ? $result['message'] : $result['code'],
					'error_code'    => $result['code'],
					'http_status'   => isset( $result['status'] ) ? $result['status'] : 500,
				);

				// Collect failed image data.
				$results['processed_images'][] = $this->get_image_result_data( $attachment_id, '', array(), false );

				// Only stop if we hit max consecutive critical errors (e.g., auth_error repeatedly).
				if ( 'auth_error' === $result['code'] || $consecutive_critical_errors >= $max_consecutive_errors ) {
					// Stop batch - but return success with partial results so frontend can continue.
					$stop_log                  = $this->log_batch_stopped( $result['code'], $consecutive_critical_errors );
					$results['log_messages'][] = array(
						'message' => sprintf(
							/* translators: %1$s: reason, %2$d: error count */
							__( 'Batch stopped: %1$s after %2$d consecutive errors', 'alt-audit' ),
							esc_html( $result['code'] ),
							absint( $consecutive_critical_errors )
						),
						'type'    => 'error',
						'data'    => $stop_log,
					);
					break;
				}

				// For rate_limit and server_error, continue to next image after a short delay.
				if ( 'rate_limit' === $result['code'] ) {
					$results['log_messages'][] = array(
						'message' => __( 'Rate limit hit, waiting 2 seconds before continuing...', 'alt-audit' ),
						'type'    => 'warning',
					);
					sleep( 2 ); // Brief pause for rate limit.
				}

				continue; // Skip to next image instead of stopping.
			}

			// Reset consecutive error counter on any non-critical response.
			$consecutive_critical_errors = 0;

			if ( ! empty( $result['alt_text'] ) ) {
				update_post_meta( $attachment_id, '_wp_attachment_image_alt', $result['alt_text'] );
				update_post_meta( $attachment_id, '_altaudit82ai_auto_generated', true );
				update_post_meta( $attachment_id, '_altaudit82ai_generation_method', 'ai' );

				// Assess quality.
				$quality_result = $this->quality_service->assess_quality( $result['alt_text'] );

				// Update taxonomy status to reflect the new alt text quality.
				// This is critical for filtering in subsequent batches.
				$quality_score  = isset( $quality_result['score'] ) ? $quality_result['score'] : 0;
				$quality_status = Altaudit82ai_Taxonomy::get_status_from_score( $quality_score, false, true );
				Altaudit82ai_Taxonomy::set_status( $attachment_id, sanitize_key( $quality_status ) );

				// Log successful generation.
				$success_log = $this->log_image_success( $attachment_id, $result['alt_text'], $quality_score );

				$results['log_messages'][] = array(
					'message' => sprintf(
						/* translators: %1$s: filename, %2$d: quality score, %3$s: alt text preview */
						__( 'Success: %1$s (score: %2$d%%) - "%3$s"', 'alt-audit' ),
						esc_html( $success_log['filename'] ),
						absint( $quality_score ),
						esc_html( wp_trim_words( $result['alt_text'], 10, '...' ) )
					),
					'type'    => 'success',
					'data'    => $success_log,
				);

				// Collect detailed image data for table view.
				$results['processed_images'][] = $this->get_image_result_data( $attachment_id, $result['alt_text'], $quality_result, true );

				++$results['success_count'];
			} else {
				++$results['error_count'];
				$error_message = isset( $result['message'] ) ? $result['message'] : ( isset( $result['error'] ) ? $result['error'] : __( 'Failed to generate alt text', 'alt-audit' ) );
				$error_code    = isset( $result['code'] ) ? $result['code'] : 'generation_failed';

				// Log the error.
				$error_log = $this->log_image_error( $attachment_id, $error_code, $error_message );

				$results['log_messages'][] = array(
					'message' => sprintf(
						/* translators: %1$s: filename, %2$s: error message */
						__( 'Failed: %1$s - %2$s', 'alt-audit' ),
						esc_html( $error_log['filename'] ),
						esc_html( $error_message )
					),
					'type'    => 'error',
					'data'    => $error_log,
				);

				$results['errors'][] = array(
					'attachment_id' => $attachment_id,
					'message'       => $error_message,
					'error_code'    => $error_code,
				);

				// Collect failed image data.
				$results['processed_images'][] = $this->get_image_result_data( $attachment_id, '', array(), false );
			}
		}

		// Calculate progress.
		$processed  = $batch_offset + count( $image_ids );
		$percentage = $total_count > 0 ? round( ( $processed / $total_count ) * 100 ) : 100;
		$remaining  = $total_count - $processed;

		// Determine if batch was stopped early due to critical errors.
		$batch_stopped_early = $consecutive_critical_errors >= $max_consecutive_errors;

		$response = array(
			'batch_complete'      => true,
			'all_complete'        => 0 >= $remaining,
			'batch_stopped_early' => $batch_stopped_early,
			'processed'           => $processed,
			'total'               => $total_count,
			'success_count'       => $results['success_count'],
			'error_count'         => $results['error_count'],
			'skipped_count'       => $results['skipped_count'],
			'percentage'          => $percentage,
			'status_message'      => sprintf(
				/* translators: %1$d: processed count, %2$d: total count */
				__( 'Processing image %1$d of %2$d...', 'alt-audit' ),
				$processed,
				$total_count
			),
			'errors'              => $results['errors'],
			'processed_images'    => $results['processed_images'],
			'log_messages'        => $results['log_messages'],
			'next_batch_offset'   => $processed,
		);

		// Add warning if batch had errors but continued.
		if ( $results['error_count'] > 0 && ! $batch_stopped_early ) {
			$response['warning'] = sprintf(
				/* translators: %d: number of errors */
				__( '%d image(s) failed but processing continued.', 'alt-audit' ),
				$results['error_count']
			);
		}

		// Log batch summary.
		if ( defined( 'WP_DEBUG_LOG' ) && WP_DEBUG_LOG ) {
			$summary = array(
				'action'          => 'batch_complete',
				'batch_offset'    => $batch_offset,
				'images_in_batch' => count( $image_ids ),
				'success'         => $results['success_count'],
				'errors'          => $results['error_count'],
				'skipped'         => $results['skipped_count'],
				'stopped_early'   => $batch_stopped_early,
				'remaining'       => $remaining,
			);
			// phpcs:ignore WordPress.PHP.DevelopmentFunctions.error_log_error_log -- Debug logging only when WP_DEBUG_LOG is enabled.
			error_log( 'Alt Audit [BATCH SUMMARY]: ' . wp_json_encode( $summary, JSON_UNESCAPED_SLASHES ) );
		}

		wp_send_json_success( $response );
	}

	/**
	 * Get images for bulk processing
	 *
	 * @param bool $skip_existing  Whether to skip images with existing alt text.
	 * @param bool $overwrite_weak Whether to include images with weak quality alt text.
	 * @param int  $offset         Offset for batch processing.
	 * @param int  $limit          Number of images to retrieve.
	 * @return array Array of attachment IDs.
	 */
	private function get_images_for_bulk_processing( $skip_existing, $overwrite_weak, $offset = 0, $limit = 10 ) {
		// Supported MIME types for AI alt text generation.
		// SVG files are excluded as AI vision APIs cannot analyze vector graphics.
		$supported_mime_types = Altaudit82ai::get_supported_mime_types();

		// If no filters, use simple WP_Query with pagination.
		if ( ! $skip_existing && ! $overwrite_weak ) {
			$images_query = new WP_Query(
				array(
					'post_type'      => 'attachment',
					'post_status'    => 'inherit',
					'post_mime_type' => $supported_mime_types,
					'posts_per_page' => $limit,
					'offset'         => $offset,
					'orderby'        => 'ID',
					'order'          => 'ASC',
					'fields'         => 'ids',
				)
			);

			return array_map( 'intval', $images_query->posts );
		}

		// For filtered queries, get all images and filter in PHP.
		// This is necessary because WP_Query cannot handle complex meta comparisons.
		$images_query = new WP_Query(
			array(
				'post_type'      => 'attachment',
				'post_status'    => 'inherit',
				'post_mime_type' => $supported_mime_types,
				'posts_per_page' => -1,
				'orderby'        => 'ID',
				'order'          => 'ASC',
				'fields'         => 'ids',
			)
		);

		$filtered_ids = array();

		foreach ( $images_query->posts as $image_id ) {
			// Use taxonomy status to match audit page behavior.
			$quality_status = Altaudit82ai_Taxonomy::get_status( $image_id );

			$include = false;

			if ( $overwrite_weak ) {
				// Include images with "weak" OR "missing" taxonomy status.
				// These images need alt text improvement (weak quality or no alt text).
				if ( 'weak' === $quality_status || 'missing' === $quality_status || null === $quality_status ) {
					$include = true;
				}
			} elseif ( $skip_existing ) {
				// Include images with "missing" taxonomy status (no alt text).
				if ( 'missing' === $quality_status || null === $quality_status ) {
					$include = true;
				}
			}

			if ( $include ) {
				$filtered_ids[] = (int) $image_id;
			}
		}

		// Apply offset and limit.
		return array_slice( $filtered_ids, $offset, $limit );
	}

	/**
	 * Get total count of images for bulk processing
	 *
	 * @param bool $skip_existing  Whether to skip images with existing alt text.
	 * @param bool $overwrite_weak Whether to include images with weak quality alt text.
	 * @return int Total count.
	 */
	private function get_total_images_count( $skip_existing, $overwrite_weak ) {
		return $this->count_images_for_bulk_processing( $skip_existing, $overwrite_weak );
	}

	/**
	 * Get detailed image result data for table view
	 *
	 * @param int    $attachment_id   Attachment ID.
	 * @param string $alt_text        Generated alt text.
	 * @param array  $quality_result  Quality assessment result.
	 * @param bool   $success         Whether generation was successful.
	 * @return array Image result data.
	 */
	private function get_image_result_data( $attachment_id, $alt_text, $quality_result, $success ) {
		$attachment    = get_post( $attachment_id );
		$image_url     = wp_get_attachment_image_src( $attachment_id, 'thumbnail' );
		$quality_score = $quality_result['score'] ?? 0;

		return array(
			'attachment_id'  => $attachment_id,
			'filename'       => $attachment ? basename( get_attached_file( $attachment_id ) ) : '',
			'title'          => $attachment ? $attachment->post_title : '',
			'thumbnail'      => $image_url ? $image_url[0] : '',
			'alt_text'       => $alt_text,
			'quality_score'  => $quality_score,
			'quality_status' => Altaudit82ai_Taxonomy::get_status_from_score( $quality_score, empty( trim( $alt_text ) ), true ),
			'success'        => $success,
			'timestamp'      => current_time( 'mysql' ),
		);
	}

	/**
	 * Log image processing start with full details
	 *
	 * @param int $attachment_id Attachment ID.
	 * @return array Log entry for frontend display.
	 */
	private function log_image_processing_start( $attachment_id ) {
		$attachment_id = absint( $attachment_id );
		$image_path    = get_attached_file( $attachment_id );
		$image_url     = wp_get_attachment_url( $attachment_id );
		$filename      = $image_path ? sanitize_file_name( basename( $image_path ) ) : 'unknown';

		$log_data = array(
			'action'        => 'processing_start',
			'attachment_id' => $attachment_id,
			'filename'      => $filename,
			'url'           => $image_url ? esc_url_raw( $image_url ) : '',
		);

		// Add file details if file exists.
		if ( $image_path && file_exists( $image_path ) ) {
			$file_size              = filesize( $image_path );
			$log_data['file_size']  = size_format( $file_size, 2 );
			$log_data['file_bytes'] = absint( $file_size );

			// Get image dimensions and MIME type.
			$image_info = wp_getimagesize( $image_path );
			if ( is_array( $image_info ) ) {
				$log_data['dimensions'] = absint( $image_info[0] ) . 'x' . absint( $image_info[1] );
				$log_data['mime_type']  = isset( $image_info['mime'] ) ? sanitize_mime_type( $image_info['mime'] ) : '';
			}
		}

		// Log to debug.log if enabled.
		if ( defined( 'WP_DEBUG_LOG' ) && WP_DEBUG_LOG ) {
			// phpcs:ignore WordPress.PHP.DevelopmentFunctions.error_log_error_log -- Debug logging only when WP_DEBUG_LOG is enabled.
			error_log( 'Alt Audit [START]: ' . wp_json_encode( $log_data, JSON_UNESCAPED_SLASHES ) );
		}

		return $log_data;
	}

	/**
	 * Log image skipped
	 *
	 * @param int    $attachment_id Attachment ID.
	 * @param string $reason        Skip reason.
	 * @param array  $extra         Extra details.
	 * @return array Log entry for frontend display.
	 */
	private function log_image_skipped( $attachment_id, $reason, $extra = array() ) {
		$attachment_id = absint( $attachment_id );
		$image_path    = get_attached_file( $attachment_id );
		$filename      = $image_path ? sanitize_file_name( basename( $image_path ) ) : 'unknown';

		$log_data = array(
			'action'        => 'skipped',
			'attachment_id' => $attachment_id,
			'filename'      => $filename,
			'reason'        => sanitize_key( $reason ),
		);

		if ( ! empty( $extra ) ) {
			$log_data = array_merge( $log_data, $extra );
		}

		// Log to debug.log if enabled.
		if ( defined( 'WP_DEBUG_LOG' ) && WP_DEBUG_LOG ) {
			// phpcs:ignore WordPress.PHP.DevelopmentFunctions.error_log_error_log -- Debug logging only when WP_DEBUG_LOG is enabled.
			error_log( 'Alt Audit [SKIP]: ' . wp_json_encode( $log_data, JSON_UNESCAPED_SLASHES ) );
		}

		return $log_data;
	}

	/**
	 * Log image processing success
	 *
	 * @param int    $attachment_id Attachment ID.
	 * @param string $alt_text      Generated alt text.
	 * @param int    $quality_score Quality score.
	 * @return array Log entry for frontend display.
	 */
	private function log_image_success( $attachment_id, $alt_text, $quality_score ) {
		$attachment_id = absint( $attachment_id );
		$image_path    = get_attached_file( $attachment_id );
		$filename      = $image_path ? sanitize_file_name( basename( $image_path ) ) : 'unknown';

		$log_data = array(
			'action'        => 'success',
			'attachment_id' => $attachment_id,
			'filename'      => $filename,
			'alt_text'      => sanitize_text_field( $alt_text ),
			'quality_score' => absint( $quality_score ),
			'alt_length'    => strlen( $alt_text ),
		);

		// Log to debug.log if enabled.
		if ( defined( 'WP_DEBUG_LOG' ) && WP_DEBUG_LOG ) {
			// phpcs:ignore WordPress.PHP.DevelopmentFunctions.error_log_error_log -- Debug logging only when WP_DEBUG_LOG is enabled.
			error_log( 'Alt Audit [SUCCESS]: ' . wp_json_encode( $log_data, JSON_UNESCAPED_SLASHES ) );
		}

		return $log_data;
	}

	/**
	 * Log image processing error
	 *
	 * @param int    $attachment_id Attachment ID.
	 * @param string $error_code    Error code.
	 * @param string $error_message Error message.
	 * @return array Log entry for frontend display.
	 */
	private function log_image_error( $attachment_id, $error_code, $error_message ) {
		$attachment_id = absint( $attachment_id );
		$image_path    = get_attached_file( $attachment_id );
		$filename      = $image_path ? sanitize_file_name( basename( $image_path ) ) : 'unknown';
		$image_url     = wp_get_attachment_url( $attachment_id );

		$log_data = array(
			'action'        => 'error',
			'attachment_id' => $attachment_id,
			'filename'      => $filename,
			'url'           => $image_url ? esc_url_raw( $image_url ) : '',
			'error_code'    => sanitize_key( $error_code ),
			'error_message' => sanitize_text_field( $error_message ),
		);

		// Add file details for debugging.
		if ( $image_path && file_exists( $image_path ) ) {
			$log_data['file_size'] = size_format( filesize( $image_path ), 2 );

			$image_info = wp_getimagesize( $image_path );
			if ( is_array( $image_info ) ) {
				$log_data['dimensions'] = absint( $image_info[0] ) . 'x' . absint( $image_info[1] );
				$log_data['mime_type']  = isset( $image_info['mime'] ) ? sanitize_mime_type( $image_info['mime'] ) : '';
			}
		}

		// Log to debug.log if enabled.
		if ( defined( 'WP_DEBUG_LOG' ) && WP_DEBUG_LOG ) {
			// phpcs:ignore WordPress.PHP.DevelopmentFunctions.error_log_error_log -- Debug logging only when WP_DEBUG_LOG is enabled.
			error_log( 'Alt Audit [ERROR]: ' . wp_json_encode( $log_data, JSON_UNESCAPED_SLASHES ) );
		}

		return $log_data;
	}

	/**
	 * Log batch processing stopped
	 *
	 * @param string $reason           Stop reason.
	 * @param int    $consecutive_errors Number of consecutive errors.
	 * @return array Log entry for frontend display.
	 */
	private function log_batch_stopped( $reason, $consecutive_errors ) {
		$log_data = array(
			'action'             => 'batch_stopped',
			'reason'             => sanitize_key( $reason ),
			'consecutive_errors' => absint( $consecutive_errors ),
			'timestamp'          => current_time( 'mysql' ),
		);

		// Log to debug.log if enabled.
		if ( defined( 'WP_DEBUG_LOG' ) && WP_DEBUG_LOG ) {
			// phpcs:ignore WordPress.PHP.DevelopmentFunctions.error_log_error_log -- Debug logging only when WP_DEBUG_LOG is enabled.
			error_log( 'Alt Audit [BATCH STOPPED]: ' . wp_json_encode( $log_data, JSON_UNESCAPED_SLASHES ) );
		}

		return $log_data;
	}

	/**
	 * Add admin notice
	 *
	 * @param string $message Notice message.
	 * @param string $type    Notice type.
	 * @return void
	 */
	private function add_admin_notice( $message, $type = 'success' ) {
		add_settings_error(
			'altaudit82ai_notices',
			'altaudit82ai_notice',
			$message,
			$type
		);
	}

	/**
	 * Get single row HTML for AJAX table updates
	 *
	 * @param int $attachment_id Attachment ID.
	 * @return string Row HTML.
	 */
	private function get_single_row_html( $attachment_id ) {
		// Ensure list table is initialized.
		if ( ! isset( $this->images_list_table ) || ! $this->images_list_table ) {
			$this->prepare_images_list_table();
		}

		// Get single row HTML from list table.
		$row_html = $this->images_list_table->get_single_row_html( $attachment_id );

		return $row_html;
	}

	/**
	 * AJAX handler for single image bulk action
	 *
	 * Processes a single image with the specified bulk action.
	 * Used for sequential processing with progress tracking.
	 *
	 * @since 1.0.0
	 * @return void
	 */
	public function ajax_single_bulk_action() {
		check_ajax_referer( 'altaudit82ai_nonce', 'nonce' );

		if ( ! current_user_can( 'upload_files' ) ) {
			wp_send_json_error( __( 'You do not have permission to perform this action.', 'alt-audit' ) );
		}

		$action        = isset( $_POST['bulk_action'] ) ? sanitize_text_field( wp_unslash( $_POST['bulk_action'] ) ) : '';
		$attachment_id = isset( $_POST['attachment_id'] ) ? intval( wp_unslash( $_POST['attachment_id'] ) ) : 0;

		if ( empty( $action ) || empty( $attachment_id ) ) {
			wp_send_json_error( __( 'Invalid action or attachment ID.', 'alt-audit' ) );
		}

		$result = array();

		switch ( $action ) {
			case 'generate_rule_based':
				$result = $this->process_single_rule_based( $attachment_id );
				break;

			case 'generate_ai':
				$result = $this->process_single_ai( $attachment_id );
				break;

			case 'mark_decorative':
				$result = $this->process_single_decorative( $attachment_id );
				break;

			case 'delete':
				$result = $this->process_single_delete( $attachment_id );
				break;

			default:
				wp_send_json_error( __( 'Invalid bulk action.', 'alt-audit' ) );
		}

		if ( $result['success'] ) {
			wp_send_json_success( $result );
		} else {
			wp_send_json_error( $result['message'] ?? __( 'Operation failed.', 'alt-audit' ) );
		}
	}

	/**
	 * Process single image with rule-based generation
	 *
	 * @param int $attachment_id Attachment ID.
	 * @return array Result array.
	 */
	private function process_single_rule_based( $attachment_id ) {
		$template_service = altaudit82ai()->get_service( 'template_service' );

		if ( ! $template_service ) {
			return array(
				'success' => false,
				'message' => __( 'Template service not available.', 'alt-audit' ),
			);
		}

		$generated_alt = $template_service->generate_from_template( $attachment_id );

		if ( ! empty( $generated_alt ) ) {
			update_post_meta( $attachment_id, '_wp_attachment_image_alt', $generated_alt );
			update_post_meta( $attachment_id, '_altaudit82ai_auto_generated', true );
			update_post_meta( $attachment_id, '_altaudit82ai_generation_method', 'rule_based' );

			// CRITICAL: Clear decorative flag when generating alt text.
			delete_post_meta( $attachment_id, '_altaudit82ai_decorative' );

			// Assess quality.
			$quality_result = $this->quality_service->assess_quality( $generated_alt );

			// Store quality score and taxonomy status.
			$quality_score  = $quality_result['score'] ?? 0;
			$quality_status = Altaudit82ai_Taxonomy::get_status_from_score( $quality_score, false, true );
			update_post_meta( $attachment_id, '_altaudit82ai_quality_score', absint( $quality_score ) );
			Altaudit82ai_Taxonomy::set_status( $attachment_id, sanitize_key( $quality_status ) );

			// Get updated row HTML for table refresh.
			$row_html = $this->get_single_row_html( $attachment_id );

			return array(
				'success'           => true,
				'alt_text'          => $generated_alt,
				'quality_score'     => $quality_result['score'] ?? 0,
				'status'            => $quality_result['status'] ?? 'unknown',
				'is_auto_generated' => true,
				'row_html'          => $row_html,
				'attachment_id'     => $attachment_id,
			);
		}

		return array(
			'success' => false,
			'message' => __( 'Failed to generate alt text.', 'alt-audit' ),
		);
	}

	/**
	 * Process single image with AI generation
	 *
	 * @param int $attachment_id Attachment ID.
	 * @return array Result array.
	 */
	private function process_single_ai( $attachment_id ) {
		$result = $this->api_client->generate_alt_text( $attachment_id );

		// Check if API request failed.
		if ( isset( $result['success'] ) && false === $result['success'] ) {
			return array(
				'success' => false,
				'message' => $result['message'] ?? __( 'API request failed.', 'alt-audit' ),
			);
		}

		if ( ! empty( $result['alt_text'] ) ) {
			update_post_meta( $attachment_id, '_wp_attachment_image_alt', $result['alt_text'] );
			update_post_meta( $attachment_id, '_altaudit82ai_auto_generated', true );
			update_post_meta( $attachment_id, '_altaudit82ai_generation_method', 'ai' );

			// CRITICAL: Clear decorative flag when generating alt text.
			delete_post_meta( $attachment_id, '_altaudit82ai_decorative' );

			// Assess quality.
			$quality_result = $this->quality_service->assess_quality( $result['alt_text'] );

			// Store quality score and taxonomy status.
			$quality_score  = $quality_result['score'] ?? 0;
			$quality_status = Altaudit82ai_Taxonomy::get_status_from_score( $quality_score, false, true );
			update_post_meta( $attachment_id, '_altaudit82ai_quality_score', absint( $quality_score ) );
			Altaudit82ai_Taxonomy::set_status( $attachment_id, sanitize_key( $quality_status ) );

			// Get updated row HTML for table refresh.
			$row_html = $this->get_single_row_html( $attachment_id );

			return array(
				'success'           => true,
				'alt_text'          => $result['alt_text'],
				'quality_score'     => $quality_result['score'] ?? 0,
				'status'            => $quality_result['status'] ?? 'unknown',
				'is_auto_generated' => true,
				'row_html'          => $row_html,
				'attachment_id'     => $attachment_id,
			);
		}

		return array(
			'success' => false,
			'message' => $result['message'] ?? __( 'Failed to generate alt text. The API returned empty result.', 'alt-audit' ),
		);
	}

	/**
	 * Process single image as decorative
	 *
	 * @param int $attachment_id Attachment ID.
	 * @return array Result array.
	 */
	private function process_single_decorative( $attachment_id ) {
		update_post_meta( $attachment_id, '_wp_attachment_image_alt', '' );
		update_post_meta( $attachment_id, '_altaudit82ai_decorative', true );

		// Get updated row HTML for table refresh.
		$row_html = $this->get_single_row_html( $attachment_id );

		return array(
			'success'       => true,
			'alt_text'      => '',
			'quality_score' => 100,
			'status'        => 'decorative',
			'row_html'      => $row_html,
			'attachment_id' => $attachment_id,
		);
	}

	/**
	 * Process single image deletion
	 *
	 * @param int $attachment_id Attachment ID.
	 * @return array Result array.
	 */
	private function process_single_delete( $attachment_id ) {
		if ( wp_delete_attachment( $attachment_id, true ) ) {
			return array(
				'success' => true,
				'deleted' => true,
			);
		}

		return array(
			'success' => false,
			'message' => __( 'Failed to delete attachment.', 'alt-audit' ),
		);
	}

	/**
	 * AJAX handler for refreshing table with new parameters
	 *
	 * Returns updated table HTML for seamless filter/sort/search/pagination updates.
	 *
	 * @since 1.0.0
	 * @return void
	 */
	public function ajax_refresh_table() {
		check_ajax_referer( 'altaudit82ai_nonce', 'nonce' );

		if ( ! current_user_can( 'upload_files' ) ) {
			wp_send_json_error( __( 'You do not have permission to perform this action.', 'alt-audit' ) );
		}

		// Get only the specific parameters needed for list table operations.
		// phpcs:ignore WordPress.Security.ValidatedSanitizedInput.InputNotSanitized -- Sanitized individually below.
		$params = isset( $_POST['params'] ) ? (array) wp_unslash( $_POST['params'] ) : array();

		// Extract and sanitize only the specific parameters we need (not the whole array).
		$sanitized_params = array();

		if ( isset( $params['page'] ) ) {
			$sanitized_params['page'] = sanitize_text_field( $params['page'] );
		}
		if ( isset( $params['paged'] ) ) {
			$sanitized_params['paged'] = absint( $params['paged'] );
		}
		if ( isset( $params['orderby'] ) ) {
			$sanitized_params['orderby'] = sanitize_key( $params['orderby'] );
		}
		if ( isset( $params['order'] ) ) {
			$sanitized_params['order'] = in_array( strtoupper( $params['order'] ), array( 'ASC', 'DESC' ), true ) ? strtoupper( $params['order'] ) : 'DESC';
		}
		if ( isset( $params['s'] ) ) {
			$sanitized_params['s'] = sanitize_text_field( $params['s'] );
		}
		if ( isset( $params['status_filter'] ) ) {
			$sanitized_params['status_filter'] = sanitize_key( $params['status_filter'] );
		}

		// Store original URI.
		// phpcs:ignore WordPress.Security.ValidatedSanitizedInput.InputNotValidated, WordPress.Security.ValidatedSanitizedInput.InputNotSanitized, WordPress.Security.ValidatedSanitizedInput.MissingUnslash -- Server variable, used for restoration only.
		$original_uri = isset( $_SERVER['REQUEST_URI'] ) ? sanitize_text_field( wp_unslash( $_SERVER['REQUEST_URI'] ) ) : '';

		// Fix: Set REQUEST_URI to the audit page URL so generated links point to the correct page.
		// This prevents links from pointing to admin-ajax.php.
		$admin_url_path         = wp_parse_url( admin_url( 'admin.php' ), PHP_URL_PATH );
		$_SERVER['REQUEST_URI'] = add_query_arg(
			array_merge(
				array( 'page' => 'alt-audit-audit' ),
				$sanitized_params
			),
			$admin_url_path
		);

		// Prepare list table with sanitized params (avoids modifying $_GET/$_REQUEST superglobals).
		$this->prepare_images_list_table( $sanitized_params );
		$this->images_list_table->prepare_items();

		// Capture complete form output for seamless replacement.
		ob_start();
		?>
		<!-- Progress Indicator (Hidden by default) -->
		<div class="alt-audit-progress-container" style="display:none;">
			<div class="alt-audit-progress-bar">
				<div class="alt-audit-progress-fill"></div>
			</div>
			<div class="alt-audit-progress-text">
				<span class="progress-message"><?php esc_html_e( 'Processing...', 'alt-audit' ); ?></span>
				<button type="button"
					class="button alt-audit-cancel-operation"><?php esc_html_e( 'Cancel', 'alt-audit' ); ?></button>
			</div>
		</div>

		<!-- Table Form -->
		<form method="get" class="alt-audit-images-form" role="search"
			aria-label="<?php esc_attr_e( 'Image search and filters', 'alt-audit' ); ?>" data-ajax-form="audit">
			<input type="hidden" name="page" value="<?php echo esc_attr( $sanitized_params['page'] ?? 'alt-audit-audit' ); ?>" />

			<?php
			// Search box.
			$this->images_list_table->search_box( __( 'Search images', 'alt-audit' ), 'altaudit82ai-images' );

			// Display the table.
			$this->images_list_table->display();
			?>
		</form>
		<?php
		$html = ob_get_clean();

		// Restore original URI.
		$_SERVER['REQUEST_URI'] = $original_uri;

		wp_send_json_success(
			array(
				'html'   => $html,
				'params' => $sanitized_params,
			)
		);
	}

	/**
	 * Maybe migrate quality scores for existing images
	 *
	 * Runs once to populate quality status and score meta for all images.
	 * This enables efficient SQL filtering for quality-based filters.
	 *
	 * @since  1.0.0
	 * @return void
	 */
	public function maybe_migrate_quality_scores() {
		// Check if migration has already run.
		$migration_done = get_option( 'altaudit82ai_quality_scores_migrated', false );

		if ( $migration_done ) {
			return;
		}

		// Only run for admins to avoid performance issues.
		if ( ! current_user_can( 'manage_options' ) ) {
			return;
		}

		// Don't run on AJAX requests (like heartbeat) to avoid critical errors.
		if ( wp_doing_ajax() || ( defined( 'DOING_AJAX' ) && DOING_AJAX ) ) {
			return;
		}

		// Process in batches of 50 to avoid memory exhaustion/timeouts.
		$batch_size = 50;

		// Get images without quality status taxonomy term using WP_Query.
		$images_query = new WP_Query(
			array(
				'post_type'      => 'attachment',
				'post_status'    => 'inherit',
				'post_mime_type' => Altaudit82ai::get_supported_mime_types(),
				'posts_per_page' => $batch_size,
				'fields'         => 'ids',
				// phpcs:ignore WordPress.DB.SlowDBQuery.slow_db_query_tax_query -- Taxonomy uses indexed term_relationships table.
				'tax_query'      => array(
					array(
						'taxonomy' => Altaudit82ai_Taxonomy::TAXONOMY,
						'operator' => 'NOT EXISTS',
					),
				),
			)
		);

		$images = $images_query->posts;

		// Process each image.
		foreach ( $images as $image_id ) {
			$image_id = absint( $image_id );

			// Skip invalid IDs.
			if ( ! $image_id ) {
				continue;
			}

			$alt_text      = get_post_meta( $image_id, '_wp_attachment_image_alt', true );
			$is_decorative = '1' === get_post_meta( $image_id, '_altaudit82ai_decorative', true );

			// Determine status and quality.
			if ( ! empty( $alt_text ) ) {
				// Has alt text - assess quality.
				if ( $this->quality_service ) {
					$quality_result = $this->quality_service->assess_quality( $alt_text );
					$status         = $quality_result['status'] ?? 'unknown';
					$quality_score  = $quality_result['score'] ?? 0;
				} else {
					// Fallback quality assessment.
					$length = strlen( $alt_text );
					if ( $length < 10 ) {
						$status        = 'weak';
						$quality_score = 30;
					} elseif ( $length < 50 ) {
						$status        = 'good';
						$quality_score = 70;
					} else {
						$status        = 'excellent';
						$quality_score = 90;
					}
				}
			} elseif ( $is_decorative ) {
				$status        = 'decorative';
				$quality_score = 100;
			} else {
				$status        = 'missing';
				$quality_score = 0;
			}

			// Store quality status as taxonomy term and score as meta.
			Altaudit82ai_Taxonomy::set_status( $image_id, sanitize_key( $status ) );
			update_post_meta( $image_id, '_altaudit82ai_quality_score', absint( $quality_score ) );
		}

		// If fewer items than batch size were returned, we are done.
		if ( count( $images ) < $batch_size ) {
			update_option( 'altaudit82ai_quality_scores_migrated', true );
		}
	}

	/**
	 * AJAX handler for scanning all images
	 *
	 * Scans all images and updates their quality scores, then returns updated statistics.
	 *
	 * @since  1.0.0
	 * @return void
	 */
	public function ajax_scan_all_images() {
		// Verify nonce for security.
		check_ajax_referer( 'altaudit82ai_scan_all', 'nonce' );

		// Check user capabilities.
		if ( ! current_user_can( 'manage_options' ) ) {
			wp_send_json_error(
				array(
					'message' => esc_html__( 'You do not have permission to perform this action.', 'alt-audit' ),
				)
			);
		}

		// Get batch size and offset from request.
		$batch_size = isset( $_POST['batch_size'] ) ? absint( $_POST['batch_size'] ) : 20;
		$offset     = isset( $_POST['offset'] ) ? absint( $_POST['offset'] ) : 0;

		// Get all image attachments.
		$images_query = new WP_Query(
			array(
				'post_type'      => 'attachment',
				'post_status'    => 'inherit',
				'post_mime_type' => Altaudit82ai::get_supported_mime_types(),
				'posts_per_page' => $batch_size,
				'offset'         => $offset,
				'fields'         => 'ids',
				'no_found_rows'  => false,
			)
		);

		$total_images    = $images_query->found_posts;
		$processed_count = 0;
		$quality_service = $this->quality_service ?? new Altaudit82ai_Quality_Service();

		// Process each image in this batch.
		foreach ( $images_query->posts as $image_id ) {
			$alt_text = get_post_meta( $image_id, '_wp_attachment_image_alt', true );

			// Skip if already has quality status and it's recent.
			$existing_status = Altaudit82ai_Taxonomy::get_status( $image_id );

			// Calculate quality score using quality service.
			if ( $quality_service ) {
				$quality_data = $quality_service->assess_quality( $alt_text, array( 'attachment_id' => $image_id ) );
				if ( ! empty( $quality_data ) && isset( $quality_data['status'], $quality_data['score'] ) ) {
					// Store the quality status and score.
					Altaudit82ai_Taxonomy::set_status( $image_id, sanitize_key( wp_slash( $quality_data['status'] ) ) );
					update_post_meta( $image_id, '_altaudit82ai_quality_score', absint( $quality_data['score'] ) );
				}
			} else {
				// Fallback to simple scoring if quality service not available.
				$is_decorative = '1' === get_post_meta( $image_id, '_altaudit82ai_decorative', true );

				if ( $is_decorative ) {
					Altaudit82ai_Taxonomy::set_status( $image_id, sanitize_key( wp_slash( 'decorative' ) ) );
					update_post_meta( $image_id, '_altaudit82ai_quality_score', 100 );
				} elseif ( empty( $alt_text ) ) {
					Altaudit82ai_Taxonomy::set_status( $image_id, sanitize_key( wp_slash( 'missing' ) ) );
					update_post_meta( $image_id, '_altaudit82ai_quality_score', 0 );
				} else {
					$length = strlen( $alt_text );
					if ( $length < 20 ) {
						Altaudit82ai_Taxonomy::set_status( $image_id, sanitize_key( wp_slash( 'weak' ) ) );
						update_post_meta( $image_id, '_altaudit82ai_quality_score', 30 );
					} elseif ( $length < 100 ) {
						Altaudit82ai_Taxonomy::set_status( $image_id, sanitize_key( wp_slash( 'good' ) ) );
						update_post_meta( $image_id, '_altaudit82ai_quality_score', 70 );
					} else {
						Altaudit82ai_Taxonomy::set_status( $image_id, sanitize_key( wp_slash( 'excellent' ) ) );
						update_post_meta( $image_id, '_altaudit82ai_quality_score', 90 );
					}
				}
			}

			++$processed_count;
		}

		// Calculate progress.
		$total_processed = $offset + $processed_count;
		$has_more        = $total_processed < $total_images;
		$progress        = $total_images > 0 ? round( ( $total_processed / $total_images ) * 100 ) : 100;

		// Get updated statistics if this is the final batch.
		$updated_stats = null;
		if ( ! $has_more ) {
			// Invalidate statistics cache to ensure fresh data.
			$this->statistics_service->invalidate_cache();
			$updated_stats = $this->statistics_service->get_statistics( true );
			// Update last scan time.
			update_option( 'altaudit82ai_last_scan_time', time() );
		}

		wp_send_json_success(
			array(
				'processed'  => $total_processed,
				'total'      => $total_images,
				'offset'     => $total_processed,
				'progress'   => $progress,
				'has_more'   => $has_more,
				'statistics' => $updated_stats,
				'message'    => sprintf(
					/* translators: 1: Number of scanned images, 2: Total images */
					esc_html__( 'Scanned %1$d images from %2$d total', 'alt-audit' ),
					$total_processed,
					$total_images
				),
			)
		);
	}

	/**
	 * AJAX handler for dismissing admin notices
	 *
	 * @since  1.0.0
	 * @return void
	 */
	public function ajax_dismiss_notice() {
		// Verify nonce for security.
		check_ajax_referer( 'altaudit82ai_dismiss_notice', 'nonce' );

		// Check user capabilities.
		if ( ! current_user_can( 'manage_options' ) ) {
			wp_send_json_error(
				array(
					'message' => esc_html__( 'You do not have permission to perform this action.', 'alt-audit' ),
				)
			);
		}

		// Get the notice type to dismiss.
		$notice_type = isset( $_POST['notice_type'] ) ? sanitize_key( $_POST['notice_type'] ) : '';

		if ( 'altaudit82ai-scan-reminder' === $notice_type ) {
			update_option( 'altaudit82ai_scan_reminder_dismissed', true );
			wp_send_json_success(
				array(
					'message' => esc_html__( 'Notice dismissed.', 'alt-audit' ),
				)
			);
		}

		wp_send_json_error(
			array(
				'message' => esc_html__( 'Invalid notice type.', 'alt-audit' ),
			)
		);
	}

	/**
	 * Show scan reminder notice
	 *
	 * Displays a notice prompting users to scan their media library for the first time.
	 * Only shown if:
	 * - User hasn't dismissed it
	 * - Last scan time is not set (never scanned)
	 * - User has manage_options capability
	 *
	 * @since  1.0.0
	 * @return void
	 */
	private function show_scan_reminder_notice() {
		// Only show to admins.
		if ( ! current_user_can( 'manage_options' ) ) {
			return;
		}

		// Check if notice was dismissed.
		$dismissed = get_option( 'altaudit82ai_scan_reminder_dismissed', false );
		if ( $dismissed ) {
			return;
		}

		// Check if initial scan has been completed.
		$last_scan_time = get_option( 'altaudit82ai_last_scan_time', false );
		if ( $last_scan_time ) {
			return;
		}

		// Get total image count.
		$images_query = new WP_Query(
			array(
				'post_type'      => 'attachment',
				'post_status'    => 'inherit',
				'post_mime_type' => Altaudit82ai::get_supported_mime_types(),
				'posts_per_page' => -1,
				'fields'         => 'ids',
				'no_found_rows'  => false,
			)
		);

		$total_images = $images_query->found_posts;

		// Don't show if no images exist.
		if ( 0 === $total_images ) {
			return;
		}

		// Display the notice.
		?>
		<div class="notice notice-info is-dismissible altaudit82ai-scan-reminder" data-notice-type="altaudit82ai-scan-reminder">
			<p>
				<strong><?php esc_html_e( 'Alt Audit Setup', 'alt-audit' ); ?></strong>
			</p>
			<p>
				<?php
				printf(
					/* translators: %d: number of images */
					esc_html__( 'Welcome to Alt Audit! You have %d images in your media library. Scan them now to analyze their accessibility and get insights.', 'alt-audit' ),
					esc_html( number_format_i18n( $total_images ) )
				);
				?>
			</p>
			<p>
				<a href="<?php echo esc_url( admin_url( 'admin.php?page=alt-audit-audit' ) ); ?>" class="button button-primary">
					<?php esc_html_e( 'Scan Images Now', 'alt-audit' ); ?>
				</a>
				<button type="button" class="button button-secondary altaudit82ai-dismiss-scan-reminder">
					<?php esc_html_e( 'Remind Me Later', 'alt-audit' ); ?>
				</button>
			</p>
		</div>
		<?php
	}
}
