<?php
/**
 * Dependency Injection Container
 *
 * @package AltAudit
 * @since 1.0.0
 */

// Prevent direct access.
if ( ! defined( 'ABSPATH' ) ) {
	exit;
}

/**
 * Dependency Injection Container class
 *
 * Implements a simple service container for managing dependencies
 * and providing dependency injection throughout the plugin.
 *
 * @since 1.0.0
 */
class Altaudit82ai_Container {

	/**
	 * Registered services
	 *
	 * @var array
	 */
	private $services = array();

	/**
	 * Service instances
	 *
	 * @var array
	 */
	private $instances = array();

	/**
	 * Service factories
	 *
	 * @var array
	 */
	private $factories = array();

	/**
	 * Singleton services
	 *
	 * @var array
	 */
	private $singletons = array();

	/**
	 * Register a service
	 *
	 * @param string $name     Service name.
	 * @param mixed  $concrete Service class name, instance, or callback.
	 * @param bool   $singleton Whether to treat as singleton.
	 * @return void
	 */
	public function register( $name, $concrete, $singleton = true ) {
		$this->services[ $name ] = $concrete;

		if ( $singleton ) {
			$this->singletons[ $name ] = true;
		}
	}

	/**
	 * Register a factory service
	 *
	 * @param string   $name     Service name.
	 * @param callable $factory  Factory callback.
	 * @return void
	 */
	public function factory( $name, $factory ) {
		$this->factories[ $name ] = $factory;
	}

	/**
	 * Get a service instance
	 *
	 * @param string $name Service name.
	 * @return mixed Service instance.
	 * @throws Exception If service not found.
	 */
	public function get( $name ) {
		// Return existing singleton instance.
		if ( isset( $this->instances[ $name ] ) && isset( $this->singletons[ $name ] ) ) {
			return $this->instances[ $name ];
		}

		// Check for factory.
		if ( isset( $this->factories[ $name ] ) ) {
			$instance = call_user_func( $this->factories[ $name ], $this );

			if ( isset( $this->singletons[ $name ] ) ) {
				$this->instances[ $name ] = $instance;
			}

			return $instance;
		}

		// Check for registered service.
		if ( ! isset( $this->services[ $name ] ) ) {
			// phpcs:ignore WordPress.Security.EscapeOutput.ExceptionNotEscaped -- Exception messages are not output to browser.
			throw new Exception( "Service '{$name}' not found in container." );
		}

		$concrete = $this->services[ $name ];

		// If it's already an instance, return it.
		if ( is_object( $concrete ) ) {
			return $concrete;
		}

		// If it's a class name, instantiate it.
		if ( is_string( $concrete ) && class_exists( $concrete ) ) {
			$instance = $this->resolve_class( $concrete );

			if ( isset( $this->singletons[ $name ] ) ) {
				$this->instances[ $name ] = $instance;
			}

			return $instance;
		}

		// If it's a callback, execute it.
		if ( is_callable( $concrete ) ) {
			$instance = call_user_func( $concrete, $this );

			if ( isset( $this->singletons[ $name ] ) ) {
				$this->instances[ $name ] = $instance;
			}

			return $instance;
		}

		// phpcs:ignore WordPress.Security.EscapeOutput.ExceptionNotEscaped -- Exception messages are not output to browser.
		throw new Exception( "Unable to resolve service '{$name}'." );
	}

	/**
	 * Check if service exists
	 *
	 * @param string $name Service name.
	 * @return bool
	 */
	public function has( $name ) {
		return isset( $this->services[ $name ] ) || isset( $this->factories[ $name ] );
	}

	/**
	 * Resolve class with dependency injection
	 *
	 * @param string $class_name Class name to resolve.
	 * @return object Class instance.
	 * @throws Exception If class cannot be resolved.
	 */
	private function resolve_class( $class_name ) {
		if ( ! class_exists( $class_name ) ) {
			// phpcs:ignore WordPress.Security.EscapeOutput.ExceptionNotEscaped -- Exception messages are not output to browser.
			throw new Exception( "Class '{$class_name}' does not exist." );
		}

		$reflection = new ReflectionClass( $class_name );

		// If no constructor, just instantiate.
		if ( ! $reflection->hasMethod( '__construct' ) ) {
			return new $class_name();
		}

		$constructor = $reflection->getMethod( '__construct' );
		$parameters  = $constructor->getParameters();

		// If no parameters, just instantiate.
		if ( empty( $parameters ) ) {
			return new $class_name();
		}

		$dependencies = array();

		foreach ( $parameters as $parameter ) {
			$dependency     = $this->resolve_parameter( $parameter );
			$dependencies[] = $dependency;
		}

		return $reflection->newInstanceArgs( $dependencies );
	}

	/**
	 * Resolve constructor parameter
	 *
	 * @param ReflectionParameter $parameter Parameter to resolve.
	 * @return mixed Resolved parameter value.
	 * @throws Exception If parameter cannot be resolved.
	 */
	private function resolve_parameter( ReflectionParameter $parameter ) {
		// Check for type hint.
		$type = $parameter->getType();

		if ( $type && ! $type->isBuiltin() ) {
			$class_name = $type->getName();

			// Try to get from container first.
			$service_name = $this->class_name_to_service_name( $class_name );
			if ( $this->has( $service_name ) ) {
				return $this->get( $service_name );
			}

			// Try to resolve class directly.
			return $this->resolve_class( $class_name );
		}

		// Check for default value.
		if ( $parameter->isDefaultValueAvailable() ) {
			return $parameter->getDefaultValue();
		}

		// Check if parameter is optional.
		if ( $parameter->isOptional() ) {
			return null;
		}

		// phpcs:ignore WordPress.Security.EscapeOutput.ExceptionNotEscaped -- Exception messages are not output to browser.
		throw new Exception( "Unable to resolve parameter '{$parameter->getName()}'." );
	}

	/**
	 * Convert class name to service name
	 *
	 * @param string $class_name Class name.
	 * @return string Service name.
	 */
	private function class_name_to_service_name( $class_name ) {
		// Remove Altaudit82ai_ prefix and convert to lowercase with underscores.
		$service_name = str_replace( 'Altaudit82ai_', '', $class_name );
		$service_name = strtolower( str_replace( '_', '_', $service_name ) );

		return $service_name;
	}

	/**
	 * Set service instance
	 *
	 * @param string $name     Service name.
	 * @param mixed  $instance Service instance.
	 * @return void
	 */
	public function set( $name, $instance ) {
		$this->instances[ $name ] = $instance;
		$this->services[ $name ]  = $instance;
	}

	/**
	 * Remove service
	 *
	 * @param string $name Service name.
	 * @return void
	 */
	public function remove( $name ) {
		unset( $this->services[ $name ] );
		unset( $this->instances[ $name ] );
		unset( $this->factories[ $name ] );
		unset( $this->singletons[ $name ] );
	}

	/**
	 * Get all registered services
	 *
	 * @return array
	 */
	public function get_services() {
		return array_keys( $this->services );
	}

	/**
	 * Clear all services
	 *
	 * @return void
	 */
	public function clear() {
		$this->services   = array();
		$this->instances  = array();
		$this->factories  = array();
		$this->singletons = array();
	}
}
