/**
 * Utility Functions Module
 *
 * Shared utility functions used across the Alt Audit admin interface.
 * Provides helper methods for UI feedback, animations, and common operations.
 *
 * @package AltAudit
 * @since 1.0.0
 */

export const Utils = {
    /**
     * Show admin notice
     *
     * @param {string} message - The message to display
     * @param {string} type - Notice type: 'info', 'success', 'warning', 'error'
     */
    showNotice: function(message, type) {
        type = type || 'info';

        const $notice = jQuery('<div class="notice notice-' + type + ' is-dismissible"><p>' + message + '</p></div>');

        jQuery('.wrap h1').first().after($notice);

        // Auto-dismiss after 5 seconds
        setTimeout(function() {
            $notice.fadeOut(function() {
                jQuery(this).remove();
            });
        }, 5000);

        // Manual dismiss
        $notice.on('click', '.notice-dismiss', function() {
            $notice.fadeOut(function() {
                jQuery(this).remove();
            });
        });
    },

    /**
     * Debounce function to limit rate of function calls
     *
     * @param {Function} func - Function to debounce
     * @param {number} wait - Wait time in milliseconds
     * @param {boolean} immediate - Execute on leading edge instead of trailing
     * @returns {Function} Debounced function
     */
    debounce: function(func, wait, immediate) {
        let timeout;
        return function() {
            const context = this, args = arguments;
            const later = function() {
                timeout = null;
                if (!immediate) func.apply(context, args);
            };
            const callNow = immediate && !timeout;
            clearTimeout(timeout);
            timeout = setTimeout(later, wait);
            if (callNow) func.apply(context, args);
        };
    },

    /**
     * Animate number count-up effect
     *
     * @param {jQuery} $element - jQuery element to animate
     * @param {number} targetNumber - Target number to count to
     */
    animateNumber: function($element, targetNumber) {
        let currentNumber = 0;
        const duration = 1000;
        const steps = 30;
        const stepTime = duration / steps;
        const increment = targetNumber / steps;

        const numberTimer = setInterval(function() {
            currentNumber += increment;

            if (currentNumber >= targetNumber) {
                currentNumber = targetNumber;
                clearInterval(numberTimer);
            }

            $element.text(Math.round(currentNumber).toLocaleString());
        }, stepTime);
    },

    /**
     * Log a message to the operation log
     *
     * @param {string} message - The message to log
     * @param {string} type - Log type: 'info', 'success', 'warning', 'error'
     */
    logMessage: function(message, type) {
        type = type || 'info';
        const timestamp = new Date().toLocaleTimeString();
        const icon = type === 'error' ? '❌' : type === 'warning' ? '⚠️' : type === 'success' ? '✅' : 'ℹ️';
        const logEntry = jQuery('<div class="log-entry log-' + type + '"></div>')
            .html('<span class="log-time">[' + timestamp + ']</span> ' + icon + ' ' + message);

        jQuery('#log-content').append(logEntry);

        // Auto-scroll to bottom
        const logContainer = jQuery('#log-content');
        logContainer.scrollTop(logContainer[0].scrollHeight);
    }
};
