/**
 * Settings Module
 *
 * Handles settings page functionality including API key toggle,
 * quality threshold slider, and form validation.
 *
 * @package AltAudit
 * @since 1.0.0
 */

import { Utils } from './utils.js';

export const Settings = {
    /**
     * Initialize settings functionality
     */
    init: function() {
        this.initApiKeyToggle();
        this.initQualityThresholdSlider();
        this.initTestConnection();
        this.initAltTextBuilder();
    },

    /**
     * Initialize API key toggle (show/hide password)
     */
    initApiKeyToggle: function() {
        const $apiKeyField = jQuery('#api_key');
        if (!$apiKeyField.length) return;

        const $toggleButton = jQuery('<button type="button" class="button">Show</button>');
        $apiKeyField.after($toggleButton);

        $toggleButton.on('click', function() {
            if ($apiKeyField.attr('type') === 'password') {
                $apiKeyField.attr('type', 'text');
                $toggleButton.text('Hide');
            } else {
                $apiKeyField.attr('type', 'password');
                $toggleButton.text('Show');
            }
        });
    },

    /**
     * Initialize quality threshold slider
     */
    initQualityThresholdSlider: function() {
        const $slider = jQuery('#quality_threshold');
        if (!$slider.length) return;

        const $display = jQuery('<span class="quality-threshold-display"></span>');
        $slider.after($display);

        function updateDisplay() {
            $display.text($slider.val() + '%');
        }

        $slider.on('input', updateDisplay);
        updateDisplay();
    },

    /**
     * Validate settings form
     *
     * @param {Event} e - Form submit event
     * @returns {boolean} Whether form is valid
     */
    validateSettings: function(e) {
        const $form = jQuery(this);
        const $apiKey = $form.find('#api_key');

        if ($apiKey.length && $apiKey.val().length > 0 && $apiKey.val().length < 20) {
            e.preventDefault();
            Utils.showNotice('API key appears to be invalid. Please check and try again.', 'error');
            return false;
        }

        return true;
    },

    /**
     * Initialize Test Connection button
     * Moved from inline script in includes/Views/Admin/settings.php:122
     */
    initTestConnection: function() {
        jQuery(document).on('click', '.test-connection', function(e) {
            e.preventDefault();
            
            const button = jQuery(this);
            const nonce = button.data('nonce');
            const adminUrl = button.data('admin-url');

            // Create form and submit
            const form = jQuery('<form>', {
                method: 'POST',
                action: adminUrl
            });

            form.append(jQuery('<input>', {
                type: 'hidden',
                name: 'action',
                value: 'altaudit82ai_test_api'
            }));

            form.append(jQuery('<input>', {
                type: 'hidden',
                name: '_wpnonce',
                value: nonce
            }));

            jQuery('body').append(form);
            form.submit();
        });
    },

    /**
     * Initialize Alt Text Builder
     * Moved from inline script in includes/Views/Admin/settings.php:406
     */
    initAltTextBuilder: function() {
        const self = this;

        // Initialize all editors on page load
        jQuery(document).ready(function() {
            ['home_images_alt', 'pages_images_alt', 'post_images_alt', 'product_images_alt', 'cpt_images_alt'].forEach(function(field) {
                self.renderTokens(field);
            });
        });

        // Add token button handlers  
        jQuery(document).on('click', '.token-btn', function() {
            const fieldName = jQuery(this).closest('.alt-builder-section').find('.pattern-editor').attr('id').replace('editor-', '');
            const tokenValue = jQuery(this).data('value');
            self.addToken(fieldName, tokenValue);
        });

        // Remove token handlers (delegated)
        jQuery(document).on('click', '.remove-token', function() {
            const chip = jQuery(this).parent();
            const editor = chip.parent();
            const fieldName = editor.attr('id').replace('editor-', '');
            const index = chip.index();
            self.removeToken(fieldName, index);
        });
    },

    /**
     * Add token to Alt Text Builder
     *
     * @param {string} fieldName - Field name
     * @param {string} tokenValue - Token value
     */
    addToken: function(fieldName, tokenValue) {
        const select = document.getElementById('input-' + fieldName);
        if (!select) return;

        const option = new Option(tokenValue, tokenValue, true, true);
        select.add(option);
        this.renderTokens(fieldName);
    },

    /**
     * Remove token from Alt Text Builder
     *
     * @param {string} fieldName - Field name
     * @param {number} index - Token index
     */
    removeToken: function(fieldName, index) {
        const select = document.getElementById('input-' + fieldName);
        if (!select) return;

        select.remove(index);
        this.renderTokens(fieldName);
    },

    /**
     * Render tokens in Alt Text Builder
     *
     * @param {string} fieldName - Field name
     */
    renderTokens: function(fieldName) {
        const select = document.getElementById('input-' + fieldName);
        const editor = document.getElementById('editor-' + fieldName);

        if (!select || !editor) return;

        editor.innerHTML = '';

        if (select.options.length === 0) {
            const emptyText = typeof altaudit82aiAdmin !== 'undefined' && altaudit82aiAdmin.strings && altaudit82aiAdmin.strings.buildPattern 
                ? altaudit82aiAdmin.strings.buildPattern 
                : 'Click tokens below to build your pattern...';
            editor.innerHTML = '<span class="builder-empty-state">' + emptyText + '</span>';
            return;
        }

        for (let i = 0; i < select.options.length; i++) {
            const val = select.options[i].value;
            const chip = document.createElement('div');
            chip.className = 'token-chip';
            chip.innerHTML = val + '<span class="remove-token">×</span>';
            editor.appendChild(chip);
        }
    }
};
