/**
 * Quality Checker Module
 *
 * Handles quality assessment and indicator updates for alt text.
 *
 * @package AltAudit
 * @since 1.0.0
 */

import { Utils } from './utils.js';

export const QualityChecker = {
    /**
     * Initialize quality checkers
     */
    init: function() {
        jQuery('.alt-audit-quality-indicator').each(function() {
            QualityChecker.updateQualityIndicator(jQuery(this));
        });
    },

    /**
     * Check alt text quality via AJAX
     *
     * @param {Event} e - Click event
     */
    checkQuality: function(e) {
        e.preventDefault();

        const $button = jQuery(this);
        const $container = $button.closest('.alt-audit-container');
        const $altTextInput = $container.find('.alt-text-input');
        const altText = $altTextInput.val();

        if (!altText.trim()) {
            Utils.showNotice('Please enter alt text to check', 'warning');
            return;
        }

        const originalText = $button.text();
        $button.text(altaudit82aiAdmin.strings.processing).prop('disabled', true);

        jQuery.ajax({
            url: altaudit82aiAdmin.ajaxUrl,
            type: 'POST',
            data: {
                action: 'altaudit82ai_check_quality',
                nonce: altaudit82aiAdmin.nonce,
                alt_text: altText
            },
            success: function(response) {
                if (response.success) {
                    const $qualityIndicator = $container.find('.alt-audit-quality-indicator');
                    QualityChecker.updateQualityDisplay($qualityIndicator, response.data);
                } else {
                    Utils.showNotice(response.data, 'error');
                }
            },
            error: function() {
                Utils.showNotice(altaudit82aiAdmin.strings.error, 'error');
            },
            complete: function() {
                $button.text(originalText).prop('disabled', false);
            }
        });
    },

    /**
     * Update quality display
     *
     * @param {jQuery} $indicator - Quality indicator element
     * @param {Object} quality - Quality data object
     */
    updateQualityDisplay: function($indicator, quality) {
        if (!$indicator.length || !quality) return;

        $indicator.removeClass('missing decorative weak good excellent')
                 .addClass(quality.status);

        const scoreText = quality.score ? ' (' + quality.score + '%)' : '';
        $indicator.find('.quality-text').text(quality.status.charAt(0).toUpperCase() + quality.status.slice(1) + scoreText);

        // Update tooltip if available
        if (quality.message) {
            $indicator.attr('title', quality.message);
        }
    },

    /**
     * Update quality indicator via AJAX
     *
     * @param {jQuery} $indicator - Quality indicator element
     */
    updateQualityIndicator: function($indicator) {
        const altText = $indicator.data('alt-text');
        if (!altText) return;

        // Trigger quality check
        jQuery.ajax({
            url: altaudit82aiAdmin.ajaxUrl,
            type: 'POST',
            data: {
                action: 'altaudit82ai_check_quality',
                nonce: altaudit82aiAdmin.nonce,
                alt_text: altText
            },
            success: function(response) {
                if (response.success) {
                    QualityChecker.updateQualityDisplay($indicator, response.data);
                }
            }
        });
    }
};
