/**
 * Inline Editor Module
 *
 * Handles inline editing functionality for the audit page list table.
 *
 * @package AltAudit
 * @since 1.0.0
 */

import { Utils } from './utils.js';

export const InlineEditor = {
    currentEditRow: null,
    currentAttachmentId: null,

    /**
     * Initialize inline editing for audit page
     */
    init: function() {
        const self = this;

        // Handle inline edit button click
        jQuery(document).on('click', '.edit-alt-text', function(e) {
            e.preventDefault();

            const $row = jQuery(this).closest('tr');
            const $altTextCell = jQuery(this).closest('.alt-audit-alt-text');
            const attachmentId = $altTextCell.attr('data-attachment-id');
            const fullAltText = $altTextCell.attr('data-full-alt-text');
            const displayText = $row.find('.alt-text-display').text();

            // Use full alt text if available, otherwise fall back to display text
            const currentAltText = fullAltText || displayText;

            console.log('Edit clicked - Full:', fullAltText, 'Display:', displayText, 'Using:', currentAltText);

            // Clone and show inline edit form
            const $editForm = jQuery('#alt-audit-inline-edit').clone().show();

            // Replace row with edit form
            const colspan = $row.find('td').length;
            const $editRow = jQuery('<tr class="inline-edit-row"><td colspan="' + colspan + '"></td></tr>');
            $editRow.find('td').html($editForm.html());

            $row.hide();
            $row.after($editRow);

            // Set the value AFTER inserting into DOM
            $editRow.find('textarea').val(currentAltText);

            self.currentEditRow = $editRow;
            self.currentAttachmentId = attachmentId;

            // Focus textarea
            $editRow.find('textarea').focus();
        });

        // Handle save inline edit
        jQuery(document).on('click', '#save-inline-edit', function() {
            self.saveInlineEdit(jQuery(this));
        });

        // Handle cancel inline edit
        jQuery(document).on('click', '#cancel-inline-edit', function() {
            self.cancelInlineEdit();
        });

        // Handle bulk actions confirmation
        jQuery(document).on('change', 'input[name="action"], input[name="action2"]', function() {
            const action = jQuery(this).val();
            if (action === 'delete') {
                // Add confirmation for delete action
                jQuery(this).closest('form').on('submit', function(e) {
                    if (!confirm(altaudit82aiAdmin.strings.confirm || 'Are you sure?')) {
                        e.preventDefault();
                    }
                });
            }
        });
    },

    /**
     * Save inline edit
     *
     * @param {jQuery} $button - Save button element
     */
    saveInlineEdit: function($button) {
        const self = this;
        const newAltText = $button.closest('.inline-edit-wrapper').find('textarea').val();
        const $spinner = $button.siblings('.spinner');

        $spinner.addClass('is-active');
        $button.prop('disabled', true);

        jQuery.post(altaudit82aiAdmin.ajaxUrl, {
            action: 'altaudit82ai_inline_edit_alt_text',
            nonce: altaudit82aiAdmin.nonce,
            attachment_id: self.currentAttachmentId,
            alt_text: newAltText
        }, function(response) {
            $spinner.removeClass('is-active');

            if (response.success) {
                // Update the row
                const $originalRow = self.currentEditRow.prev();
                const displayText = newAltText.length > 100 ? newAltText.substring(0, 100) + '...' : newAltText;
                $originalRow.find('.alt-text-display').text(displayText);
                $originalRow.find('.alt-audit-alt-text').attr('data-full-alt-text', newAltText);

                // Update quality score if provided
                if (response.data.quality_score !== undefined) {
                    const $scoreElement = $originalRow.find('.alt-audit-score .score-number');
                    const $scoreFill = $originalRow.find('.score-fill');
                    const $statusElement = $originalRow.find('.alt-audit-status');

                    $scoreElement.text(response.data.quality_score);
                    $scoreFill.css('width', response.data.quality_score + '%');

                    if (response.data.status) {
                        $statusElement.removeClass().addClass('alt-audit-status alt-audit-status-' + response.data.status);
                        $statusElement.text(response.data.status.charAt(0).toUpperCase() + response.data.status.slice(1));
                    }
                }

                // Clean up
                self.currentEditRow.remove();
                $originalRow.show();
                self.currentEditRow = null;
                self.currentAttachmentId = null;

                // Show success message
                Utils.showNotice(altaudit82aiAdmin.strings.success || 'Alt text updated successfully', 'success');

                // Refresh statistics (with small delay to ensure DB is updated)
                if (jQuery('.alt-audit-statistics-section').length > 0) {
                    setTimeout(function() {
                        // Import AuditPage dynamically to avoid circular dependency
                        import('./audit-page.js').then(module => {
                            module.AuditPage.refreshAuditStatistics();
                        });
                    }, 300);
                }
            } else {
                alert(response.data || altaudit82aiAdmin.strings.error);
                $button.prop('disabled', false);
            }
        }).fail(function() {
            $spinner.removeClass('is-active');
            $button.prop('disabled', false);
            alert(altaudit82aiAdmin.strings.error || 'An error occurred');
        });
    },

    /**
     * Cancel inline edit
     */
    cancelInlineEdit: function() {
        if (this.currentEditRow) {
            this.currentEditRow.prev().show();
            this.currentEditRow.remove();
            this.currentEditRow = null;
            this.currentAttachmentId = null;
        }
    }
};
