/**
 * Help Page Module
 * Moved from inline <script> tag in includes/Views/Admin/help.php
 *
 * Handles help page functionality including navigation, accordion, and system info copy.
 *
 * @package AltAudit
 * @since 1.0.1
 */

export const HelpPage = {
    /**
     * Initialize help page functionality
     */
    init: function() {
        if (jQuery('.alt-audit-help').length === 0) {
            return;
        }

        this.initNavigation();
        this.initAccordion();
        this.initCopySystemInfo();
    },

    /**
     * Initialize navigation tabs
     */
    initNavigation: function() {
        const navLinks = document.querySelectorAll('.help-nav .nav-link');
        const sections = document.querySelectorAll('.help-section');

        navLinks.forEach(function(link) {
            link.addEventListener('click', function() {
                const targetSection = this.getAttribute('data-section');

                // Update navigation
                navLinks.forEach(function(nav) {
                    nav.classList.remove('active');
                    nav.setAttribute('aria-selected', 'false');
                });
                this.classList.add('active');
                this.setAttribute('aria-selected', 'true');

                // Update sections
                sections.forEach(function(section) {
                    section.classList.remove('active');
                });
                const target = document.getElementById(targetSection);
                if (target) {
                    target.classList.add('active');
                }
            });
        });
    },

    /**
     * Initialize accordion functionality
     */
    initAccordion: function() {
        const accordionItems = document.querySelectorAll('.troubleshooting-item');

        accordionItems.forEach(function(item) {
            const header = item.querySelector('.trouble-header');
            const button = item.querySelector('.toggle-button');
            const content = item.querySelector('.trouble-content');

            if (!header || !button || !content) {
                return;
            }

            header.addEventListener('click', function() {
                const isExpanded = button.getAttribute('aria-expanded') === 'true';

                // Close all items
                accordionItems.forEach(function(otherItem) {
                    const otherButton = otherItem.querySelector('.toggle-button');
                    const otherContent = otherItem.querySelector('.trouble-content');
                    if (otherButton && otherContent) {
                        otherButton.setAttribute('aria-expanded', 'false');
                        otherContent.classList.remove('open');
                        otherItem.classList.remove('open');
                    }
                });

                // Toggle current item
                if (!isExpanded) {
                    button.setAttribute('aria-expanded', 'true');
                    content.classList.add('open');
                    item.classList.add('open');
                }
            });
        });
    },

    /**
     * Initialize copy system info button
     */
    initCopySystemInfo: function() {
        const copyButton = document.getElementById('copy-system-info');
        const copyFeedback = document.getElementById('copy-feedback');

        if (!copyButton) {
            return;
        }

        copyButton.addEventListener('click', function() {
            const infoItems = document.querySelectorAll('.info-item');
            let infoText = 'Alt Audit System Information\n\n';

            infoItems.forEach(function(item) {
                const label = item.querySelector('.info-label');
                const value = item.querySelector('.info-value');
                if (label && value) {
                    infoText += label.textContent + ': ' + value.textContent + '\n';
                }
            });

            // Try modern clipboard API first
            if (navigator.clipboard && navigator.clipboard.writeText) {
                navigator.clipboard.writeText(infoText)
                    .then(function() {
                        HelpPage.showCopyFeedback(copyFeedback);
                    })
                    .catch(function() {
                        HelpPage.fallbackCopy(infoText, copyFeedback);
                    });
            } else {
                HelpPage.fallbackCopy(infoText, copyFeedback);
            }
        });
    },

    /**
     * Fallback copy method for older browsers
     * 
     * @param {string} text - Text to copy
     * @param {HTMLElement} copyFeedback - Feedback element
     */
    fallbackCopy: function(text, copyFeedback) {
        const textarea = document.createElement('textarea');
        textarea.value = text;
        textarea.style.position = 'fixed';
        textarea.style.opacity = '0';
        document.body.appendChild(textarea);
        textarea.select();

        try {
            document.execCommand('copy');
            this.showCopyFeedback(copyFeedback);
        } catch (err) {
            console.error('Copy failed:', err);
        }

        document.body.removeChild(textarea);
    },

    /**
     * Show copy success feedback
     * 
     * @param {HTMLElement} copyFeedback - Feedback element
     */
    showCopyFeedback: function(copyFeedback) {
        if (copyFeedback) {
            copyFeedback.style.display = 'inline-block';
            setTimeout(function() {
                copyFeedback.style.display = 'none';
            }, 3000);
        }
    }
};
