/**
 * Generation Mode Module
 *
 * Handles generation mode selection (Rule-based vs AI) and cost preview calculations.
 *
 * @package AltAudit
 * @since 1.0.0
 */

export const GenerationMode = {
    /**
     * Initialize generation mode functionality
     */
    init: function() {
        // Initialize cost preview
        this.updateCostPreview();

        // Set initial state
        const $ruleBasedRadio = jQuery('input[name="generation_mode"][value="rule_based"]');
        if ($ruleBasedRadio.length && $ruleBasedRadio.is(':checked')) {
            jQuery('#alt-audit-credit-info').hide();
            jQuery('#alt-audit-cost-preview').text('Rule-based generation is free!').css('color', '#46b450');
        }
    },

    /**
     * Handle generation mode change
     */
    handleGenerationModeChange: function() {
        const mode = jQuery(this).val();
        const $costPreview = jQuery('#alt-audit-cost-preview');

        if (mode === 'ai') {
            // Show credit requirements
            GenerationMode.updateCostPreview();
            jQuery('#alt-audit-credit-info').show();
        } else {
            // Hide cost preview for rule-based (free)
            $costPreview.text('Rule-based generation is free!').css('color', '#46b450');
            jQuery('#alt-audit-credit-info').hide();
        }
    },

    /**
     * Update cost preview based on selected images
     */
    updateCostPreview: function() {
        const mode = jQuery('input[name="generation_mode"]:checked').val();
        const $costPreview = jQuery('#alt-audit-cost-preview');

        if (mode === 'rule_based') {
            $costPreview.text('Rule-based generation is free!').css('color', '#46b450');
            return;
        }

        // Count selected images
        let selectedCount = jQuery('.wp-list-table input[type="checkbox"]:checked').length;

        // Don't count the header checkbox
        const headerCheckbox = jQuery('.wp-list-table .check-column input[type="checkbox"]').first();
        if (headerCheckbox.is(':checked')) {
            selectedCount--;
        }

        if (selectedCount <= 0) {
            $costPreview.text('');
            return;
        }

        const creditCount = parseInt(jQuery('#alt-audit-credit-count').text() || '0');
        const remainingCredits = creditCount - selectedCount;

        let message = selectedCount + ' credits will be used';
        if (remainingCredits < 0) {
            message += ' (⚠️ ' + Math.abs(remainingCredits) + ' credits over limit)';
            $costPreview.css('color', '#d63638');
        } else {
            message += ' (' + remainingCredits + ' remaining)';
            $costPreview.css('color', '#666');
        }

        $costPreview.text(message);
    }
};
