/**
 * Dashboard Module
 *
 * Handles dashboard-specific functionality including stats, score animations,
 * and card interactions.
 *
 * @package AltAudit
 * @since 1.0.0
 */

import { Utils } from './utils.js';

export const Dashboard = {
    /**
     * Initialize dashboard functionality
     */
    init: function () {
        if (!jQuery('.alt-audit-dashboard').length) return;
        this.initScoreCircle();
        this.initCardInteractions();
        this.refreshStats();
        this.initStatsRefresh();
        this.fetchCredits();
    },

    /**
     * Initialize score circle animation
     */
    initScoreCircle: function () {
        const $scoreCircle = jQuery('.score-circle');
        if (!$scoreCircle.length) return;

        $scoreCircle.each(function () {
            const $circle = jQuery(this);
            const score = $circle.data('score') || 0;

            // Animate the circular progress
            Dashboard.animateScoreCircle($circle, score);
        });
    },

    /**
     * Animate score circle
     *
     * @param {jQuery} $circle - Circle element
     * @param {number} targetScore - Target score percentage
     */
    animateScoreCircle: function ($circle, targetScore) {
        let currentScore = 0;
        const duration = 1500; // Animation duration in ms
        const steps = 60; // Number of animation steps
        const stepTime = duration / steps;
        const increment = targetScore / steps;

        // Set initial state
        $circle.css('--score-percentage', 0);

        // Animate to target score
        const animationTimer = setInterval(function () {
            currentScore += increment;

            if (currentScore >= targetScore) {
                currentScore = targetScore;
                clearInterval(animationTimer);
            }

            // Update CSS custom property for the conic gradient
            $circle.css('--score-percentage', currentScore);

            // Update the displayed percentage
            $circle.find('.score-percentage').text(Math.round(currentScore) + '%');
        }, stepTime);
    },

    /**
     * Update dashboard statistics with animation
     *
     * @param {Object} stats - Statistics data
     */
    updateDashboardStats: function (stats) {
        // Update status cards with count-up animation
        jQuery('.status-card').each(function () {
            const $card = jQuery(this);
            const $number = $card.find('.card-number');
            const cardType = $card.attr('class').match(/card-(\w+)/);

            if (cardType && stats[cardType[1] + '_alt'] !== undefined) {
                Utils.animateNumber($number, stats[cardType[1] + '_alt']);
            }
        });

        // Update quality score circle
        const qualityScore = stats.quality_score || 0;
        const $scoreCircle = jQuery('.score-circle');
        if ($scoreCircle.length) {
            Dashboard.animateScoreCircle($scoreCircle, qualityScore);
        }
    },

    /**
     * Add hover effects for cards
     */
    initCardInteractions: function () {
        jQuery('.status-card, .quality-score-card, .quick-actions-card, .recent-activity-card, .priority-issues-card').on('mouseenter', function () {
            jQuery(this).addClass('card-hover');
        }).on('mouseleave', function () {
            jQuery(this).removeClass('card-hover');
        });

        // Add click ripple effect for action buttons
        jQuery('.action-btn, .btn-primary, .btn-secondary').on('click', function (e) {
            const $button = jQuery(this);
            const $ripple = jQuery('<span class="ripple"></span>');
            const buttonOffset = $button.offset();
            const x = e.pageX - buttonOffset.left;
            const y = e.pageY - buttonOffset.top;

            $ripple.css({
                left: x + 'px',
                top: y + 'px'
            });

            $button.append($ripple);

            setTimeout(function () {
                $ripple.remove();
            }, 600);
        });
    },

    /**
     * Refresh dashboard stats
     */
    refreshStats: function () {
        const $statsContainer = jQuery('.alt-audit-stats-grid');
        if (!$statsContainer.length) return;

        $statsContainer.addClass('alt-audit-loading');

        // This would be an AJAX call to refresh stats
        setTimeout(function () {
            $statsContainer.removeClass('alt-audit-loading');
        }, 1000);
    },

    /**
     * Fetch user credits from API
     */
    fetchCredits: function () {
        const creditsElement = jQuery('#dashboard-credits');
        if (!creditsElement.length) return;
        jQuery.ajax({
            url: altaudit82aiAdmin.ajaxUrl,
            type: 'POST',
            data: {
                action: 'altaudit82ai_get_user_credits',
                nonce: altaudit82aiAdmin.nonce
            },
            success: function (response) {
                if (response.success && response.data && typeof response.data.credits !== 'undefined') {
                    creditsElement.text(parseInt(response.data.credits).toLocaleString());
                } else {
                    creditsElement.text('--');
                }
            },
            error: function () {
                creditsElement.text('--');
            }
        });
    },

    /**
     * Initialize stats auto-refresh
     */
    initStatsRefresh: function () {
        // Refresh stats every 5 minutes
        setInterval(() => this.refreshStats(), 300000);
    }
};
