/**
 * Bulk Tools Module
 *
 * Handles bulk operations for generating and processing alt text for multiple images.
 * Includes progress tracking, batch processing, and result reporting.
 *
 * @package AltAudit
 * @since 1.0.0
 */

import { Utils } from './utils.js';
import { GenerationMode } from './generation-mode.js';

export const BulkTools = {
    bulkOperationState: null,
    allProcessedImages: [],
    currentView: 'card',
    currentFilter: 'all',
    currentSort: 'filename-asc',
    searchQuery: '',
    currentPage: 1,
    itemsPerPage: 10,

    /**
     * Initialize bulk tools page
     */
    init: function () {
        // Only run on bulk tools page
        if (!jQuery('.alt-audit-bulk').length) {
            return;
        }

        const self = this;

        // Fetch and display user credits
        self.fetchUserCredits();

        // Fetch queue status (Phase 1B)
        self.fetchQueueStatus();

        // Initial count on page load for both sections
        self.updateImageCount('rule');
        self.updateImageCount('ai');

        // Make checkboxes mutually exclusive and update count - Rule-based generation
        jQuery('#rule_skip_existing').on('change', function () {
            if (jQuery(this).is(':checked')) {
                jQuery('#rule_overwrite_weak').prop('checked', false);
            }
            self.updateImageCount('rule');
        });

        jQuery('#rule_overwrite_weak').on('change', function () {
            if (jQuery(this).is(':checked')) {
                jQuery('#rule_skip_existing').prop('checked', false);
            }
            self.updateImageCount('rule');
        });

        // Make checkboxes mutually exclusive and update count - AI generation
        jQuery('#skip_existing').on('change', function () {
            if (jQuery(this).is(':checked')) {
                jQuery('#overwrite_weak').prop('checked', false);
            }
            self.updateImageCount('ai');
        });

        jQuery('#overwrite_weak').on('change', function () {
            if (jQuery(this).is(':checked')) {
                jQuery('#skip_existing').prop('checked', false);
            }
            self.updateImageCount('ai');
        });

        // Handle bulk generation button clicks
        jQuery('#start-bulk-rule-generate').on('click', function (e) {
            e.preventDefault();
            self.startBulkGeneration('rule');
        });

        jQuery('#start-bulk-generate').on('click', function (e) {
            e.preventDefault();
            self.startBulkGeneration('ai');
        });

        // Handle data export button
        jQuery('#export-data').on('click', function (e) {
            e.preventDefault();
            self.exportData();
        });

        // Handle cancel operation
        jQuery('#cancel-bulk').on('click', function (e) {
            e.preventDefault();
            self.cancelBulkOperation();
        });

        // Handle start new operation
        jQuery('#start-new-operation').on('click', function (e) {
            e.preventDefault();
            self.resetBulkInterface();
        });

        // Handle log toggle
        jQuery('#show-log-btn').on('click', function () {
            jQuery('#progress-log').slideDown();
            jQuery(this).hide();
        });

        jQuery('#hide-log-btn').on('click', function () {
            jQuery('#progress-log').slideUp();
            jQuery('#show-log-btn').show();
        });

        // Initialize toolbar handlers
        self.bindToolbarHandlers();

        // Initialize pagination handlers
        self.bindPaginationHandlers();

        // Handle download report button
        jQuery('#download-report').on('click', function (e) {
            e.preventDefault();
            self.downloadReport();
        });

        // Handle detail tab switching
        jQuery(document).on('click', '.detail-tab', function (e) {
            e.preventDefault();
            const tab = jQuery(this).data('tab');
            self.switchTab(tab);
        });
    },

    /**
     * Update image count for bulk operations
     *
     * @param {string} type - 'rule' for rule-based, 'ai' for AI generation
     */
    updateImageCount: function (type) {
        let skipExisting, overwriteWeak, countElement, costElement;

        if (type === 'rule') {
            skipExisting = jQuery('#rule_skip_existing').is(':checked');
            overwriteWeak = jQuery('#rule_overwrite_weak').is(':checked');
            countElement = jQuery('#rule-estimated-images');
            costElement = null; // No cost for rule-based
        } else {
            skipExisting = jQuery('#skip_existing').is(':checked');
            overwriteWeak = jQuery('#overwrite_weak').is(':checked');
            countElement = jQuery('#estimated-images');
            costElement = jQuery('#estimated-cost'); // AI generation has cost
        }

        // Show loading state
        countElement.text('...');
        if (costElement) {
            costElement.text('...');
        }

        jQuery.ajax({
            url: altaudit82aiAdmin.ajaxUrl,
            type: 'POST',
            data: {
                action: 'altaudit82ai_count_images',
                nonce: altaudit82aiAdmin.nonce,
                skip_existing: skipExisting ? 'true' : 'false',
                overwrite_weak: overwriteWeak ? 'true' : 'false'
            },
            success: function (response) {
                if (response.success && response.data && typeof response.data.count !== 'undefined') {
                    const imageCount = response.data.count;
                    countElement.text(imageCount.toLocaleString());

                    // Update cost element for AI generation (1 credit per image)
                    if (costElement) {
                        costElement.text(imageCount.toLocaleString());
                    }
                } else {
                    countElement.text('--');
                    if (costElement) {
                        costElement.text('--');
                    }
                }
            },
            error: function (_xhr, _status, error) {
                console.error('Failed to count images:', error);
                countElement.text('--');
                if (costElement) {
                    costElement.text('--');
                }
            }
        });
    },

    /**
     * Fetch user credits from API and update display
     */
    fetchUserCredits: function () {
        const creditsElement = jQuery('#available-credits');
        const buyButton = jQuery('#buy-credits-button');
        const buttonText = buyButton.find('.button-text');

        // Show loading state
        creditsElement.text('...');

        jQuery.ajax({
            url: altaudit82aiAdmin.ajaxUrl,
            type: 'POST',
            data: {
                action: 'altaudit82ai_get_user_credits',
                nonce: altaudit82aiAdmin.nonce
            },
            success: function (response) {
                if (response.success && response.data && typeof response.data.credits !== 'undefined') {
                    const credits = parseInt(response.data.credits);
                    creditsElement.text(credits.toLocaleString());

                    // Add visual indicator based on credit level to stat item
                    const statItem = creditsElement.closest('.stat-item');
                    statItem.removeClass('credits-low credits-empty');

                    // Update button styling and text based on credit level
                    buyButton.removeClass('credits-normal credits-low credits-empty button-secondary');

                    if (credits === 0) {
                        // Empty: Urgent red button
                        statItem.addClass('credits-empty');
                        buyButton.addClass('credits-empty button-primary');
                        buttonText.text('Add Credits Now');
                    } else if (credits < 50) {
                        // Low: Warning orange/yellow button
                        statItem.addClass('credits-low');
                        buyButton.addClass('credits-low');
                        buttonText.text('Get More Credits');
                    } else {
                        // Normal: Standard secondary button
                        buyButton.addClass('credits-normal button-secondary');
                        buttonText.text('Buy More Credits');
                    }
                } else {
                    creditsElement.text('--');
                    buyButton.addClass('button-secondary');
                }
            },
            error: function (_xhr, _status, error) {
                console.error('Failed to fetch credits:', error);
                creditsElement.text('--');
                buyButton.addClass('button-secondary');
            }
        });
    },

    /**
     * Fetch and display queue status (Phase 1B)
     */
    fetchQueueStatus: function () {
        const self = this;

        jQuery.ajax({
            url: altaudit82aiAdmin.ajaxUrl,
            type: 'POST',
            data: {
                action: 'altaudit82ai_get_queue_status',
                nonce: altaudit82aiAdmin.nonce
            },
            success: function (response) {
                if (response.success && response.data) {
                    self.displayQueueStatus(response.data);
                }
            },
            error: function () {
                // Silently fail - queue status is nice-to-have
            }
        });
    },

    /**
     * Display queue status in UI
     */
    displayQueueStatus: function (queueData) {
        // Remove any existing queue notice
        jQuery('.queue-status-notice').remove();

        if (queueData.queue_size > 0) {
            const statusClass = queueData.status === 'busy' ? 'notice-warning' : 'notice-info';
            const statusIcon = queueData.status === 'busy' ? '⚠️' : 'ℹ️';

            const message = jQuery('<div>')
                .addClass('notice ' + statusClass + ' queue-status-notice')
                .css({ 'margin': '15px 0 10px 0', 'padding': '12px' })
                .html(
                    '<p><strong>' + statusIcon + ' Queue Status:</strong> ' + queueData.message + '</p>' +
                    '<p style="margin: 5px 0 0 0;"><small>' +
                    'Queue: ' + queueData.queue_size.toLocaleString() + ' images | ' +
                    'Status: <strong>' + queueData.status.toUpperCase() + '</strong> | ' +
                    'Processing: ' + queueData.processing_rate +
                    '</small></p>'
                );

            // Add before operation cards
            jQuery('.bulk-operations-grid').before(message);

            // Auto-refresh queue status every 30 seconds
            setTimeout(function () {
                self.fetchQueueStatus();
            }, 30000);
        }
    },

    /**
     * Start bulk generation operation
     *
     * @param {string} type - 'rule' for rule-based, 'ai' for AI generation
     */
    startBulkGeneration: function (type) {
        const self = this;
        let skipExisting, overwriteWeak, countElement;

        if (type === 'rule') {
            skipExisting = jQuery('#rule_skip_existing').is(':checked');
            overwriteWeak = jQuery('#rule_overwrite_weak').is(':checked');
            countElement = jQuery('#rule-estimated-images');
        } else {
            skipExisting = jQuery('#skip_existing').is(':checked');
            overwriteWeak = jQuery('#overwrite_weak').is(':checked');
            countElement = jQuery('#estimated-images');
        }

        const countText = countElement.text().trim();
       

        // Check if count is still loading or unavailable
        if (countText === '...' || countText === '--' || countText === '') {
            alert('Please wait while we count the images...');
            // Trigger count update and wait
            self.updateImageCount(type);
            return;
        }

        // Remove all non-numeric characters (commas, spaces, nbsp, etc.) before parsing
        const totalImages = parseInt(countText.replace(/[\s,\u00A0]/g, '')) || 0;
        if (totalImages === 0) {
            alert('No images found to process. Please adjust your filter options.');
            return;
        }

        // Confirm before starting
        const confirmMessage = type === 'rule'
            ? 'Start free rule-based generation for ' + totalImages + ' images?'
            : 'Start AI generation for ' + totalImages + ' images? This will use ' + totalImages + ' credits.';

        if (!confirm(confirmMessage)) {
            return;
        }

        // Hide operation cards and show progress
        jQuery('.bulk-operations-grid').slideUp();
        jQuery('#bulk-progress').slideDown();
        jQuery('#bulk-results').hide();

        // Reset processed images array
        self.allProcessedImages = [];

        // Initialize progress state
        self.bulkOperationState = {
            type: type,
            skipExisting: skipExisting,
            overwriteWeak: overwriteWeak,
            totalImages: totalImages,
            processed: 0,
            successCount: 0,
            errorCount: 0,
            batchOffset: 0,
            cancelled: false,
            errors: []
        };

        // Reset progress UI
        self.updateProgressUI(0, totalImages, 0, 0, 'Starting operation...');
        jQuery('#log-content').empty();

        // Start processing
        self.processBatch();
    },

    /**
     * Process a batch of images
     */
    processBatch: function () {
        const self = this;
        const state = self.bulkOperationState;

        if (state.cancelled) {
            Utils.logMessage('Operation cancelled by user.', 'warning');
            self.showResults();
            return;
        }

        const action = state.type === 'rule' ? 'altaudit82ai_bulk_generate_rule' : 'altaudit82ai_bulk_generate_ai';

        jQuery.ajax({
            url: altaudit82aiAdmin.ajaxUrl,
            type: 'POST',
            data: {
                action: action,
                nonce: altaudit82aiAdmin.nonce,
                skip_existing: state.skipExisting ? 'true' : 'false',
                overwrite_weak: state.overwriteWeak ? 'true' : 'false',
                batch_offset: state.batchOffset,
                processed_count: state.processed,
                error_count: state.errorCount,
                total_images: state.totalImages
            },
            success: function (response) {
                if (response.success && response.data) {
                    const data = response.data;

                    // Update state
                    state.processed = data.processed || 0;
                    state.successCount += data.success_count || 0;
                    state.errorCount += data.error_count || 0;
                    state.batchOffset = data.next_batch_offset || state.processed;

                    // Collect processed images for table view
                    if (data.processed_images && data.processed_images.length > 0) {
                        self.allProcessedImages = self.allProcessedImages.concat(data.processed_images);
                    }

                    // Display detailed log messages from backend
                    if (data.log_messages && data.log_messages.length > 0) {
                        data.log_messages.forEach(function (logEntry) {
                            Utils.logMessage(logEntry.message, logEntry.type || 'info');
                        });
                    }

                    // Log errors to state for tracking
                    if (data.errors && data.errors.length > 0) {
                        data.errors.forEach(function (error) {
                            state.errors.push(error);
                        });
                    }

                    // Log batch summary with skipped count if any
                    let summaryMsg = 'Batch complete: ' + data.success_count + ' successful, ' + data.error_count + ' errors';
                    if (data.skipped_count && data.skipped_count > 0) {
                        summaryMsg += ', ' + data.skipped_count + ' skipped';
                    }
                    Utils.logMessage(summaryMsg, 'info');

                    // Update progress UI
                    self.updateProgressUI(
                        state.processed,
                        state.totalImages,
                        state.successCount,
                        state.errorCount,
                        data.status_message || 'Processing...'
                    );

                    // Check if operation is complete
                    if (data.all_complete) {
                        Utils.logMessage('All images processed successfully!', 'success');
                        self.showResults();
                    } else {
                        // Process next batch after delay to avoid API rate limits
                        // Delay accounts for: 10 images * 30ms = 300ms processing + 100ms buffer
                        setTimeout(function () {
                            self.processBatch();
                        }, 400);
                    }
                } else {
                    // Handle different error types with user-friendly messages
                    if (response.data && response.data.error_type) {
                        const errorType = response.data.error_type;
                        const errorMessage = response.data.message || 'An error occurred';

                        // Show error message with appropriate styling
                        if (errorType === 'rate_limit') {
                            Utils.logMessage('⚠️ ' + errorMessage, 'error');
                            Utils.logMessage('Please wait a few minutes before trying again.', 'warning');
                        } else if (errorType === 'server_error') {
                            Utils.logMessage('🔧 ' + errorMessage, 'error');
                            Utils.logMessage('The service may be temporarily down. Please try again in a few minutes.', 'warning');
                        } else if (errorType === 'auth_error') {
                            Utils.logMessage('🔑 ' + errorMessage, 'error');
                            Utils.logMessage('Please check your API key in plugin settings.', 'warning');
                        } else {
                            Utils.logMessage('❌ ' + errorMessage, 'error');
                        }

                        // Show processing stats if available
                        if (response.data.success_count > 0) {
                            Utils.logMessage(
                                'Successfully processed ' + response.data.success_count + ' images before error occurred.',
                                'info'
                            );
                        }
                    } else {
                        // Generic error handling
                        const errorMsg = response.data && response.data.message
                            ? response.data.message
                            : 'Unknown error occurred';
                        Utils.logMessage('❌ Error: ' + errorMsg, 'error');
                    }

                    self.showResults();
                }
            },
            error: function (xhr, status, error) {
                // Track retry attempts for timeout errors.
                if (!state.retryCount) {
                    state.retryCount = 0;
                }

                // For 504/502/503 errors, retry up to 3 times with increasing delay.
                if ((xhr.status === 502 || xhr.status === 503 || xhr.status === 504) && state.retryCount < 3) {
                    state.retryCount++;
                    const retryDelay = state.retryCount * 3000; // 3s, 6s, 9s delays.

                    Utils.logMessage(
                        '⚠️ Server timeout (' + xhr.status + '). Retrying in ' + (retryDelay / 1000) + ' seconds... (attempt ' + state.retryCount + '/3)',
                        'warning'
                    );

                    setTimeout(function () {
                        self.processBatch();
                    }, retryDelay);
                    return;
                }

                // Reset retry count on other errors.
                state.retryCount = 0;

                // Provide user-friendly error messages based on error type.
                let errorMessage = '';
                if (xhr.status === 0) {
                    errorMessage = 'Network error. Please check your internet connection.';
                } else if (xhr.status === 429) {
                    errorMessage = 'Rate limit exceeded. Please wait a few minutes before trying again.';
                } else if (xhr.status === 500) {
                    errorMessage = 'Server error (500). The Alt Audit service may be experiencing issues.';
                } else if (xhr.status === 502 || xhr.status === 503 || xhr.status === 504) {
                    errorMessage = 'Service temporarily unavailable (' + xhr.status + ') after 3 retries. Please try again later.';
                } else if (status === 'timeout') {
                    errorMessage = 'Request timed out. The server took too long to respond.';
                } else if (status === 'parsererror') {
                    errorMessage = 'Invalid response from server. Please contact support if this persists.';
                } else {
                    errorMessage = 'AJAX error: ' + error + ' (Status: ' + xhr.status + ')';
                }

                Utils.logMessage('❌ ' + errorMessage, 'error');

                // Show success count if any images were processed.
                if (state.successCount > 0) {
                    Utils.logMessage(
                        '✓ Successfully processed ' + state.successCount + ' images before error occurred.',
                        'info'
                    );
                }

                self.showResults();
            }
        });
    },

    /**
     * Update progress UI elements
     *
     * @param {number} processed - Number of images processed
     * @param {number} total - Total number of images
     * @param {number} successCount - Number of successful operations
     * @param {number} errorCount - Number of failed operations
     * @param {string} statusMessage - Current status message
     */
    updateProgressUI: function (processed, total, successCount, errorCount, statusMessage) {
        const percentage = total > 0 ? Math.round((processed / total) * 100) : 0;

        // Update progress bar
        jQuery('#bulk-progress-fill').css('width', percentage + '%');
        jQuery('#bulk-progress-percentage').text(percentage + '%');

        // Update progress text
        jQuery('#bulk-progress-text').text(statusMessage);

        // Update stats
        jQuery('#processed-count').text(processed);
        jQuery('#total-count').text(total);
        jQuery('#success-count').text(successCount);
        jQuery('#error-count').text(errorCount);

        // Update progress title with spinning icon during processing
        if (percentage < 100) {
            jQuery('#progress-title .dashicons').addClass('dashicons-update-alt').css('animation', 'spin 1s linear infinite');
        } else {
            jQuery('#progress-title .dashicons').removeClass('dashicons-update-alt').css('animation', '');
        }
    },

    /**
     * Show results after operation completion
     */
    showResults: function () {
        const self = this;
        const state = self.bulkOperationState;

        // Hide progress, show results
        jQuery('#bulk-progress').slideUp(function () {
            jQuery('#bulk-results').slideDown();
        });

        // Determine default view based on number of images
        const defaultView = self.allProcessedImages.length > 1 ? 'table' : 'card';
        self.currentView = defaultView;

        // Update results summary with view toggle
        let summaryHtml = '<div class="results-header-flex">';
        summaryHtml += '<div class="results-stats-grid">';
        summaryHtml += '<div class="result-stat result-stat-processed"><div class="stat-number">' + state.processed + '</div><div class="stat-label">Images Processed</div></div>';
        summaryHtml += '<div class="result-stat result-stat-success"><div class="stat-number">' + state.successCount + '</div><div class="stat-label">Successful</div></div>';
        summaryHtml += '<div class="result-stat result-stat-errors"><div class="stat-number">' + state.errorCount + '</div><div class="stat-label">Errors</div></div>';
        summaryHtml += '</div>';

        // Add view toggle buttons if we have processed images
        if (self.allProcessedImages.length > 0) {
            summaryHtml += '<div class="view-toggle-container">';
            summaryHtml += '<button type="button" class="view-toggle-btn ' + (defaultView === 'card' ? 'active' : '') + '" data-view="card" title="Card View"><span class="dashicons dashicons-grid-view"></span></button>';
            summaryHtml += '<button type="button" class="view-toggle-btn ' + (defaultView === 'table' ? 'active' : '') + '" data-view="table" title="Table View"><span class="dashicons dashicons-list-view"></span></button>';
            summaryHtml += '</div>';
        }
        summaryHtml += '</div>';

        if (state.cancelled) {
            summaryHtml += '<p class="results-message warning">Operation was cancelled by user.</p>';
        } else if (state.errorCount === 0) {
            summaryHtml += '<p class="results-message success">All images processed successfully!</p>';
        } else {
            summaryHtml += '<p class="results-message warning">Operation completed with some errors. See detailed log for more information.</p>';
        }

        jQuery('#results-summary').html(summaryHtml);

        // Initialize views if we have processed images
        if (self.allProcessedImages.length > 0) {
            self.renderTableView();
            self.renderCardView();

            // Show default view immediately
            self.switchView(defaultView);
        }

        // Bind view toggle handlers
        jQuery('.view-toggle-btn').on('click', function () {
            const view = jQuery(this).data('view');
            self.switchView(view);
        });

        // Populate log panel
        jQuery('#results-log-content').val(jQuery('#log-content').text());

        // Populate errors panel
        if (state.errors.length > 0) {
            let errorsHtml = '<ul class="error-items">';
            state.errors.forEach(function (error) {
                errorsHtml += '<li><strong>Image #' + error.attachment_id + ':</strong> ' + error.message + '</li>';
            });
            errorsHtml += '</ul>';
            jQuery('#error-list').html(errorsHtml);
        } else {
            jQuery('#error-list').html('<p class="no-errors">No errors occurred during processing.</p>');
        }

        // Update image counts
        self.updateImageCount('rule');
        self.updateImageCount('ai');

        // Refresh user credits display
        self.fetchUserCredits();

        // Refresh statistics if on audit page
        if (jQuery('.alt-audit-statistics-section').length > 0) {
            import('./audit-page.js').then(module => {
                module.AuditPage.refreshAuditStatistics();
            });
        }
    },

    /**
     * Switch between detail tabs (Summary, Log, Errors)
     *
     * @param {string} tab - Tab identifier ('summary', 'log', 'errors')
     */
    switchTab: function (tab) {
        // Update tab buttons
        jQuery('.detail-tab').removeClass('active');
        jQuery('.detail-tab[data-tab="' + tab + '"]').addClass('active');

        // Hide all panels
        jQuery('.detail-panel').removeClass('active').hide();

        // Show selected panel
        jQuery('#' + tab + '-panel').addClass('active').fadeIn(300);
    },

    /**
     * Download comprehensive report of bulk operation results
     */
    downloadReport: function () {
        const self = this;
        const state = self.bulkOperationState;

        if (!state || !self.allProcessedImages.length) {
            Utils.showNotice('No data available to download', 'warning');
            return;
        }

        // Determine report type based on operation
        const operationType = state.type === 'rule' ? 'Rule-Based' : 'AI-Powered';
        const timestamp = new Date().toISOString().replace(/[:.]/g, '-').slice(0, 19);

        // Generate comprehensive CSV report
        const headers = [
            'Operation Type',
            'Attachment ID',
            'Filename',
            'Generated Alt Text',
            'Quality Score',
            'Quality Status',
            'Success',
            'Timestamp'
        ];

        const rows = self.allProcessedImages.map(img => [
            operationType,
            img.attachment_id || '',
            '"' + (img.filename || '').replace(/"/g, '""') + '"',
            '"' + (img.alt_text || '').replace(/"/g, '""') + '"',
            img.quality_score || '0',
            img.quality_status || 'unknown',
            img.success ? 'Yes' : 'No',
            new Date().toISOString()
        ]);

        // Add summary section at the top
        let csvContent = '# Alt Audit Bulk Operation Report\n';
        csvContent += '# Generated: ' + new Date().toLocaleString() + '\n';
        csvContent += '# Operation Type: ' + operationType + '\n';
        csvContent += '# Total Processed: ' + state.processed + '\n';
        csvContent += '# Successful: ' + state.successCount + '\n';
        csvContent += '# Errors: ' + state.errorCount + '\n';
        csvContent += '\n';

        // Add data table
        csvContent += headers.join(',') + '\n';
        rows.forEach(row => {
            csvContent += row.join(',') + '\n';
        });

        // Add errors section if any
        if (state.errors && state.errors.length > 0) {
            csvContent += '\n# Errors:\n';
            csvContent += 'Attachment ID,Error Message\n';
            state.errors.forEach(error => {
                csvContent += error.attachment_id + ',"' + (error.message || '').replace(/"/g, '""') + '"\n';
            });
        }

        // Create and download file
        const blob = new Blob([csvContent], { type: 'text/csv;charset=utf-8;' });
        const link = document.createElement('a');
        const url = URL.createObjectURL(blob);

        link.setAttribute('href', url);
        link.setAttribute('download', 'alt-audit-report-' + timestamp + '.csv');
        link.style.visibility = 'hidden';

        document.body.appendChild(link);
        link.click();
        document.body.removeChild(link);

        // Revoke object URL to free memory
        URL.revokeObjectURL(url);

        Utils.showNotice('Report downloaded successfully', 'success');
    },

    /**
     * Cancel bulk operation
     */
    cancelBulkOperation: function () {
        if (confirm('Are you sure you want to cancel this operation?')) {
            if (this.bulkOperationState) {
                this.bulkOperationState.cancelled = true;
                Utils.logMessage('Cancelling operation...', 'warning');
            }
        }
    },

    /**
     * Reset bulk interface to initial state
     */
    resetBulkInterface: function () {
        jQuery('#bulk-results').slideUp();
        jQuery('.bulk-operations-grid').slideDown();

        if (this.bulkOperationState) {
            this.bulkOperationState = null;
        }
    },

    /**
     * Handle bulk actions from list table
     *
     * @param {Event} e - Click event
     */
    handleBulkAction: function (e) {
        e.preventDefault();

        const $button = jQuery(this);
        const action = $button.data('action');
        const $checkboxes = jQuery('.wp-list-table input[type="checkbox"]:checked');

        if ($checkboxes.length === 0) {
            Utils.showNotice(altaudit82aiAdmin.strings.noImagesFound, 'warning');
            return;
        }

        if (!confirm(altaudit82aiAdmin.strings.confirm)) {
            return;
        }

        const attachmentIds = [];
        $checkboxes.each(function () {
            const val = jQuery(this).val();
            if (val && val !== 'on' && !isNaN(val)) { // Exclude header checkbox
                attachmentIds.push(val);
            }
        });

        if (attachmentIds.length === 0) {
            Utils.showNotice('No images selected', 'warning');
            return;
        }

        // Get selected generation mode for generation actions
        const generationMode = jQuery('input[name="generation_mode"]:checked').val();
        let bulkAction = action;

        // Map generation actions to the correct bulk action based on mode
        if (action === 'generate_alt_text' || action === 'generate') {
            bulkAction = generationMode === 'ai' ? 'generate_ai' : 'generate_rule_based';
        }

        // Check credits for AI generation
        if (bulkAction === 'generate_ai') {
            const creditCount = parseInt(jQuery('#alt-audit-credit-count').text() || '0');
            if (creditCount < attachmentIds.length) {
                const shortage = attachmentIds.length - creditCount;
                if (!confirm('You need ' + shortage + ' more credits. Continue anyway?')) {
                    return;
                }
            }
        }

        const originalText = $button.text();
        $button.text(altaudit82aiAdmin.strings.processing || 'Processing...').prop('disabled', true);

        jQuery.ajax({
            url: altaudit82aiAdmin.ajaxUrl,
            type: 'POST',
            data: {
                action: 'altaudit82ai_bulk_process',
                nonce: altaudit82aiAdmin.nonce,
                attachment_ids: attachmentIds,
                bulk_action: bulkAction
            },
            success: function (response) {
                if (response.success) {
                    BulkTools.handleBulkSuccess(response.data);
                } else {
                    Utils.showNotice(response.data, 'error');
                }
            },
            error: function () {
                Utils.showNotice(altaudit82aiAdmin.strings.error, 'error');
            },
            complete: function () {
                $button.text(originalText).prop('disabled', false);
            }
        });
    },

    /**
     * Handle bulk operation success
     *
     * @param {Object} data - Response data
     */
    handleBulkSuccess: function (data) {
        let message = 'Processed ' + data.processed + ' images';
        if (data.errors > 0) {
            message += ' with ' + data.errors + ' errors';
        }

        Utils.showNotice(message, data.errors > 0 ? 'warning' : 'success');

        // Refresh the page if needed
        if (data.remaining === 0) {
            setTimeout(function () {
                location.reload();
            }, 2000);
        }
    },

    /**
     * Render table view for bulk results
     */
    renderTableView: function () {
        const self = this;

        const summaryPanel = jQuery('#summary-panel');

        if (summaryPanel.length === 0) {
            return;
        }

        // Create table view container if it doesn't exist
        if (jQuery('#bulk-results-table-view').length === 0) {
            const tableHtml = `
                <div id="bulk-results-table-view" class="bulk-results-table-container" style="display:none;">
                    ${self.renderToolbar()}
                    <div class="table-actions">
                        <button type="button" class="button button-secondary" id="copy-all-alt-text">
                            <span class="dashicons dashicons-clipboard"></span>
                            Copy All Alt Text
                        </button>
                        <button type="button" class="button button-secondary" id="export-table-csv">
                            <span class="dashicons dashicons-download"></span>
                            Export CSV
                        </button>
                    </div>
                    <div class="table-wrapper">
                        <table class="bulk-results-table">
                            <thead>
                                <tr>
                                    <th class="column-thumbnail">Thumbnail</th>
                                    <th class="column-filename">Filename</th>
                                    <th class="column-alt-text">Generated Alt Text</th>
                                    <th class="column-quality">Quality Score</th>
                                    <th class="column-actions">Actions</th>
                                </tr>
                            </thead>
                            <tbody id="table-results-body">
                            </tbody>
                        </table>
                    </div>
                    <div class="table-footer">
                        <div class="table-summary">
                            Showing <strong id="table-count">0</strong> processed images
                        </div>
                    </div>
                </div>
            `;
            summaryPanel.append(tableHtml);

        } else {

        }

        // Populate table rows
        self.populateTableRows();

        // Bind action handlers
        jQuery('#copy-all-alt-text').off('click').on('click', function () {
            self.copyAllAltText();
        });

        jQuery('#export-table-csv').off('click').on('click', function () {
            self.exportTableCSV();
        });
    },

    /**
     * Get filtered and sorted images based on current state
     *
     * @return {Array} Filtered and sorted images
     */
    getFilteredAndSortedImages: function () {
        const self = this;
        let images = [...self.allProcessedImages];

        // Apply search filter
        if (self.searchQuery.trim() !== '') {
            const query = self.searchQuery.toLowerCase();
            images = images.filter(function (image) {
                const filename = (image.filename || '').toLowerCase();
                const altText = (image.alt_text || '').toLowerCase();
                return filename.includes(query) || altText.includes(query);
            });
        }

        // Apply status filter
        if (self.currentFilter !== 'all') {
            images = images.filter(function (image) {
                if (self.currentFilter === 'success') {
                    return image.success === true;
                } else if (self.currentFilter === 'error') {
                    return image.success === false;
                } else if (self.currentFilter === 'missing') {
                    return image.quality_status === 'missing';
                } else if (self.currentFilter === 'weak') {
                    return image.quality_status === 'weak';
                } else if (self.currentFilter === 'good') {
                    return image.quality_status === 'good';
                } else if (self.currentFilter === 'excellent') {
                    return image.quality_status === 'excellent';
                }
                return true;
            });
        }

        // Apply sorting
        images.sort(function (a, b) {
            switch (self.currentSort) {
                case 'filename-asc':
                    return (a.filename || '').localeCompare(b.filename || '');
                case 'filename-desc':
                    return (b.filename || '').localeCompare(a.filename || '');
                case 'quality-high':
                    return (b.quality_score || 0) - (a.quality_score || 0);
                case 'quality-low':
                    return (a.quality_score || 0) - (b.quality_score || 0);
                case 'date-newest':
                    return (b.attachment_id || 0) - (a.attachment_id || 0);
                case 'date-oldest':
                    return (a.attachment_id || 0) - (b.attachment_id || 0);
                default:
                    return 0;
            }
        });

        return images;
    },

    /**
     * Get paginated images for current page
     *
     * @return {Array} Images for current page
     */
    getPaginatedImages: function () {
        const self = this;
        const filteredImages = self.getFilteredAndSortedImages();

        // If "All" is selected, return all filtered images
        if (self.itemsPerPage === 'all') {
            return filteredImages;
        }

        const startIndex = (self.currentPage - 1) * self.itemsPerPage;
        const endIndex = startIndex + self.itemsPerPage;

        return filteredImages.slice(startIndex, endIndex);
    },

    /**
     * Get total number of pages
     *
     * @return {number} Total pages
     */
    getTotalPages: function () {
        const self = this;
        const filteredImages = self.getFilteredAndSortedImages();

        // If "All" is selected, return 1 page
        if (self.itemsPerPage === 'all') {
            return 1;
        }

        return Math.ceil(filteredImages.length / self.itemsPerPage);
    },

    /**
     * Change to specific page
     *
     * @param {number} pageNumber - Page number to change to
     */
    changePage: function (pageNumber) {
        const self = this;
        const totalPages = self.getTotalPages();

        // Validate page number
        if (pageNumber < 1 || pageNumber > totalPages) {
            return;
        }

        self.currentPage = pageNumber;
        self.refreshResults();

        // Smooth scroll to top of results
        const resultsContainer = jQuery('#bulk-results-table-view, #bulk-results-card-view').filter(':visible');
        if (resultsContainer.length > 0) {
            resultsContainer.get(0).scrollIntoView({ behavior: 'smooth', block: 'start' });
        }
    },

    /**
     * Change items per page
     *
     * @param {number|string} perPage - Items per page (number or 'all')
     */
    changeItemsPerPage: function (perPage) {
        const self = this;

        self.itemsPerPage = perPage === 'all' ? 'all' : parseInt(perPage);
        self.currentPage = 1; // Reset to first page
        self.refreshResults();
    },

    /**
     * Update results count display
     *
     * @param {number} filteredCount - Number of filtered results
     * @param {number} totalCount - Total number of results
     */
    updateResultsCount: function (filteredCount, totalCount) {
        const countText = filteredCount === totalCount
            ? filteredCount + ' results'
            : filteredCount + ' of ' + totalCount + ' results';

        jQuery('#results-count').text(countText);
    },

    /**
     * Render pagination controls
     *
     * @return {string} Pagination HTML
     */
    renderPagination: function () {
        const self = this;
        const filteredImages = self.getFilteredAndSortedImages();
        const totalResults = filteredImages.length;
        const totalPages = self.getTotalPages();

        // Hide pagination if no results or showing all
        if (totalResults === 0 || self.itemsPerPage === 'all') {
            return '';
        }

        // Hide pagination if results fit on one page
        if (totalResults <= 10 && self.itemsPerPage === 10) {
            return '';
        }

        const startResult = totalResults > 0 ? (self.currentPage - 1) * self.itemsPerPage + 1 : 0;
        const endResult = Math.min(self.currentPage * self.itemsPerPage, totalResults);

        let html = '<div class="results-pagination">';

        // Pagination info
        html += '<div class="pagination-info">';
        html += 'Showing ' + startResult + '-' + endResult + ' of ' + totalResults + ' results';
        html += '</div>';

        // Pagination controls
        html += '<div class="pagination-controls">';

        // Previous button
        const prevDisabled = self.currentPage === 1 ? ' disabled' : '';
        html += '<button type="button" class="pagination-btn prev"' + prevDisabled + ' data-page="' + (self.currentPage - 1) + '">';
        html += '<span class="dashicons dashicons-arrow-left-alt2"></span>';
        html += '<span class="btn-text">Previous</span>';
        html += '</button>';

        // Page numbers
        html += '<div class="pagination-numbers">';
        html += self.renderPageNumbers(totalPages);
        html += '</div>';

        // Next button
        const nextDisabled = self.currentPage === totalPages ? ' disabled' : '';
        html += '<button type="button" class="pagination-btn next"' + nextDisabled + ' data-page="' + (self.currentPage + 1) + '">';
        html += '<span class="btn-text">Next</span>';
        html += '<span class="dashicons dashicons-arrow-right-alt2"></span>';
        html += '</button>';

        html += '</div>'; // End pagination-controls

        // Per-page selector
        html += '<div class="pagination-per-page">';
        html += '<label>Show:</label>';
        html += '<select class="per-page-select">';
        html += '<option value="10"' + (self.itemsPerPage === 10 ? ' selected' : '') + '>10</option>';
        html += '<option value="25"' + (self.itemsPerPage === 25 ? ' selected' : '') + '>25</option>';
        html += '<option value="50"' + (self.itemsPerPage === 50 ? ' selected' : '') + '>50</option>';
        html += '<option value="100"' + (self.itemsPerPage === 100 ? ' selected' : '') + '>100</option>';
        html += '<option value="all"' + (self.itemsPerPage === 'all' ? ' selected' : '') + '>All</option>';
        html += '</select>';
        html += '<span>per page</span>';
        html += '</div>';

        html += '</div>'; // End results-pagination

        return html;
    },

    /**
     * Render page number buttons with smart ellipsis
     *
     * @param {number} totalPages - Total number of pages
     * @return {string} Page numbers HTML
     */
    renderPageNumbers: function (totalPages) {
        const self = this;
        const current = self.currentPage;
        let html = '';

        if (totalPages <= 7) {
            // Show all pages if 7 or fewer
            for (let i = 1; i <= totalPages; i++) {
                const activeClass = i === current ? ' active' : '';
                html += '<button type="button" class="page-btn' + activeClass + '" data-page="' + i + '">' + i + '</button>';
            }
        } else {
            // Show first page
            html += '<button type="button" class="page-btn' + (current === 1 ? ' active' : '') + '" data-page="1">1</button>';

            // Show ellipsis if needed
            if (current > 3) {
                html += '<span class="pagination-ellipsis">...</span>';
            }

            // Show pages around current page
            const startPage = Math.max(2, current - 1);
            const endPage = Math.min(totalPages - 1, current + 1);

            for (let i = startPage; i <= endPage; i++) {
                const activeClass = i === current ? ' active' : '';
                html += '<button type="button" class="page-btn' + activeClass + '" data-page="' + i + '">' + i + '</button>';
            }

            // Show ellipsis if needed
            if (current < totalPages - 2) {
                html += '<span class="pagination-ellipsis">...</span>';
            }

            // Show last page
            html += '<button type="button" class="page-btn' + (current === totalPages ? ' active' : '') + '" data-page="' + totalPages + '">' + totalPages + '</button>';
        }

        return html;
    },

    /**
     * Bind pagination event handlers
     */
    bindPaginationHandlers: function () {
        const self = this;

        // Page number buttons
        jQuery(document).on('click', '.page-btn', function (e) {
            e.preventDefault();
            const page = parseInt(jQuery(this).data('page'));
            self.changePage(page);
        });

        // Previous/Next buttons
        jQuery(document).on('click', '.pagination-btn', function (e) {
            e.preventDefault();
            if (!jQuery(this).prop('disabled')) {
                const page = parseInt(jQuery(this).data('page'));
                self.changePage(page);
            }
        });

        // Per-page selector
        jQuery(document).on('change', '.per-page-select', function () {
            const perPage = jQuery(this).val();
            self.changeItemsPerPage(perPage);
        });
    },

    /**
     * Populate table rows with processed image data
     */
    populateTableRows: function () {
        const self = this;
        const tbody = jQuery('#table-results-body');
        tbody.empty();

        const paginatedImages = self.getPaginatedImages();
        const filteredImages = self.getFilteredAndSortedImages();

        if (filteredImages.length === 0) {
            tbody.append('<tr><td colspan="5" class="no-results">No results found</td></tr>');
            jQuery('#table-count').text('0');

            // Remove pagination if exists
            jQuery('.results-pagination').remove();
            return;
        }

        paginatedImages.forEach(function (image) {
            const statusClass = image.quality_status || 'unknown';
            const score = image.quality_score || 0;
            const successClass = image.success ? 'success' : 'error';

            const row = `
                <tr class="result-row ${successClass}" data-image-id="${image.attachment_id}">
                    <td class="column-thumbnail">
                        <img src="${image.thumbnail}" alt="" loading="lazy" />
                    </td>
                    <td class="column-filename">
                        <strong>${image.filename}</strong>
                        <span class="image-id">ID: ${image.attachment_id}</span>
                    </td>
                    <td class="column-alt-text">
                        <div class="alt-text-content">${image.alt_text || '<em>No alt text</em>'}</div>
                    </td>
                    <td class="column-quality">
                        <div class="quality-badge quality-${statusClass}">
                            <span class="quality-score">${score}</span>
                            <span class="quality-status">${statusClass}</span>
                        </div>
                    </td>
                    <td class="column-actions">
                        <button type="button" class="button button-small copy-alt-text" data-alt-text="${image.alt_text}" title="Copy Alt Text">
                            <span class="dashicons dashicons-clipboard"></span>
                        </button>
                    </td>
                </tr>
            `;
            tbody.append(row);
        });

        // Update count
        jQuery('#table-count').text(filteredImages.length);
        self.updateResultsCount(filteredImages.length, self.allProcessedImages.length);

        // Render and append pagination
        jQuery('.table-wrapper').next('.results-pagination').remove(); // Remove old pagination
        const paginationHtml = self.renderPagination();
        if (paginationHtml) {
            jQuery('.table-wrapper').after(paginationHtml);
        }

        // Bind copy buttons
        jQuery('.copy-alt-text').off('click').on('click', function (e) {
            e.preventDefault();
            const altText = jQuery(this).data('alt-text');
            self.copyToClipboard(altText);
            jQuery(this).addClass('copied');
            setTimeout(function () {
                jQuery('.copy-alt-text').removeClass('copied');
            }, 2000);
        });
    },

    /**
     * Render card view for bulk results
     */
    renderCardView: function () {
        const self = this;

        const summaryPanel = jQuery('#summary-panel');

        if (summaryPanel.length === 0) {
            return;
        }

        // Create card view container if it doesn't exist
        if (jQuery('#bulk-results-card-view').length === 0) {
            const cardHtml = `
                <div id="bulk-results-card-view" class="bulk-results-card-container" style="display:none;">
                    ${self.renderToolbar()}
                    <div class="card-grid">
                        <!-- Cards will be populated here -->
                    </div>
                </div>
            `;
            summaryPanel.append(cardHtml);
        }

        // Populate cards
        self.populateCardView();
    },

    /**
     * Populate card view with processed image data
     */
    populateCardView: function () {
        const self = this;
        const cardGrid = jQuery('#bulk-results-card-view .card-grid');
        cardGrid.empty();

        const paginatedImages = self.getPaginatedImages();
        const filteredImages = self.getFilteredAndSortedImages();

        if (filteredImages.length === 0) {
            cardGrid.append('<div class="no-results">No results found</div>');

            // Remove pagination if exists
            jQuery('#bulk-results-card-view').find('.results-pagination').remove();
            return;
        }

        paginatedImages.forEach(function (image) {
            const statusClass = image.quality_status || 'unknown';
            const score = image.quality_score || 0;
            const successClass = image.success ? 'success' : 'error';

            const card = `
                <div class="result-card ${successClass}" data-image-id="${image.attachment_id}">
                    <div class="card-thumbnail">
                        <img src="${image.thumbnail}" alt="" loading="lazy" />
                    </div>
                    <div class="card-content">
                        <div class="card-header">
                            <strong class="card-filename">${image.filename}</strong>
                            <span class="card-image-id">ID: ${image.attachment_id}</span>
                        </div>
                        <div class="card-alt-text">
                            <label>Alt Text:</label>
                            <p>${image.alt_text || '<em>No alt text</em>'}</p>
                        </div>
                        <div class="card-footer">
                            <div class="quality-badge quality-${statusClass}">
                                <span class="quality-score">${score}</span>
                                <span class="quality-status">${statusClass}</span>
                            </div>
                            <button type="button" class="button button-small copy-alt-text copy-btn" data-alt-text="${image.alt_text}" title="Copy Alt Text">
                                <span class="dashicons dashicons-clipboard"></span>
                                Copy
                            </button>
                        </div>
                    </div>
                </div>
            `;
            cardGrid.append(card);
        });

        // Update results count
        self.updateResultsCount(filteredImages.length, self.allProcessedImages.length);

        // Render and append pagination AFTER populating cards
        jQuery('#bulk-results-card-view').find('.results-pagination').remove(); // Remove old pagination
        const paginationHtml = self.renderPagination();
        if (paginationHtml) {
            jQuery('#bulk-results-card-view .card-grid').after(paginationHtml);
        }

        // Bind copy buttons
        jQuery('#bulk-results-card-view .copy-alt-text').off('click').on('click', function (e) {
            e.preventDefault();
            const altText = jQuery(this).data('alt-text');
            self.copyToClipboard(altText);
            jQuery(this).addClass('copied').text('Copied!');
            setTimeout(function () {
                jQuery('#bulk-results-card-view .copy-alt-text').removeClass('copied').html('<span class="dashicons dashicons-clipboard"></span> Copy');
            }, 2000);
        });

    },

    /**
     * Switch between card and table views
     *
     * @param {string} view - View type ('card' or 'table')
     */
    switchView: function (view) {
        const self = this;
        self.currentView = view;


        // Update toggle buttons
        jQuery('.view-toggle-btn').removeClass('active');
        jQuery('.view-toggle-btn[data-view="' + view + '"]').addClass('active');

        const tableView = jQuery('#bulk-results-table-view');
        const cardView = jQuery('#bulk-results-card-view');


        if (view === 'table') {
            cardView.hide();
            tableView.fadeIn(300);
        } else {
            tableView.hide();
            cardView.fadeIn(300);
        }
    },

    /**
     * Copy all alt text to clipboard
     */
    copyAllAltText: function () {
        const self = this;
        const altTexts = self.allProcessedImages
            .filter(img => img.alt_text && img.success)
            .map(img => img.filename + ': ' + img.alt_text)
            .join('\n');

        self.copyToClipboard(altTexts);
        Utils.showNotice('All alt text copied to clipboard', 'success');
    },

    /**
     * Export table data to CSV
     */
    exportTableCSV: function () {
        const self = this;

        // Create CSV content
        const headers = ['Attachment ID', 'Filename', 'Alt Text', 'Quality Score', 'Quality Status', 'Success'];
        const rows = self.allProcessedImages.map(img => [
            img.attachment_id,
            img.filename,
            '"' + (img.alt_text || '').replace(/"/g, '""') + '"',
            img.quality_score,
            img.quality_status,
            img.success ? 'Yes' : 'No'
        ]);

        let csvContent = headers.join(',') + '\n';
        rows.forEach(row => {
            csvContent += row.join(',') + '\n';
        });

        // Create download link
        const blob = new Blob([csvContent], { type: 'text/csv;charset=utf-8;' });
        const link = document.createElement('a');
        const url = URL.createObjectURL(blob);

        link.setAttribute('href', url);
        link.setAttribute('download', 'alt-audit-results-' + Date.now() + '.csv');
        link.style.visibility = 'hidden';

        document.body.appendChild(link);
        link.click();
        document.body.removeChild(link);

        Utils.showNotice('CSV file downloaded successfully', 'success');
    },

    /**
     * Render results toolbar with filters, sorting, and search
     *
     * @return {string} HTML string for toolbar
     */
    renderToolbar: function () {
        const self = this;
        return `
            <div class="results-toolbar">
                <div class="toolbar-left">
                    <div class="segmented-control view-toggle-segmented">
                        <button type="button" class="segmented-btn ${self.currentView === 'card' ? 'active' : ''}" data-view="card">
                            <span class="dashicons dashicons-grid-view"></span>
                            <span class="btn-label">Cards</span>
                        </button>
                        <button type="button" class="segmented-btn ${self.currentView === 'table' ? 'active' : ''}" data-view="table">
                            <span class="dashicons dashicons-list-view"></span>
                            <span class="btn-label">Table</span>
                        </button>
                    </div>
                    <div class="toolbar-divider"></div>
                    <div class="results-count-display">
                        <span class="dashicons dashicons-images-alt2"></span>
                        <span id="results-count">0 results</span>
                    </div>
                </div>
                <div class="toolbar-right">
                    <div class="search-box">
                        <span class="dashicons dashicons-search"></span>
                        <input type="search" id="results-search" placeholder="Search images..." value="${self.searchQuery}" />
                        <button type="button" class="clear-search ${self.searchQuery ? '' : 'hidden'}" title="Clear search">
                            <span class="dashicons dashicons-no-alt"></span>
                        </button>
                    </div>
                    <select id="results-filter" class="toolbar-select">
                        <option value="all" ${self.currentFilter === 'all' ? 'selected' : ''}>All Results</option>
                        <option value="success" ${self.currentFilter === 'success' ? 'selected' : ''}>✓ Success Only</option>
                        <option value="error" ${self.currentFilter === 'error' ? 'selected' : ''}>✗ Errors Only</option>
                        <option value="missing" ${self.currentFilter === 'missing' ? 'selected' : ''}>Missing Alt</option>
                        <option value="weak" ${self.currentFilter === 'weak' ? 'selected' : ''}>Weak Alt</option>
                        <option value="good" ${self.currentFilter === 'good' ? 'selected' : ''}>Good Alt</option>
                        <option value="excellent" ${self.currentFilter === 'excellent' ? 'selected' : ''}>Excellent Alt</option>
                    </select>
                    <select id="results-sort" class="toolbar-select">
                        <option value="filename-asc" ${self.currentSort === 'filename-asc' ? 'selected' : ''}>Filename A-Z</option>
                        <option value="filename-desc" ${self.currentSort === 'filename-desc' ? 'selected' : ''}>Filename Z-A</option>
                        <option value="quality-high" ${self.currentSort === 'quality-high' ? 'selected' : ''}>Quality High-Low</option>
                        <option value="quality-low" ${self.currentSort === 'quality-low' ? 'selected' : ''}>Quality Low-High</option>
                        <option value="date-newest" ${self.currentSort === 'date-newest' ? 'selected' : ''}>Date Newest</option>
                        <option value="date-oldest" ${self.currentSort === 'date-oldest' ? 'selected' : ''}>Date Oldest</option>
                    </select>
                </div>
            </div>
        `;
    },

    /**
     * Bind toolbar event handlers
     */
    bindToolbarHandlers: function () {
        const self = this;

        // View toggle
        jQuery(document).on('click', '.segmented-btn', function () {
            const view = jQuery(this).data('view');
            self.switchView(view);
        });

        // Filter change
        jQuery(document).on('change', '#results-filter', function () {
            self.currentFilter = jQuery(this).val();
            self.currentPage = 1; // Reset to first page
            self.refreshResults();
        });

        // Sort change
        jQuery(document).on('change', '#results-sort', function () {
            self.currentSort = jQuery(this).val();
            self.currentPage = 1; // Reset to first page
            self.refreshResults();
        });

        // Search input with debouncing
        let searchTimeout;
        jQuery(document).on('input', '#results-search', function () {
            clearTimeout(searchTimeout);
            const value = jQuery(this).val();

            searchTimeout = setTimeout(function () {
                self.searchQuery = value;
                self.currentPage = 1; // Reset to first page
                self.refreshResults();

                // Show/hide clear button
                if (value.trim() !== '') {
                    jQuery('.clear-search').removeClass('hidden');
                } else {
                    jQuery('.clear-search').addClass('hidden');
                }
            }, 300);
        });

        // Clear search
        jQuery(document).on('click', '.clear-search', function () {
            jQuery('#results-search').val('');
            self.searchQuery = '';
            self.currentPage = 1; // Reset to first page
            self.refreshResults();
            jQuery(this).addClass('hidden');
        });
    },

    /**
     * Refresh results after filter/sort/search change
     */
    refreshResults: function () {
        const self = this;

        if (self.currentView === 'table') {
            self.populateTableRows();
        } else {
            self.populateCardView();
        }
    },

    /**
     * Copy text to clipboard using modern Clipboard API with fallback
     *
     * @param {string} text - Text to copy
     * @return {Promise} Promise that resolves when text is copied
     */
    copyToClipboard: async function (text) {
        // Use modern Clipboard API if available
        if (navigator.clipboard && navigator.clipboard.writeText) {
            try {
                await navigator.clipboard.writeText(text);
            } catch (err) {
                console.error('Failed to copy text:', err);
                // If modern API fails, try fallback
                return BulkTools.copyToClipboardFallback(text);
            }
        } else {
            // Use fallback for older browsers
            return BulkTools.copyToClipboardFallback(text);
        }
    },

    /**
     * Fallback clipboard copy method for older browsers
     *
     * @param {string} text - Text to copy
     * @return {Promise} Promise that resolves when text is copied
     */
    copyToClipboardFallback: function (text) {
        return new Promise(function (resolve, reject) {
            try {
                const textarea = document.createElement('textarea');
                textarea.value = text;
                textarea.style.position = 'fixed';
                textarea.style.opacity = '0';
                textarea.setAttribute('readonly', '');
                textarea.style.top = '-9999px';
                textarea.style.left = '-9999px';

                document.body.appendChild(textarea);

                // Select text
                textarea.focus();
                textarea.select();
                textarea.setSelectionRange(0, textarea.value.length);

                // Try to copy
                let success = false;
                try {
                    // Intentional use of deprecated execCommand as fallback for older browsers
                    // Modern browsers use Clipboard API (handled above)
                    // @ts-ignore - execCommand is deprecated but needed for legacy browser support
                    success = document.execCommand('copy');
                } catch (err) {
                    console.error('execCommand failed:', err);
                }

                document.body.removeChild(textarea);

                if (success) {
                    resolve();
                } else {
                    reject(new Error('Copy command failed'));
                }
            } catch (err) {
                reject(err);
            }
        });
    },

    /**
     * Export all data (from Data Export & Backup section)
     */
    exportData: function () {
        const self = this;
        const format = jQuery('input[name="export_format"]:checked').val() || 'csv';
        const button = jQuery('#export-data');
        const buttonText = button.html();

        // Show loading state
        button.prop('disabled', true).html('<span class="dashicons dashicons-update-alt spin"></span> Generating Export...');

        jQuery.ajax({
            url: altaudit82aiAdmin.ajaxUrl,
            type: 'POST',
            data: {
                action: 'altaudit82ai_export_data',
                nonce: altaudit82aiAdmin.nonce,
                format: format
            },
            success: function (response) {
                if (response.success && response.data) {
                    // Create download link
                    const blob = new Blob([response.data.content], { type: response.data.mime_type });
                    const url = window.URL.createObjectURL(blob);
                    const a = document.createElement('a');
                    a.href = url;
                    a.download = response.data.filename;
                    document.body.appendChild(a);
                    a.click();
                    window.URL.revokeObjectURL(url);
                    document.body.removeChild(a);

                    // Show success message
                    Utils.showNotice('Data exported successfully!', 'success');
                } else {
                    Utils.showNotice('Error: ' + (response.data || 'Failed to export data'), 'error');
                }
            },
            error: function () {
                Utils.showNotice('Failed to export data. Please try again.', 'error');
            },
            complete: function () {
                // Restore button
                button.prop('disabled', false).html(buttonText);
            }
        });
    }
};
