/**
 * Audit Page Module
 *
 * Handles audit page specific functionality including table row updates
 * and statistics refreshing.
 *
 * @package AltAudit
 * @since 1.0.0
 */

import { Utils } from './utils.js';

export const AuditPage = {
    /**
     * Update audit table row after alt text generation
     *
     * @param {jQuery} $row - Table row element
     * @param {Object} data - Response data
     */
    updateAuditTableRow: function($row, data) {
        // Find the alt text cell
        const $altTextCell = $row.find('td').filter(function() {
            return jQuery(this).find('.alt-text-display, .alt-audit-missing').length > 0;
        });

        // Check if alt text was missing and needs to be replaced with proper structure
        const $missingSpan = $altTextCell.find('.alt-audit-missing');
        const attachmentId = $row.find('input[type="checkbox"]').val();

        if ($missingSpan.length > 0 && data.alt_text) {
            // Replace "Missing" span with proper alt text structure
            const displayText = data.alt_text.length > 100 ? data.alt_text.substring(0, 100) + '...' : data.alt_text;
            const $newContainer = jQuery('<div class="alt-audit-alt-text"></div>')
                .attr('data-attachment-id', attachmentId)
                .attr('data-full-alt-text', data.alt_text);
            const $displaySpan = jQuery('<span class="alt-text-display"></span>').text(displayText);
            const $autoGenIcon = jQuery('<span class="alt-audit-auto-generated" title="Auto-generated">🤖</span>');
            const $editButton = jQuery('<button type="button" class="button-link edit-alt-text" title="Edit alt text"></button>');
            const $editIcon = jQuery('<span class="dashicons dashicons-edit"></span>');

            $editButton.append($editIcon);
            $newContainer.append($displaySpan).append(' ').append($autoGenIcon).append($editButton);
            $altTextCell.html($newContainer);
        } else {
            // Update existing alt text display
            const $altTextContainer = $row.find('.alt-audit-alt-text');
            const $altTextDisplay = $row.find('.alt-text-display');
            if ($altTextDisplay.length && data.alt_text) {
                const displayText = data.alt_text.length > 100 ? data.alt_text.substring(0, 100) + '...' : data.alt_text;
                $altTextDisplay.text(displayText);
                $altTextContainer.attr('data-full-alt-text', data.alt_text);
            }
        }

        // Update quality score if provided
        if (data.quality) {
            const quality = data.quality;
            const $scoreElement = $row.find('.alt-audit-score .score-number');
            const $scoreFill = $row.find('.score-fill');
            const $statusElement = $row.find('.alt-audit-status');
            const $scoreContainer = $row.find('.alt-audit-score');

            // Update score number and bar
            if ($scoreElement.length && quality.score !== undefined) {
                $scoreElement.text(quality.score);
                $scoreFill.css('width', quality.score + '%');

                // Update score class
                $scoreContainer.removeClass('score-missing score-weak score-good score-excellent');
                if (quality.score >= 80) {
                    $scoreContainer.addClass('score-excellent');
                } else if (quality.score >= 60) {
                    $scoreContainer.addClass('score-good');
                } else if (quality.score > 0) {
                    $scoreContainer.addClass('score-weak');
                } else {
                    $scoreContainer.addClass('score-missing');
                }
            }

            // Update status
            if ($statusElement.length && quality.status) {
                $statusElement.removeClass().addClass('alt-audit-status alt-audit-status-' + quality.status);
                $statusElement.text(quality.status.charAt(0).toUpperCase() + quality.status.slice(1));
            }
        }

        // Show success message
        Utils.showNotice(data.message || 'Alt text updated successfully', 'success');

        // Refresh statistics on audit page (with small delay to ensure DB is updated)
        if (jQuery('.alt-audit-statistics-section').length > 0) {
            setTimeout(() => {
                this.refreshAuditStatistics();
            }, 300);
        }
    },

    /**
     * Refresh audit page statistics
     */
    refreshAuditStatistics: function() {
        const self = this;

        jQuery.ajax({
            url: altaudit82aiAdmin.ajaxUrl,
            type: 'POST',
            data: {
                action: 'altaudit82ai_get_statistics',
                nonce: altaudit82aiAdmin.nonce
            },
            success: function(response) {
                if (response.success && response.data) {
                    self.updateAuditStatistics(response.data);
                }
            },
            error: function(xhr, status, error) {
                console.log('Failed to refresh statistics:', error);
            }
        });
    },

    /**
     * Update audit statistics with animation
     *
     * @param {Object} stats - Statistics data
     */
    updateAuditStatistics: function(stats) {
        // Update ring score with animation
        this.updateRingScore(stats.avg_score);

        // Update status breakdown counts
        this.updateStatusCounts(stats);
    },

    /**
     * Update ring score with animation
     *
     * @param {number} newScore - New score value
     */
    updateRingScore: function(newScore) {
        const $ringScore = jQuery('.ring-score');
        const $ringProgress = jQuery('.ring-progress');

        if ($ringScore.length === 0 || $ringProgress.length === 0) {
            return;
        }

        // Get current score
        const currentScore = parseInt($ringScore.text()) || 0;

        // Animate score number
        jQuery({ score: currentScore }).animate(
            { score: newScore },
            {
                duration: 1000,
                easing: 'swing',
                step: function() {
                    $ringScore.text(Math.round(this.score));
                },
                complete: function() {
                    $ringScore.text(Math.round(newScore));
                }
            }
        );

        // Animate ring progress
        const circumference = 326.73; // 2 * PI * 52
        const dashOffset = circumference - (circumference * newScore / 100);
        $ringProgress.css({
            'stroke-dashoffset': dashOffset
        });

        // Update description text
        const $description = jQuery('.alt-audit-overall-score .stat-description');
        if ($description.length > 0) {
            let descriptionText = '';
            if (newScore >= 80) {
                descriptionText = 'Excellent accessibility';
            } else if (newScore >= 60) {
                descriptionText = 'Good accessibility';
            } else if (newScore >= 40) {
                descriptionText = 'Needs improvement';
            } else {
                descriptionText = 'Action required';
            }
            $description.text(descriptionText);
        }
    },

    /**
     * Update status breakdown counts with animation
     *
     * @param {Object} stats - Statistics data
     */
    updateStatusCounts: function(stats) {
        const statusMap = {
            'missing': stats.missing || 0,
            'weak': stats.weak || 0,
            'good': stats.good || 0,
            'excellent': stats.excellent || 0,
            // 'decorative': stats.decorative || 0 // Removed - simplified status
        };

        jQuery.each(statusMap, function(status, newCount) {
            const $statusItem = jQuery('.status-item.status-' + status);
            const $countElement = $statusItem.find('.status-count');

            if ($countElement.length === 0) {
                return;
            }

            // Get current count
            const currentCount = parseInt($countElement.text()) || 0;

            // Animate count
            jQuery({ count: currentCount }).animate(
                { count: newCount },
                {
                    duration: 800,
                    easing: 'swing',
                    step: function() {
                        $countElement.text(Math.round(this.count));
                    },
                    complete: function() {
                        $countElement.text(newCount);
                    }
                }
            );
        });
    }
};
