/**
 * Alt Audit AJAX Audit Page Module
 *
 * Handles all AJAX operations for the Accessibility Audit page.
 * Provides seamless, no-reload user experience with progress tracking,
 * notifications, and real-time table updates.
 *
 * @package AltAudit
 * @since 1.0.0
 */

import { Utils } from './utils.js';

export const AuditPageAjax = {
	/**
	 * Current processing state
	 */
	processing: false,

	/**
	 * Current request abort controller
	 */
	abortController: null,

	/**
	 * Initialize the module
	 */
	init: function() {
		this.bindEvents();
		this.initProgress();
	},

	/**
	 * Bind event handlers
	 */
	bindEvents: function() {
		const $ = jQuery;

		// Bulk actions form submit
		$(document).on('click', '#doaction, #doaction2', this.handleBulkActionClick.bind(this));

		// Search form submit
		$(document).on('submit', '.alt-audit-images-form', this.handleSearchSubmit.bind(this));

		// Filter button click - apply status filter and search
		$(document).on('click', 'input[name="filter_action"]', this.handleFilterButtonClick.bind(this));

		// Column header sorting - separate handlers for ASC/DESC indicators
		$(document).on('click', '.wp-list-table thead .sorting-indicator, .wp-list-table tfoot .sorting-indicator', this.handleSortIndicatorClick.bind(this));

		// Column header sorting - fallback for regular link click
		$(document).on('click', '.wp-list-table thead th.sortable a, .wp-list-table tfoot th.sortable a', this.handleSortClick.bind(this));

		// Pagination links
		$(document).on('click', '.tablenav-pages a', this.handlePaginationClick.bind(this));

		// Per page change
		$(document).on('change', '#altaudit82ai_images_per_page', this.handlePerPageChange.bind(this));

		// Individual row actions
		$(document).on('click', '.alt-audit-generate-rule', this.handleGenerateRuleClick.bind(this));
		$(document).on('click', '.alt-audit-generate-ai', this.handleGenerateAIClick.bind(this));

		// Scan All Images button
		$(document).on('click', '#altaudit82ai-scan-all-btn', this.handleScanAllImagesClick.bind(this));

		// View image in lightbox
		$(document).on('click', '.row-actions .view a', this.handleViewImageClick.bind(this));

		// Close lightbox
		$(document).on('click', '.alt-audit-lightbox-close, .alt-audit-lightbox-backdrop', this.closeLightbox.bind(this));

		// Cancel operation
		$(document).on('click', '.alt-audit-cancel-operation', this.cancelOperation.bind(this));
	},

	/**
	 * Initialize progress indicators
	 */
	initProgress: function() {
		const $ = jQuery;

		// Create progress container if it doesn't exist
		if ($('.alt-audit-progress-container').length === 0) {
			const progressHtml = `
				<div class="alt-audit-progress-container" style="display:none;">
					<div class="alt-audit-progress-bar">
						<div class="alt-audit-progress-fill"></div>
					</div>
					<div class="alt-audit-progress-text">
						<span class="progress-message"></span>
						<button type="button" class="button alt-audit-cancel-operation">Cancel</button>
					</div>
				</div>
			`;
			$('.audit-table-section').prepend(progressHtml);
		}

		// Create notification container if it doesn't exist
		if ($('.alt-audit-notifications').length === 0) {
			const notificationsHtml = '<div class="alt-audit-notifications"></div>';
			$('.wrap.alt-audit-section').prepend(notificationsHtml);
		}
	},

	/**
	 * Handle bulk action button click
	 *
	 * @param {Event} e Click event
	 */
	handleBulkActionClick: function(e) {
		e.preventDefault();
		const $ = jQuery;

		// Determine which bulk action selector to use
		const actionSelector = $(e.target).attr('id') === 'doaction' ?
			'select[name="action"]' :
			'select[name="action2"]';

		const action = $(actionSelector).val();

		if (!action || action === '-1') {
			return;
		}

		// Get selected attachment IDs
		const attachmentIds = [];
		$('input[name="attachment_id[]"]:checked').each(function() {
			attachmentIds.push($(this).val());
		});

		if (attachmentIds.length === 0) {
			this.showNotification('Please select at least one image.', 'warning');
			return;
		}

		// Confirm delete action
		if (action === 'delete') {
			const confirmMessage = attachmentIds.length === 1 ?
				'Are you sure you want to delete this image?' :
				`Are you sure you want to delete ${attachmentIds.length} images?`;

			if (!confirm(confirmMessage)) {
				return;
			}
		}

		// Show AI credit warning for AI generation
		if (action === 'generate_ai') {
			const confirmMessage = attachmentIds.length === 1 ?
				'This will use 1 credit to generate AI-powered alt text. Continue?' :
				`This will use ${attachmentIds.length} credits to generate AI-powered alt text. Continue?`;

			if (!confirm(confirmMessage)) {
				return;
			}
		}

		// Process bulk action
		this.processBulkAction(action, attachmentIds);
	},

	/**
	 * Process bulk action via AJAX
	 *
	 * @param {string} action Bulk action name
	 * @param {Array} attachmentIds Array of attachment IDs
	 */
	processBulkAction: async function(action, attachmentIds) {
		const $ = jQuery;

		if (this.processing) {
			this.showNotification('Another operation is in progress.', 'warning');
			return;
		}

		this.processing = true;
		this.toggleFormState(true);
		this.showProgress(true);

		const total = attachmentIds.length;
		let processed = 0;
		let successCount = 0;
		let errorCount = 0;
		const errors = [];

		// Create abort controller for cancellation
		this.abortController = new AbortController();

		try {
			// Process images sequentially to show progress
			for (let i = 0; i < attachmentIds.length; i++) {
				const attachmentId = attachmentIds[i];

				// Check if operation was cancelled
				if (this.abortController.signal.aborted) {
					throw new Error('Operation cancelled by user');
				}

				this.updateProgress(i + 1, total);

				try {
					const result = await this.processSingleImage(action, attachmentId);

					if (result.success) {
						successCount++;
						// Update the table row with new data
						if (result.data) {
							this.updateTableRow(attachmentId, result.data);
						}
					} else {
						errorCount++;
						errors.push({
							id: attachmentId,
							message: result.message || 'Unknown error'
						});
					}
				} catch (error) {
					errorCount++;
					errors.push({
						id: attachmentId,
						message: error.message
					});
				}

				processed++;
			}

			// Show completion message
			this.showCompletionMessage(action, successCount, errorCount, errors);

			// Refresh statistics after bulk operation
			this.refreshStatistics();

			// Uncheck all checkboxes
			$('input[name="attachment_id[]"]').prop('checked', false);
			$('#cb-select-all-1, #cb-select-all-2').prop('checked', false);

		} catch (error) {
			if (error.message === 'Operation cancelled by user') {
				this.showNotification(
					`Operation cancelled. Processed ${processed} of ${total} images.`,
					'info'
				);
			} else {
				this.showNotification(
					`Error: ${error.message}`,
					'error'
				);
			}
		} finally {
			this.processing = false;
			this.toggleFormState(false);
			this.showProgress(false);
			this.abortController = null;
		}
	},

	/**
	 * Process single image with specific action
	 *
	 * @param {string} action Action to perform
	 * @param {number} attachmentId Attachment ID
	 * @returns {Promise} AJAX promise
	 */
	processSingleImage: function(action, attachmentId) {
		const $ = jQuery;

		console.log('=== AJAX Request Starting ===');
		console.log('Action:', action);
		console.log('Attachment ID:', attachmentId);
		console.log('Ajax URL:', window.altaudit82aiAdmin.ajaxUrl);

		return new Promise((resolve, reject) => {
			$.ajax({
				url: window.altaudit82aiAdmin.ajaxUrl,
				type: 'POST',
				data: {
					action: 'altaudit82ai_single_bulk_action',
					nonce: window.altaudit82aiAdmin.nonce,
					bulk_action: action,
					attachment_id: attachmentId
				},
				signal: this.abortController ? this.abortController.signal : undefined,
				success: function(response) {
					console.log('=== AJAX Response Received ===');
					console.log('Full Response:', response);
					console.log('Response.success:', response.success);
					console.log('Response.data:', response.data);

					if (response.data) {
						console.log('Has row_html:', !!response.data.row_html);
						console.log('row_html length:', response.data.row_html ? response.data.row_html.length : 0);
						console.log('Attachment ID in response:', response.data.attachment_id);
						console.log('Alt text:', response.data.alt_text);
						console.log('Status:', response.data.status);
						console.log('Quality score:', response.data.quality_score);
					}

					if (response.success) {
						resolve({
							success: true,
							data: response.data
						});
					} else {
						// Handle both string and object error responses
						let errorMessage = 'Operation failed';
						if (typeof response.data === 'string') {
							errorMessage = response.data;
						} else if (response.data && response.data.message) {
							errorMessage = response.data.message;
						}

						console.error('=== API Error ===');
						console.error('Error Message:', errorMessage);
						if (response.data && response.data.debug) {
							console.error('Debug Data:', response.data.debug);
						}

						resolve({
							success: false,
							message: errorMessage,
							debug: response.data
						});
					}
				},
				error: function(_xhr, status, error) {
					console.error('=== AJAX Error ===');
					console.error('Status:', status);
					console.error('Error:', error);

					if (status === 'abort') {
						reject(new Error('Operation cancelled'));
					} else {
						reject(new Error(error || 'AJAX request failed'));
					}
				}
			});
		});
	},

	/**
	 * Handle individual row rule-based generation
	 *
	 * @param {Event} e Click event
	 */
	handleGenerateRuleClick: function(e) {
		e.preventDefault();
		const $ = jQuery;
		const attachmentId = $(e.currentTarget).data('attachment-id');

		if (!attachmentId) {
			return;
		}

		this.processSingleImage('generate_rule_based', attachmentId)
			.then(result => {
				if (result.success && result.data) {
					this.updateTableRow(attachmentId, result.data);
					this.showNotification('Alt text generated successfully!', 'success');
					this.refreshStatistics();
				} else {
					this.showNotification(
						result.message || 'Failed to generate alt text.',
						'error'
					);
				}
			})
			.catch(error => {
				this.showNotification(
					`Error: ${error.message}`,
					'error'
				);
			});
	},

	/**
	 * Handle individual row AI generation
	 *
	 * @param {Event} e Click event
	 */
	handleGenerateAIClick: function(e) {
		e.preventDefault();
		const $ = jQuery;
		const attachmentId = $(e.currentTarget).data('attachment-id');

		if (!attachmentId) {
			return;
		}

		if (!confirm('This will use 1 credit to generate AI-powered alt text. Continue?')) {
			return;
		}

		this.processSingleImage('generate_ai', attachmentId)
			.then(result => {
				if (result.success && result.data) {
					this.updateTableRow(attachmentId, result.data);
					this.showNotification('AI-powered alt text generated successfully!', 'success');
					this.refreshStatistics();
				} else {
					this.showNotification(
						result.message || 'Failed to generate alt text.',
						'error'
					);
				}
			})
			.catch(error => {
				this.showNotification(
					`Error: ${error.message}`,
					'error'
				);
			});
	},

	/**
	 * Handle search form submit
	 *
	 * Triggers when user presses Enter in search field or clicks "Search images" button.
	 * Applies both search term and current status filter.
	 *
	 * @param {Event} e Submit event
	 */
	handleSearchSubmit: function(e) {
		e.preventDefault();
		const $ = jQuery;

		const searchTerm = $('input[name="s"]').val();
		const currentParams = this.getCurrentParams();

		// Update search term
		if (searchTerm) {
			currentParams.s = searchTerm;
		} else {
			delete currentParams.s;
		}

		// Reset to first page on search
		currentParams.paged = 1;

		this.refreshTable(currentParams);
	},

	/**
	 * Handle filter button click
	 *
	 * @param {Event} e Click event
	 */
	handleFilterButtonClick: function(e) {
		e.preventDefault();
		const $ = jQuery;

		const status = $('#status_filter').val();
		const searchTerm = $('input[name="s"]').val();
		const currentParams = this.getCurrentParams();

		// Update status filter
		if ( status ) {
			currentParams.status_filter = status;
		} else {
			delete currentParams.status_filter;
		}

		// Update search term
		if ( searchTerm ) {
			currentParams.s = searchTerm;
		} else {
			delete currentParams.s;
		}

		currentParams.paged = 1; // Reset to first page on filter change

		this.refreshTable(currentParams);
	},

	/**
	 * Handle sort indicator click (separate ASC/DESC arrows)
	 *
	 * @param {Event} e Click event
	 */
	handleSortIndicatorClick: function(e) {
		e.preventDefault();
		e.stopPropagation();
		const $ = jQuery;

		console.log('=== Sort Indicator Click Handler Triggered ===');

		const $indicator = $(e.currentTarget);
		const $th = $indicator.closest('th.sortable');

		// Get the column ID from the th element's class
		const columnClass = $th.attr('class').split(' ').find(cls => cls.startsWith('column-'));
		if (!columnClass) {
			console.warn('No column class found');
			return;
		}

		const columnId = columnClass.replace('column-', '');
		console.log('Column ID:', columnId);

		// Determine desired order from clicked indicator
		const newOrder = $indicator.hasClass('asc') ? 'asc' : 'desc';
		console.log('Clicked indicator order:', newOrder);

		// Get current form state (filters, search)
		const currentParams = this.getCurrentParams();

		// Set sort parameters
		currentParams.orderby = columnId;
		currentParams.order = newOrder;
		currentParams.paged = 1; // Reset to page 1 when sorting

		console.log('New sort params:', { orderby: columnId, order: newOrder });

		this.refreshTable(currentParams);
	},

	/**
	 * Handle column header sort click (fallback for clicking outside indicators)
	 *
	 * @param {Event} e Click event
	 */
	handleSortClick: function(e) {
		// Check if click originated from a sorting indicator
		const $ = jQuery;
		if ($(e.target).hasClass('sorting-indicator') || $(e.target).closest('.sorting-indicator').length > 0) {
			// Let the indicator handler deal with it
			return;
		}

		e.preventDefault();
		e.stopPropagation(); // Ensure event doesn't bubble up

		console.log('=== Sort Click Handler Triggered ===');

		const $link = $(e.currentTarget);
		const href = $link.attr('href');

		console.log('Sort Link Href:', href);

		if (!href) {
			console.warn('No href found on sort link');
			return;
		}

		// Parse URL parameters from sort link
		const url = new URL(href, window.location.origin);
		const sortParams = Object.fromEntries(url.searchParams.entries());



		// Get current form state (filters, search)
		const currentParams = this.getCurrentParams();
		console.log('Current params:', currentParams);
		console.log('Sort params:', sortParams);

		// Merge current params with sort params (sort overrides)
		// Reset to page 1 when sorting
		const mergedParams = {...currentParams, ...sortParams, paged: 1};

		this.refreshTable(mergedParams);
	},

	/**
	 * Handle pagination link click
	 *
	 * @param {Event} e Click event
	 */
	handlePaginationClick: function(e) {
		e.preventDefault();
		const $ = jQuery;

		console.log('=== Pagination Click Handler ===');
		console.log('Event:', e);

		const $link = $(e.currentTarget);
		const href = $link.attr('href');

		console.log('Link:', $link.get(0));
		console.log('Href:', href);

		if (!href) {
			console.warn('No href found on pagination link');
			return;
		}

		// Parse URL parameters from pagination link
		const url = new URL(href, window.location.origin);
		const paginationParams = Object.fromEntries(url.searchParams.entries());

		console.log('Parsed URL:', url.toString());
		console.log('Extracted pagination params:', paginationParams);

		// Get current form state (filters, search)
		const currentParams = this.getCurrentParams();
		console.log('Current form params:', currentParams);

		// Merge current params with pagination params (pagination overrides)
		const mergedParams = {...currentParams, ...paginationParams};
		console.log('Merged params:', mergedParams);

		this.refreshTable(mergedParams);

		// Scroll to top of table
		$('html, body').animate({
			scrollTop: $('.audit-table-section').offset().top - 100
		}, 300);
	},

	/**
	 * Handle per page change
	 *
	 * @param {Event} e Change event
	 */
	handlePerPageChange: function(e) {
		const $ = jQuery;

		const perPage = $(e.target).val();
		const currentParams = this.getCurrentParams();

		currentParams.per_page = perPage;
		currentParams.paged = 1; // Reset to first page

		this.refreshTable(currentParams);
	},

	/**
	 * Refresh table with new parameters
	 *
	 * @param {Object} params URL parameters
	 */
	refreshTable: function(params) {
		const $ = jQuery;

		console.log('=== refreshTable called ===');
		console.log('Params:', params);

		// Show loading state
		$('.wp-list-table').addClass('alt-audit-loading');

		$.ajax({
			url: window.altaudit82aiAdmin.ajaxUrl,
			type: 'POST',
			data: {
				action: 'altaudit82ai_refresh_table',
				nonce: window.altaudit82aiAdmin.nonce,
				params: params
			},
			success: (response) => {
				

				if (response.success && response.data.html) {
					// Replace table content
					console.log('Replacing .audit-table-section content');
					console.log('HTML length:', response.data.html.length);

					$('.audit-table-section').html(response.data.html);

					// Update URL without reload
					const newUrl = this.buildUrl(params);
					console.log('Updating URL to:', newUrl);
				//	window.history.pushState({}, '', newUrl);

					// Refresh statistics
					this.refreshStatistics();

					console.log('Table refresh complete');
				} else {
					console.error('Table refresh failed:', response);
					this.showNotification(
						response.data || 'Failed to refresh table.',
						'error'
					);
				}
			},
			error: (xhr, status, error) => {
				console.error('=== AJAX Error ===');
				console.error('Status:', status);
				console.error('Error:', error);
				console.error('Response:', xhr.responseText);

				this.showNotification(
					'Failed to refresh table. Please reload the page.',
					'error'
				);
			},
			complete: () => {
				$('.wp-list-table').removeClass('alt-audit-loading');
			}
		});
	},

	/**
	 * Update single table row
	 *
	 * @param {number} attachmentId Attachment ID
	 * @param {Object} data Updated row data
	 */
	updateTableRow: function(attachmentId, data) {
		const $ = jQuery;

		console.log('=== updateTableRow Called ===');
		console.log('Attachment ID:', attachmentId);
		console.log('Data:', data);
		console.log('Has row_html:', !!data.row_html);

		// Find row by data-attachment-id attribute
		const $row = $(`tr[data-attachment-id="${attachmentId}"]`);

		console.log('Selector used:', `tr[data-attachment-id="${attachmentId}"]`);
		console.log('Rows found:', $row.length);

		if ($row.length > 0) {
			console.log('Row found:', $row.get(0));
			console.log('Row HTML:', $row.get(0).outerHTML.substring(0, 200));
		} else {
			console.warn('=== Row NOT Found - Debugging ===');
			console.log('All table rows:', $('table.wp-list-table tbody tr').length);

			// Check if ANY rows have the data-attachment-id attribute
			$('table.wp-list-table tbody tr').each(function(index) {
				const $tr = $(this);
				const dataId = $tr.attr('data-attachment-id');
				const id = $tr.attr('id');
				console.log(`Row ${index}: id="${id}", data-attachment-id="${dataId}"`);
			});

			return;
		}

		// If we have full row HTML, replace the entire row
		if (data.row_html) {
			console.log('Replacing row with new HTML...');
			console.log('New row HTML length:', data.row_html.length);
			console.log('New row HTML preview:', data.row_html.substring(0, 200));

			const $newRow = $(data.row_html);
			console.log('Parsed new row:', $newRow.length, 'elements');
			console.log('New row attributes:', $newRow.attr('data-attachment-id'));

			$row.replaceWith($newRow);

			// Add flash animation to new row
			const $replacedRow = $(`tr[data-attachment-id="${attachmentId}"]`);
			console.log('After replace, found:', $replacedRow.length, 'rows');

			$replacedRow.addClass('alt-audit-row-updated');
			setTimeout(() => {
				$replacedRow.removeClass('alt-audit-row-updated');
			}, 2000);

			console.log('Row successfully updated for attachment ID:', attachmentId);
			return;
		}

		// Fallback: Update individual cells if row_html not provided
		// Update alt text column
		const $altTextCell = $row.find('.column-alt_text');
		if (data.alt_text !== undefined) {
			const displayText = data.alt_text.length > 100 ?
				data.alt_text.substring(0, 100) + '...' :
				data.alt_text;

			const autoGenIndicator = data.is_auto_generated ?
				' <span class="alt-audit-auto-generated" title="Auto-generated">🤖</span>' :
				'';

			$altTextCell.html(`
				<div class="alt-audit-alt-text" data-attachment-id="${attachmentId}" data-full-alt-text="${Utils.escapeHtml(data.alt_text)}">
					<span class="alt-text-display">${Utils.escapeHtml(displayText)}</span>${autoGenIndicator}
					<button type="button" class="button-link edit-alt-text" title="Edit alt text">
						<span class="dashicons dashicons-edit"></span>
					</button>
				</div>
			`);
		}

		// Update status column
		if (data.status) {
			const statusLabels = {
				missing: 'Missing',
				weak: 'Weak',
				good: 'Good',
				excellent: 'Excellent',
				// decorative: 'Decorative' // Removed - simplified status
			};

			const statusLabel = statusLabels[data.status] || data.status;

			$row.find('.column-status').html(`
				<span class="alt-audit-status alt-audit-status-${data.status}">${statusLabel}</span>
			`);
		}

		// Update quality score column
		if (data.quality_score !== undefined) {
			const score = data.quality_score;
			let scoreClass = 'alt-audit-score';

			if (score >= 80) {
				scoreClass += ' score-excellent';
			} else if (score >= 60) {
				scoreClass += ' score-good';
			} else if (score > 0) {
				scoreClass += ' score-weak';
			} else {
				scoreClass += ' score-missing';
			}

			$row.find('.column-quality_score').html(`
				<div class="${scoreClass}">
					<span class="score-number">${score}</span>
					<span class="score-bar">
						<span class="score-fill" style="width: ${score}%"></span>
					</span>
				</div>
			`);
		}

		// Add flash animation
		$row.addClass('alt-audit-row-updated');
		setTimeout(() => {
			$row.removeClass('alt-audit-row-updated');
		}, 2000);

		console.log(`Row cells updated for attachment ID: ${attachmentId}`);
	},

	/**
	 * Refresh statistics section
	 */
	refreshStatistics: function() {
		const $ = jQuery;

		$.ajax({
			url: window.altaudit82aiAdmin.ajaxUrl,
			type: 'POST',
			data: {
				action: 'altaudit82ai_get_statistics',
				nonce: window.altaudit82aiAdmin.nonce
			},
			success: (response) => {
				if (response.success && response.data) {
					this.updateStatistics(response.data);
				}
			}
		});
	},

	/**
	 * Update statistics display
	 *
	 * @param {Object} stats Statistics data
	 */
	updateStatistics: function(stats) {
		const $ = jQuery;

		// Update overall score
		if (stats.avg_score !== undefined) {
			$('.ring-score').text(stats.avg_score);

			// Update ring progress
			const $ring = $('.ring-progress');
			$ring.css('--score', stats.avg_score);
		}

		// Update total images
		if (stats.total !== undefined) {
			$('.alt-audit-stat-card .stat-value').text(stats.total.toLocaleString());
		}

		// Update status counts
		if (stats.missing !== undefined) {
			$('.status-item.status-missing .status-count').text(stats.missing);
		}
		if (stats.weak !== undefined) {
			$('.status-item.status-weak .status-count').text(stats.weak);
		}
		if (stats.good !== undefined) {
			$('.status-item.status-good .status-count').text(stats.good);
		}
		if (stats.excellent !== undefined) {
			$('.status-item.status-excellent .status-count').text(stats.excellent);
		}
	},

	/**
	 * Show completion message after bulk operation
	 *
	 * @param {string} action Action performed
	 * @param {number} successCount Number of successful operations
	 * @param {number} errorCount Number of failed operations
	 * @param {Array} errors Array of error details
	 */
	showCompletionMessage: function(action, successCount, errorCount, errors) {
		const actionMessages = {
			generate_rule_based: 'Generated rule-based alt text',
			generate_ai: 'Generated AI-powered alt text',
			delete: 'Deleted'
		};

		const actionMessage = actionMessages[action] || 'Processed';

		let message = '';
		let type = 'success';

		if (errorCount === 0) {
			message = `${actionMessage} for ${successCount} image${successCount !== 1 ? 's' : ''}.`;
		} else if (successCount === 0) {
			message = `Failed to process ${errorCount} image${errorCount !== 1 ? 's' : ''}.`;
			type = 'error';
		} else {
			message = `${actionMessage} for ${successCount} image${successCount !== 1 ? 's' : ''}, ${errorCount} failed.`;
			type = 'warning';
		}

		// Add error details if present
		if (errors.length > 0 && errors.length <= 5) {
			message += '<br><br>Errors:<br>';
			errors.forEach(error => {
				message += `ID ${error.id}: ${error.message}<br>`;
			});
		} else if (errors.length > 5) {
			message += `<br><br>${errors.length} images failed to process.`;
		}

		this.showNotification(message, type);
	},

	/**
	 * Show notification message
	 *
	 * @param {string} message Notification message
	 * @param {string} type Notification type (success, error, warning, info)
	 */
	showNotification: function(message, type = 'info') {
		const $ = jQuery;

		const $notification = $(`
			<div class="alt-audit-notification alt-audit-notification-${type}">
				<span class="notification-icon"></span>
				<span class="notification-message">${message}</span>
				<button type="button" class="notification-dismiss" aria-label="Dismiss">
					<span class="dashicons dashicons-no-alt"></span>
				</button>
			</div>
		`);

		$('.alt-audit-notifications').append($notification);

		// Slide in animation
		setTimeout(() => {
			$notification.addClass('alt-audit-notification-show');
		}, 10);

		// Auto-dismiss: errors stay longer (10s) so users can read them
		const dismissTime = (type === 'error' || type === 'warning') ? 10000 : 5000;
		setTimeout(() => {
			this.dismissNotification($notification);
		}, dismissTime);

		// Manual dismiss
		$notification.find('.notification-dismiss').on('click', () => {
			this.dismissNotification($notification);
		});
	},

	/**
	 * Dismiss notification
	 *
	 * @param {jQuery} $notification Notification element
	 */
	dismissNotification: function($notification) {
		$notification.removeClass('alt-audit-notification-show');
		setTimeout(() => {
			$notification.remove();
		}, 300);
	},

	/**
	 * Show progress indicator
	 *
	 * @param {boolean} show Whether to show progress
	 */
	showProgress: function(show) {
		const $ = jQuery;

		if (show) {
			// Hide the scan prompt
			$('.alt-audit-scan-prompt').fadeOut(200);
			// Show progress container
			$('.alt-audit-progress-container').slideDown(200);
			// Hide the table during processing
			$('.alt-audit-images-form').fadeOut(200);
		} else {
			$('.alt-audit-progress-container').slideUp(200);
			// Show the table after processing (scan prompt stays hidden)
			$('.alt-audit-images-form').fadeIn(200);
		}
	},

	/**
	 * Update progress indicator
	 *
	 * @param {number} current Current progress
	 * @param {number} total Total items
	 * @param {string} message Optional custom message
	 */
	updateProgress: function(current, total, message) {
		const $ = jQuery;

		const percentage = Math.round((current / total) * 100);

		$('.alt-audit-progress-fill').css('width', `${percentage}%`);

		if (message) {
			$('.progress-message').text(message);
		} else {
			$('.progress-message').text(`Processing ${current} of ${total} images...`);
		}
	},

	/**
	 * Toggle form state (disabled/enabled)
	 *
	 * @param {boolean} disabled Whether to disable form
	 */
	toggleFormState: function(disabled) {
		const $ = jQuery;

		if (disabled) {
			$('.alt-audit-images-form input, .alt-audit-images-form select, .alt-audit-images-form button').prop('disabled', true);
			$('.wp-list-table').addClass('alt-audit-processing');
		} else {
			$('.alt-audit-images-form input, .alt-audit-images-form select, .alt-audit-images-form button').prop('disabled', false);
			$('.wp-list-table').removeClass('alt-audit-processing');
		}
	},

	/**
	 * Cancel current operation
	 */
	cancelOperation: function() {
		if (this.abortController) {
			this.abortController.abort();
		}
	},

	/**
	 * Get current URL parameters
	 *
	 * @returns {Object} Current parameters
	 */
	getCurrentParams: function() {
		const $ = jQuery;
		const params = {};

		// Get page
		params.page = 'alt-audit-audit';

		// Get search term
		const searchTerm = $('input[name="s"]').val();
		if (searchTerm) {
			params.s = searchTerm;
		}

		// Get status filter
		const statusFilter = $('#status_filter').val();
		if (statusFilter) {
			params.status_filter = statusFilter;
		}

		// Get sort parameters
		const urlParams = new URLSearchParams(window.location.search);
		console.log('URL dd params:', urlParams);
		if (urlParams.has('orderby')) {
			params.orderby = urlParams.get('orderby');
		}
		if (urlParams.has('order')) {
			params.order = urlParams.get('order');
			console.log('Order:', params.order);
		}
		console.log('Params:', params);
		// Get current page number
		if (urlParams.has('paged')) {
			params.paged = urlParams.get('paged');
		}

		return params;
	},

	/**
	 * Build URL from parameters
	 *
	 * @param {Object} params URL parameters
	 * @returns {string} Built URL
	 */
	buildUrl: function(params) {
		const baseUrl = window.location.pathname;
		const queryString = new URLSearchParams(params).toString();
		return `${baseUrl}?${queryString}`;
	},

	/**
	 * Handle view image click - open in lightbox
	 *
	 * @param {Event} e Click event
	 */
	handleViewImageClick: function(e) {
		e.preventDefault();
		const $ = jQuery;

		const $link = $(e.currentTarget);
		const imageUrl = $link.attr('href');
		const $row = $link.closest('tr');
		const imageTitle = $row.find('.column-filename strong a').text();
		const altText = $row.find('.column-alt_text .alt-text-display').text() || 'No alt text';

		this.openLightbox(imageUrl, imageTitle, altText);
	},

	/**
	 * Open lightbox with image
	 *
	 * @param {string} imageUrl Image URL
	 * @param {string} imageTitle Image title
	 * @param {string} altText Alt text
	 */
	openLightbox: function(imageUrl, imageTitle, altText) {
		const $ = jQuery;

		// Create lightbox if it doesn't exist
		if ($('#alt-audit-lightbox').length === 0) {
			const lightboxHtml = `
				<div id="alt-audit-lightbox" class="alt-audit-lightbox">
					<div class="alt-audit-lightbox-backdrop"></div>
					<div class="alt-audit-lightbox-content">
						<button class="alt-audit-lightbox-close" aria-label="Close">
							<span class="dashicons dashicons-no-alt"></span>
						</button>
						<div class="alt-audit-lightbox-image-wrapper">
							<img src="" alt="" class="alt-audit-lightbox-image" />
						</div>
						<div class="alt-audit-lightbox-info">
							<h3 class="alt-audit-lightbox-title"></h3>
							<p class="alt-audit-lightbox-alt"></p>
						</div>
					</div>
				</div>
			`;
			$('body').append(lightboxHtml);
		}

		// Update lightbox content
		$('#alt-audit-lightbox .alt-audit-lightbox-image').attr('src', imageUrl).attr('alt', altText);
		$('#alt-audit-lightbox .alt-audit-lightbox-title').text(imageTitle);
		$('#alt-audit-lightbox .alt-audit-lightbox-alt').html('<strong>Alt Text:</strong> ' + altText);

		// Show lightbox
		$('#alt-audit-lightbox').fadeIn(200).addClass('active');
		$('body').addClass('alt-audit-lightbox-open');
	},

	/**
	 * Close lightbox
	 *
	 * @param {Event} e Click event
	 */
	closeLightbox: function(e) {
		if (e) {
			e.preventDefault();
		}
		const $ = jQuery;

		$('#alt-audit-lightbox').fadeOut(200).removeClass('active');
		$('body').removeClass('alt-audit-lightbox-open');
	},

	/**
	 * Handle Scan All Images button click
	 *
	 * @param {Event} e Click event
	 */
	handleScanAllImagesClick: function(e) {
		e.preventDefault();
		const $ = jQuery;
		const $button = $(e.currentTarget);
		const nonce = $button.data('nonce');

		// Prevent double clicks
		if (this.processing) {
			this.showNotification('Another operation is in progress.', 'warning');
			return;
		}

		// Confirm action
		if (!confirm('This will scan all images and update their quality scores. This may take a few minutes depending on the number of images. Continue?')) {
			return;
		}

		// Start processing
		this.processing = true;
		$button.prop('disabled', true).addClass('processing');

		// Show progress indicator
		this.showProgress(true);
		this.updateProgress(0, 100);

		// Start batch processing
		this.processScanBatch(0, nonce);
	},

	/**
	 * Process a batch of images for scanning
	 *
	 * @param {number} offset Current offset
	 * @param {string} nonce Security nonce
	 */
	processScanBatch: function(offset, nonce) {
		const $ = jQuery;
		const self = this;
		const batchSize = 20;

		$.ajax({
			url: window.altaudit82aiAdmin.ajaxUrl,
			type: 'POST',
			data: {
				action: 'altaudit82ai_scan_all_images',
				nonce: nonce,
				batch_size: batchSize,
				offset: offset
			},
			success: function(response) {
				if (response.success) {
					const data = response.data;

					// Update progress
					if (data.processed !== undefined && data.total !== undefined) {
						self.updateProgress(data.processed, data.total, data.message);
					}

					// Continue processing if there are more images
					if (data.has_more) {
						self.processScanBatch(data.offset, nonce);
					} else {
						// All done - update statistics and refresh table
						self.completeScan(data.statistics);
					}
				} else {
					self.showProgress(false);
					self.processing = false;
					$('#altaudit82ai-scan-all-btn').prop('disabled', false).removeClass('processing');
					self.showNotification(response.data.message || 'Failed to scan images.', 'error');
				}
			},
			error: function(xhr, status, error) {
				console.error('Scan AJAX Error:', xhr, status, error);
				self.showProgress(false);
				self.processing = false;
				$('#altaudit82ai-scan-all-btn').prop('disabled', false).removeClass('processing');
				self.showNotification('An error occurred while scanning images: ' + error, 'error');
			}
		});
	},

	/**
	 * Complete the scan process
	 *
	 * @param {Object} statistics Updated statistics
	 */
	completeScan: function(statistics) {
		const $ = jQuery;

		// Hide progress indicator
		this.showProgress(false);
		this.processing = false;
		$('#altaudit82ai-scan-all-btn').prop('disabled', false).removeClass('processing');

		// Show success notification
		this.showNotification('Successfully scanned all images!', 'success');

		// Update statistics cards if they exist
		if (statistics) {
			this.updateStatisticsCards(statistics);
		}

		// Refresh the table to show updated quality scores
		const currentParams = this.getCurrentParams();
		this.refreshTable(currentParams);
	},

	/**
	 * Update statistics cards with new data
	 *
	 * @param {Object} stats Statistics object
	 */
	updateStatisticsCards: function(stats) {
		const $ = jQuery;

		// Update overall score
		if ($('.ring-score').length) {
			$('.ring-score').text(stats.avg_score);
			$('.ring-progress').css('--score', stats.avg_score);
		}

		// Update total images
		if ($('.alt-audit-stat-card .stat-value').length) {
			$('.alt-audit-stat-card .stat-value').first().text(stats.total.toLocaleString());
		}

		// Update status breakdown
		if ($('.status-item').length) {
			$('.status-item.status-missing .status-count').text(stats.missing);
			$('.status-item.status-weak .status-count').text(stats.weak);
			$('.status-item.status-good .status-count').text(stats.good);
			$('.status-item.status-excellent .status-count').text(stats.excellent);
		}
	}
};

// Export for use in other modules
export default AuditPageAjax;
