/**
 * AJAX Handlers Module
 *
 * Handles all AJAX operations for API communication.
 *
 * @package AltAudit
 * @since 1.0.0
 */

import { Utils } from './utils.js';
import { QualityChecker } from './quality-checker.js';

export const AjaxHandlers = {
    /**
     * Test API connection
     *
     * @param {Event} e - Click event
     */
    testApiConnection: function(e) {
        e.preventDefault();

        const $button = jQuery(this);
        const originalText = $button.text();

        $button.text(altaudit82aiAdmin.strings.processing).prop('disabled', true);

        jQuery.ajax({
            url: altaudit82aiAdmin.ajaxUrl,
            type: 'POST',
            data: {
                action: 'altaudit82ai_test_connection',
                nonce: altaudit82aiAdmin.nonce
            },
            success: function(response) {
                if (response.success) {
                    Utils.showNotice(response.data.message, 'success');
                } else {
                    Utils.showNotice(response.data, 'error');
                }
            },
            error: function() {
                Utils.showNotice(altaudit82aiAdmin.strings.error, 'error');
            },
            complete: function() {
                $button.text(originalText).prop('disabled', false);
            }
        });
    },

    /**
     * Generate alt text for image
     *
     * @param {Event} e - Click event
     */
    generateAltText: function(e) {
        e.preventDefault();

        const $button = jQuery(this);
        const attachmentId = $button.data('attachment-id');
        const context = $button.data('context') || '';
        const style = $button.data('style') || 'descriptive';

        if (!attachmentId) {
            Utils.showNotice('Invalid attachment ID', 'error');
            return;
        }

        const originalText = $button.text();
        $button.text(altaudit82aiAdmin.strings.generating).prop('disabled', true);

        jQuery.ajax({
            url: altaudit82aiAdmin.ajaxUrl,
            type: 'POST',
            data: {
                action: 'altaudit82ai_generate',
                nonce: altaudit82aiAdmin.nonce,
                attachment_id: attachmentId,
                context: context,
                style: style
            },
            success: function(response) {
                if (response.success) {
                    AjaxHandlers.handleGenerateSuccess(response.data, $button);
                } else {
                    Utils.showNotice(response.data, 'error');
                }
            },
            error: function() {
                Utils.showNotice(altaudit82aiAdmin.strings.error, 'error');
            },
            complete: function() {
                $button.text(originalText).prop('disabled', false);
            }
        });
    },

    /**
     * Handle successful alt text generation
     *
     * @param {Object} data - Response data
     * @param {jQuery} $button - Button element that triggered the action
     */
    handleGenerateSuccess: function(data, $button) {
        const $container = $button.closest('.alt-audit-container');
        const $altTextInput = $container.find('.alt-text-input');
        const $qualityIndicator = $container.find('.alt-audit-quality-indicator');

        // Update alt text input
        if ($altTextInput.length) {
            $altTextInput.val(data.alt_text);
        }

        // Update quality indicator
        if ($qualityIndicator.length && data.quality) {
            QualityChecker.updateQualityDisplay($qualityIndicator, data.quality);
        }

        // Show success message
        Utils.showNotice(data.message, 'success');

        // Trigger custom event
        jQuery(document).trigger('altaudit82aiGenerated', [data]);
    },

    /**
     * Save alt text
     *
     * @param {Event} e - Click event
     */
    saveAltText: function(e) {
        e.preventDefault();

        const $button = jQuery(this);
        const $container = $button.closest('.alt-audit-container');
        const $altTextInput = $container.find('.alt-text-input');
        const attachmentId = $button.data('attachment-id');
        const altText = $altTextInput.val();

        if (!attachmentId) {
            Utils.showNotice('Invalid attachment ID', 'error');
            return;
        }

        const originalText = $button.text();
        $button.text(altaudit82aiAdmin.strings.processing).prop('disabled', true);

        jQuery.ajax({
            url: altaudit82aiAdmin.ajaxUrl,
            type: 'POST',
            data: {
                action: 'altaudit82ai_save_alt_text',
                nonce: altaudit82aiAdmin.nonce,
                attachment_id: attachmentId,
                alt_text: altText
            },
            success: function(response) {
                if (response.success) {
                    Utils.showNotice(response.data.message, 'success');

                    // Update quality indicator if available
                    if (response.data.quality) {
                        const $qualityIndicator = $container.find('.alt-audit-quality-indicator');
                        QualityChecker.updateQualityDisplay($qualityIndicator, response.data.quality);
                    }
                } else {
                    Utils.showNotice(response.data, 'error');
                }
            },
            error: function() {
                Utils.showNotice(altaudit82aiAdmin.strings.error, 'error');
            },
            complete: function() {
                $button.text(originalText).prop('disabled', false);
            }
        });
    },

    /**
     * Generate rule-based alt text
     *
     * @param {Event} e - Click event
     */
    generateRuleBasedAltText: function(e) {
        e.preventDefault();

        const $button = jQuery(this);
        const attachmentId = $button.data('attachment-id');
        const template = $button.data('template') || '';
        const $row = $button.closest('tr');
        const isAuditPage = $row.length > 0;

        if (!attachmentId) {
            Utils.showNotice('Invalid attachment ID', 'error');
            return;
        }

        const originalText = $button.text();
        $button.text('Generating...').prop('disabled', true);

        jQuery.ajax({
            url: altaudit82aiAdmin.ajaxUrl,
            type: 'POST',
            data: {
                action: 'altaudit82ai_generate_rule_based',
                nonce: altaudit82aiAdmin.nonce,
                attachment_id: attachmentId,
                template: template
            },
            success: function(response) {
                if (response.success) {
                    if (isAuditPage) {
                        // Import AuditPage dynamically to avoid circular dependency
                        import('./audit-page.js').then(module => {
                            module.AuditPage.updateAuditTableRow($row, response.data);
                        });
                    } else {
                        AjaxHandlers.handleGenerateSuccess(response.data, $button);
                    }
                } else {
                    Utils.showNotice(response.data, 'error');
                }
            },
            error: function() {
                Utils.showNotice('Error generating alt text', 'error');
            },
            complete: function() {
                $button.text(originalText).prop('disabled', false);
            }
        });
    },

    /**
     * Generate AI-powered alt text
     *
     * @param {Event} e - Click event
     */
    generateAIAltText: function(e) {
        e.preventDefault();

        const $button = jQuery(this);
        const attachmentId = $button.data('attachment-id');
        const context = $button.data('context') || '';
        const style = $button.data('style') || 'descriptive';
        const $row = $button.closest('tr');
        const isAuditPage = $row.length > 0;

        if (!attachmentId) {
            Utils.showNotice('Invalid attachment ID', 'error');
            return;
        }

        // Check if user has credits
        const creditCount = parseInt(jQuery('#alt-audit-credit-count').text() || '0');
        if (creditCount <= 0) {
            if (!confirm('You have no credits remaining. This will still attempt to generate alt text. Continue?')) {
                return;
            }
        }

        const originalText = $button.text();
        $button.text('Generating...').prop('disabled', true);

        jQuery.ajax({
            url: altaudit82aiAdmin.ajaxUrl,
            type: 'POST',
            data: {
                action: 'altaudit82ai_generate',
                nonce: altaudit82aiAdmin.nonce,
                attachment_id: attachmentId,
                context: context,
                style: style
            },
            success: function(response) {
                if (response.success) {
                    if (isAuditPage) {
                        // Import AuditPage dynamically to avoid circular dependency
                        import('./audit-page.js').then(module => {
                            module.AuditPage.updateAuditTableRow($row, response.data);
                        });
                    } else {
                        AjaxHandlers.handleGenerateSuccess(response.data, $button);
                    }

                    // Update credit count if provided
                    if (response.data.credits_remaining !== undefined) {
                        jQuery('#alt-audit-credit-count').text(response.data.credits_remaining);
                    }
                } else {
                    Utils.showNotice(response.data, 'error');
                }
            },
            error: function() {
                Utils.showNotice('Error generating alt text', 'error');
            },
            complete: function() {
                $button.text(originalText).prop('disabled', false);
            }
        });
    },

    /**
     * Load user credits
     */
    loadUserCredits: function() {
        jQuery.ajax({
            url: altaudit82aiAdmin.ajaxUrl,
            type: 'POST',
            data: {
                action: 'altaudit82ai_get_user_info',
                nonce: altaudit82aiAdmin.nonce
            },
            success: function(response) {
                if (response.success && response.data.credits !== undefined) {
                    jQuery('#alt-audit-credit-count').text(response.data.credits);
                    // Trigger update of cost preview if GenerationMode is loaded
                    if (window.Altaudit82aiAdminApp && window.Altaudit82aiAdminApp.GenerationMode) {
                        window.Altaudit82aiAdminApp.GenerationMode.updateCostPreview();
                    }
                }
            },
            error: function() {
                // Silently fail, credits will show as 0
            }
        });
    }
};
