/**
 * Alt Audit Media Library JavaScript
 *
 * Handles AJAX functionality for media edit page buttons
 *
 * @package AltAudit
 * @since 1.0.0
 */

(function($) {
	'use strict';

	const Altaudit82aiMedia = {
		/**
		 * Initialize
		 */
		init: function() {
			this.bindEvents();
		},

		/**
		 * Bind event handlers
		 */
		bindEvents: function() {
			// Unbind first to prevent duplicate handlers
			$(document).off('click.altaudit', '.alt-audit-generate-ai-btn');
			$(document).off('click.altaudit', '.alt-audit-generate-rule-btn');
			$(document).off('click.altaudit', '.alt-audit-audit-quality-btn');

			// Bind with namespaced events
			$(document).on('click.altaudit', '.alt-audit-generate-ai-btn', this.handleGenerateAI.bind(this));
			$(document).on('click.altaudit', '.alt-audit-generate-rule-btn', this.handleGenerateRule.bind(this));
			$(document).on('click.altaudit', '.alt-audit-audit-quality-btn', this.handleAuditQuality.bind(this));
		},

		/**
		 * Handle AI generation button click
		 */
		handleGenerateAI: function(e) {
			e.preventDefault();
			const $button = $(e.currentTarget);
			const attachmentId = $button.data('attachment-id');

			if (!confirm('This will use 1 credit to generate AI-powered alt text. Continue?')) {
				return;
			}

			this.performAction($button, attachmentId, 'generate_ai', 'altaudit82ai_media_generate_ai');
		},

		/**
		 * Handle rule-based generation button click
		 */
		handleGenerateRule: function(e) {
			e.preventDefault();
			const $button = $(e.currentTarget);
			const attachmentId = $button.data('attachment-id');

			this.performAction($button, attachmentId, 'generate_rule', 'altaudit82ai_media_generate_rule');
		},

		/**
		 * Handle audit quality button click
		 */
		handleAuditQuality: function(e) {
			e.preventDefault();
			const $button = $(e.currentTarget);
			const attachmentId = $button.data('attachment-id');

			this.performAction($button, attachmentId, 'audit_quality', 'altaudit82ai_media_audit_quality');
		},

		/**
		 * Perform AJAX action
		 */
		performAction: function($button, attachmentId, action, ajaxAction) {
			const $container = $button.closest('.alt-audit-field');
			const $loading = $container.find('.alt-audit-loading');
			const $altTextDisplay = $container.find('.alt-audit-alt-text');
			const $qualityDisplay = $container.find('.alt-audit-quality-score');
			const $actions = $container.find('.alt-audit-actions');

			// Show loading state
			$actions.hide();
			$loading.show();
			$button.prop('disabled', true);

			$.ajax({
				url: altaudit82aiMedia.ajaxUrl,
				type: 'POST',
				data: {
					action: ajaxAction,
					nonce: altaudit82aiMedia.nonce,
					attachment_id: attachmentId
				},
				success: (response) => {
					if (response.success) {
						// Update alt text display
						if (response.data.alt_text) {
							$altTextDisplay.text(response.data.alt_text);

							// Update the main WordPress alt text field
							let $altField = null;

							// For Media Modal (popup)
							const $modal = $('.media-modal:visible');
							if ($modal.length > 0) {
								// In media modal, data-setting="alt" is on the parent span
								// Find textarea inside the alt-text setting span
								$altField = $modal.find('.setting[data-setting="alt"] textarea, #attachment-details-two-column-alt-text').first();

								if ($altField.length > 0) {
									// Update the field
									$altField.val(response.data.alt_text).trigger('change').trigger('input');

									// Also update the Backbone model directly
									const model = wp.media.frame?.state()?.get('selection')?.first();
									if (model) {
										model.set('alt', response.data.alt_text);
									}
								}
							}

							// For Regular Edit Page and Compat Fields
							if (!$altField || $altField.length === 0) {
								// Try regular edit page selectors
								$altField = $(
									'#attachment_alt, ' +
									'textarea[name="_wp_attachment_image_alt"], ' +
									'#attachments-' + attachmentId + '-alt'
								).first();

								// If still not found, check if we're in compat-attachment-fields context
								if ($altField.length === 0) {
									// Look in the same compat fields table row
									const $compatRow = $container.closest('tr');
									if ($compatRow.length > 0) {
										// Find alt text field in previous rows of the same table
										$altField = $compatRow.closest('table')
											.find('textarea[name="_wp_attachment_image_alt"], #attachment_alt')
											.first();
									}
								}

								if ($altField.length > 0) {
									$altField.val(response.data.alt_text).trigger('change').trigger('input');
								}
							}

							if ($altField && $altField.length > 0) {
								console.log('Alt text field updated:', $altField);
							} else {
								console.warn('Could not find alt text field');
							}
						}

						// Update quality display
						if (response.data.quality) {
							const quality = response.data.quality;
							const qualityHtml = `
								<strong>Quality Score:</strong>
								<span class="alt-audit-score alt-audit-score-${quality.status.toLowerCase()}">
									${quality.score}%
									(${quality.status})
								</span>
							`;

							if ($qualityDisplay.length > 0) {
								$qualityDisplay.html(qualityHtml);
							} else {
								// Create quality display if it doesn't exist
								$altTextDisplay.after('<div class="alt-audit-quality-score">' + qualityHtml + '</div>');
							}
						}

						// Show success message
						this.showNotice(response.data.message, 'success');
					} else {
						this.showNotice(response.data || 'Action failed', 'error');
					}
				},
				error: (xhr, status, error) => {
					this.showNotice('Error: ' + error, 'error');
				},
				complete: () => {
					$loading.hide();
					$actions.show();
					$button.prop('disabled', false);
				}
			});
		},

		/**
		 * Show notice message
		 */
		showNotice: function(message, type) {
			const $notice = $('<div class="notice notice-' + type + ' is-dismissible"><p>' + message + '</p></div>');

			// Find the best container for the notice
			let $container = null;

			// For media modal
			if ($('.media-modal:visible').length > 0) {
				$container = $('.media-modal-content .attachment-info');
				if ($container.length === 0) {
					$container = $('.media-sidebar');
				}
				if ($container.length > 0) {
					$container.prepend($notice);
				}
			}
			// For compat fields (edit media page)
			else if ($('.compat-attachment-fields').length > 0) {
				// Insert BEFORE the table, not inside it
				const $table = $('.compat-attachment-fields');
				$table.before($notice);
			}
			// Fallback
			else {
				$container = $('.wrap');
				if ($container.length > 0) {
					$container.prepend($notice);
				} else {
					alert(message);
				}
			}

			// Auto-dismiss after 5 seconds
			setTimeout(() => {
				$notice.fadeOut(() => {
					$notice.remove();
				});
			}, 5000);
		}
	};

	// Initialize when document is ready
	$(document).ready(function() {
		Altaudit82aiMedia.init();
	});

	// Also initialize when media modal opens
	$(document).on('click', '.attachment', function() {
		setTimeout(() => {
			Altaudit82aiMedia.init();
		}, 100);
	});

})(jQuery);
