/**
 * Alt Audit Admin JavaScript - Main Entry Point
 *
 * Modular architecture for the Alt Audit admin interface.
 * Imports and initializes all feature modules.
 *
 * @package AltAudit
 * @since 1.0.0
 */

// Import all modules
import { Utils } from './modules/utils.js';
import { Dashboard } from './modules/dashboard.js';
import { Settings } from './modules/settings.js';
import { QualityChecker } from './modules/quality-checker.js';
import { GenerationMode } from './modules/generation-mode.js';
import { InlineEditor } from './modules/inline-editor.js';
import { BulkTools } from './modules/bulk-tools.js';
import { AjaxHandlers } from './modules/ajax-handlers.js';
import { AuditPage } from './modules/audit-page.js';
import { AuditPageAjax } from './modules/audit-page-ajax.js';
import { HelpPage } from './modules/help-page.js';

(function($) {
    'use strict';

    /**
     * Global Alt Audit Admin object
     * Aggregates all module functionality for WordPress compatibility
     */
    window.Altaudit82aiAdminApp = {
        // Module references
        Utils: Utils,
        Dashboard: Dashboard,
        Settings: Settings,
        QualityChecker: QualityChecker,
        GenerationMode: GenerationMode,
        InlineEditor: InlineEditor,
        BulkTools: BulkTools,
        AjaxHandlers: AjaxHandlers,
        AuditPage: AuditPage,
        AuditPageAjax: AuditPageAjax,
        HelpPage: HelpPage,

        /**
         * Initialize the admin interface
         */
        init: function() {
            this.bindEvents();
            this.initDashboard();
            this.initSettings();
            this.initQualityCheckers();
            this.initGenerationMode();
            this.initInlineEditing();
            this.initBulkTools();
            this.initAuditPageAjax();
            this.initHelpPage();
        },

        /**
         * Bind event handlers
         */
        bindEvents: function() {
            // Test API connection
            $(document).on('click', '.alt-audit-test-api', AjaxHandlers.testApiConnection);

            // Generate alt text
            // Note: .alt-audit-generate-rule and .alt-audit-generate-ai are handled by AuditPageAjax module
            $(document).on('click', '.alt-audit-generate', AjaxHandlers.generateAltText);

            // Check quality
            $(document).on('click', '.alt-audit-check-quality', QualityChecker.checkQuality);

            // Save alt text
            $(document).on('click', '.alt-audit-save', AjaxHandlers.saveAltText);

            // Bulk operations
            $(document).on('click', '.alt-audit-bulk-action', BulkTools.handleBulkAction);

            // Generation mode and cost preview
            $(document).on('change', 'input[name="generation_mode"]', GenerationMode.handleGenerationModeChange);
            $(document).on('change', 'input[type="checkbox"][name^="bulk_ids"]', GenerationMode.updateCostPreview.bind(GenerationMode));
            $(document).on('change', '.wp-list-table input[type="checkbox"]', GenerationMode.updateCostPreview.bind(GenerationMode));

            // Settings validation
            $(document).on('submit', '.alt-audit-settings form', Settings.validateSettings);

            // Dismissible notices
            $(document).on('click', '.notice.is-dismissible[data-dismissible] .notice-dismiss', this.handleNoticeDismiss);

            // Scan reminder dismiss button
            $(document).on('click', '.altaudit82ai-dismiss-scan-reminder', this.handleScanReminderDismiss);
        },

        /**
         * Handle dismissible notice clicks
         */
        handleNoticeDismiss: function(e) {
            const $notice = $(this).closest('.notice');
            const noticeType = $notice.data('dismissible');
            const nonce = $notice.data('nonce');

            if (!noticeType || !nonce) {
                return;
            }

            // Send AJAX request to save dismissal
            $.ajax({
                url: window.altaudit82aiAdmin.ajaxUrl,
                type: 'POST',
                data: {
                    action: 'altaudit82ai_dismiss_notice',
                    notice_type: noticeType,
                    nonce: nonce
                }
            });
        },

        /**
         * Handle scan reminder "Remind Me Later" button click
         */
        handleScanReminderDismiss: function(e) {
            e.preventDefault();

            const $button = $(this);
            const $notice = $button.closest('.altaudit82ai-scan-reminder');
            const noticeType = $notice.data('notice-type');

            // Disable button during AJAX request
            $button.prop('disabled', true);

            // Send AJAX request to dismiss the notice
            $.ajax({
                url: window.altaudit82aiAdmin.ajaxUrl,
                type: 'POST',
                data: {
                    action: 'altaudit82ai_dismiss_notice',
                    notice_type: noticeType,
                    nonce: window.altaudit82aiAdmin.nonce
                },
                success: function() {
                    // Fade out and remove the notice
                    $notice.fadeOut(300, function() {
                        $(this).remove();
                    });
                },
                error: function() {
                    // Re-enable button on error
                    $button.prop('disabled', false);
                }
            });
        },

        /**
         * Initialize dashboard functionality
         */
        initDashboard: function() {
            Dashboard.init();
        },

        /**
         * Initialize settings functionality
         */
        initSettings: function() {
            Settings.init();
        },

        /**
         * Initialize quality checkers
         */
        initQualityCheckers: function() {
            QualityChecker.init();
        },

        /**
         * Initialize generation mode functionality
         */
        initGenerationMode: function() {
            GenerationMode.init();
            // Load user credits
            AjaxHandlers.loadUserCredits();
        },

        /**
         * Initialize inline editing
         */
        initInlineEditing: function() {
            InlineEditor.init();
        },

        /**
         * Initialize bulk tools
         */
        initBulkTools: function() {
            BulkTools.init();
        },

        /**
         * Initialize AJAX audit page functionality
         */
        initAuditPageAjax: function() {
            // Only initialize on audit page
            if ($('.alt-audit-audit[data-page="audit"]').length > 0) {
                AuditPageAjax.init();
            }
        },

        /**
         * Initialize help page functionality
         */
        initHelpPage: function() {
            HelpPage.init();
        },

        // Expose utility methods for backwards compatibility
        showNotice: Utils.showNotice,
        debounce: Utils.debounce,
        animateNumber: Utils.animateNumber,
        logMessage: Utils.logMessage,

        // Expose quality checker methods
        updateQualityDisplay: QualityChecker.updateQualityDisplay.bind(QualityChecker),
        updateQualityIndicator: QualityChecker.updateQualityIndicator.bind(QualityChecker),

        // Expose AJAX handler methods
        handleGenerateSuccess: AjaxHandlers.handleGenerateSuccess.bind(AjaxHandlers),
        loadUserCredits: AjaxHandlers.loadUserCredits.bind(AjaxHandlers),

        // Expose generation mode methods
        handleGenerationModeChange: GenerationMode.handleGenerationModeChange.bind(GenerationMode),
        updateCostPreview: GenerationMode.updateCostPreview.bind(GenerationMode),

        // Expose bulk tools methods
        handleBulkAction: BulkTools.handleBulkAction.bind(BulkTools),
        handleBulkSuccess: BulkTools.handleBulkSuccess.bind(BulkTools),
        updateImageCount: BulkTools.updateImageCount.bind(BulkTools),

        // Expose dashboard methods
        updateDashboardStats: Dashboard.updateDashboardStats.bind(Dashboard),
        animateScoreCircle: Dashboard.animateScoreCircle.bind(Dashboard),
        initScoreCircle: Dashboard.initScoreCircle.bind(Dashboard),
        initCardInteractions: Dashboard.initCardInteractions.bind(Dashboard),
        refreshStats: Dashboard.refreshStats.bind(Dashboard),
        initStatsRefresh: Dashboard.initStatsRefresh.bind(Dashboard),

        // Expose audit page methods
        updateAuditTableRow: AuditPage.updateAuditTableRow.bind(AuditPage),
        refreshAuditStatistics: AuditPage.refreshAuditStatistics.bind(AuditPage),
        updateAuditStatistics: AuditPage.updateAuditStatistics.bind(AuditPage),
        updateRingScore: AuditPage.updateRingScore.bind(AuditPage),
        updateStatusCounts: AuditPage.updateStatusCounts.bind(AuditPage)
    };

    // Initialize when document is ready
    $(document).ready(function() {
        Altaudit82aiAdminApp.init();
    });

})(jQuery);
