<?php
// includes/ajax-card-details-admin.php

if ( ! defined( 'ABSPATH' ) ) exit;

// Helper: Time ago or formatted date
function alba_board_time_ago_or_date( $datetime ) {
    $timestamp = strtotime( $datetime );
    $now = current_time( 'timestamp' );
    $diff = $now - $timestamp;

    if ( $diff < 60 ) {
        return __( 'Just now', 'alba-board' );
    } elseif ( $diff < 3600 ) {
        $mins = floor( $diff / 60 );
        return sprintf( _n( '%d minute ago', '%d minutes ago', $mins, 'alba-board' ), $mins );
    } elseif ( $diff < 86400 ) {
        $hours = floor( $diff / 3600 );
        return sprintf( _n( '%d hour ago', '%d hours ago', $hours, 'alba-board' ), $hours );
    } else {
        return date_i18n( 'Y-m-d H:i', $timestamp );
    }
}

// Output modal HTML for admin card details
function alba_output_card_details_admin_modal( $card_id, $force_author = null ) {
    $card_id = absint( $card_id );
    if ( ! $card_id || get_post_type( $card_id ) !== 'alba_card' ) {
        echo esc_html__( 'Invalid card.', 'alba-board' );
        return;
    }

    $card = get_post( $card_id );
    if ( ! $card ) {
        echo esc_html__( 'Card not found.', 'alba-board' );
        return;
    }

    $author_id = $force_author !== null ? absint( $force_author ) : intval( $card->post_author );
    
    echo '<form id="alba-card-details-form" class="alba-card-details-form">';
    echo '<input type="hidden" id="alba-current-card-id" name="card_id" value="' . esc_attr( $card_id ) . '">';

    // Title
    echo '<div class="alba-form-group">';
    echo '<label>' . esc_html__( 'Title:', 'alba-board' ) . '</label>';
    echo '<input type="text" name="card_title" class="alba-form-input-text" value="' . esc_attr( $card->post_title ) . '" required>';
    echo '</div>';

    // Author dropdown
    echo '<div class="alba-form-group">';
    echo '<label>' . esc_html__( 'Assignee:', 'alba-board' ) . '</label>';
    echo '<select name="card_author" class="alba-select2">';
    $users = get_users( [ 'fields' => [ 'ID', 'display_name' ] ] );
    foreach ( $users as $user ) {
        $selected = $user->ID == $author_id ? 'selected' : '';
        echo '<option value="' . esc_attr( $user->ID ) . '" ' . esc_attr( $selected ) . '>' . esc_html( $user->display_name ) . '</option>';
    }
    echo '</select>';
    echo '</div>';

    // Content/description
    echo '<div class="alba-form-group">';
    echo '<label>' . esc_html__( 'Description:', 'alba-board' ) . '</label>';
    echo '<textarea name="card_content" class="alba-form-input-text" rows="3">' . esc_textarea( $card->post_content ) . '</textarea>';
    echo '</div>';

    // Hook for Add-ons (Tags, etc.)
    do_action('alba_admin_card_modal_after_description', $card_id);

    // Attachments Section
    echo '<div class="alba-form-group" style="margin-bottom: 15px;">';
    echo '<label>' . esc_html__( 'Attachments:', 'alba-board' ) . '</label>';
    
    $attachments = get_post_meta( $card_id, 'alba_card_attachments' );
    echo '<div id="alba-attachments-list" style="display: flex; flex-direction: column; gap: 8px; margin-bottom: 10px;">';
    if ( ! empty( $attachments ) ) {
        foreach ( $attachments as $att_id ) {
            $file_url = wp_get_attachment_url( $att_id );
            $file_title = get_the_title( $att_id );
            if ($file_url) {
                echo '<div class="alba-attachment-item" id="alba-attachment-' . esc_attr($att_id) . '" style="display: flex; justify-content: space-between; align-items: center; background: #fff; padding: 8px 14px; border-radius: 12px; box-shadow: 2px 2px 6px #e3e7ef, -2px -2px 6px #fff;">';
                echo '<a href="' . esc_url($file_url) . '" target="_blank" style="text-decoration: none; color: #3d4b5c; font-weight: 600; font-size: 0.95em;">📄 ' . esc_html($file_title) . '</a>';
                echo '<button type="button" class="alba-delete-attachment-btn" data-attachment-id="' . esc_attr($att_id) . '" style="background: none; border: none; color: #e05c4d; cursor: pointer; font-size: 1.2em; outline: none;" title="' . esc_attr__('Delete file', 'alba-board') . '">&times;</button>';
                echo '</div>';
            }
        }
    } else {
        echo '<div id="alba-no-attachments-msg" style="color: #8c98a9; font-size: 0.9em; font-style: italic;">' . esc_html__( 'No files attached.', 'alba-board' ) . '</div>';
    }
    echo '</div>'; // #alba-attachments-list

    // Upload controls
    echo '<input type="file" id="alba-file-upload-input" style="display: none;">';
    echo '<button type="button" id="alba-trigger-upload-btn" class="alba-btn-cancel" style="padding: 6px 12px !important; font-size: 0.9em;">+ ' . esc_html__( 'Add File', 'alba-board' ) . '</button>';
    echo '<div id="alba-upload-feedback" style="margin-top: 8px; font-size: 0.9em; font-weight: 600;"></div>';
    echo '</div>'; 

    // Comments section
    $comments = get_post_meta( $card_id, 'alba_comments', true );
    if ( ! is_array( $comments ) ) {
        $comments = @unserialize( $comments );
        if ( ! is_array( $comments ) ) $comments = [];
    }

    echo '<div class="alba-form-group">';
    echo '<label class="alba-comment-label">' . esc_html__( 'Activity & Comments:', 'alba-board' ) . '</label>';
    echo '<div class="alba-card-comments-scrollable" id="alba-comments-list" style="background: #fdfdfd; padding: 10px; border-radius: 12px; box-shadow: inset 2px 2px 5px #e3e7ef, inset -2px -2px 5px #ffffff; max-height: 150px; overflow-y: auto;">';
    if ( $comments ) {
        foreach ( $comments as $c ) {
            $author = isset( $c['author'] ) ? $c['author'] : '';
            $when = isset( $c['date'] ) ? alba_board_time_ago_or_date( $c['date'] ) : '';
            $text = isset( $c['text'] ) ? $c['text'] : '';
            echo '<div class="alba-board-comment" style="margin-bottom: 10px; border-bottom: 1px solid #e3e7ef; padding-bottom: 8px;">';
            echo '<strong style="color: #3d4b5c;">' . esc_html( $author ) . '</strong> ';
            echo '<span class="alba-comment-date" style="font-size: 0.8em; color: #8c98a9;">' . esc_html( $when ) . '</span>';
            echo '<div class="alba-comment-text" style="margin-top: 4px; color: #475569;">' . esc_html( $text ) . '</div>';
            echo '</div>';
        }
    } else {
        echo '<div style="color: #8c98a9; font-style: italic;">' . esc_html__( 'No activity yet.', 'alba-board' ) . '</div>';
    }
    echo '</div>'; 
    echo '</div>';

    // Add comment
    echo '<div class="alba-form-group">';
    echo '<label class="alba-comment-label">' . esc_html__( 'Write a comment:', 'alba-board' ) . '</label>';
    echo '<textarea id="alba-new-comment" name="new_comment" class="alba-form-input-text" rows="2" placeholder="' . esc_attr__('Type here...', 'alba-board') . '"></textarea>';
    echo '</div>';

    // 🔥 NEUMORPHIC SAVE BUTTON (COMPACT) 🔥
    echo '<button type="submit" id="alba-card-save-btn" class="alba-btn-neumorphic alba-btn-compact">' . esc_html__( 'Save Changes', 'alba-board' ) . '</button>';
    echo '</form>';
}

// Secure AJAX endpoint for admin modal
add_action( 'wp_ajax_alba_get_card_details_admin', function() {
    $nonce = isset( $_POST['nonce'] ) ? sanitize_text_field( wp_unslash( $_POST['nonce'] ) ) : '';
    if ( empty( $nonce ) || ! wp_verify_nonce( $nonce, 'alba_get_card_details_admin' ) ) {
        wp_send_json_error( [ 'message' => __( 'Invalid nonce.', 'alba-board' ) ] );
    }
    if ( ! current_user_can( 'edit_cards' ) ) {
        wp_send_json_error( [ 'message' => __( 'Permission denied.', 'alba-board' ) ] );
    }
    $card_id = isset( $_POST['card_id'] ) ? absint( $_POST['card_id'] ) : 0;
    $force_author = isset( $_POST['force_author'] ) ? absint( $_POST['force_author'] ) : null;
    
    alba_output_card_details_admin_modal( $card_id, $force_author );
    wp_die();
});