// assets/js/alba-backend-kanban.js

document.addEventListener('DOMContentLoaded', () => {
    
    // --- 1. SORTABLE CARDS (Drag & Drop inside lists) ---
    document.querySelectorAll('.alba-cards-container').forEach(list => {
        if (typeof Sortable !== "undefined") {
            new Sortable(list, {
                group: 'alba-cards',
                animation: 150,
                ghostClass: 'sortable-ghost',
                dragClass: 'sortable-drag',
                filter: '.alba-no-cards-msg',

                onEnd: function (evt) {
                    const card = evt.item;
                    const cardId = card.dataset.cardId;
                    const newListId = evt.to.dataset.listId;
                    const orderedCardIds = Array.from(evt.to.querySelectorAll('.alba-card')).map(el => el.dataset.cardId);

                    [evt.to, evt.from].forEach(listContainer => {
                        const cardsCount = listContainer.querySelectorAll('.alba-card').length;
                        const noCardsMsg = listContainer.querySelector('.alba-no-cards-msg');
                        if (noCardsMsg) {
                            if (cardsCount > 0) {
                                noCardsMsg.classList.add('alba-is-hidden');
                            } else {
                                noCardsMsg.classList.remove('alba-is-hidden');
                            }
                        }
                    });

                    if (cardId && newListId) {
                        const params = new URLSearchParams();
                        params.append('action', 'alba_move_card');
                        params.append('card_id', cardId);
                        params.append('new_list_id', newListId);
                        params.append('nonce', albaBoard.nonce);

                        orderedCardIds.forEach((id, index) => {
                            params.append(`order[${index}]`, id);
                        });

                        fetch(albaBoard.ajaxurl, {
                            method: 'POST',
                            headers: { 'Content-Type': 'application/x-www-form-urlencoded' },
                            body: params
                        })
                        .then(res => res.json())
                        .then(response => {
                            if (!response.success && response.data && response.data.message) {
                                alert(response.data.message);
                            }
                        });
                    }
                }
            });
        }
    });

    // 👉 NEW: 2. SORTABLE LISTS (Drag & Drop columns) 👈
    const boardWrapper = document.querySelector('.alba-board-wrapper');
    if (boardWrapper && typeof Sortable !== "undefined") {
        new Sortable(boardWrapper, {
            group: 'alba-lists-group',
            animation: 150,
            draggable: '.alba-list-scrollable', // Only drag actual lists, ignore "Add List" button
            handle: '.alba-list-header',        // Magic: Only drag from the header title area
            filter: '.alba-delete-list-btn',    // Don't drag if clicking the delete button
            ghostClass: 'sortable-ghost',
            onEnd: function (evt) {
                const orderedListIds = Array.from(boardWrapper.querySelectorAll('.alba-list-scrollable')).map(el => el.dataset.listId);
                
                const params = new URLSearchParams();
                params.append('action', 'alba_move_list_action');
                params.append('nonce', albaBoard.move_list_nonce);
                orderedListIds.forEach((id, index) => {
                    params.append(`order[${index}]`, id);
                });

                fetch(albaBoard.ajaxurl, {
                    method: 'POST',
                    headers: { 'Content-Type': 'application/x-www-form-urlencoded' },
                    body: params
                });
            }
        });
    }

    // --- 3. UI Interaction (Forms Toggle via CSS Classes) ---
    const boardSelector = document.querySelector('.alba-auto-submit-select');
    if(boardSelector) {
        boardSelector.addEventListener('change', function() {
            this.form.submit();
        });
    }

    document.addEventListener('click', function(e) {
        // Toggle New Board Form
        if (e.target.id === 'alba-show-new-board-btn') {
            e.preventDefault();
            const form = document.getElementById('alba-new-board-form');
            if(form) {
                form.classList.remove('alba-is-hidden');
                e.target.classList.add('alba-is-hidden');
                const input = form.querySelector('input[type="text"]');
                if (input) input.focus();
            }
        }
        if (e.target.id === 'alba-cancel-new-board-btn') {
            e.preventDefault();
            const form = document.getElementById('alba-new-board-form');
            const btn = document.getElementById('alba-show-new-board-btn');
            if(form && btn) {
                form.classList.add('alba-is-hidden');
                btn.classList.remove('alba-is-hidden');
            }
        }

        // Toggle New List Form
        const showListBtn = e.target.closest('.alba-show-add-list-btn');
        if (showListBtn) {
            e.preventDefault();
            const form = showListBtn.nextElementSibling;
            if(form) {
                form.classList.remove('alba-is-hidden');
                showListBtn.classList.add('alba-is-hidden');
                const input = form.querySelector('input[type="text"]');
                if (input) input.focus();
            }
        }

        const cancelListBtn = e.target.closest('.alba-cancel-new-list-btn');
        if (cancelListBtn) {
            e.preventDefault();
            const form = cancelListBtn.closest('form');
            if(form) {
                form.classList.add('alba-is-hidden');
                if(form.previousElementSibling) form.previousElementSibling.classList.remove('alba-is-hidden');
            }
        }

        // Toggle New Card Form
        const showCardBtn = e.target.closest('.alba-show-add-card-btn');
        if (showCardBtn) {
            e.preventDefault();
            const form = showCardBtn.nextElementSibling;
            if(form) {
                form.classList.remove('alba-is-hidden');
                showCardBtn.classList.add('alba-is-hidden');
                const input = form.querySelector('input[type="text"]');
                if (input) input.focus();
            }
        }

        const cancelCardBtn = e.target.closest('.alba-cancel-new-card-btn');
        if (cancelCardBtn) {
            e.preventDefault();
            const form = cancelCardBtn.closest('form');
            if(form) {
                form.classList.add('alba-is-hidden');
                if(form.previousElementSibling) form.previousElementSibling.classList.remove('alba-is-hidden');
            }
        }

        // DELETE LIST LOGIC
        const deleteListBtn = e.target.closest('.alba-delete-list-btn');
        if (deleteListBtn) {
            e.preventDefault();
            if (confirm('Are you sure you want to delete this list and ALL its cards?')) {
                const listId = deleteListBtn.dataset.listId;
                const listContainer = deleteListBtn.closest('.alba-list');
                
                deleteListBtn.style.opacity = '0.5';
                deleteListBtn.style.pointerEvents = 'none';

                const formData = new URLSearchParams();
                formData.append('action', 'alba_delete_list_action');
                formData.append('list_id', listId);
                formData.append('nonce', albaBoard.delete_list_nonce);

                fetch(albaBoard.ajaxurl, {
                    method: 'POST',
                    headers: { 'Content-Type': 'application/x-www-form-urlencoded' },
                    body: formData
                })
                .then(res => res.json())
                .then(response => {
                    if (response.success) {
                        listContainer.style.transition = 'all 0.3s ease';
                        listContainer.style.opacity = '0';
                        listContainer.style.transform = 'scale(0.9)';
                        setTimeout(() => listContainer.remove(), 300);
                    } else {
                        alert(response.data?.message || 'Error deleting list.');
                        deleteListBtn.style.opacity = '1';
                        deleteListBtn.style.pointerEvents = 'auto';
                    }
                });
            }
        }
    });

    // --- 4. Modal logic (card details) ---
    const modalAdmin = document.getElementById('alba-card-modal-admin');
    const modalBody = document.getElementById('alba-modal-body-admin');
    let currentOpenedCard = null;

    document.querySelectorAll('.alba-card').forEach(card => {
        card.addEventListener('click', function(e) {
            if (e.target.closest('.alba-card-action-btn') || e.target.closest('a') || e.target.closest('button')) return;
            
            const cardId = card.dataset.cardId;
            currentOpenedCard = card;
            
            if(modalAdmin) {
                modalAdmin.classList.remove('alba-is-hidden');
                modalAdmin.style.display = 'flex'; 
                modalAdmin.classList.add('active');
            }
            if(modalBody) modalBody.innerHTML = albaBoard.loading || 'Loading...';

            const formData = new FormData();
            formData.append('action', 'alba_get_card_details_admin');
            formData.append('card_id', cardId);
            formData.append('nonce', albaBoard.get_card_details_nonce);

            fetch(albaBoard.ajaxurl, {
                method: 'POST',
                body: formData
            })
            .then(res => res.text())
            .then(html => {
                if(modalBody) modalBody.innerHTML = html;
                bindSaveButtonHandler();
                bindAttachmentHandlers(); 
                jQuery(document).trigger('alba_modal_loaded');
            });
        });
    });

    // --- 5. Close modal handlers ---
    const closeModal = () => {
        if(modalAdmin) {
            modalAdmin.classList.add('alba-is-hidden');
            modalAdmin.style.display = '';
            modalAdmin.classList.remove('active');
        }
        currentOpenedCard = null;
    };

    document.getElementById('alba-modal-close-admin')?.addEventListener('click', closeModal);
    window.addEventListener('click', (e) => {
        if (e.target.id === 'alba-card-modal-admin') closeModal();
    });

    // --- 6. ACCESSIBILITY: Safe Keyboard Shortcuts ---
    document.addEventListener('keydown', (e) => {
        const isModalOpen = modalAdmin && !modalAdmin.classList.contains('alba-is-hidden');

        if (e.key === 'Escape') {
            if (isModalOpen) {
                closeModal();
                return;
            }
            document.querySelectorAll('.alba-stacked-form:not(.alba-is-hidden), .alba-inline-form:not(.alba-is-hidden)').forEach(form => {
                form.classList.add('alba-is-hidden');
                if (form.previousElementSibling && form.previousElementSibling.tagName === 'BUTTON') {
                    form.previousElementSibling.classList.remove('alba-is-hidden');
                }
            });
        }

        if ((e.ctrlKey || e.metaKey) && e.key === 'Enter') {
            if (isModalOpen) {
                e.preventDefault(); 
                const saveBtn = document.getElementById('alba-card-save-btn');
                if (saveBtn) {
                    saveBtn.focus();
                    saveBtn.click();
                }
            }
        }
    });

    // --- 7. Save button handler ---
    function bindSaveButtonHandler() {
        const saveBtn = document.getElementById('alba-card-save-btn');
        if (!saveBtn) return;

        saveBtn.onclick = function (e) {
            e.preventDefault();
            const form = document.getElementById('alba-card-details-form');
            if (!form) return;

            const formData = new FormData(form);
            formData.append('action', 'alba_save_card_details_admin');
            formData.append('nonce', albaBoard.save_card_details_nonce);

            const newCommentField = document.getElementById('alba-new-comment');
            if (newCommentField) {
                formData.set('new_comment', newCommentField.value);
            }

            saveBtn.textContent = 'Saving...';
            saveBtn.style.opacity = '0.7';

            fetch(albaBoard.ajaxurl, {
                method: 'POST',
                body: formData
            })
            .then(res => res.json())
            .then(response => {
                if (response && response.success) {
                    if(modalBody) modalBody.innerHTML = response.data.html || '';
                    bindSaveButtonHandler();
                    bindAttachmentHandlers();
                    jQuery(document).trigger('alba_modal_loaded');
                    location.reload(); 
                } else {
                    saveBtn.textContent = 'Save Changes';
                    saveBtn.style.opacity = '1';
                }
            });
        }
    }

    // --- 8. File Upload & Delete Handlers ---
    function bindAttachmentHandlers() {
        const fileInput = document.getElementById('alba-file-upload-input');
        const triggerBtn = document.getElementById('alba-trigger-upload-btn');
        const feedbackDiv = document.getElementById('alba-upload-feedback');
        const cardIdInput = document.getElementById('alba-current-card-id');

        if (triggerBtn && fileInput) triggerBtn.onclick = () => fileInput.click();

        if (fileInput) {
            fileInput.onchange = function(e) {
                const file = e.target.files[0];
                if (!file || !cardIdInput) return;

                feedbackDiv.textContent = albaBoard.uploading || 'Uploading...';
                feedbackDiv.style.color = '#2271b1';

                const formData = new FormData();
                formData.append('action', 'alba_upload_attachment');
                formData.append('card_id', cardIdInput.value);
                formData.append('nonce', albaBoard.upload_attachment_nonce);
                formData.append('file', file);

                fetch(albaBoard.ajaxurl, {
                    method: 'POST',
                    body: formData
                })
                .then(res => res.json())
                .then(response => {
                    if (response.success) {
                        feedbackDiv.textContent = '';
                        const listDiv = document.getElementById('alba-attachments-list');
                        const noMsg = document.getElementById('alba-no-attachments-msg');
                        if (noMsg) noMsg.remove();

                        const newItem = document.createElement('div');
                        newItem.className = 'alba-attachment-item';
                        newItem.id = 'alba-attachment-' + response.data.attachment_id;
                        newItem.style = 'display: flex; justify-content: space-between; align-items: center; background: #fff; padding: 8px 14px; border-radius: 12px; box-shadow: 2px 2px 6px #e3e7ef, -2px -2px 6px #fff; margin-top: 8px;';
                        
                        newItem.innerHTML = `
                            <a href="${response.data.file_url}" target="_blank" style="text-decoration: none; color: #3d4b5c; font-weight: 600; font-size: 0.95em;">📄 ${response.data.file_name}</a>
                            <button type="button" class="alba-delete-attachment-btn" data-attachment-id="${response.data.attachment_id}" style="background: none; border: none; color: #e05c4d; cursor: pointer; font-size: 1.2em; outline: none;">&times;</button>
                        `;
                        
                        listDiv.appendChild(newItem);
                        bindDeleteButtons();
                    } else {
                        feedbackDiv.textContent = response.data?.message || 'Error uploading.';
                        feedbackDiv.style.color = '#e05c4d';
                    }
                    fileInput.value = '';
                });
            };
        }

        bindDeleteButtons();
    }

    function bindDeleteButtons() {
        document.querySelectorAll('.alba-delete-attachment-btn').forEach(btn => {
            btn.onclick = function(e) {
                e.preventDefault();
                const attachmentId = this.dataset.attachmentId;
                const cardIdInput = document.getElementById('alba-current-card-id');
                const itemDiv = document.getElementById('alba-attachment-' + attachmentId);

                if (!attachmentId || !cardIdInput) return;
                
                this.textContent = '...';
                this.disabled = true;

                const formData = new FormData();
                formData.append('action', 'alba_delete_attachment');
                formData.append('card_id', cardIdInput.value);
                formData.append('attachment_id', attachmentId);
                formData.append('nonce', albaBoard.delete_attachment_nonce);

                fetch(albaBoard.ajaxurl, {
                    method: 'POST',
                    body: formData
                })
                .then(res => res.json())
                .then(response => {
                    if (response.success && itemDiv) {
                        itemDiv.remove();
                    } else {
                        alert(response.data?.message || 'Failed to delete.');
                        this.textContent = '✖';
                        this.disabled = false;
                    }
                });
            };
        });
    }

    // --- 9. Select2 Initialization for Authors ---
    jQuery(document).on('DOMNodeInserted', function(e) {
        jQuery(e.target).find('.alba-select2:not(.alba-tags-select2)').each(function() {
            if (!jQuery(this).hasClass('select2-hidden-accessible')) {
                jQuery(this).select2({
                    width: '100%',
                    dropdownParent: jQuery('#alba-card-modal-admin .alba-modal-content')
                });
            }
        });
    });
});