<?php
/**
 * Akkedli Settings Handler
 * Manages plugin settings (API Key and Webhook URL)
 */

if (!defined('ABSPATH')) {
    exit;
}

class Akkedli_Settings
{

    private $option_group = 'akkedli_settings';

    public function __construct()
    {
        add_action('admin_init', array($this, 'register_settings'));
    }

    /**
     * Register plugin settings
     */
    public function register_settings()
    {
        // Register settings
        register_setting($this->option_group, 'akkedli_api_key', array(
            'type' => 'string',
            'sanitize_callback' => 'sanitize_text_field',
            'default' => ''
        ));

        register_setting($this->option_group, 'akkedli_webhook_url', array(
            'type' => 'string',
            'sanitize_callback' => 'esc_url_raw',
            'default' => ''
        ));

        register_setting($this->option_group, 'akkedli_send_pending', array(
            'type' => 'boolean',
            'sanitize_callback' => 'rest_sanitize_boolean',
            'default' => true
        ));

        register_setting($this->option_group, 'akkedli_send_processing', array(
            'type' => 'boolean',
            'sanitize_callback' => 'rest_sanitize_boolean',
            'default' => true
        ));

        // Add settings section
        add_settings_section(
            'akkedli_main_section',
            __('Connection Settings', 'akkedli-connector'),
            array($this, 'section_callback'),
            'akkedli-connector'
        );

        // Add settings fields
        add_settings_field(
            'akkedli_api_key',
            __('Akkedli API Key', 'akkedli-connector'),
            array($this, 'api_key_field_callback'),
            'akkedli-connector',
            'akkedli_main_section'
        );

        add_settings_field(
            'akkedli_webhook_url',
            __('Webhook URL', 'akkedli-connector'),
            array($this, 'webhook_url_field_callback'),
            'akkedli-connector',
            'akkedli_main_section'
        );

        add_settings_field(
            'akkedli_order_triggers',
            __('Send orders when status is', 'akkedli-connector'),
            array($this, 'order_triggers_field_callback'),
            'akkedli-connector',
            'akkedli_main_section'
        );
    }

    /**
     * Section description callback
     */
    public function section_callback()
    {
        echo '<p>' . esc_html__('Connect your store to Akkedli to automate order verification via WhatsApp.', 'akkedli-connector') . '</p>';
    }

    /**
     * API Key field callback
     */
    public function api_key_field_callback()
    {
        $value = get_option('akkedli_api_key', '');
        echo '<input type="text" id="akkedli_api_key" name="akkedli_api_key" value="' . esc_attr($value) . '" class="regular-text" placeholder="ak_xxxxxxxxxxxxxxxx" />';
        echo '<p class="description">' . esc_html__('Enter your Akkedli API Key from the dashboard.', 'akkedli-connector') . '</p>';
    }

    /**
     * Webhook URL field callback
     */
    public function webhook_url_field_callback()
    {
        $value = get_option('akkedli_webhook_url', '');
        echo '<input type="url" id="akkedli_webhook_url" name="akkedli_webhook_url" value="' . esc_attr($value) . '" class="regular-text" placeholder="https://your-webhook-url.com/..." />';
        echo '<p class="description">' . esc_html__('Enter the Webhook URL provided by Akkedli.', 'akkedli-connector') . '</p>';
    }

    /**
     * Order triggers field callback
     */
    public function order_triggers_field_callback()
    {
        $send_pending = get_option('akkedli_send_pending', true);
        $send_processing = get_option('akkedli_send_processing', true);

        echo '<label style="display: block; margin-bottom: 8px;">';
        echo '<input type="checkbox" name="akkedli_send_pending" value="1" ' . checked($send_pending, true, false) . ' />';
        echo ' ' . esc_html__('Pending', 'akkedli-connector');
        echo '</label>';

        echo '<label style="display: block;">';
        echo '<input type="checkbox" name="akkedli_send_processing" value="1" ' . checked($send_processing, true, false) . ' />';
        echo ' ' . esc_html__('Processing', 'akkedli-connector');
        echo '</label>';

        echo '<p class="description">' . esc_html__('Select which order statuses should trigger the webhook.', 'akkedli-connector') . '</p>';
    }

    /**
     * Get API Key
     */
    public static function get_api_key()
    {
        return get_option('akkedli_api_key', '');
    }

    /**
     * Get Webhook URL
     */
    public static function get_webhook_url()
    {
        return get_option('akkedli_webhook_url', '');
    }

    /**
     * Check if settings are configured
     */
    public static function is_configured()
    {
        return !empty(self::get_api_key()) && !empty(self::get_webhook_url());
    }

    /**
     * Check if API key is validated (via auto-connect)
     */
    public static function is_validated()
    {
        return (bool) get_option('akkedli_api_key_validated', false);
    }

    /**
     * Get connection timestamp
     */
    public static function get_connected_at()
    {
        return get_option('akkedli_connected_at', '');
    }
}
