<?php
/**
 * Akkedli Order Handler
 * Sends new orders to the configured webhook
 */

if (!defined('ABSPATH')) {
    exit;
}

class Akkedli_Order_Handler
{

    public function __construct()
    {
        $this->init_hooks();
    }

    private function init_hooks()
    {
        // Hook into new order creation (HPOS compatible)
        add_action('woocommerce_new_order', array($this, 'handle_new_order'), 10, 2);

        // HPOS Store API hook
        add_action('woocommerce_store_api_checkout_order_processed', array($this, 'handle_new_order_hpos'), 10, 1);

        // Hook for status changes to processing
        add_action('woocommerce_order_status_processing', array($this, 'handle_order_processing'), 10, 2);
    }

    /**
     * Handle new order creation
     */
    public function handle_new_order($order_id, $order = null)
    {
        if (!$order) {
            $order = wc_get_order($order_id);
        }

        if (!$order) {
            return;
        }

        // Check if plugin is configured
        if (!Akkedli_Settings::is_configured()) {
            return;
        }

        // Check if we should send pending orders
        $send_pending = get_option('akkedli_send_pending', true);
        if (!$send_pending) {
            return;
        }

        // Prevent duplicate sends
        $transient_key = 'akkedli_order_sent_' . $order_id;
        if (get_transient($transient_key)) {
            return;
        }

        // Set transient to prevent duplicates (5 minutes)
        set_transient($transient_key, true, 5 * MINUTE_IN_SECONDS);

        // Send to webhook
        $this->send_order_to_webhook($order);
    }

    /**
     * Handle HPOS Store API order
     */
    public function handle_new_order_hpos($order)
    {
        if ($order instanceof WC_Order) {
            $this->handle_new_order($order->get_id(), $order);
        }
    }

    /**
     * Handle order status change to processing
     */
    public function handle_order_processing($order_id, $order = null)
    {
        if (!$order) {
            $order = wc_get_order($order_id);
        }

        if (!$order) {
            return;
        }

        // Check if plugin is configured
        if (!Akkedli_Settings::is_configured()) {
            return;
        }

        // Check if we should send processing orders
        $send_processing = get_option('akkedli_send_processing', true);
        if (!$send_processing) {
            return;
        }

        // Prevent duplicate sends
        $transient_key = 'akkedli_order_sent_' . $order_id;
        if (get_transient($transient_key)) {
            return;
        }

        set_transient($transient_key, true, 5 * MINUTE_IN_SECONDS);

        $this->send_order_to_webhook($order);
    }

    /**
     * Send order data to webhook
     */
    private function send_order_to_webhook($order)
    {
        $webhook_url = Akkedli_Settings::get_webhook_url();
        $api_key = Akkedli_Settings::get_api_key();

        if (empty($webhook_url)) {
            return;
        }

        // Build order data payload
        $order_data = array(
            'event' => 'order.created',
            'api_key' => $api_key,
            'domain' => home_url(),
            'timestamp' => current_time('c'),
            'order' => array(
                'id' => $order->get_id(),
                'number' => $order->get_order_number(),
                'status' => $order->get_status(),
                'currency' => $order->get_currency(),
                'total' => $order->get_total(),
                'subtotal' => $order->get_subtotal(),
                'shipping_total' => $order->get_shipping_total(),
                'payment_method' => $order->get_payment_method(),
                'payment_title' => $order->get_payment_method_title(),
                'date_created' => $order->get_date_created() ? $order->get_date_created()->format('c') : '',
                'customer_note' => $order->get_customer_note(),
                'billing' => array(
                    'first_name' => $order->get_billing_first_name(),
                    'last_name' => $order->get_billing_last_name(),
                    'company' => $order->get_billing_company(),
                    'address_1' => $order->get_billing_address_1(),
                    'address_2' => $order->get_billing_address_2(),
                    'city' => $order->get_billing_city(),
                    'state' => $order->get_billing_state(),
                    'postcode' => $order->get_billing_postcode(),
                    'country' => $order->get_billing_country(),
                    'email' => $order->get_billing_email(),
                    'phone' => $order->get_billing_phone()
                ),
                'shipping' => array(
                    'first_name' => $order->get_shipping_first_name(),
                    'last_name' => $order->get_shipping_last_name(),
                    'address_1' => $order->get_shipping_address_1(),
                    'address_2' => $order->get_shipping_address_2(),
                    'city' => $order->get_shipping_city(),
                    'state' => $order->get_shipping_state(),
                    'postcode' => $order->get_shipping_postcode(),
                    'country' => $order->get_shipping_country(),
                    'phone' => $order->get_shipping_phone()
                ),
                'line_items' => array()
            )
        );

        // Add line items
        foreach ($order->get_items() as $item_id => $item) {
            $product = $item->get_product();

            $item_data = array(
                'id' => $item_id,
                'name' => $item->get_name(),
                'product_id' => $item->get_product_id(),
                'variation_id' => $item->get_variation_id(),
                'quantity' => $item->get_quantity(),
                'subtotal' => $item->get_subtotal(),
                'total' => $item->get_total(),
                'sku' => $product ? $product->get_sku() : '',
                'price' => $product ? $product->get_price() : ''
            );

            $order_data['order']['line_items'][] = $item_data;
        }

        // Send webhook request
        $response = wp_remote_post($webhook_url, array(
            'body' => wp_json_encode($order_data),
            'headers' => array(
                'Content-Type' => 'application/json',
                'User-Agent' => 'Akkedli-Connector/' . AKKEDLI_VERSION,
                'X-API-Key' => $api_key
            ),
            'timeout' => 30
        ));

        // Log result
        if (is_wp_error($response)) {
            $order->add_order_note(
                sprintf(
                    /* translators: %1$s: error message from webhook */
                    __('Akkedli webhook failed: %1$s', 'akkedli-connector'),
                    $response->get_error_message()
                )
            );
        } else {
            $response_code = wp_remote_retrieve_response_code($response);
            if ($response_code >= 200 && $response_code < 300) {
                $order->add_order_note(__('Order sent to Akkedli successfully.', 'akkedli-connector'));
            } else {
                $order->add_order_note(
                    sprintf(
                        /* translators: %1$d: HTTP response status code */
                        __('Akkedli webhook returned status: %1$d', 'akkedli-connector'),
                        $response_code
                    )
                );
            }
        }
    }
}
