<?php

if (!defined('WP_UNINSTALL_PLUGIN')) {
    exit;
}

// Include the Config class file
require_once plugin_dir_path(__FILE__) . 'includes/class-config.php';
// Include the Article class file
require_once plugin_dir_path(__FILE__) . 'includes/class-article.php';
// Include the DBHandler class file
require_once plugin_dir_path(__FILE__) . 'includes/class-dbhandler.php';
// Include config for API endpoint constants
require_once plugin_dir_path(__FILE__) . 'config.php';

/**
 * Delete Qdrant collection for the current site
 */
function aizl_delete_qdrant_collection() {
    if (!defined('AIZL_API_COLLECTION_ENDPOINT')) {
        return;
    }

    $db_handler = new AIZLabs\ChatAgent\DBHandler();
    $access_key = $db_handler->get_access_key();

    if ($access_key) {
        // Call API to delete Qdrant collection
        wp_remote_post(AIZL_API_COLLECTION_ENDPOINT, array(
            'body' => wp_json_encode(array('action' => 'delete')),
            'headers' => array(
                'Content-Type' => 'application/json',
                'x-access-key' => $access_key,
                'x-plugin-source' => 'aizlabs_chat_agent'
            ),
            'timeout' => 60,
            'blocking' => false,
        ));
    }
}

/**
 * Delete all OpenAI files from articles
 */
function aizl_delete_openai_files() {
    global $wpdb;
    $table_article = AIZLabs\ChatAgent\Config::get_table_name('article');

    $db_handler = new AIZLabs\ChatAgent\DBHandler();
    $access_key = $db_handler->get_access_key();

    if (!$access_key) {
        return;
    }

    // Get all file_ids from articles
    // phpcs:ignore WordPress.DB.DirectDatabaseQuery.DirectQuery, WordPress.DB.DirectDatabaseQuery.NoCaching, WordPress.DB.PreparedSQL.InterpolatedNotPrepared -- Reading file_ids before uninstall cleanup
    $articles = $wpdb->get_results("SELECT file_id FROM `{$table_article}` WHERE file_id != ''");

    foreach ($articles as $article) {
        if (!empty($article->file_id)) {
            wp_remote_post(AIZL_API_DELETE_ENDPOINT, array(
                'body' => wp_json_encode(array(
                    'action' => 'delete_file',
                    'file_id' => $article->file_id
                )),
                'headers' => array(
                    'Content-Type' => 'application/json',
                    'x-access-key' => $access_key,
                    'x-plugin-source' => 'aizlabs_chat_agent'
                ),
                'timeout' => 30,
                'blocking' => false,
            ));
        }
    }
}

/**
 * Delete all OpenAI vector stores from agents
 */
function aizl_delete_openai_vector_stores() {
    global $wpdb;
    $table_agent = AIZLabs\ChatAgent\Config::get_table_name('agent');

    $db_handler = new AIZLabs\ChatAgent\DBHandler();
    $access_key = $db_handler->get_access_key();

    if (!$access_key) {
        return;
    }

    // Get all vector_store_ids from agents
    // phpcs:ignore WordPress.DB.DirectDatabaseQuery.DirectQuery, WordPress.DB.DirectDatabaseQuery.NoCaching, WordPress.DB.PreparedSQL.InterpolatedNotPrepared -- Reading vector_store_ids before uninstall cleanup
    $agents = $wpdb->get_results("SELECT vector_store_id FROM `{$table_agent}` WHERE vector_store_id != ''");

    foreach ($agents as $agent) {
        if (!empty($agent->vector_store_id)) {
            wp_remote_post(AIZL_API_DELETE_ENDPOINT, array(
                'body' => wp_json_encode(array(
                    'action' => 'delete_agent',
                    'vector_store_id' => $agent->vector_store_id
                )),
                'headers' => array(
                    'Content-Type' => 'application/json',
                    'x-access-key' => $access_key,
                    'x-plugin-source' => 'aizlabs_chat_agent'
                ),
                'timeout' => 30,
                'blocking' => false,
            ));
        }
    }
}

/**
 * Drop database tables and delete pages for the current site
 */
function aizl_cleanup_site_data() {
    global $wpdb;

    // Get table names for current site
    $table_article = AIZLabs\ChatAgent\Config::get_table_name('article');
    $table_agent = AIZLabs\ChatAgent\Config::get_table_name('agent');
    $table_conversation = AIZLabs\ChatAgent\Config::get_table_name('conversation');
    $table_function = AIZLabs\ChatAgent\Config::get_table_name('function');
    $table_global = AIZLabs\ChatAgent\Config::get_table_name('global');

    // phpcs:ignore WordPress.DB.DirectDatabaseQuery.DirectQuery, WordPress.DB.DirectDatabaseQuery.NoCaching, WordPress.DB.DirectDatabaseQuery.SchemaChange, WordPress.DB.PreparedSQL.InterpolatedNotPrepared -- Dropping custom plugin tables on uninstall
    $wpdb->query("DROP TABLE IF EXISTS " . esc_sql($table_article));
    // phpcs:ignore WordPress.DB.DirectDatabaseQuery.DirectQuery, WordPress.DB.DirectDatabaseQuery.NoCaching, WordPress.DB.DirectDatabaseQuery.SchemaChange, WordPress.DB.PreparedSQL.InterpolatedNotPrepared -- Dropping custom plugin tables on uninstall
    $wpdb->query("DROP TABLE IF EXISTS " . esc_sql($table_agent));
    // phpcs:ignore WordPress.DB.DirectDatabaseQuery.DirectQuery, WordPress.DB.DirectDatabaseQuery.NoCaching, WordPress.DB.DirectDatabaseQuery.SchemaChange, WordPress.DB.PreparedSQL.InterpolatedNotPrepared -- Dropping custom plugin tables on uninstall
    $wpdb->query("DROP TABLE IF EXISTS " . esc_sql($table_conversation));
    // phpcs:ignore WordPress.DB.DirectDatabaseQuery.DirectQuery, WordPress.DB.DirectDatabaseQuery.NoCaching, WordPress.DB.DirectDatabaseQuery.SchemaChange, WordPress.DB.PreparedSQL.InterpolatedNotPrepared -- Dropping custom plugin tables on uninstall
    $wpdb->query("DROP TABLE IF EXISTS " . esc_sql($table_function));
    // phpcs:ignore WordPress.DB.DirectDatabaseQuery.DirectQuery, WordPress.DB.DirectDatabaseQuery.NoCaching, WordPress.DB.DirectDatabaseQuery.SchemaChange, WordPress.DB.PreparedSQL.InterpolatedNotPrepared -- Dropping custom plugin tables on uninstall
    $wpdb->query("DROP TABLE IF EXISTS " . esc_sql($table_global));

    // Delete the privacy policy page
    $privacy = AIZLabs\ChatAgent\Article::privacy_policy();
    $page = get_page_by_path($privacy->page_slug);
    if ($page) {
        wp_delete_post($page->ID, true);
    }

    // Delete the custom function doc page
    $custom_function = AIZLabs\ChatAgent\Article::custom_function_doc();
    $page = get_page_by_path($custom_function->page_slug);
    if ($page) {
        wp_delete_post($page->ID, true);
    }
}

/**
 * Delete the plugin's uploads folder
 */
function aizl_delete_uploads_folder() {
    $upload_dir = wp_upload_dir();
    $aizlabs_dir = $upload_dir['basedir'] . '/aizlabs-chat-agent';

    if (is_dir($aizlabs_dir)) {
        // Delete all files in the folder
        $files = glob($aizlabs_dir . '/*');
        foreach ($files as $file) {
            if (is_file($file)) {
                wp_delete_file($file);
            }
        }
        // Remove the directory
        rmdir($aizlabs_dir);
    }
}

// Remove custom database tables, external resources, and Qdrant collections
if (is_multisite()) {
    $aizl_sites = get_sites();
    foreach ($aizl_sites as $aizl_site) {
        switch_to_blog($aizl_site->blog_id);

        // 1. Delete Qdrant collection (needs access key from global table)
        aizl_delete_qdrant_collection();

        // 2. Delete all OpenAI files (needs access key + article table)
        aizl_delete_openai_files();

        // 3. Delete all OpenAI vector stores (needs access key + agent table)
        aizl_delete_openai_vector_stores();

        // 4. Drop tables and delete pages (last, after external cleanup)
        aizl_cleanup_site_data();

        // 5. Delete uploads folder
        aizl_delete_uploads_folder();

        restore_current_blog();
    }
} else {
    // 1. Delete Qdrant collection (needs access key from global table)
    aizl_delete_qdrant_collection();

    // 2. Delete all OpenAI files (needs access key + article table)
    aizl_delete_openai_files();

    // 3. Delete all OpenAI vector stores (needs access key + agent table)
    aizl_delete_openai_vector_stores();

    // 4. Drop tables and delete pages (last, after external cleanup)
    aizl_cleanup_site_data();

    // 5. Delete uploads folder
    aizl_delete_uploads_folder();
}
