<?php
namespace AIZLabs\ChatAgent;

class Upload {
    private $table_article;

    public function __construct() {
        $this->table_article = Config::get_table_name('article');
        add_action('wp_ajax_aizl_article_upload', array($this, 'aizl_article_upload'));
        add_action('wp_ajax_aizl_process_s3_file', array($this, 'aizl_process_s3_file'));
        add_action('wp_ajax_aizl_get_article_list', array($this, 'aizl_get_article_list'));
        add_action('wp_ajax_aizl_delete_article', array($this, 'aizl_delete_article'));
    }

    public function aizl_article_upload() {
        if (!check_ajax_referer('aizl_setting', 'nonce', false)) {
            wp_send_json_error('Invalid nonce.');
            return;
        }

        if (!isset($_POST['file_name']) || !isset($_POST['file_size'])) {
            wp_send_json_error('No file information provided.');
            return;
        }

        $db_handler = new DBHandler();
        $access_key = $db_handler->get_access_key();
        if(!$access_key) {
            wp_send_json_error('Invalid Access Key');
            return;
        }

        $file_name = sanitize_text_field(wp_unslash($_POST['file_name']));
        $file_size = intval($_POST['file_size']);

        // Normalize filename extension to lowercase for OpenAI compatibility
        $name_parts = explode('.', $file_name);
        if (count($name_parts) > 1) {
            $name_parts[count($name_parts) - 1] = strtolower($name_parts[count($name_parts) - 1]);
            $file_name = implode('.', $name_parts);
        }

        // Check file size against processing limit
        if ($file_size > 10 * 1024 * 1024) {
            wp_send_json_error('File too large. Maximum 10MB allowed.');
            return;
        }

        // Get presigned URL for S3 upload
        $presigned_response = $this->get_presigned_url($file_name, $file_size, $access_key);

        if (is_wp_error($presigned_response)) {
            wp_send_json_error('Failed to get upload URL.');
            return;
        }

        if (wp_remote_retrieve_response_code($presigned_response) != 200) {
            wp_send_json_error('Failed to get upload URL.');
            return;
        }

        $presigned_data = json_decode(wp_remote_retrieve_body($presigned_response), true);

        // Return upload URL to frontend for S3 upload
        wp_send_json_success(array(
            'upload_url' => $presigned_data['upload_url'],
            'file_key' => $presigned_data['file_key'],
            'file_name' => $file_name,
            'file_size' => $file_size,
            'action' => 'upload_to_s3'
        ));
    }

    private function get_presigned_url($file_name, $file_size, $access_key) {
        $api_url = Config::get_api_endpoint('presign');
        $api_response = wp_remote_post($api_url, array(
            'method' => 'POST',
            'body' => json_encode(array(
                'file_name' => $file_name,
                'file_size' => $file_size
            )),
            'headers' => array(
                'Content-Type' => 'application/json',
                'x-access-key' => $access_key,
                'x-plugin-source' => 'aizlabs_chat_agent'
            ),
            'timeout' => 60,
        ));
        return $api_response;
    }

    public function aizl_process_s3_file() {
        if (!check_ajax_referer('aizl_setting', 'nonce', false)) {
            wp_send_json_error('Invalid nonce.');
            return;
        }

        if (!isset($_POST['file_key']) || !isset($_POST['file_name'])) {
            wp_send_json_error('Missing file information.');
            return;
        }

        $file_key = sanitize_text_field(wp_unslash($_POST['file_key']));
        $file_name = sanitize_text_field(wp_unslash($_POST['file_name']));
        $file_size = isset($_POST['file_size']) ? intval($_POST['file_size']) : 0;

        $db_handler = new DBHandler();
        $access_key = $db_handler->get_access_key();
        if(!$access_key) {
            wp_send_json_error('Invalid Access Key');
            return;
        }

        // Call vector store lambda with S3 reference
        $api_url = Config::get_api_endpoint('upload');
        $api_response = wp_remote_post($api_url, array(
            'body' => json_encode(array(
                's3_bucket' => 'aizlabs-ai-agent-bucket',
                's3_key' => $file_key,
                'file_name' => $file_name
            )),
            'headers' => array(
                'Content-Type' => 'application/json',
                'x-access-key' => $access_key,
                'x-plugin-source' => 'aizlabs_chat_agent'
            ),
            'timeout' => 60,
        ));

        if (is_wp_error($api_response)) {
            $api_error_msg = $api_response->get_error_message();
            wp_send_json_error('Failed to process file: ' . $api_error_msg);
            return;
        }

        if (wp_remote_retrieve_response_code($api_response) != 200) {
            error_log('Upload API Error Response: ' . print_r($api_response, true));
            wp_send_json_error('Failed to create file.');
            return;
        }

        $api_response_body = json_decode(wp_remote_retrieve_body($api_response), true);
        $file_id = $api_response_body['file_id'];

        // Save to database
        // phpcs:ignore WordPress.DB.DirectDatabaseQuery.DirectQuery
        $article_id = $this->save_article_s3($file_name, $file_size, $file_id);

        if ($article_id) {
            wp_send_json_success('File uploaded and processed successfully.');
        } else {
            wp_send_json_error('File processed but failed to save to database.');
        }
    }

    private function save_article_s3($file_name, $file_size, $file_id) {
        global $wpdb;

        // phpcs:ignore WordPress.DB.DirectDatabaseQuery.DirectQuery -- Inserting into custom plugin table
        $result = $wpdb->insert($this->table_article, array(
            'file_type' => 'application/octet-stream',
            'file_name' => $file_name,
            'file_size' => $file_size,
            'file_id' => $file_id,
            'created_time' => gmdate('Y-m-d H:i:s'),
        ));

        if ($result === false) {
// phpcs:ignore WordPress.PHP.DevelopmentFunctions.error_log_error_log
            error_log('Database insert error: ' . $wpdb->last_error);
            return false;
        }

        return $wpdb->insert_id;
    }
    
    public function aizl_get_article_list() {
        if (!check_ajax_referer('aizl_setting', 'nonce', false)) {
            wp_send_json_error('Invalid nonce.');
            return;
        }
        //require_once __DIR__ . '/class-dbhandler.php';
        $db_handler = new DBHandler();
        $articles = $db_handler->get_articles();
        ob_start();
        if ($articles && count($articles) > 0) {
            foreach ($articles as $article) {
                ?>
                <div class="aizl-row aizl-article-row">
                    <div class="aizl-article-content">
                        <input type="checkbox" id="article_<?php echo esc_attr($article->id); ?>" name="articles[]" value="<?php echo esc_attr($article->id); ?>">
                        <label for="article_<?php echo esc_attr($article->id); ?>">
                            <?php echo esc_html(strtoupper($article->file_name)); ?>
                            <?php if (strtolower($article->file_name) === 'website_content.md') : ?>
                                <?php
                                $upload_dir = wp_upload_dir();
                                $review_file_path = $upload_dir['basedir'] . '/aizlabs-chat-agent/website_content.md';
                                if (file_exists($review_file_path)) :
                                    $review_file_url = $upload_dir['baseurl'] . '/aizlabs-chat-agent/website_content.md';
                                ?>
                                <a href="<?php echo esc_url($review_file_url); ?>" target="_blank" class="aizl-review-link" title="Download to review">[Review]</a>
                                <?php endif; ?>
                            <?php endif; ?>
                            <span class="aizl-tooltip">i
                                <span class="aizl-tooltip-text">Uploaded: <?php echo esc_html(wp_date('H:i:s Y-m-d', strtotime($article->created_time))); ?></span>
                            </span>
                        </label>                    </div>
                    <div class="aizl-article-actions">
                        <button type="button" class="aizl-delete-article-btn" data-article-id="<?php echo esc_attr($article->id); ?>" data-article-name="<?php echo esc_attr($article->file_name); ?>" title="<?php echo $article->in_use ? 'Cannot delete - file is used by agent(s)' : 'Delete Article'; ?>" <?php echo $article->in_use ? 'disabled' : ''; ?>>
                            <span class="aizl-trash-icon"></span>
                        </button>
                    </div>
                </div>
                <?php
            }
        } else {
            echo wp_kses_post('<div>No files uploaded yet.</div>');
        }
        $html = ob_get_clean();
        wp_send_json_success($html);
    }

    public function aizl_delete_article() {
        if (!check_ajax_referer('aizl_setting', 'nonce', false)) {
            wp_send_json_error('Invalid nonce.');
            return;
        }

        if (!isset($_POST['article_id'])) {
            wp_send_json_error('No article ID provided.');
            return;
        }

        $article_id = intval($_POST['article_id']);
        
        // Get article file_id for API cleanup
        global $wpdb;
        // phpcs:ignore WordPress.DB.DirectDatabaseQuery.DirectQuery, WordPress.DB.DirectDatabaseQuery.NoCaching, WordPress.DB.PreparedSQL.NotPrepared, WordPress.DB.PreparedSQL.InterpolatedNotPrepared -- Custom table, property is safe
        $article = $wpdb->get_row($wpdb->prepare("SELECT file_id FROM {$this->table_article} WHERE id = %d", $article_id));

        if (!$article) {
            wp_send_json_error('Article not found.');
            return;
        }

        // Check if article is in use by any agent
        $db_handler = new DBHandler();
        if ($db_handler->is_article_in_use($article_id)) {
            wp_send_json_error('Cannot delete - file is currently used by one or more agents.');
            return;
        }
        $access_key = $db_handler->get_access_key();
        if(!$access_key) {
            wp_send_json_error('Invalid Access Key');
            return;
        }

        // Call API to delete file from OpenAI (not from vector store, just the file object)
        $api_url = Config::get_api_endpoint('delete');
        $api_response = wp_remote_post($api_url, array(
            'method' => 'POST',
            'body' => json_encode(array(
                'action' => 'delete_file',
                'file_id' => $article->file_id
            )),
            'headers' => array(
                'Content-Type' => 'application/json',
                'x-access-key' => $access_key,
                'x-plugin-source' => 'aizlabs_chat_agent'
            ),
            'timeout' => 60,
        ));

        // Delete from local database regardless of API response
        // phpcs:ignore WordPress.DB.DirectDatabaseQuery.DirectQuery, WordPress.DB.DirectDatabaseQuery.NoCaching
        $result = $wpdb->delete($this->table_article, array('id' => $article_id), array('%d'));

        if ($result === false) {
            wp_send_json_error('Failed to delete article from database.');
            return;
        }

        if (is_wp_error($api_response) || wp_remote_retrieve_response_code($api_response) != 200) {
            wp_send_json_success('Article deleted from database successfully.');
            return;
        }

        wp_send_json_success('Article deleted completely.');
    }


}