<?php
namespace AIZLabs\ChatAgent;

class Dashboard {
    public function __construct() {
        add_action('wp_ajax_aizl_get_dashboard_metrics', array($this, 'get_dashboard_metrics'));
        add_action('wp_ajax_aizl_get_dashboard_timeseries', array($this, 'get_dashboard_timeseries'));
        add_action('wp_ajax_aizl_get_conversation_history', array($this, 'get_conversation_history'));
        add_action('wp_ajax_aizl_delete_qdrant_cache', array($this, 'delete_qdrant_cache'));
    }

    public function get_dashboard_metrics() {
        if (!check_ajax_referer('aizl_dashboard', 'nonce', false)) {
            wp_send_json_error('Invalid nonce.');
            return;
        }

        if (!current_user_can('manage_options')) {
            wp_send_json_error('Unauthorized', 403);
        }
        $timezone = isset($_POST['timezone']) ? sanitize_text_field(wp_unslash($_POST['timezone'])) : 'UTC';
        $from = isset($_POST['from']) ? sanitize_text_field(wp_unslash($_POST['from'])) : null;
        $to = isset($_POST['to']) ? sanitize_text_field(wp_unslash($_POST['to'])) : null;
        $db = new DBHandler();
        $data = [
            'total_agents' => $db->get_total_agents(),
            'published_agents' => $db->get_published_agents(),
            'unique_users' => $db->get_unique_users($timezone, $from, $to),
            'total_interactions' => $db->get_total_interactions($timezone, $from, $to),
        ];
        wp_send_json_success($data);
    }

    public function get_dashboard_timeseries() {
        if (!check_ajax_referer('aizl_dashboard', 'nonce', false)) {
            wp_send_json_error('Invalid nonce.');
            return;
        }

        if (!current_user_can('manage_options')) {
            wp_send_json_error('Unauthorized', 403);
        }
        $timezone = isset($_POST['timezone']) ? sanitize_text_field(wp_unslash($_POST['timezone'])) : 'UTC';
        $from = isset($_POST['from']) ? sanitize_text_field(wp_unslash($_POST['from'])) : null;
        $to = isset($_POST['to']) ? sanitize_text_field(wp_unslash($_POST['to'])) : null;
        $bucket = isset($_POST['bucket']) ? sanitize_text_field(wp_unslash($_POST['bucket'])) : 'day';
        $db = new DBHandler();
        $result = $db->get_timeseries_data($timezone, $from, $to, $bucket);
        wp_send_json_success($result);
    }

    public function get_agents_dashboard() {
        if (!check_ajax_referer('aizl_dashboard', 'nonce', false)) {
            wp_send_json_error('Invalid nonce.');
            return;
        }

        if (!current_user_can('manage_options')) {
            wp_send_json_error('Unauthorized', 403);
        }
        $aizl_search = isset($_POST['search']) ? sanitize_text_field(wp_unslash($_POST['search'])) : '';
        $aizl_sort = isset($_POST['sort']) ? sanitize_text_field(wp_unslash($_POST['sort'])) : 'name';
        $aizl_order = isset($_POST['order']) ? sanitize_text_field(wp_unslash($_POST['order'])) : 'ASC';
        $aizl_page = isset($_POST['page']) ? intval($_POST['page']) : 1;
        $aizl_per_page = isset($_POST['per_page']) ? intval($_POST['per_page']) : 10;
        $aizl_db = new DBHandler();
        $aizl_result = $aizl_db->get_agents_dashboard($aizl_search, $aizl_sort, $aizl_order, $aizl_page, $aizl_per_page);
        wp_send_json_success($aizl_result);
    }

    public function get_conversation_history() {
        if (!check_ajax_referer('aizl_dashboard', 'nonce', false)) {
            wp_send_json_error('Invalid nonce.');
            return;
        }

        if (!current_user_can('manage_options')) {
            wp_send_json_error('Unauthorized', 403);
        }
        $agent_id = isset($_POST['agent_id']) ? intval($_POST['agent_id']) : 0;
        if (!$agent_id) {
            wp_send_json_error('Missing agent_id', 400);
        }

        $timezone = isset($_POST['timezone']) ? sanitize_text_field(wp_unslash($_POST['timezone'])) : 'UTC';
        $from = isset($_POST['from']) ? sanitize_text_field(wp_unslash($_POST['from'])) : null;
        $to = isset($_POST['to']) ? sanitize_text_field(wp_unslash($_POST['to'])) : null;
        $db = new DBHandler();
        $result = $db->get_conversations($agent_id, $timezone, $from, $to);
        wp_send_json_success($result);
    }

    public function delete_qdrant_cache() {
        if (!check_ajax_referer('aizl_dashboard', 'nonce', false)) {
            wp_send_json_error('Invalid nonce.');
            return;
        }

        if (!current_user_can('manage_options')) {
            wp_send_json_error('Unauthorized', 403);
        }

        $conversation_id = isset($_POST['conversation_id']) ? intval($_POST['conversation_id']) : 0;
        if (!$conversation_id) {
            wp_send_json_error('Missing conversation_id', 400);
        }

        // Fetch the conversation to get the content (question)
        $db = new DBHandler();
        $conversation = $db->get_conversation_by_id($conversation_id);
        if (!$conversation) {
            wp_send_json_error('Conversation not found', 404);
        }

        if ($conversation->source !== 'qdrant') {
            wp_send_json_error('This conversation is not from cache', 400);
        }

        $content = $conversation->content;

        // Get access key
        $global_setting = $db->get_global_setting();
        if (!$global_setting || empty($global_setting->access_key)) {
            wp_send_json_error('Access key not configured', 400);
        }

        // Call the collection API with delete_points action
        $api_url = Config::get_api_endpoint('collection');
        $api_response = wp_remote_post($api_url, array(
            'method' => 'POST',
            'body' => wp_json_encode(array(
                'action' => 'delete_points',
                'content' => $content,
            )),
            'headers' => array(
                'Content-Type' => 'application/json',
                'x-access-key' => $global_setting->access_key,
                'x-plugin-source' => 'aizlabs_chat_agent'
            ),
            'timeout' => 60,
        ));

        if (is_wp_error($api_response)) {
            wp_send_json_error('API Error: ' . $api_response->get_error_message());
            return;
        }

        $response_code = wp_remote_retrieve_response_code($api_response);
        $response_body = json_decode(wp_remote_retrieve_body($api_response), true);

        if ($response_code !== 200) {
            $error_msg = isset($response_body['message']) ? $response_body['message'] : 'Failed to delete cached response';
            wp_send_json_error($error_msg);
            return;
        }

        // Update the conversation source to 'qdrant_cleared'
        $db->update_conversation_source($conversation_id, 'qdrant_cleared');

        wp_send_json_success(isset($response_body['message']) ? $response_body['message'] : 'Cache deleted successfully');
    }
} 