// Global utility functions
function showNotification(message, type = 'info') {
    const notification = jQuery(`
        <div class="aizl-notification aizl-notification-${type}">
            ${message}
        </div>
    `);
    jQuery('#aizl-notification-container').append(notification);
    setTimeout(() => {
        notification.fadeOut(400, function() { jQuery(this).remove(); });
    }, 3000);
}

function reloadArticleList() {
    jQuery.ajax({
        url: ajaxurl,
        type: 'POST',
        data: {
            action: 'aizl_get_article_list',
            nonce: aizl_setting_nonce.nonce
        },
        success: function(response) {
            if (response.success) {
                jQuery('#aizl-article-list').html(response.data);
            }
        }
    });
}

jQuery(document).ready(function($) {

    // Listen for clicks on the menu items
    $('#aizl_setting_menu li').on('click', function() {
        var page = $(this).data('page');

        // Hide all pages first
        $('#aizl_settings_content .aizl-page').hide();

        // Show the selected page
        $('#aizl_' + page).show();

        // Remove 'active' class from all menu items and add to the clicked one
        $('#aizl_setting_menu li').removeClass('active');
        $(this).addClass('active');
    });

    // Listen for clicks on the agent list
    $('.aizl-agent-list1 li, .aizl-agent-list2 li, .aizl-agent-list3 li').on('click', function() {
        // Remove 'active' class from all agents and add to the clicked one
        $('.aizl-agent-list1 li, .aizl-agent-list2 li, .aizl-agent-list3 li').removeClass('active');
        $(this).addClass('active');
    });


    // Trigger click on the first item to show it by default (Upload Article)
    $('#aizl_setting_menu li:first').trigger('click');

    // Load agent scope when agent is selected
    $('select[name="agent_id"]').on('change', function() {
        var agent_id = $(this).val();
        
        // Clear previous selections
        $('select[name="selected_pages[]"] option').prop('selected', false);
        $('select[name="selected_admin_pages[]"] option').prop('selected', false);
        
        if (agent_id) {

            $('select[name="selected_pages[]"], select[name="selected_admin_pages[]"]').prop('disabled', false);
            
            // Request agent scope via AJAX
            $.ajax({
                url: ajaxurl,
                type: 'POST',
                data: {
                    action: 'aizl_get_agent_scope',
                    nonce: aizl_setting_nonce.nonce,
                    agent_id: agent_id
                },
                success: function(response) {
                    if (response.success && response.data) {
                        // Select the pages from the response
                        var scope = response.data;
                        
                        // Loop through scope array and select matching options
                        $.each(scope, function(index, value) {
                            // Check if it's an admin page (containing .php)
                            if (isNaN(value)) {
                                $('select[name="selected_admin_pages[]"] option[value="' + value + '"]').prop('selected', true);
                            } else {
                                // It's a page ID, frontend page
                                $('select[name="selected_pages[]"] option[value="' + value + '"]').prop('selected', true);
                            }
                        });
                    }
                },
                error: function() {
                    $('.aizl-plugin-container form').prepend('<div class="error"><p>Error loading agent scope.</p></div>');
                }
            });
        }
    });


    // Custom multi-select functionality for the publish page
    $(document).on('mousedown', 'select[name="selected_pages[]"] option, select[name="selected_admin_pages[]"] option', function(e) {
        e.preventDefault();

        var $option = $(this);
        var $select = $option.parent();
        var scroll = $select.scrollTop();

        // Toggle selection
        $option.prop('selected', !$option.prop('selected'));

        // Blur and refocus to prevent jump
        $select.blur();
        setTimeout(function() {
            $select.focus();
            $select.scrollTop(scroll);
        }, 1);

        return false;
    });


    // Create Agent
    $('#aizl_create_agent_form').submit(function(event) {
        event.preventDefault();

        $('#aizl_create_agent_button').prop('disabled', true);
        button_text = $('#aizl_create_agent_button').text();
        if (button_text === 'Create Chat Agent') {
            $('#aizl_create_agent_button').text('Creating...');
        } else {
            $('#aizl_create_agent_button').text('Updating...');
        }

        const checkboxes = document.querySelectorAll('input[name="articles[]"]');
        let checkedOne = false;
        checkboxes.forEach(function(checkbox) {
            if (checkbox.checked) {
                checkedOne = true;
            }
        });
        if (!checkedOne) {
            showNotification('Please select a knowledge article or upload first.', 'error');
            setTimeout(function() {
                $('#aizl_create_agent_button').prop('disabled', false);
                $('#aizl_create_agent_button').text(button_text);
            }, 2000);
            return;
        }

        var formData = new FormData(this);
        formData.append('action', 'aizl_create_agent');
        formData.append('nonce', aizl_setting_nonce.nonce);

        $.ajax({
            url: ajaxurl,
            type: 'POST',
            data: formData,
            processData: false,
            contentType: false,
            success: function(response) {
                // Reset button first
                $('#aizl_create_agent_button').prop('disabled', false);
                $('#aizl_create_agent_button').text(button_text);

                if (response.success) {
                    setTimeout(function() {
                        showNotification(response.data, 'success');
                    }, 1000);
                    setTimeout(function() {
                        location.reload();
                    }, 2000);
                } else {
                    showNotification(response.data, 'error');
                }
            },
            error: function(response) {
                // Reset button first
                $('#aizl_create_agent_button').prop('disabled', false);
                $('#aizl_create_agent_button').text(button_text);

                showNotification(response.data, 'error');
            }
        });

    });

    // Knowledge Base Link
    $(document).on('click', '#aizl-kb-link', function(e) {
        e.preventDefault();
        // Modal HTML
        var modalHtml = `
            <div class="aizl-modal-overlay">
                <div class="aizl-modal">
                    <button class="aizl-modal-close" title="Close">&times;</button>
                    <div class="aizl-modal-header">
                        <h2>Upload Knowledge Article</h2>
                    </div>
                    <hr class="aizl-hr">
                    <div class="aizl-modal-content">
                        <div class="aizl-modal-description">
                            <h4>Upload your knowledge articles to enrich your Agent's contextual understanding, enabling it to deliver more accurate and intelligent responses.</h4>
                        </div>
                        <form id="aizl_upload_article_form" method="post" enctype="multipart/form-data">
                            <div class="aizl-file-upload">
                                <input type="file" name="article_file" id="article_file" accept=".txt,.doc,.docx,.pdf,.pptx,.md,.html,.json" required>
                                <button type="submit" class="aizl-upload-btn" id="aizl_upload_article_button">Upload</button>
                            </div>
                        </form>
                        <hr>
                        <div class="aizl-modal-footer">
                            <p><b>Suggested Content:</b> Product catalogs, manuals, user guides, and technical documentation
                            </br><b>Supported Formats:</b> .txt, .doc, .docx, .pdf, .pptx, .md, .html, .json
                            </br><b>File Size Limit:</b> Maximum 10MB per file</p>
                        </div>
                    </div>
                </div>
            </div>
        `;
        // Remove any existing modal
        $('.aizl-modal-overlay').remove();
        // Append modal to body
        $('body').append(modalHtml);

    });

    // Handle upload in modal - S3 workflow
    $(document).on('submit', '#aizl_upload_article_form', function(event) {
        event.preventDefault();

        var fileInput = document.getElementById('article_file');
        var file = fileInput.files[0];

        if (!file) {
            showNotification('Please select a file to upload.', 'error');
            return;
        }

        // Client-side validation
        if (file.size > 10 * 1024 * 1024) {
            showNotification('File too large. Maximum 10MB allowed.', 'error');
            // Clear the file input
            fileInput.value = '';
            return;
        }

        updateUploadStatus('Processing...');
        $('#aizl_upload_article_button').prop('disabled', true);

        // Step 1: Get presigned URL
        getPresignedUrl(file)
            .then(function(data) {
                updateUploadStatus('Uploading...');
                // Step 2: Upload to S3
                return uploadToS3(file, data.upload_url)
                    .then(function() { return data; }); // Pass file_key forward
            })
            .then(function(data) {
                updateUploadStatus('Embedding...');
                // Step 3: Process with OpenAI
                return processS3File(data.file_key, data.file_name, file.size);
            })
            .then(function() {
                showNotification('File uploaded and processed successfully!', 'success');
                reloadArticleList();
                $('.aizl-modal-overlay').remove();
            })
            .catch(function(error) {
                showNotification('Upload failed: ' + error.message, 'error');
            })
            .finally(function() {
                $('#aizl_upload_article_button').prop('disabled', false);
                $('#aizl_upload_article_button').text('Upload');
            });
    });

    function updateUploadStatus(message) {
        $('#aizl_upload_article_button').text(message);
    }

    function getPresignedUrl(file) {
        return new Promise(function(resolve, reject) {
            $.ajax({
                url: ajaxurl,
                type: 'POST',
                data: {
                    action: 'aizl_article_upload',
                    nonce: aizl_setting_nonce.nonce,
                    file_name: file.name,
                    file_size: file.size
                },
                success: function(response) {
                    if (response.success && response.data.action === 'upload_to_s3') {
                        resolve(response.data);
                    } else {
                        reject(new Error(response.data || 'Failed to get upload URL'));
                    }
                },
                error: function(xhr, status, error) {
                    reject(new Error('Failed to get upload URL: ' + error));
                }
            });
        });
    }

    function uploadToS3(file, uploadUrl) {
        return new Promise(function(resolve, reject) {
            var xhr = new XMLHttpRequest();
            
            xhr.upload.addEventListener('progress', function(e) {
                if (e.lengthComputable) {
                    var percent = Math.round((e.loaded / e.total) * 100);
                    updateUploadStatus('Uploading: ' + percent + '%');
                }
            });

            xhr.onload = function() {
                if (xhr.status === 200) {
                    resolve();
                } else {
                    reject(new Error('S3 upload failed: ' + xhr.status + ' ' + xhr.responseText));
                }
            };

            xhr.onerror = function() {
                reject(new Error('S3 upload failed: Network error'));
            };

            xhr.open('PUT', uploadUrl);
            xhr.setRequestHeader('Content-Type', 'application/octet-stream');
            xhr.send(file);
        });
    }

    function processS3File(fileKey, fileName, fileSize) {
        return new Promise(function(resolve, reject) {
            $.ajax({
                url: ajaxurl,
                type: 'POST',
                data: {
                    action: 'aizl_process_s3_file',
                    nonce: aizl_setting_nonce.nonce,
                    file_key: fileKey,
                    file_name: fileName,
                    file_size: fileSize
                },
                success: function(response) {
                    if (response.success) {
                        resolve(response.data);
                    } else {
                        reject(new Error(response.data || 'Failed to process file'));
                    }
                },
                error: function() {
                    reject(new Error('Failed to process file'));
                }
            });
        });
    }


    // Agent List on Create Page
    $(document).on('click', '.aizl-agent-list2 li', function(e) {
        // If the click was on a child <a>, prevent default
        if ($(e.target).is('a')) {
            e.preventDefault();
        }
        var agent_id = $(this).find('a').data('agent_id');
        
        // If clicking "New Agent", just open create page with blank form
        if (agent_id === 'new') {
            $('#aizl_setting_menu li[data-page="create"]').trigger('click');
            // Clear form fields
            $('#agent_id').val('');
            $('#name').val('');
            $('#instructions').val('');
            $('#greeting_message').val('');
            $('#model').val('gpt-4o-mini'); // Set default model
            // Uncheck all articles and functions
            $('input[name="articles[]"]').prop('checked', false);
            $('input[name="functions[]"]').prop('checked', false);
            // Set page title for new agent
            $('#aizl_create_agent_title').text('Create Your Agent');
            $('#aizl_create_agent_button').text('Create Agent');
            $('#delete_agent_link').hide();

            $('#aizl_update_agent_info').hide();
            return;
        }

        //disable the agent form first
        $('#aizl_create_agent_form').find('input, textarea, select').prop('disabled', true);
        // Load agent info from database
        $.ajax({
            url: ajaxurl,
            type: 'POST',
            data: {
                action: 'aizl_get_agent',
                nonce: aizl_setting_nonce.nonce,
                agent_id: agent_id
            },
            success: function(response) {
                if (response.success) {
                    // Fill form with agent data
                    $('#aizl_create_agent_title').text('Update Your Agent');
                    $('#aizl_create_agent_button').text('Update Agent');

                    $('#agent_id').val(response.data.id);
                    $('#name').val(response.data.name);
                    $('#instructions').val(response.data.instructions);
                    $('#greeting_message').val(response.data.greeting_message);
                    $('#model').val(response.data.model);
                    $('#delete_agent_link').show();
                    $('#delete_agent_link').data('agent_id', response.data.id);

                    $('#aizl_update_agent_info').show();
                    
                    // Check selected articles
                    $('input[name="articles[]"]').prop('checked', false);
                    if (response.data.article_ids) {
                        var articleIds = JSON.parse(response.data.article_ids);
                        articleIds.forEach(function(article_id) {
                            $('#article_' + article_id).prop('checked', true);
                        });
                    }

                    // First, check all previously selected functions (from database)
                    $('input[name="functions[]"]').prop('checked', false);
                    var selectedFunctionIds = response.data.function_ids ? JSON.parse(response.data.function_ids) : [];
                    
                    // Check all previously selected functions first
                    selectedFunctionIds.forEach(function(function_id) {
                        $('#function_' + function_id).prop('checked', true);
                    });
                    
                    // Then check current function availability and update disabled state
                    $.ajax({
                        url: ajaxurl,
                        type: 'POST',
                        data: {
                            action: 'aizl_check_function_availability',
                            nonce: aizl_setting_nonce.nonce
                        },
                        success: function(availabilityResponse) {
                            if (availabilityResponse.success) {
                                var availability = availabilityResponse.data;
                                // Update disabled state for all functions based on current availability
                                Object.keys(availability).forEach(function(function_id) {
                                    var checkbox = $('#function_' + function_id);
                                    var isAvailable = availability[function_id];
                                    // Set disabled state - selected functions remain checked but greyed out if unavailable
                                    checkbox.prop('disabled', !isAvailable);
                                });
                            }
                        }
                    });

                } else {
                    showNotification('Error loading agent data', 'error');
                }
            },
            error: function() {
                showNotification('Error loading agent data', 'error');
            },
            complete: function() {
                $('#aizl_create_agent_form').find('input, textarea, select').prop('disabled', false);
            }
        });
    });


    // Agent List on Publish Page
    $(document).on('click', '.aizl-agent-list3 li', function(e) {
        // If the click was on a child <a>, prevent default
        if ($(e.target).is('a')) {
            e.preventDefault();
        }
        var agent_id = $(this).find('a').data('agent_id');
        
        // Update selected state in the list
        $('.aizl-agent-list3 li').removeClass('active');
        $(this).addClass('active');
        
        // Update the select and trigger change
        $('select[name="agent_id"]').val(agent_id).trigger('change');
    });


    // Publish Agent
    $('#aizl-publish-agent-form').on('submit', function(e) {
        e.preventDefault();

        $('#aizl_publish_agent_button').prop('disabled', true);
        var dots = 1;
        var publishingInterval = setInterval(function() {
            var text = 'Publishing' + '.'.repeat(dots);
            $('#aizl_publish_agent_button').text(text);
            dots = dots === 3 ? 1 : dots + 1;
        }, 500);

        var formData = new FormData(this);
        formData.append('action', 'aizl_publish_agent');
        formData.append('nonce', aizl_setting_nonce.nonce);

        $.ajax({
            url: ajaxurl,
            type: 'POST',
            data: formData,
            processData: false,
            contentType: false,
            success: function(response) {
                if (response.success) {
                    showNotification(response.data, 'success');
                } else {
                    showNotification('Error: ' + response.data, 'error');
                }
            },
            error: function(xhr) {
                showNotification('Error publishing agent. Please try again.', 'error');
            },
            complete: function() {
                $('#aizl_publish_agent_button').prop('disabled', false);
                $('#aizl_publish_agent_button').text('Publish Agent');
                clearInterval(publishingInterval);
            }
        });

    });
    

    
    // Add notification container if not present
    if ($('#aizl-notification-container').length === 0) {
        $('body').append('<div id="aizl-notification-container"></div>');
    }

    // Delete Agent Confirmation Popup
    $(document).on('click', '.aizl-delete-agent-link', function(e) {
        e.preventDefault();
        var agent_id = $(this).data('agent_id');
        
        // Remove any existing modal
        $('.aizl-modal-overlay').remove();
        
        // Modal HTML
        var modalHtml = `
            <div class="aizl-modal-overlay">
                <div class="aizl-modal">
                    <button class="aizl-modal-close" title="Close">&times;</button>
                    <div class="aizl-modal-header">
                        <h3>Confirm Deletion</h3>
                    </div>
                    <hr class="aizl-hr">
                    <div class="aizl-modal-content">
                        <p>Are you sure you want to delete this agent?</p>
                        <div class="aizl-modal-button">
                            <button id="aizl-confirm-delete" data-agent_id="${agent_id}">Delete</button>
                            <button id="aizl-confirm-cancel">Cancel</button>
                        </div>
                    </div>
                    <hr>
                    <div class="aizl-modal-footer">
                        <p>The assigned article and its functions will remain intact, but the conversation history with this agent will be deleted.</p>
                    </div>
                </div>
            </div>
        `;
        $('body').append(modalHtml);
    });

    // Handle confirm delete
    $(document).on('click', '#aizl-confirm-delete', function(e) {
        e.preventDefault();
        var agent_id = $(this).data('agent_id');

        $.ajax({
            url: ajaxurl,
            type: 'POST',
            data: {
                action: 'aizl_delete_agent',
                nonce: aizl_setting_nonce.nonce,
                agent_id: agent_id
            },
            success: function(response) {
                if (response.success) {
                    showNotification(response.data, 'success');
                    setTimeout(() => {
                        location.reload();
                    }, 2000);
                } else {
                    showNotification(response.data, 'error');
                }
            },
            error: function() {
                showNotification('Error deleting agent. Please try again.', 'error');
            }
        });
        $('.aizl-modal-overlay').remove();
    });


    // Close modal on click
    $(document).on('click', '.aizl-modal-close, .aizl-modal-overlay, #aizl-confirm-cancel', function(e) {
        if ($(e.target).hasClass('aizl-modal-close') || $(e.target).hasClass('aizl-modal-overlay') || $(e.target).is('#aizl-confirm-cancel')) {
            $('.aizl-modal-overlay').remove();
        }
    });


    // Save UI
    $('#aizl_save_ui_form').on('submit', function(e) {
        e.preventDefault();

        $('#aizl_save_ui_button').prop('disabled', true);
        var dots = 1;
        var savingInterval = setInterval(function() {
            var text = 'Saving' + '.'.repeat(dots);
            $('#aizl_save_ui_button').text(text);
            dots = dots === 3 ? 1 : dots + 1;
        }, 500);

        var agents_header = $('#agents_header').val();
        $.ajax({
            url: ajaxurl,
            type: 'POST',
            data: {
                action: 'aizl_save_ui',
                nonce: aizl_setting_nonce.nonce,
                agents_header: agents_header
            },
            success: function(response) {
                if (response.success) {
                    showNotification(response.data, 'success');
                } else {
                    showNotification(response.data, 'error');
                }
            },
            error: function(xhr) {
                showNotification('Please try again.', 'error');
            },
            complete: function() {
                $('#aizl_save_ui_button').prop('disabled', false);
                $('#aizl_save_ui_button').text('Save Appearance');
                clearInterval(savingInterval);
            }
        });
    });


    // Save Access Key
    $('#aizl_save_key_form').on('submit', function(e) {
        e.preventDefault();
        $('#aizl_save_key_button').prop('disabled', true);
        var dots = 1;
        var savingInterval = setInterval(function() {
            var text = 'Validating' + '.'.repeat(dots);
            $('#aizl_save_key_button').text(text);
            dots = dots === 3 ? 1 : dots + 1;
        }, 500);


        var access_key = $('#access_key').val();
        $.ajax({
            url: ajaxurl,
            type: 'POST',
            data: {
                action: 'aizl_save_key',
                nonce: aizl_setting_nonce.nonce,
                access_key: access_key
            },
            success: function(response) {
                if (response.success) {
                    showNotification(response.data, 'success');
                } else {
                    showNotification(response.data, 'error');
                }
            },
            error: function(xhr) {
                showNotification('Error validating access key. Please try again.', 'error');
            },
            complete: function() {
                $('#aizl_save_key_button').prop('disabled', false);
                $('#aizl_save_key_button').text('Save Key');
                clearInterval(savingInterval);
            }
        });
    });



    // Delete Article Confirmation
    $(document).on('click', '.aizl-delete-article-btn', function(e) {
        e.preventDefault();
        var articleId = $(this).data('article-id');
        var articleName = $(this).data('article-name');
        
        // Remove any existing modal
        $('.aizl-modal-overlay').remove();
        
        // Modal HTML
        var modalHtml = `
            <div class="aizl-modal-overlay">
                <div class="aizl-modal">
                    <button class="aizl-modal-close" title="Close">&times;</button>
                    <div class="aizl-modal-header">
                        <h3>Delete Article</h3>
                    </div>
                    <hr class="aizl-hr">
                    <div class="aizl-modal-content">
                        <p>Are you sure you want to delete "<strong>${articleName}</strong>" completely?</p>
                        <div class="aizl-modal-button">
                            <button id="aizl-confirm-delete-article" data-article-id="${articleId}">Delete</button>
                            <button id="aizl-confirm-cancel">Cancel</button>
                        </div>
                    </div>
                </div>
            </div>
        `;
        
        $('body').append(modalHtml);
    });

    // Confirm delete article
    $(document).on('click', '#aizl-confirm-delete-article', function() {
        var articleId = $(this).data('article-id');
        
        $(this).prop('disabled', true).text('Deleting...');
        
        $.ajax({
            url: ajaxurl,
            type: 'POST',
            data: {
                action: 'aizl_delete_article',
                nonce: aizl_setting_nonce.nonce,
                article_id: articleId
            },
            success: function(response) {
                if (response.success) {
                    showNotification(response.data, 'success');
                    reloadArticleList();
                    $('.aizl-modal-overlay').remove();
                } else {
                    showNotification('Error: ' + response.data, 'error');
                }
            },
            error: function() {
                showNotification('Error deleting article. Please try again.', 'error');
            },
            complete: function() {
                $('#aizl-confirm-delete-article').prop('disabled', false).text('Delete');
            }
        });
    });

});


