// Utility: Create an element with optional class and innerHTML
function createElement(tag, className = '', innerHTML = '') {
    const el = document.createElement(tag);
    if (className) el.className = className;
    if (innerHTML) el.innerHTML = innerHTML;
    return el;
}

// Utility: Scroll chat history to bottom
function scrollToBottom(el) {
    el.scrollTop = el.scrollHeight;
}

// Utility: Render a chat message (user or agent)
function renderMessage({ type, name, message, isError = false, message_time }) {
    const formatted_time = message_time ? message_time.split(', ')[1] : '';
    const wrapper = createElement('div', `aizl-chat-${type}`);
    
    // Create header row container
    const headerRow = createElement('div', `aizl-chat-${type}-header`);
    
    // Create profile photo div
    //const photoDiv = createElement('div', `aizl-chat-${type}-photo`);
    //const photoImg = createElement('img');
    //photoImg.src = type === 'agent' ? '/path/to/agent-avatar.png' : '/path/to/user-avatar.png';
    //photoDiv.appendChild(photoImg);
    
    const nameTimeContainer = createElement('div', `aizl-chat-${type}-name-time`);
    const nameDiv = createElement('div', `aizl-chat-${type}-name`, `<b>${name}</b>`);
    const timeDiv = createElement('div', `aizl-chat-${type}-time`, formatted_time);
    
    nameTimeContainer.appendChild(nameDiv);
    nameTimeContainer.appendChild(timeDiv);
    
    //headerRow.appendChild(photoDiv);
    headerRow.appendChild(nameTimeContainer);
    
    const msgDiv = createElement('div', `aizl-chat-${type}-message${isError ? ' aizl-error' : ''}`);
    const safeMessage = linkify(message);
    msgDiv.innerHTML = safeMessage;
    
    wrapper.appendChild(headerRow);
    wrapper.appendChild(msgDiv);
    return wrapper;
}

// Utility: Show/hide elements
function show(el) { el.style.display = 'block'; }
function hide(el) { el.style.display = 'none'; }
function flex(el) { el.style.display = 'flex'; }


function linkify(text) {
    const urlPattern = /\b(?:https?:\/\/|www\.)[^\s<>\]]+\b(?<!\.|,|;|:)/gi;
    let linked = text.replace(urlPattern, function(match) {
        let href = match;
        let displayUrl = match;
        if (!/^https?:\/\//i.test(href)) {
            href = 'https://' + href;
        }
        href = href.replace(/\/+$/, '');
        displayUrl = displayUrl.replace(/\/+$/, '');
        displayUrl = displayUrl.replace(/[.,;:!?]+$/, '');
        return `<a href="${href}" target="_blank" rel="noopener noreferrer">${displayUrl}</a>`;
    });
    // Unescape single-escaped \'
    linked = linked.replace(/\\'/g, "'");
    // Unescape double-escaped \""
    linked = linked.replace(/\\"/g, '"');
    // Replace newlines with <br>
    return linked.replace(/\n/g, '<br>');
}


jQuery(document).ready(function($) {
    // Capture the current page slug
    const currentPageSlug = window.location.pathname + window.location.search;

    // Check if current page is in scope via AJAX
    $.ajax({
        url: aizl_request_nonce.ajaxurl,
        type: 'POST',
        data: {
            action: 'aizl_check_agent_scope',
            nonce: aizl_request_nonce.nonce,
            current_url: window.location.href,
            page_slug: currentPageSlug
        },
        success: function(response) {
            if (!response.success) {
                return;
            }
            
            const agentId = response.data.agent_id;
            const agentsHeader = response.data.agents_header;
            const agentName = response.data.name;
            const agentGreetingMessage = response.data.greeting_message;

            window.agentId = agentId;
            window.agentName = agentName;
            window.agentGreetingMessage = agentGreetingMessage;

            // --- UI Elements ---
            const chatIcon = createElement('div', 'aizl-chat-icon', `
                <svg viewBox="0 0 24 24" fill="none" xmlns="http://www.w3.org/2000/svg">
                    <path d="M20 2H4C2.9 2 2 2.9 2 4V22L6 18H20C21.1 18 22 17.1 22 16V4C22 2.9 21.1 2 20 2Z" fill="currentColor"/>
                </svg>
            `);
            chatIcon.id = 'aizl-chat-icon';
            
            const chatPopup = createElement('div', 'aizl-chat-popup', `
                <div id="aizl-chat-header" class="aizl-chat-header">
                    <span>${agentsHeader}</span>
                    <button id="aizl-chat-close-button" class="aizl-chat-close-button">-</button>
                </div>
                <div id="aizl-chat-history" class="aizl-chat-history"></div>
                <div class="aizl-chat-input-container">
                    <textarea id="aizl-chat-input" placeholder="Type your message..."></textarea>
                    <button id="aizl-chat-send-button">Send</button>
                </div>
            `);
            chatPopup.id = 'aizl-chat-popup';

            // --- Append elements to body ---
            document.body.appendChild(chatIcon);
            document.body.appendChild(chatPopup);
            // Initially hide the chat popup
            hide(chatPopup);
            hide(chatIcon);
            
            // --- DOM References ---
            const sendButton = document.getElementById('aizl-chat-send-button');
            const chatInput = document.getElementById('aizl-chat-input');
            const chatHistory = document.getElementById('aizl-chat-history');
            const closeButton = document.getElementById('aizl-chat-close-button');

            window.sendButton = sendButton;
            window.chatInput = chatInput;
            window.chatHistory = chatHistory;
            window.closeButton = closeButton;
            
            // --- UI Events ---
            chatIcon.addEventListener('click', function() {
                show(chatPopup);
                scrollToBottom(chatHistory);
                hide(chatIcon);
                chatInput.focus();
            });
            closeButton.addEventListener('click', function() {
                hide(chatPopup);
                flex(chatIcon);
            });
            
            // Auto-resize textarea as user types
            chatInput.addEventListener('input', function() {
                chatInput.style.height = 'auto';
                chatInput.style.height = `${chatInput.scrollHeight}px`;
            });
            
            // Send message on button click or Enter (not Shift+Enter)
            sendButton.addEventListener('click', function(e) {
                if (chatInput.value.trim() !== '') {
                    e.preventDefault();
                    sendMessage();
                }
            });
            chatInput.addEventListener('keypress', function(e) {
                if (e.key === 'Enter' && !e.shiftKey && chatInput.value.trim() !== '') {
                    e.preventDefault();
                    sendMessage();
                }
            });
            

            // Copy theme styles to Chat Panel
            copyThemeStyles();
            
            // --- Load chat history from server ---
            const sessionId = aizl_request_nonce.session_id;
            loadChatHistoryFromServer(agentId, agentName, agentGreetingMessage, sessionId, chatHistory, function() {
                flex(chatIcon);
                hide(chatPopup);
            });
        },
        error: function(xhr, status, error) {
            console.error('Error checking page scope:', error);
        }
    });


    function loadChatHistoryFromServer(agentId, agentName, agentGreetingMessage, sessionId, chatHistoryElement, callback) {
        jQuery.ajax({
            url: aizl_request_nonce.ajaxurl,
            type: 'POST',
            data: {
                action: 'aizl_get_chat_history',
                nonce: aizl_request_nonce.nonce,
                agent_id: agentId,
                session_id: sessionId
            },
            success: function(response) {
                chatHistoryElement.innerHTML = '';
                let hasHistory = false;
                if (response.success && Array.isArray(response.data) && response.data.length > 0) {
                    hasHistory = true;
                    response.data.forEach(function(entry) {
                        if (entry.content) {
                            const content_time = new Date(entry.content_time + 'Z').toLocaleString();
                            chatHistoryElement.appendChild(renderMessage({ type: 'user', name: 'You', message: entry.content, message_time: content_time }));
                        }
                        if (entry.response) {
                            const response_time = new Date(entry.response_time + 'Z').toLocaleString();
                            chatHistoryElement.appendChild(renderMessage({ type: 'agent', name: agentName, message: entry.response, message_time: response_time }));
                        }
                    });
                }

                // If no history, show greeting
                if (!hasHistory) {
                    const now = new Date();
                    const message_time = now.toLocaleString();
                    const message_time_iso = now.toISOString();
                    chatHistoryElement.appendChild(renderMessage({ type: 'agent', name: agentName, message: agentGreetingMessage, message_time: message_time }));
                    // Save the greeting message to the database for history display across sessions
                    jQuery.ajax({
                        url: aizl_request_nonce.ajaxurl,
                        type: 'POST',
                        data: {
                            action: 'aizl_save_conversation',
                            nonce: aizl_request_nonce.nonce,
                            agent_id: agentId,
                            session_id: sessionId,
                            content: null,
                            content_time: message_time_iso,
                            response_time: message_time_iso,
                            api_msg: agentGreetingMessage
                        }
                    });
                }

                scrollToBottom(chatHistoryElement);
                if (typeof callback === 'function') callback();
            },
            error: function() {
                if (typeof callback === 'function') callback();
            }
        });
    }


    function copyThemeStyles() {
        $.ajax({
            url: aizl_request_nonce.ajaxurl,
            type: 'POST',
            data: {
                action: 'aizl_get_theme_color',
                nonce: aizl_request_nonce.nonce
            },
            success: function(response) {

                const $majorColor = response.data;
                const $chatHeader = $('#aizl-chat-header');
                const $chatSendButton = $('#aizl-chat-send-button');
                const $chatIcon = $('.aizl-chat-icon');

                $chatHeader.css({
                    'background': $majorColor,
                });
                $chatSendButton.css({
                    'background': $majorColor,
                });
                $chatIcon.css({
                    'background': $majorColor,
                });
                
                return true;
                
            },
            error: function(response) {
                console.log(response);
            }
        });
    }


    // --- Chat Logic ---
    function sendMessage(userMessage) {
        if (!userMessage) {
            userMessage = chatInput.value.trim();
        }
        
        // Add user message
        const now = new Date();
        const message_time = now.toLocaleString();
        const message_time_iso = now.toISOString();
        chatHistory.appendChild(renderMessage({ type: 'user', name: 'You', message: userMessage, message_time: message_time }));
        scrollToBottom(chatHistory);
        chatInput.value = '';
        
        // Add loading indicator with typing animation delay
        const loadingMsg = renderMessage({ type: 'agent', name: agentName, message: 'Typing...', isError: false, message_time: message_time });
        let loadingDiv = null;
        let loadingInterval = null;

        setTimeout(() => {
            chatHistory.appendChild(loadingMsg);
            loadingDiv = loadingMsg.querySelector('.aizl-chat-agent-message');
            if (loadingDiv) loadingDiv.classList.add('loading');
            scrollToBottom(chatHistory);

            // Animate loading (start interval only after loadingDiv is available)
            let dotCount = 0;
            const maxDots = 3;
            const baseText = 'Typing';
            loadingInterval = setInterval(() => {
                if (loadingDiv) {
                    loadingDiv.innerHTML = baseText + '.'.repeat(dotCount);
                    dotCount = dotCount < maxDots ? dotCount + 1 : 1;
                }
            }, 500);
        }, 500);
        
        // AJAX: run agent
        $.ajax({
            url: aizl_request_nonce.ajaxurl,
            type: 'POST',
            data: {
                action: 'aizl_run_agent',
                nonce: aizl_request_nonce.nonce,
                agent_id: agentId,
                content: userMessage,
                content_time: message_time_iso
            },
            success: function(response) {
                clearInterval(loadingInterval);
                if (chatHistory.contains(loadingMsg)) {
                    chatHistory.removeChild(loadingMsg);
                }
                if (response.success) {
                    const response_time = new Date(response.data.response_time + 'Z').toLocaleString();
                    chatHistory.appendChild(renderMessage({ type: 'agent', name: agentName, message: response.data.message, message_time: response_time }));
                } else {
                    const response_time = new Date().toLocaleString();
                    chatHistory.appendChild(renderMessage({ type: 'agent', name: agentName, message: `<b>[System Message]</b> ${response.data}`, isError: true, message_time: response_time }));
                }
                scrollToBottom(chatHistory);
            },
            error: function() {
                clearInterval(loadingInterval);
                if (chatHistory.contains(loadingMsg)) {
                    chatHistory.removeChild(loadingMsg);
                }
                const response_time = new Date().toLocaleString();
                chatHistory.appendChild(renderMessage({ type: 'agent', name: agentName, message: '<b>[System Message]</b> An error occurred. Please try again later.', isError: true, message_time: response_time }));
                scrollToBottom(chatHistory);
            }
        });
    }


    // --- Function Link Click Handler ---
    $(document).on('click', '.aizl-function-link', handleFunctionLinkClick);

    // --- Function Tools ---
    function handleFunctionLinkClick(event) {
        event.preventDefault();
        const functionName = event.target.getAttribute('data-function');
        const functionDescription = event.target.textContent;
        
        // Add user message with square brackets to indicate function call
        sendMessage(`[${functionDescription}]`);
        
        // Clear any existing function links
        /*
        const existingLinks = chatHistory.querySelectorAll('.aizl-function-link');
        existingLinks.forEach(link => {
            link.style.pointerEvents = 'none';
            link.style.opacity = '0.5';
        });
        */
        
        // Scroll chat to bottom
        scrollToBottom(chatHistory);
    }


    

    
});
