    // Get UTC date range for last 24 hours
    function getLast24HoursUTCRange() {
        const now = new Date();
        const twentyFourHoursAgo = new Date(now.getTime() - (24 * 60 * 60 * 1000));
        return {
            from: formatDateTimeUTC(twentyFourHoursAgo),
            to: formatDateTimeUTC(now),
            bucket: 'hour'
        };
    }
jQuery(document).ready(function($) {
    if (!$('#aizl-dashboard-chart').length) return;

    // --- Chart Configuration ---
    let chart;
    let chartDataCache = {};
    let currentRange = '24h';
    const colorInteractions = '#2271b1';
    const colorUsers = '#d63638';

    // --- Date Utilities ---
    function formatLocalDateTime(date) {
        const pad = n => String(n).padStart(2, '0');
        return `${date.getFullYear()}-${pad(date.getMonth() + 1)}-${pad(date.getDate())} ` +
               `${pad(date.getHours())}:${pad(date.getMinutes())}:${pad(date.getSeconds())}`;
    }

    // Get current local time
    function getCurrentLocalTime() {
        return formatLocalDateTime(new Date());
    }

    // Get local time range for last X hours
    function getLastHoursLocalRange(hours) {
        const now = new Date();
        const start = new Date(now.getTime() - (hours * 60 * 60 * 1000));
        return {
            from: formatLocalDateTime(start),
            to: formatLocalDateTime(now),
            bucket: hours <= 24 ? 'hour' : 'day'
        };
    }

    // Get local date range for "n days ago"
    function getLocalDaysAgoRange(n) {
        const now = new Date();
        const start = new Date(now);
        start.setDate(now.getDate() - n);
        start.setHours(0, 0, 0, 0);
        
        return {
            from: formatLocalDateTime(start),
            to: formatLocalDateTime(now),
            bucket: detectBucket(formatLocalDateTime(start), formatLocalDateTime(now))
        };
    }

    // Format local date as YYYY-MM-DD
    function formatLocalDate(date) {
        const pad = n => String(n).padStart(2, '0');
        return `${date.getFullYear()}-${pad(date.getMonth() + 1)}-${pad(date.getDate())}`;
    }

    // Format local date and time as YYYY-MM-DDTHH:mm
    function formatLocalDateTimePicker(date) {
        const pad = n => n < 10 ? '0' + n : n;
        return date.getFullYear() + '-' +
            pad(date.getMonth() + 1) + '-' +
            pad(date.getDate()) + 'T' +
            pad(date.getHours()) + ':' +
            pad(date.getMinutes());
    }

    // --- Time Range Presets (all in local time) ---
    const rangePresets = {
        '24h': () => getLastHoursLocalRange(24),
        '7d': () => getLocalDaysAgoRange(6),
        '30d': () => getLocalDaysAgoRange(29),
        '90d': () => getLocalDaysAgoRange(89),
        '1y': () => getLocalDaysAgoRange(364),
        'all': () => {
            const now = new Date();
            const tenYearsAgo = new Date(now);
            tenYearsAgo.setFullYear(now.getFullYear() - 10);
            
            return {
                from: formatLocalDateTime(tenYearsAgo),
                to: formatLocalDateTime(now),
                bucket: 'year'
            };
        }
    };

    // --- Bucket Detection ---
    function detectBucket(fromDate, toDate) {
        const from = new Date(fromDate);
        const to = new Date(toDate);
        const diffHours = Math.ceil((to - from) / (1000 * 60 * 60));
        
        if (diffHours <= 24) return 'hour';
        if (diffHours <= (7 * 24)) return 'day';
        if (diffHours <= (30 * 24)) return 'day';
        if (diffHours <= (90 * 24)) return 'month';
        if (diffHours <= (365 * 24)) return 'month';
        return 'year';
    }

    // --- Data Fetching with Caching ---
    function fetchChartData(params, callback) {
        const { from, to, bucket } = params;
        const cacheKey = `${from || 'null'}_${to || 'null'}_${bucket}`;
        
        if (chartDataCache[cacheKey]) {
            callback(chartDataCache[cacheKey]);
            return;
        }

        $('#aizl-dashboard-chart').addClass('aizl-chart-loading');
        
        $.ajax({
            url: ajaxurl,
            type: 'POST',
            data: {
                action: 'aizl_get_dashboard_timeseries',
                nonce: aizl_dashboard_nonce.nonce,
                timezone: Intl.DateTimeFormat().resolvedOptions().timeZone,
                from: from,
                to: to,
                bucket: bucket
            },
            success: (response) => {
                if (response.success) {
                    chartDataCache[cacheKey] = response.data;
                    callback(response.data);
                } else {
                    console.error('API Error:', response);
                    callback([]);
                }
            },
            error: (xhr, status, error) => {
                console.error('AJAX Error:', error);
                callback([]);
            },
            complete: () => {
                $('#aizl-dashboard-chart').removeClass('aizl-chart-loading');
            }
        });
    }

    // --- Label Formatting ---
    function formatLabel(label, bucket) {
        if (!label) return '';
        try {
            const date = new Date(label);
            switch (bucket) {
                case 'hour':
                    return `${date.getHours().toString().padStart(2, '0')}:${date.getMinutes().toString().padStart(2, '0')}`;
                case 'day':
                    return formatLocalDate(date);
                case 'month':
                    return date.toLocaleDateString([], { month: 'short', year: 'numeric' });
                case 'year':
                    return date.getFullYear().toString();
                default:
                    return label;
            }
        } catch (e) {
            return label;
        }
    }

    // --- Chart Rendering ---
    function renderChart(data, bucket) {

        const ctx = $('#aizl-dashboard-chart')[0].getContext('2d');
        const labels = data.map(item => formatLabel(item.label, bucket));
        const interactions = data.map(item => item.request_count || 0);
        const users = data.map(item => item.unique_users || 0);

        if (chart) chart.destroy();

        chart = new Chart(ctx, {
            type: 'bar',
            data: {
                labels: labels,
                datasets: [
                    {
                        type: 'line',
                        label: 'Visitors',
                        data: users,
                        borderColor: colorUsers,
                        backgroundColor: 'transparent',
                        borderWidth: 2,
                        borderDash: [5, 3],
                        tension: 0.3,
                        pointRadius: 4,
                        pointBackgroundColor: colorUsers,
                        yAxisID: 'y1'
                    },
                    {
                        type: 'bar',
                        label: 'Interactions',
                        data: interactions,
                        backgroundColor: colorInteractions,
                        borderColor: colorInteractions,
                        borderWidth: 1,
                        borderRadius: 4,
                        yAxisID: 'y',
                        barPercentage: 0.3,
                        categoryPercentage: 0.8
                    }
                ]
            },
            options: {
                responsive: true,
                maintainAspectRatio: false,
                plugins: {
                    title: {
                        display: true,
                        text: 'Agent Interaction Overview',
                        font: { size: 16 },
                        padding: { top: 10, bottom: 20 }
                    },
                    legend: {
                        position: 'bottom',
                        labels: {
                            boxWidth: 12,
                            padding: 20,
                            usePointStyle: true,
                            font: { size: 12 }
                        }
                    },
                    tooltip: {
                        mode: 'index',
                        intersect: false,
                        callbacks: {
                            label: (context) => {
                                let label = context.dataset.label || '';
                                if (label) label += ': ';
                                label += context.parsed.y.toLocaleString();
                                return label;
                            }
                        },
                        titleFont: { size: 14 },
                        bodyFont: { size: 14 }
                    }
                },
                scales: {
                    x: {
                        type: 'category',
                        grid: { display: false },
                        ticks: {
                            callback: function(value, index, ticks) {
                                return this.chart.data.labels[index];
                            },
                            font: { size: 13 }
                        }
                    },
                    y: {
                        type: 'linear',
                        display: true,
                        position: 'left',
                        title: { 
                            display: true, 
                            text: 'Interactions',
                            font: { size: 14 }
                        },
                        beginAtZero: true,
                        ticks: { font: { size: 13 } }
                    },
                    y1: {
                        type: 'linear',
                        display: true,
                        position: 'right',
                        title: { 
                            display: true,
                            text: 'Visitors',
                            font: { size: 14 }
                        },
                        beginAtZero: true,
                        grid: { drawOnChartArea: false },
                        ticks: { font: { size: 13 } }
                    }
                }
            }
        });
    }

    // --- Time Range Handling ---
    function updateChartForRange(range, customFrom, customTo) {
        let params;
        
        if (range === 'custom' && customFrom && customTo) {
            params = {
                from: customFrom,
                to: customTo,
                bucket: detectBucket(customFrom, customTo)
            };
        } else if (rangePresets[range]) {
            params = rangePresets[range]();
        } else {
            params = rangePresets['30d']();
        }

        $('#aizl-chart-date-from').val(params.from).attr('max', params.to);
        $('#aizl-chart-date-to').val(params.to).attr('max', params.to);

        fetchChartData(params, (data) => {
            renderChart(data, params.bucket);
            getDashboardMetrics();
            getConversationHistory();
        });
    }

    // --- UI Initialization ---
    function initializeDatePickers() {
        const now = new Date();
        // user function getLastHoursLocalRange
        const range = getLastHoursLocalRange(24);
        $('#aizl-chart-date-from').val(range.from).attr('max', range.to);
        $('#aizl-chart-date-to').val(range.to);
    }

    // --- UI Event Handlers ---
    $('#aizl-chart-range').on('change', function() {
        currentRange = $(this).val();
        
        if (currentRange === 'custom') {
            $('#aizl-chart-date-range-selector').show();
        } else {
            $('#aizl-chart-date-range-selector').hide();
            updateChartForRange(currentRange);
        }
    });

    $('#aizl-chart-apply-custom-range').on('click', function() {
        const from = $('#aizl-chart-date-from').val();
        const to = $('#aizl-chart-date-to').val();
        
        if (from && to) {
            updateChartForRange('custom', from, to);
        }
    });

    // --- Initial Load ---
    initializeDatePickers();
    updateChartForRange(currentRange);
});