<?php
/**
 * Plugin Name: AIZ Labs Chat Agent
 * Plugin URI: https://aizlabs.com
 * Description: An AI-powered chat agent for WordPress that's easy to set up, providing instant, accurate, and personalized support to handle customer inquiries and requests effortlessly.
 * Version: 1.1.2
 * Stable tag: 1.1.2
 * Author: AI Zone Labs
 * Author URI: https://aizlabs.com/about
 * License: GPL-2.0+
 * License URI: http://www.gnu.org/licenses/gpl-2.0.txt
 * Text Domain: aizlabs-chat-agent
 * Domain Path: /languages
 */

namespace AIZLabs\ChatAgent;

if (!defined('ABSPATH')) {
    exit;
}

// Include the configuration file
require_once plugin_dir_path(__FILE__) . 'config.php';

// Define constants
define('AIZL_CHAT_AGENT_VERSION', '1.0.0');
define('AIZL_CHAT_AGENT_PLUGIN_FILE', __FILE__);
define('AIZL_CHAT_AGENT_PLUGIN_URL', plugin_dir_url(__FILE__));
define('AIZL_CHAT_AGENT_PLUGIN_DIR', plugin_dir_path(__FILE__));

// Class Autoloader
spl_autoload_register(function ($class) {
    $prefix = 'AIZLabs\\ChatAgent\\';
    $base_dir = AIZL_CHAT_AGENT_PLUGIN_DIR . 'includes/';
    $len = strlen($prefix);
    if (strncmp($prefix, $class, $len) !== 0) {
        return;
    }
    $relative_class = substr($class, $len);
    $file = $base_dir . 'class-' . strtolower($relative_class) . '.php';
    if (file_exists($file)) {
        require $file;
    }
});

// Activate the plugin
function aizl_agent_activate($network_wide) {
    // Initialize database
    $db = new DB();
    $db->network_activate($network_wide);

    // Create privacy policy page
    $aizl_privacy = \AIZLabs\ChatAgent\Article::privacy_policy();
    $aizl_privacy->create_page($network_wide);

    // Create custom function page
    $aizl_custom_function = \AIZLabs\ChatAgent\Article::custom_function_doc();
    $aizl_custom_function->create_page($network_wide);

    // Create uploads folder for scanned content
    $upload_dir = wp_upload_dir();
    $aizlabs_dir = $upload_dir['basedir'] . '/aizlabs-chat-agent';
    if (!file_exists($aizlabs_dir)) {
        wp_mkdir_p($aizlabs_dir);
    }
}
register_activation_hook(__FILE__, __NAMESPACE__ . '\\aizl_agent_activate');
add_action('init', ['\\AIZLabs\\ChatAgent\\Article', 'register_shortcodes']);

// Initialize the plugin
function aizl_agent_init() {
    // Check if API configuration is complete
    if (!\AIZLabs\ChatAgent\Config::is_api_configured()) {
        add_action('admin_notices', function() {
            echo wp_kses_post('<div class="notice notice-warning is-dismissible"><p><strong>AIZLabs Chat Agent:</strong> Please configure the API Gateway settings in <code>config.php</code>. Replace <code>YOUR_API_GATEWAY_KEY_HERE</code> with the valid API key.</p></div>');
        });
    }

    // Add service dependency notice
    add_action('admin_notices', function() {
        $screen = get_current_screen();
        if ($screen && strpos($screen->id, 'aizlabs-chat-agent') !== false) {
            $db_handler = new \AIZLabs\ChatAgent\DBHandler();
            $access_key = $db_handler->get_access_key();
            if (empty($access_key)) {
                echo wp_kses_post('<div class="notice notice-info update-nag inline"><p><strong>AIZLabs Chat Agent - External Service Required:</strong> This plugin requires an active API key from AI Zone Labs to function. User data will be transmitted to external services for AI processing. Please review our <a href="https://aizlabs.com/terms-of-service" target="_blank">Terms of Service</a> and <a href="https://aizlabs.com/privacy-policy" target="_blank">Privacy Policy</a>. To obtain an API key, visit <a href="https://aizlabs.com/#pricing" target="_blank">aizlabs.com</a>.</p></div>');
            }
        }
    });

    $ai_agent = new \AIZLabs\ChatAgent\ChatAgent();
    $ai_agent->init_plugin();

    //user custom function handler
    do_action('aizl_custom_function_handler');
}
add_action('init', __NAMESPACE__ . '\\aizl_agent_init', 1);
















