<?php
if ( ! defined( 'ABSPATH' ) ) {
	exit;
}

class AISP_CC_Email_Tools {

	public function __construct() {
		add_action( 'admin_init', array( $this, 'handle_tools_post' ) );
		add_action( 'admin_post_aisp_cc_email_export', array( $this, 'handle_export' ) );
	}

	/**
	 * Process Tools POST Actions.
	 */
	public function handle_tools_post() {

		// Nonce verification.
		if ( ! isset( $_POST['aisp_cc_email_tools_nonce'] ) ) {
			return;
		}

		$nonce = sanitize_text_field( wp_unslash( $_POST['aisp_cc_email_tools_nonce'] ) );

		if ( ! wp_verify_nonce( $nonce, 'aisp_cc_email_tools' ) ) {
			return;
		}

		if ( ! current_user_can( 'manage_options' ) ) {
			return;
		}

		global $wpdb;

		/**
		 * ==========================================================
		 * 1) BULK EMAIL & STATUS UPDATE (COMBINÉ CC1 + CC2)
		 * ==========================================================
		 */
		 
		if ( isset( $_POST['aisp_cc_email_bulk_replace_submit'] ) ) {

			$old_raw = isset( $_POST['aisp_cc_email_bulk_old'] )
				? sanitize_text_field( wp_unslash( $_POST['aisp_cc_email_bulk_old'] ) )
				: '';

			$new_raw = isset( $_POST['aisp_cc_email_bulk_new'] )
				? sanitize_text_field( wp_unslash( $_POST['aisp_cc_email_bulk_new'] ) )
				: '';

			$old = sanitize_email( $old_raw );
			$new = sanitize_email( $new_raw );

	$add_raw = array();
		if ( isset( $_POST['aisp_cc_email_status_add'] ) ) {
			$add_raw = array_map( 'sanitize_text_field', (array) wp_unslash( $_POST['aisp_cc_email_status_add'] ) );
		}

		$remove_raw = array();
		if ( isset( $_POST['aisp_cc_email_status_remove'] ) ) {
			$remove_raw = array_map( 'sanitize_text_field', (array) wp_unslash( $_POST['aisp_cc_email_status_remove'] ) );
		}

			$add_statuses    = array_map( 'sanitize_text_field', $add_raw );
			$remove_statuses = array_map( 'sanitize_text_field', $remove_raw );

			$has_new_email    = ! empty( $new );
			$has_status_changes = ! empty( $add_statuses ) || ! empty( $remove_statuses );

		
			if ( empty( $old ) ) {
				add_action(
					'admin_notices',
					function() {
						echo '<div class="notice notice-error"><p>' .
							 esc_html__( 'Old email is required for bulk update.', 'aisp-cc-email-manager' ) .
							 '</p></div>';
					}
				);
				return;
			}

		
			if ( ! $has_new_email && ! $has_status_changes ) {
				add_action(
					'admin_notices',
					function() {
						echo '<div class="notice notice-error"><p>' .
							 esc_html__( 'Please provide a new email and/or select statuses to add or remove.', 'aisp-cc-email-manager' ) .
							 '</p></div>';
					}
				);
				return;
			}

			
			$user_query = new WP_User_Query(
				array(
					'meta_query' => array(
						'relation' => 'OR',
						array(
							'key'     => 'cc_email_1',
							'value'   => $old,
							'compare' => '=',
						),
						array(
							'key'     => 'cc_email_2',
							'value'   => $old,
							'compare' => '=',
						),
					),
					'fields' => 'ID',
					'number' => -1,
				)
			);

			$replaced_count       = 0;
			$status_updated_count = 0;

			if ( ! empty( $user_query->results ) ) {
				foreach ( $user_query->results as $user_id ) {

					// CC1.
					$cc1 = sanitize_email( get_user_meta( $user_id, 'cc_email_1', true ) );
					if ( $cc1 === $old ) {

						// Remplacement email si demandé.
						if ( $has_new_email ) {
							update_user_meta( $user_id, 'cc_email_1', $new );
							$replaced_count++;
						}

						// Mise à jour des statuts si demandé.
						if ( $has_status_changes ) {
							$current_statuses = (array) get_user_meta( $user_id, 'cc_email_1_statuses', true );

							// Ajouter.
							foreach ( $add_statuses as $status ) {
								if ( ! in_array( $status, $current_statuses, true ) ) {
									$current_statuses[] = $status;
								}
							}

							// Retirer.
							if ( ! empty( $remove_statuses ) ) {
								$current_statuses = array_values(
									array_filter(
										$current_statuses,
										function( $s ) use ( $remove_statuses ) {
											return ! in_array( $s, $remove_statuses, true );
										}
									)
								);
							}

							update_user_meta( $user_id, 'cc_email_1_statuses', $current_statuses );
							$status_updated_count++;
						}
					}

				
					$cc2 = sanitize_email( get_user_meta( $user_id, 'cc_email_2', true ) );
					if ( $cc2 === $old ) {

					
						if ( $has_new_email ) {
							update_user_meta( $user_id, 'cc_email_2', $new );
							$replaced_count++;
						}

						
						if ( $has_status_changes ) {
							$current_statuses = (array) get_user_meta( $user_id, 'cc_email_2_statuses', true );

						
							foreach ( $add_statuses as $status ) {
								if ( ! in_array( $status, $current_statuses, true ) ) {
									$current_statuses[] = $status;
								}
							}

							
							if ( ! empty( $remove_statuses ) ) {
								$current_statuses = array_values(
									array_filter(
										$current_statuses,
										function( $s ) use ( $remove_statuses ) {
											return ! in_array( $s, $remove_statuses, true );
										}
									)
								);
							}

							update_user_meta( $user_id, 'cc_email_2_statuses', $current_statuses );
							$status_updated_count++;
						}
					}
				}
			}

			
			if ( 0 === $replaced_count && 0 === $status_updated_count ) {
				add_action(
					'admin_notices',
					function() {
						echo '<div class="notice notice-warning"><p>' .
							 esc_html__( 'No CC entries found for the specified email.', 'aisp-cc-email-manager' ) .
							 '</p></div>';
					}
				);
				return;
			}

			
			add_action(
				'admin_notices',
				function() use ( $replaced_count, $status_updated_count ) {
					$message = '';

					if ( $replaced_count > 0 ) {
						
						$message .= sprintf( // translators: %d is number of CC email fields replaced. 
							esc_html__(
								'%d CC email field replaced.',
								'aisp-cc-email-manager'
							),
							$replaced_count
						);
					}

					if ( $status_updated_count > 0 ) {
						if ( '' !== $message ) {
							$message .= ' ';
						}
						
						$message .= sprintf( // translators: %d is number of CC status configurations updated.
							esc_html__(
								'%d CC status configuration updated.',
								'aisp-cc-email-manager'
							),
							$status_updated_count
						);
					}

					echo '<div class="notice notice-success"><p>' . esc_html( $message ) . '</p></div>';

				}
			);
		}

		/**
		 * ==========================================================
		 * 2) CLEAR ALL CC
		 * ==========================================================
		 */
		if ( isset( $_POST['aisp_cc_email_clear_all_cc'] ) ) {

			$keys = array(
				'cc_email_1',
				'cc_email_2',
				'cc_email_1_statuses',
				'cc_email_2_statuses',
			);

			foreach ( $keys as $key ) {
				// phpcs:ignore WordPress.DB.DirectDatabaseQuery.DirectQuery, WordPress.DB.DirectDatabaseQuery.NoCaching
				$wpdb->query(
					$wpdb->prepare(
						"/* no-cache */ DELETE FROM {$wpdb->usermeta} WHERE meta_key = %s",
						$key
					)
				);
			}

			add_action(
				'admin_notices',
				function() {
					echo '<div class="notice notice-warning"><p>' .
						 esc_html__( 'All CC data cleared.', 'aisp-cc-email-manager' ) .
						 '</p></div>';
				}
			);
		}

		/**
		 * ==========================================================
		 * 3) IMPORT CSV — Sanitized & Validated
		 * ==========================================================
		 */
		if ( isset( $_POST['aisp_cc_email_import_submit'] ) && isset( $_FILES['aisp_cc_email_import_file'] ) ) {

			
			$tmp_name = isset( $_FILES['aisp_cc_email_import_file']['tmp_name'] )
				? sanitize_text_field( wp_unslash( $_FILES['aisp_cc_email_import_file']['tmp_name'] ) )
				: '';

			if ( empty( $tmp_name ) || ! is_uploaded_file( $tmp_name ) ) {
				add_action(
					'admin_notices',
					function() {
						echo '<div class="notice notice-error"><p>' .
							 esc_html__( 'Invalid CSV upload.', 'aisp-cc-email-manager' ) .
							 '</p></div>';
					}
				);
				return;
			}

	
			$file_name = isset( $_FILES['aisp_cc_email_import_file']['name'] )
				? sanitize_text_field( wp_unslash( $_FILES['aisp_cc_email_import_file']['name'] ) )
				: '';

			$filetype = wp_check_filetype( $file_name, array( 'csv' => 'text/csv' ) );

			if ( empty( $filetype['ext'] ) || 'csv' !== $filetype['ext'] ) {
				add_action(
					'admin_notices',
					function() {
						echo '<div class="notice notice-error"><p>' .
							 esc_html__( 'Please upload a valid CSV file.', 'aisp-cc-email-manager' ) .
							 '</p></div>';
					}
				);
				return;
			}

		
			if ( ! function_exists( 'WP_Filesystem' ) ) {
				require_once ABSPATH . 'wp-admin/includes/file.php';
			}

			WP_Filesystem();
			global $wp_filesystem;

			$content = $wp_filesystem->get_contents( $tmp_name );

			if ( empty( $content ) ) {
				add_action(
					'admin_notices',
					function() {
						echo '<div class="notice notice-error"><p>' .
							 esc_html__( 'Unable to read the CSV file.', 'aisp-cc-email-manager' ) .
							 '</p></div>';
					}
				);
				return;
			}

			$lines   = explode( "\n", $content );
			$updated = 0;

			foreach ( $lines as $line ) {

				$row_raw = str_getcsv( $line );
				$row     = array_map( 'sanitize_text_field', $row_raw );

				if ( empty( $row ) || ! isset( $row[0] ) ) {
					continue;
				}

				if ( strtolower( trim( $row[0] ) ) === 'user_id' ) {
					continue;
				}

				$user_id_raw = sanitize_text_field( $row[0] );
				$user_id     = intval( $user_id_raw );

				if ( $user_id <= 0 ) {
					continue;
				}

		
				$cc1_raw = isset( $row[2] ) ? sanitize_text_field( $row[2] ) : '';
				update_user_meta( $user_id, 'cc_email_1', sanitize_email( $cc1_raw ) );

	
				$cc2_raw = isset( $row[3] ) ? sanitize_text_field( $row[3] ) : '';
				update_user_meta( $user_id, 'cc_email_2', sanitize_email( $cc2_raw ) );

				
				$st1 = isset( $row[4] ) ? array_map( 'sanitize_text_field', array_map( 'trim', explode( ',', $row[4] ) ) ) : array();
				$st2 = isset( $row[5] ) ? array_map( 'sanitize_text_field', array_map( 'trim', explode( ',', $row[5] ) ) ) : array();

				update_user_meta( $user_id, 'cc_email_1_statuses', $st1 );
				update_user_meta( $user_id, 'cc_email_2_statuses', $st2 );

				$updated++;
			}

			add_action(
				'admin_notices',
				function() use ( $updated ) {
					printf(
						'<div class="notice notice-success"><p>%s</p></div>',
						esc_html(
							sprintf(
								/* translators: %d is number of users updated by CSV import. */
								_n( '%d user updated from CSV.', '%d users updated from CSV.', $updated, 'aisp-cc-email-manager' ),
								$updated
							)
						)
					);
				}
			);
		}
	}

	/**
	 * ==========================================================
	 * 4) EXPORT CSV HANDLER
	 * ==========================================================
	 */
	public function handle_export() {

		if ( ! current_user_can( 'manage_options' ) ) {
			wp_die( esc_html__( 'No permission.', 'aisp-cc-email-manager' ) );
		}

		if ( ! isset( $_POST['aisp_cc_email_export_nonce'] ) ) {
			wp_die( esc_html__( 'Security error.', 'aisp-cc-email-manager' ) );
		}

		$nonce = sanitize_text_field( wp_unslash( $_POST['aisp_cc_email_export_nonce'] ) );

		if ( ! wp_verify_nonce( $nonce, 'aisp_cc_email_export' ) ) {
			wp_die( esc_html__( 'Security error.', 'aisp-cc-email-manager' ) );
		}

		$filename = 'cc-email-export-' . gmdate( 'Y-m-d-His' ) . '.csv';

		header( 'Content-Type: text/csv' );
		header( 'Content-Disposition: attachment; filename=' . $filename );

		$output = fopen( 'php://output', 'w' );

		fputcsv(
			$output,
			array(
				'user_id',
				'user_email',
				'cc_email_1',
				'cc_email_2',
				'cc_email_1_statuses',
				'cc_email_2_statuses',
			)
		);

		$users = get_users( array( 'fields' => array( 'ID', 'user_email' ) ) );

		foreach ( $users as $user ) {

			$cc1 = get_user_meta( $user->ID, 'cc_email_1', true );
			$cc2 = get_user_meta( $user->ID, 'cc_email_2', true );

			$st1 = implode( ',', (array) get_user_meta( $user->ID, 'cc_email_1_statuses', true ) );
			$st2 = implode( ',', (array) get_user_meta( $user->ID, 'cc_email_2_statuses', true ) );

			$cc1_safe = sanitize_email( $cc1 );
			$cc2_safe = sanitize_email( $cc2 );
			$st1_safe = sanitize_text_field( $st1 );
			$st2_safe = sanitize_text_field( $st2 );

			fputcsv(
				$output,
				array(
					intval( $user->ID ),
					sanitize_email( $user->user_email ),
					$cc1_safe,
					$cc2_safe,
					$st1_safe,
					$st2_safe,
				)
			);
		}

		exit;
	}
}
