<?php
/**
 * Admin settings page
 *
 * @package Airgentic
 */

if ( ! defined( 'ABSPATH' ) ) {
	exit;
}

/**
 * Airgentic Admin Settings Class
 */
class Airgentic_Admin_Settings {

	/**
	 * Option group name
	 *
	 * @var string
	 */
	private const OPTION_GROUP = 'airgentic';

	/**
	 * Option name
	 *
	 * @var string
	 */
	private const OPTION_NAME = 'airgentic_options';

	/**
	 * Constructor
	 */
	public function __construct() {
		add_action( 'admin_menu', array( $this, 'add_settings_page' ) );
		add_action( 'admin_init', array( $this, 'register_settings' ) );
	}

	/**
	 * Add settings page to WordPress admin
	 */
	public function add_settings_page() {
		add_options_page(
			__( 'Airgentic Settings', 'airgentic' ),
			__( 'Airgentic', 'airgentic' ),
			'manage_options',
			'airgentic',
			array( $this, 'render_settings_page' )
		);
	}

	/**
	 * Register plugin settings
	 */
	public function register_settings() {
		register_setting(
			self::OPTION_GROUP,
			self::OPTION_NAME,
			array(
				'type'              => 'array',
				'sanitize_callback' => array( $this, 'sanitize_options' ),
				'default'           => array(
					'account_id'       => '',
					'service_id'       => '',
					'search_input_id'  => '',
					'search_button_id' => '',
					'enable_on_front'  => 1,
				),
			)
		);

		add_settings_section(
			'airgentic_main',
			__( 'Configuration', 'airgentic' ),
			array( $this, 'render_section_info' ),
			self::OPTION_GROUP
		);

		// Account ID
		add_settings_field(
			'account_id',
			__( 'Account ID', 'airgentic' ) . ' <span style="color:red;">*</span>',
			array( $this, 'render_text_field' ),
			self::OPTION_GROUP,
			'airgentic_main',
			array(
				'label_for'   => 'account_id',
				'description' => __( 'Required. Your Airgentic account identifier.', 'airgentic' ),
			)
		);

		// Service ID
		add_settings_field(
			'service_id',
			__( 'Service ID', 'airgentic' ) . ' <span style="color:red;">*</span>',
			array( $this, 'render_text_field' ),
			self::OPTION_GROUP,
			'airgentic_main',
			array(
				'label_for'   => 'service_id',
				'description' => __( 'Required. Your service identifier (typically "website").', 'airgentic' ),
			)
		);

		// Search Input ID (optional)
		add_settings_field(
			'search_input_id',
			__( 'Search Input Element ID', 'airgentic' ),
			array( $this, 'render_text_field' ),
			self::OPTION_GROUP,
			'airgentic_main',
			array(
				'label_for'   => 'search_input_id',
				'placeholder' => 'searchInput',
				'description' => __( 'Optional. The HTML ID of your search input field.', 'airgentic' ),
			)
		);

		// Search Button ID (optional)
		add_settings_field(
			'search_button_id',
			__( 'Search Button Element ID', 'airgentic' ),
			array( $this, 'render_text_field' ),
			self::OPTION_GROUP,
			'airgentic_main',
			array(
				'label_for'   => 'search_button_id',
				'placeholder' => 'searchButton',
				'description' => __( 'Optional. The HTML ID of your search button.', 'airgentic' ),
			)
		);

		// Enable on front-end
		add_settings_field(
			'enable_on_front',
			__( 'Enable on Front-end', 'airgentic' ),
			array( $this, 'render_checkbox_field' ),
			self::OPTION_GROUP,
			'airgentic_main',
			array( 'label_for' => 'enable_on_front' )
		);
	}

	/**
	 * Render section information
	 */
	public function render_section_info() {
		echo '<p>' . esc_html__( 'Configure your Airgentic account details below. You can find these in your Airgentic admin dashboard.', 'airgentic' ) . '</p>';
	}

	/**
	 * Render text field
	 *
	 * @param array $args Field arguments.
	 */
	public function render_text_field( $args ) {
		$options = get_option( self::OPTION_NAME, array() );
		$id      = $args['label_for'];
		$value   = isset( $options[ $id ] ) ? $options[ $id ] : '';
		$placeholder = isset( $args['placeholder'] ) ? $args['placeholder'] : '';

		printf(
			'<input type="text" id="%1$s" name="%2$s[%1$s]" value="%3$s" class="regular-text" placeholder="%4$s" />',
			esc_attr( $id ),
			esc_attr( self::OPTION_NAME ),
			esc_attr( $value ),
			esc_attr( $placeholder )
		);

		if ( ! empty( $args['description'] ) ) {
			echo '<p class="description">' . esc_html( $args['description'] ) . '</p>';
		}
	}

	/**
	 * Render checkbox field
	 *
	 * @param array $args Field arguments.
	 */
	public function render_checkbox_field( $args ) {
		$options = get_option( self::OPTION_NAME, array() );
		$id      = $args['label_for'];
		$checked = ! empty( $options[ $id ] );

		printf(
			'<label><input type="checkbox" id="%1$s" name="%2$s[%1$s]" value="1" %3$s /> %4$s</label>',
			esc_attr( $id ),
			esc_attr( self::OPTION_NAME ),
			checked( $checked, true, false ),
			esc_html__( 'Load Airgentic widget on all front-end pages', 'airgentic' )
		);
	}

	/**
	 * Sanitize options
	 *
	 * @param array $input Raw input values.
	 * @return array Sanitized values.
	 */
	public function sanitize_options( $input ) {
		$sanitized = array();

		// Sanitize text fields
		$sanitized['account_id'] = isset( $input['account_id'] ) ? sanitize_text_field( $input['account_id'] ) : '';
		$sanitized['service_id'] = isset( $input['service_id'] ) ? sanitize_text_field( $input['service_id'] ) : '';

		// Sanitize HTML IDs (preserve A-Z, a-z, 0-9, underscore, hyphen)
		$sanitized['search_input_id']  = isset( $input['search_input_id'] ) ? $this->sanitize_html_id( $input['search_input_id'] ) : '';
		$sanitized['search_button_id'] = isset( $input['search_button_id'] ) ? $this->sanitize_html_id( $input['search_button_id'] ) : '';

		// Sanitize checkbox
		$sanitized['enable_on_front'] = ! empty( $input['enable_on_front'] ) ? 1 : 0;

		return $sanitized;
	}

	/**
	 * Sanitize HTML ID value
	 *
	 * @param string $value Raw value.
	 * @return string Sanitized value.
	 */
	private function sanitize_html_id( $value ) {
		$value = is_string( $value ) ? trim( $value ) : '';
		return preg_replace( '/[^A-Za-z0-9_-]/', '', $value );
	}

	/**
	 * Render settings page
	 */
	public function render_settings_page() {
		if ( ! current_user_can( 'manage_options' ) ) {
			return;
		}

		// Check if settings were saved
		// phpcs:ignore WordPress.Security.NonceVerification.Recommended -- Checking WordPress-set flag, Settings API handles nonces via settings_fields()
		if ( isset( $_GET['settings-updated'] ) ) {
			add_settings_error(
				'airgentic_messages',
				'airgentic_message',
				__( 'Settings saved successfully.', 'airgentic' ),
				'updated'
			);
		}

		settings_errors( 'airgentic_messages' );
		?>
		<div class="wrap">
			<h1><?php echo esc_html( get_admin_page_title() ); ?></h1>
			
			<form action="options.php" method="post">
				<?php
				settings_fields( self::OPTION_GROUP );
				do_settings_sections( self::OPTION_GROUP );
				submit_button( __( 'Save Settings', 'airgentic' ) );
				?>
			</form>

			<hr style="margin: 30px 0;">

			<h2><?php esc_html_e( 'How to Use', 'airgentic' ); ?></h2>
			
			<h3><?php esc_html_e( 'Automatic Integration', 'airgentic' ); ?></h3>
			<p><?php esc_html_e( 'Once you enable the plugin and enter your Account ID and Service ID, the Airgentic widget will automatically appear on all pages of your website.', 'airgentic' ); ?></p>

			<h3><?php esc_html_e( 'Service Hub Shortcode', 'airgentic' ); ?></h3>
			<p><?php esc_html_e( 'To display the Service Hub container on a specific page or post, use the following shortcode:', 'airgentic' ); ?></p>
			<p><code>[airgentic_service_hub]</code></p>

			<h3><?php esc_html_e( 'Getting Your Credentials', 'airgentic' ); ?></h3>
			<p>
				<?php
				printf(
					/* translators: %s: Airgentic website URL */
					esc_html__( 'You can find your Account ID and Service ID in your %s.', 'airgentic' ),
					'<a href="https://www.airgentic.com" target="_blank" rel="noopener noreferrer">' . esc_html__( 'Airgentic admin dashboard', 'airgentic' ) . '</a>'
				);
				?>
			</p>

			<hr style="margin: 30px 0;">

			<h2><?php esc_html_e( 'Need Help?', 'airgentic' ); ?></h2>
			<p>
				<?php
				printf(
					/* translators: %s: Airgentic support URL */
					esc_html__( 'Visit %s for documentation and support.', 'airgentic' ),
					'<a href="https://www.airgentic.com" target="_blank" rel="noopener noreferrer">www.airgentic.com</a>'
				);
				?>
			</p>
		</div>
		<?php
	}
}

// Initialize admin settings
new Airgentic_Admin_Settings();

