<?php
/**
 * AINP: AI Native Publisher Uninstall Script
 *
 * This script runs only when the plugin is explicitly uninstalled (deleted)
 * via the WordPress Plugins page. It cleans up all plugin-specific data
 * from the database, including options, transients, scheduled cron events,
 * and user preferences.
 *
 * @package           AINP_AI_Native_Publisher
 * @author            AI News Publisher
 * @copyright         2025, AI News Publisher
 * @license           GPL-2.0+
 */

// Exit if accessed directly or not during uninstall process.
if ( ! defined( 'WP_UNINSTALL_PLUGIN' ) ) {
	exit;
}

// --- 1. Clear Scheduled Cron Jobs ---
// It is critical to clear these to prevent WP from trying to run missing code.
$ainp_cron_hooks = array(
	'ainp_auto_fetch_cron_hook',
	'ainp_generate_image_hook',
	'ainp_process_batch_hook',
);

foreach ( $ainp_cron_hooks as $ainp_hook ) {
	wp_clear_scheduled_hook( $ainp_hook );
}

// --- 2. Delete Plugin Options ---
// Array containing all option names used by the plugin in the wp_options table.
$ainp_options_to_delete = array(
	'ainp_settings',              // Main settings array
	'ainp_logs',                  // Log entries
	'ainp_processing_status',     // Current status of fetching
	'ainp_cancel_flag',           // Cancel signal flag
	'ainp_domain_timestamps',     // Cooldown tracking for domains
	'ainp_pending_image_tasks',   // Queue counter for images
	'ainp_execution_start_time',  // Timeout guard timestamp
);

foreach ( $ainp_options_to_delete as $ainp_option ) {
	delete_option( $ainp_option );
}

// --- 3. Delete Transients ---
// Transients used for caching feeds, queues, and dashboard stats.
$ainp_transients_to_delete = array(
	'ainp_feed_queue_transient',
	'ainp_manual_context',
	'ainp_fallback_cron_check',
	'ainp_total_plugin_posts_count',
	'ainp_total_ia_generated_images_count',
	'ainp_total_active_sources_count',
);

foreach ( $ainp_transients_to_delete as $ainp_transient ) {
	delete_transient( $ainp_transient );
}

// Clear any feed-specific cache transients (prefix-based cleanup)
global $wpdb;
// phpcs:ignore WordPress.DB.DirectDatabaseQuery.DirectQuery, WordPress.DB.DirectDatabaseQuery.NoCaching
$wpdb->query( "DELETE FROM {$wpdb->options} WHERE option_name LIKE '_transient_ainp_feed_%' OR option_name LIKE '_transient_timeout_ainp_feed_%'" );


// --- 4. Clear User Meta (Preferences) ---
// Remove the 'Dismiss Welcome Panel' preference for all users.
// This ensures if they reinstall, they see the welcome panel again.
$ainp_users = get_users( array( 'fields' => 'ID' ) );
foreach ( $ainp_users as $ainp_user_id ) {
	delete_user_meta( $ainp_user_id, 'ainp_hide_welcome_panel' );
}

// --- Optional: Remove Post Meta Data ---
// This section is commented out by default as deleting post meta is destructive
// and usually not desired upon simple uninstallation (posts might be kept).
// Uncomment *only* if you want to strictly remove ALL data attached to posts.
/*
// Define the meta keys added by the plugin to posts.
$ainp_meta_keys_to_delete = array(
	'_ainp_guid',               // Original item GUID
	'_ainp_original_link',      // Link to the original article
	'_ainp_source_url',         // URL of the feed/scraper source
	'_ainp_source_name',        // Name of the feed/scraper source
	'_ainp_image_source',       // Source of the featured image ('feed', 'ia_generated', etc.)
	'_ainp_feed_item_xml',      // Original XML string of the item (temporary)
	'_ainp_final_post_status',  // Intended final status (temporary)
);

// Format keys for SQL IN clause.
$placeholders = implode( ',', array_fill( 0, count( $ainp_meta_keys_to_delete ), '%s' ) );

// phpcs:ignore WordPress.DB.DirectDatabaseQuery.DirectQuery
$wpdb->query( $wpdb->prepare(
	"DELETE FROM {$wpdb->postmeta} WHERE meta_key IN ($placeholders)",
	$ainp_meta_keys_to_delete
) );
*/