/**
 * Admin UI Helper Functions
 *
 * Provides utility functions for managing common UI elements and interactions
 * within the AINP: AI Native Publisher admin pages, such as confirmation modals,
 * button loading states, status messages, and clipboard operations.
 * Also handles dynamic interactions for the Sources settings tab.
 *
 * @package           AINP_AI_Native_Publisher
 * @author            AI News Publisher
 * @copyright         2025, AI News Publisher
 * @license           GPL-2.0+
 */

/* global jQuery, ainp_ajax */

/**
 * Displays a confirmation modal dialog.
 *
 * @param {string}    message   The confirmation message to display.
 * @param {function} callback   Function to call with the result (boolean).
 */
function showConfirmationModal( message, callback ) {
	// Remove any existing modal first.
	jQuery( '#ainp-confirmation-modal' ).remove();

	const title = ( typeof ainp_ajax !== 'undefined' && ainp_ajax.i18n ) ? ainp_ajax.i18n.processing : 'Confirmation';
	const btnYes = ( typeof ainp_ajax !== 'undefined' && ainp_ajax.i18n ) ? ainp_ajax.i18n.confirm_yes : 'Yes';
	const btnNo = ( typeof ainp_ajax !== 'undefined' && ainp_ajax.i18n ) ? ainp_ajax.i18n.confirm_no : 'No';

	const modalHTML = `
        <div id="ainp-confirmation-modal" style="display:none; position:fixed; top:0; left:0; width:100%; height:100%; background:rgba(0,0,0,0.6); z-index:10001; justify-content:center; align-items:center; backdrop-filter: blur(3px);">
            <div style="background:#fff; padding:25px 35px; border-radius:8px; box-shadow:0 4px 15px rgba(0,0,0,0.2); max-width:450px; width:90%; text-align:center;">
                <h3 style="margin-top:0; color:#333; font-size:18px;">${title}</h3>
                <p style="font-size:15px; color:#555; line-height:1.5;">${message}</p>
                <div style="margin-top:25px; display:flex; justify-content:center; gap:15px;">
                    <button id="ainp-modal-cancel" class="button button-secondary button-large">${btnNo}</button>
                    <button id="ainp-modal-confirm" class="button button-primary button-large">${btnYes}</button>
                </div>
            </div>
        </div>
    `;

	jQuery( 'body' ).append( modalHTML );
	const $modal = jQuery( '#ainp-confirmation-modal' );

	$modal.css( 'display', 'flex' ).hide().fadeIn( 200 );

	jQuery( '#ainp-modal-cancel' ).on( 'click', function () {
		$modal.fadeOut( 200, function () {
			$modal.remove();
		} );
		if ( typeof callback === 'function' ) callback( false );
	} );

	jQuery( '#ainp-modal-confirm' ).on( 'click', function () {
		$modal.fadeOut( 200, function () {
			$modal.remove();
		} );
		if ( typeof callback === 'function' ) callback( true );
	} );
}

/**
 * Toggles the loading state of a button.
 *
 * @param {jQuery}  $btn       The jQuery object of the button.
 * @param {boolean} isLoading True to set loading state, false to reset.
 */
function setButtonLoadingState( $btn, isLoading ) {
	if ( ! $btn || ! $btn.length ) return;

	if ( isLoading ) {
		$btn.prop( 'disabled', true ).addClass('is-busy');
		$btn.find( '.spinner' ).addClass( 'is-active' );
	} else {
		$btn.prop( 'disabled', false ).removeClass('is-busy');
		$btn.find( '.spinner' ).removeClass( 'is-active' );
	}
}

/**
 * Displays a status message in a specific container.
 *
 * @param {string} type      Message type ('success', 'error', 'warning', 'info').
 * @param {string} message   The message text to display.
 * @param {string} targetId  The ID of the container element.
 */
function showStatusMessage( type, message, targetId ) {
	const $target = jQuery( '#' + targetId );
	if ( ! $target.length ) return;

	const noticeClass = ( type === 'error' ) ? 'notice-error' :
		( type === 'success' ) ? 'notice-success' :
		( type === 'warning' ) ? 'notice-warning' : 'notice-info';

	const dismissText = ( typeof ainp_ajax !== 'undefined' && ainp_ajax.i18n ) ? ainp_ajax.i18n.dismiss_notice : 'Dismiss';

	const html = `
        <div class="notice ${noticeClass} is-dismissible" style="margin: 10px 0 20px 0; animation: fadeIn 0.3s;">
            <p>${message}</p>
            <button type="button" class="notice-dismiss" onclick="jQuery(this).parent().remove();">
                <span class="screen-reader-text">${dismissText}</span>
            </button>
        </div>
    `;

	$target.html( html );

	if ( type === 'success' ) {
		setTimeout( function () {
			$target.find( '.notice' ).fadeOut( 500, function () {
				jQuery( this ).remove();
			} );
		}, 5000 );
	}
}

/**
 * Updates the enabled/disabled state of global control buttons.
 *
 * @param {boolean} isProcessing True if a process is active, false otherwise.
 */
function updateControlsState( isProcessing ) {
	const $startBtn = jQuery( '#ainp-dashboard-fetch-all' ); 
	const $fetchNextBtn = jQuery( '#ainp-dashboard-fetch-next' );
	const $cancelBtns = jQuery( '.ainp-cancel-button, #ainp-cancel-search' );

	if ( isProcessing ) {
		$startBtn.prop( 'disabled', true ).addClass( 'disabled' );
		$fetchNextBtn.prop( 'disabled', true ).addClass( 'disabled' );
		$cancelBtns.prop( 'disabled', false ).removeClass( 'disabled' ).show();
	} else {
		$startBtn.prop( 'disabled', false ).removeClass( 'disabled' );
		$fetchNextBtn.prop( 'disabled', false ).removeClass( 'disabled' );
		$cancelBtns.hide();
	}
}

/**
 * Copies text content to the system clipboard.
 *
 * @param {string} textToCopy The text to copy.
 */
function copyLogToClipboard( textToCopy ) {
	if ( ! textToCopy ) return;

	const successMsg = ( typeof ainp_ajax !== 'undefined' && ainp_ajax.i18n ) ? ainp_ajax.i18n.success_generic : 'Success';
	const errorMsg = ( typeof ainp_ajax !== 'undefined' && ainp_ajax.i18n ) ? ainp_ajax.i18n.error_generic : 'Error';

	if ( navigator.clipboard && window.isSecureContext ) {
		navigator.clipboard.writeText( textToCopy ).then( function () {
			showStatusMessage( 'success', successMsg, 'ainp-status-logs-action-status' );
		}, function ( err ) {
			console.error( 'CopyLog Error:', err );
			showStatusMessage( 'error', errorMsg, 'ainp-status-logs-action-status' );
		} );
	} else {
		const textArea = document.createElement( 'textarea' );
		textArea.value = textToCopy;
		textArea.style.position = 'fixed';
		textArea.style.left = '-9999px';
		document.body.appendChild( textArea );
		textArea.select();
		try {
			document.execCommand( 'copy' );
			showStatusMessage( 'success', successMsg, 'ainp-status-logs-action-status' );
		} catch ( err ) {
			console.error( 'CopyLog Fallback Error:', err );
			showStatusMessage( 'error', errorMsg, 'ainp-status-logs-action-status' );
		} finally {
			document.body.removeChild( textArea );
		}
	}
}

/**
 * Sources Tab Logic - Dynamic Row Management
 * Handles adding and removing source rows with NO LIMITS.
 */
jQuery( document ).ready( function( $ ) {
	
	// Use event delegation for the Add Source button to ensures it works even if DOM is slow
	$( document ).on( 'click', '#ainp-add-source-btn', function() {
		const $btn = $( this );
		
		// Retrieve data attributes (using .attr to ensure we get fresh string values)
		let rowIndex = parseInt( $btn.attr( 'data-next-index' ) ) || 0;
		
		// Template for the new row
		const newRowHTML = `
			<tr class="source-row" style="display:none;">
				<td style="padding: 12px 15px; vertical-align: top;">
					<label for="source-url-${rowIndex}" class="screen-reader-text">Source URL</label>
					<input type="url" id="source-url-${rowIndex}" name="sources[${rowIndex}][url]" class="large-text" required placeholder="e.g., https://example.com/feed.xml" style="width: 100%; max-width: 450px;">
				</td>
				<td style="padding: 12px 15px; vertical-align: top;">
					<label for="source-type-${rowIndex}" class="screen-reader-text">Source Type</label>
					<select id="source-type-${rowIndex}" name="sources[${rowIndex}][type]" class="source-type-select" style="width: 100%; max-width: 450px;">
						<option value="rss" selected>RSS/Atom Feed</option>
					</select>
				</td>
				<td style="padding: 12px 15px; vertical-align: top;">
					<button type="button" class="button button-secondary remove-source" style="margin-top: 5px;">Remove</button>
				</td>
			</tr>
		`;

		$( '#ainp-sources-tbody' ).append( newRowHTML );
		$( '#ainp-sources-tbody' ).find( '.source-row' ).last().fadeIn( 300 );
		
		// Increment the index for the next add
		rowIndex++;
		$btn.attr( 'data-next-index', rowIndex );
	} );

	// Event Delegation for "Remove Source" button
	$( document ).on( 'click', '.remove-source', function() {
		const $row = $( this ).closest( 'tr.source-row' );
		
		$row.fadeOut( 300, function() {
			$( this ).remove();
		} );
	} );

} );