/**
 * Admin Event Listeners
 *
 * Attaches jQuery event listeners (click, change) to buttons, toggles, and
 * other interactive elements on the AINP: AI Native Publisher admin pages.
 * Delegates actions to functions defined in admin-ajax.js and admin-ui.js.
 * Initializes status and log polling on page load.
 *
 * @package           AINP_AI_Native_Publisher
 * @author            AI News Publisher
 * @copyright         2025, AI News Publisher
 * @license           GPL-2.0+
 */

jQuery( document ).ready( function ( $ ) {
	
	// --- Initialize Global State Variables ---
	// Ensures clean state on page load/reload.
	window.ainpStatusInterval = null; 
	window.ainpLogsInterval = null;   
	window.ainpCurrentAjaxRequest = null; 
	window.isAinpProcessing = false;  

	// --- Manual Fetch Actions (Dashboard) ---

	// 1. Fetch All Sources
	$( '#ainp-dashboard-fetch-all' ).on( 'click', function ( e ) {
		e.preventDefault();
		
		// Capture search depth settings safely
		// Default to '1' if the element is missing (Free version restriction)
		const depthChoice = $( 'input[name="ainp_search_depth_choice"]:checked' ).val() || '1';
		const depthCustom = 5; // Default fallback, ignored by backend when choice is '1'

		// handleAjaxAction is defined in admin-ajax.js
		if ( typeof handleAjaxAction === 'function' ) {
			handleAjaxAction(
				$( this ),
				'ainp_start_batch_search',
				'ainp-dashboard-fetch-status',
				{
					search_depth_choice: depthChoice,
					search_depth_custom: depthCustom
				}
			);
		}
	} );

	// 2. Fetch Next Single Item
	$( '#ainp-dashboard-fetch-next' ).on( 'click', function ( e ) {
		e.preventDefault();
		if ( typeof handleAjaxAction === 'function' ) {
			handleAjaxAction(
				$( this ),
				'ainp_fetch_next_news_item',
				'ainp-dashboard-fetch-status'
			);
		}
	} );

	// 3. Cancel Fetch
	// Handles button on Dashboard and Status Logs tab
	$( '.ainp-cancel-button, #ainp-cancel-search' ).on( 'click', function ( e ) {
		e.preventDefault();
		if ( typeof handleAjaxAction === 'function' ) {
			handleAjaxAction(
				$( this ),
				'ainp_cancel_search',
				'ainp-dashboard-fetch-status', // Might target different divs depending on page, but generic feedback is ok
				{},
				true, // Maintenance Action
				ainp_ajax.i18n.cancel_confirm
			);
		}
	} );


	// --- Maintenance Tools (Status & Logs Tab) ---

	$( '#ainp-reset-jobs' ).on( 'click', function ( e ) {
		e.preventDefault();
		if ( typeof handleAjaxAction === 'function' ) {
			handleAjaxAction(
				$( this ),
				'ainp_reset_jobs',
				'ainp-maintenance-status',
				{},
				true, // Maintenance
				ainp_ajax.i18n.confirm_reset_jobs
			);
		}
	} );

	$( '#ainp-clear-logs' ).on( 'click', function ( e ) {
		e.preventDefault();
		if ( typeof handleAjaxAction === 'function' ) {
			handleAjaxAction(
				$( this ),
				'ainp_clear_logs',
				'ainp-maintenance-status',
				{},
				true, // Maintenance
				ainp_ajax.i18n.confirm_clear_logs
			);
		}
	} );

	$( '#ainp-reset-status' ).on( 'click', function ( e ) {
		e.preventDefault();
		if ( typeof handleAjaxAction === 'function' ) {
			handleAjaxAction(
				$( this ),
				'ainp_reset_processing_status',
				'ainp-maintenance-status',
				{},
				true, // Maintenance
				ainp_ajax.i18n.confirm_reset_status
			);
		}
	} );


	// --- Diagnostics & Testing ---

	$( '#ainp-test-api-button' ).on( 'click', function ( e ) {
		e.preventDefault();
		const $btn = $( this );
		const $statusDiv = $( '#ainp-api-test-status' );
		
		// setButtonLoadingState is defined in admin-ui.js
		if ( typeof setButtonLoadingState === 'function' ) {
			setButtonLoadingState( $btn, true );
		}
		$statusDiv.html( '<p class="description">' + ainp_ajax.i18n.testing_apis + '</p>' );

		$.post( ainp_ajax.ajax_url, {
			action: 'ainp_test_api_connection',
			_ajax_nonce: ainp_ajax.nonce
		}, function ( response ) {
			if ( typeof setButtonLoadingState === 'function' ) {
				setButtonLoadingState( $btn, false );
			}
			if ( response.success ) {
				let html = '<ul style="margin-top:10px;">';
				$.each( response.data.results, function ( key, result ) {
					const color = ( result.status === 'success' ) ? '#46b450' : '#d63638';
					const icon  = ( result.status === 'success' ) ? 'dashicons-yes' : 'dashicons-no';
					// Sanitization Note: result.provider is hardcoded PHP side, result.message is sanitized PHP side.
					html += `<li style="color:${color};"><span class="dashicons ${icon}"></span> <strong>${result.provider}:</strong> ${result.message}</li>`;
				} );
				html += '</ul>';
				$statusDiv.html( html );
			} else {
				if ( typeof showStatusMessage === 'function' ) {
					showStatusMessage( 'error', ainp_ajax.i18n.error_generic, 'ainp-api-test-status' );
				}
			}
		} ).fail( function () {
			if ( typeof setButtonLoadingState === 'function' ) {
				setButtonLoadingState( $btn, false );
			}
			if ( typeof showStatusMessage === 'function' ) {
				showStatusMessage( 'error', ainp_ajax.i18n.error_ajax, 'ainp-api-test-status' );
			}
		} );
	} );

	$( '#ainp-test-write-permission-button' ).on( 'click', function ( e ) {
		e.preventDefault();
		if ( typeof handleAjaxAction === 'function' ) {
			handleAjaxAction(
				$( this ),
				'ainp_test_write_permission',
				'ainp-write-permission-test-status'
			);
		}
	} );


	// --- Clipboard Copy ---
	$( '#ainp-copy-log' ).on( 'click', function ( e ) {
		e.preventDefault();
		// Extract text from the log area
		const logText = $( '#ainp-log-area' ).text();
		// copyLogToClipboard is defined in admin-ui.js
		if ( typeof copyLogToClipboard === 'function' ) {
			copyLogToClipboard( logText );
		}
	} );


	// --- Quick Settings Toggles (Dashboard) ---
	
	/**
	 * Generic handler for toggle switches.
	 * @param {HTMLElement} toggleElement The checkbox input.
	 * @param {string} actionName The AJAX action to trigger.
	 */
	const handleToggleClick = function( toggleElement, actionName ) {
		const $toggle = $( toggleElement );
		const isEnabled = $toggle.is( ':checked' );
		
		// Temporarily disable to prevent spamming
		$toggle.prop( 'disabled', true );

		$.post( ainp_ajax.ajax_url, {
			action: actionName,
			is_enabled: isEnabled,
			_ajax_nonce: ainp_ajax.nonce
		}, function( response ) {
			$toggle.prop( 'disabled', false );
			if ( response.success ) {
				if ( typeof showStatusMessage === 'function' ) {
					showStatusMessage( 'success', response.data.message, 'ainp-quick-settings-status' );
				}
			} else {
				// Revert toggle on error
				$toggle.prop( 'checked', !isEnabled );
				if ( typeof showStatusMessage === 'function' ) {
					showStatusMessage( 'error', response.data.message || ainp_ajax.i18n.error_generic, 'ainp-quick-settings-status' );
				}
			}
		}).fail( function() {
			$toggle.prop( 'disabled', false );
			$toggle.prop( 'checked', !isEnabled );
			if ( typeof showStatusMessage === 'function' ) {
				showStatusMessage( 'error', ainp_ajax.i18n.error_ajax, 'ainp-quick-settings-status' );
			}
		});
	};

	// Only one toggle remains active in the Free version dashboard
	$( '#ainp-toggle-image-fallback' ).on( 'change', function () {
		handleToggleClick( this, 'ainp_toggle_image_fallback' );
	} );


	// --- Welcome Panel Dismiss ---
	$( '#ainp-dismiss-welcome-panel' ).on( 'click', function ( e ) {
		e.preventDefault();
		const $panel = $( this ).closest( '.ainp-welcome-panel' );

		// Animation: Fade out and slide up
		$panel.fadeTo( 100, 0, function () {
			$panel.slideUp( 100, function () {
				$panel.remove();
			} );
		} );

		// Send AJAX request to save preference
		$.post( ainp_ajax.ajax_url, {
			action: 'ainp_dismiss_welcome_panel',
			_ajax_nonce: ainp_ajax.nonce
		} );
	} );


	// --- Initial Data Polling ---
	// If the status area exists (Dashboard/Logs tab), fetch initial status.
	if ( $( '#ainp-status-area' ).length || $( '.processing-status' ).length || $( '#ainp-dashboard-fetch-status' ).length ) {
		// Fetch status immediately (fetchStatus is in admin-ajax.js)
		if ( typeof fetchStatus === 'function' ) {
			fetchStatus();
			// Set interval for regular updates
			window.ainpStatusInterval = setInterval( fetchStatus, 4000 );
		}
	}
	
	// If the log area exists, fetch logs (fetchLogs is in admin-ajax.js)
	if ( $( '#ainp-log-area' ).length ) {
		if ( typeof fetchLogs === 'function' ) {
			fetchLogs();
			window.ainpLogsInterval = setInterval( fetchLogs, 5000 );
		}
	}

} );