/**
 * Admin AJAX Handler
 *
 * Manages AJAX requests for plugin actions like fetching, cancelling,
 * maintenance, and polling for status/logs. Interacts with admin-ui.js
 * for UI updates (button states, modals, messages).
 *
 * @package           AINP_AI_Native_Publisher
 * @author            AI News Publisher
 * @copyright         2025, AI News Publisher
 * @license           GPL-2.0+
 */

/**
 * Handles a generic AJAX action triggered by a button click.
 *
 * Includes confirmation prompts, button loading states, AJAX call execution,
 * and status message display. Also manages polling intervals for search actions.
 *
 * @param {jQuery} $clickedButton       jQuery object of the button that was clicked.
 * @param {string} ajaxAction           WordPress AJAX action name (e.g., 'ainp_start_batch_search').
 * @param {string} statusDivId          ID of the HTML element where status messages should be displayed.
 * @param {Object} [additionalData={}] Optional. Extra data to send with the AJAX request.
 * @param {boolean} [isMaintenanceAction=false] Optional. True if using the maintenance nonce. Default false.
 * @param {string|null} [confirmationMessage=null] Optional. If provided, displays a confirmation modal before proceeding.
 */
function handleAjaxAction( $clickedButton, ajaxAction, statusDivId, additionalData = {}, isMaintenanceAction = false, confirmationMessage = null ) {

	// Ensure localization object exists
	if ( typeof ainp_ajax === 'undefined' ) {
		console.error( 'AINP: Localization object (ainp_ajax) not found.' );
		return;
	}

	/**
	 * Executes the core AJAX call after checks.
	 */
	const executeAjax = function () {
		// setButtonLoadingState defined in admin-ui.js
		if ( typeof setButtonLoadingState === 'function' ) {
			setButtonLoadingState( $clickedButton, true );
		}

		// Prepare data object
		const data = {
			action: ajaxAction,
			...additionalData
		};

		// Add appropriate nonce
		if ( isMaintenanceAction ) {
			data._ajax_nonce = ainp_ajax.maintenance_nonce;
		} else {
			data._ajax_nonce = ainp_ajax.nonce;
		}

		jQuery.ajax( {
			url: ainp_ajax.ajax_url,
			type: 'POST',
			data: data,
			dataType: 'json',
			success: function ( response ) {
				if ( typeof setButtonLoadingState === 'function' ) {
					setButtonLoadingState( $clickedButton, false );
				}

				if ( response.success ) {
					if ( typeof showStatusMessage === 'function' ) {
						showStatusMessage( 'success', response.data.message || ainp_ajax.i18n.success_generic, statusDivId );
					}

					// Special handling for search start: Start polling
					if ( ajaxAction === 'ainp_start_batch_search' ) {
						startStatusPolling();
						startLogsPolling();
					}
					// Special handling for cancellation or reset: Stop polling immediately
					else if ( ajaxAction === 'ainp_cancel_search' || ajaxAction === 'ainp_reset_jobs' || ajaxAction === 'ainp_reset_processing_status' ) {
						// Force a status update to 'Idle' immediately in UI
						jQuery( '.processing-status' ).text( ainp_ajax.i18n.status_idle );
						jQuery( '.processing-progress' ).text( '--' );
						jQuery( '.processing-progress-bar' ).css( 'width', '0%' );
						if ( typeof updateControlsState === 'function' ) {
							updateControlsState( false ); // Re-enable start buttons
						}
					}
					// Special handling for log clear
					else if ( ajaxAction === 'ainp_clear_logs' ) {
						jQuery( '#ainp-log-area' ).html( '<p>' + ainp_ajax.i18n.status_idle + '</p>' );
					}

				} else {
					if ( typeof showStatusMessage === 'function' ) {
						showStatusMessage( 'error', response.data.message || ainp_ajax.i18n.error_generic, statusDivId );
					}
				}
			},
			error: function ( jqXHR, textStatus, errorThrown ) {
				if ( typeof setButtonLoadingState === 'function' ) {
					setButtonLoadingState( $clickedButton, false );
				}
				console.error( 'AJAX Error:', textStatus, errorThrown );
				if ( typeof showStatusMessage === 'function' ) {
					showStatusMessage( 'error', ainp_ajax.i18n.error_ajax, statusDivId );
				}
			}
		} );
	};

	// Logic flow: Confirmation -> Execution
	if ( confirmationMessage ) {
		// showConfirmationModal defined in admin-ui.js
		if ( typeof showConfirmationModal === 'function' ) {
			showConfirmationModal( confirmationMessage, function ( confirmed ) {
				if ( confirmed ) {
					executeAjax();
				}
			} );
		} else {
			// Fallback if UI script missing
			if ( confirm( confirmationMessage ) ) {
				executeAjax();
			}
		}
	} else {
		executeAjax();
	}
}

// Global intervals to manage polling (so we can stop them if needed)
window.ainpStatusInterval = null;
window.ainpLogsInterval = null;

/**
 * Starts the interval to poll status and logs.
 * Clears existing intervals first to prevent duplicates.
 */
function startStatusPolling() {
	if ( window.ainpStatusInterval ) clearInterval( window.ainpStatusInterval );
	fetchStatus(); // Run immediately
	window.ainpStatusInterval = setInterval( fetchStatus, 4000 ); // Poll every 4 seconds
}

function startLogsPolling() {
	if ( window.ainpLogsInterval ) clearInterval( window.ainpLogsInterval );
	fetchLogs(); // Run immediately
	window.ainpLogsInterval = setInterval( fetchLogs, 5000 ); // Poll logs every 5 seconds
}

/**
 * Fetches the current processing status from the backend via AJAX.
 * Updates UI elements (.processing-status, .processing-progress, progress bar).
 * Handles stopping the poll when status is 'idle' or 'completed'.
 */
function fetchStatus() {
	if ( typeof ainp_ajax === 'undefined' ) return;

	jQuery.ajax( {
		url: ainp_ajax.ajax_url,
		type: 'POST',
		data: { action: 'ainp_get_status', _ajax_nonce: ainp_ajax.nonce }, // Use standard nonce
		dataType: 'json',
		success: function ( response ) {
			if ( response && response.success && response.data && response.data.status ) {
				const statusData = response.data.status;

				// Update UI Elements
				jQuery( '.processing-status' ).text( statusData.status_display );
				jQuery( '.processing-message' ).html( statusData.message ); // Use html() for links/formatting

				if ( statusData.total > 0 ) {
					const progressText = statusData.current + ' / ' + statusData.total + ' (' + statusData.progress + '%)';
					jQuery( '.processing-progress' ).text( progressText );
					jQuery( '.processing-progress-bar' ).css( 'width', statusData.progress + '%' );
				} else {
					jQuery( '.processing-progress' ).text( '--' );
					jQuery( '.processing-progress-bar' ).css( 'width', '0%' );
				}

				// Determine if processing is active
				const isProcessing = ( statusData.status === 'processing' || statusData.status === 'processing_images' );
				
				if ( typeof updateControlsState === 'function' ) {
					updateControlsState( isProcessing );
				}

				// Stop polling if finished
				if ( statusData.status === 'idle' || statusData.status === 'completed' ) {
					if ( window.ainpStatusInterval ) {
						clearInterval( window.ainpStatusInterval );
						window.ainpStatusInterval = null;
					}
				}
			}
		},
		error: function ( jqXHR, textStatus, errorThrown ) {
			// Handle AJAX transport errors during status polling. Silent fail or console log only.
			console.error( 'Status Poll AJAX Error:', textStatus, errorThrown );
		}
	} );
} // --- End of fetchStatus ---

/**
 * Fetches the latest log entries HTML from the backend via AJAX.
 * Updates the content of the log display area.
 */
function fetchLogs() {
	if ( typeof ainp_ajax === 'undefined' ) return;

	// Simple check: only fetch if the log area exists on the current page.
	const $logArea = jQuery( '#ainp-log-area' );
	if ( ! $logArea.length ) {
		// If log area doesn't exist, stop polling for logs.
		if ( window.ainpLogsInterval ) {
			clearInterval( window.ainpLogsInterval );
			window.ainpLogsInterval = null;
		}
		return;
	}

	jQuery.ajax( {
		url: ainp_ajax.ajax_url,
		type: 'POST',
		data: { action: 'ainp_get_logs', _ajax_nonce: ainp_ajax.nonce }, // Use standard nonce
		dataType: 'json',
		success: function ( response ) {
			if ( response && response.success && response.data && typeof response.data.logs !== 'undefined' ) {
				// Update the log area content with the received HTML.
				$logArea.html( response.data.logs );
			}
		},
		error: function ( jqXHR, textStatus, errorThrown ) {
			console.error( 'Logs Poll AJAX Error:', textStatus, errorThrown );
		}
	} );
}