<?php
/**
 * Status Logger Class
 *
 * This class handles logging of plugin activities and manages the real-time
 * processing status displayed in the admin dashboard.
 *
 * @package           AINP_AI_Native_Publisher
 * @author            AI News Publisher
 * @copyright         2025, AI News Publisher
 * @license           GPL-2.0+
 */

// Exit if accessed directly.
if ( ! defined( 'ABSPATH' ) ) {
	exit;
}

/**
 * Handles all logging and status update functionalities.
 */
class AINP_Status_Logger {

	/**
	 * The option key for storing log entries.
	 * @var string
	 */
	const LOG_OPTION_KEY = 'ainp_logs';

	/**
	 * The option key for storing the current processing status.
	 * @var string
	 */
	const STATUS_OPTION_KEY = 'ainp_processing_status';

	/**
	 * The maximum number of log entries to store.
	 * @var int
	 */
	const MAX_LOG_ENTRIES = 250;

	/**
	 * The option key for the cancellation flag.
	 * @var string
	 */
	const CANCEL_FLAG_OPTION = 'ainp_cancel_flag';

	/**
	 * Adds a new entry to the activity log.
	 * Determines context (manual user action or system/cron) and saves the log entry.
	 *
	 * @param string $type     Log entry type (info, success, warning, error).
	 * @param string $message  Main log message.
	 * @param mixed  $details  Optional. Additional details or WP_Error object.
	 * @param array  $metadata Optional. Metadata (e.g., post_id).
	 */
	public function add_log_entry( $type, $message, $details = '', $metadata = array() ) {
		// Fetch existing logs without autoloading if possible (though get_option fetches it anyway if not loaded)
		$logs = get_option( self::LOG_OPTION_KEY, array() );

		// Determine context: 'user' or 'cron'.
		$context = 'cron'; // Default
		if ( get_transient( 'ainp_manual_context' ) || ! ( defined( 'DOING_CRON' ) && DOING_CRON ) ) {
			$context = 'user';
		}

		$entry = array(
			'timestamp' => time(),
			'type'      => strtolower( sanitize_key( $type ) ),
			'message'   => $message, // Saved raw, sanitized on output
			'context'   => $context,
			'details'   => is_wp_error( $details ) ? 'WP_Error: ' . $details->get_error_message() : $details,
			'metadata'  => (array) $metadata,
		);

		// Prepend new entry
		array_unshift( $logs, $entry );
		// Trim log to max entries
		$logs = array_slice( $logs, 0, self::MAX_LOG_ENTRIES );

		// PERFORMANCE FIX: Explicitly set autoload to 'no'.
		// This prevents the log data from loading on every page view of the website.
		update_option( self::LOG_OPTION_KEY, $logs, 'no' );
	}

	/**
	 * Generates the HTML representation of the log entries for display.
	 *
	 * @return string HTML for the log list.
	 */
	public function get_log_entries_html() {
		$logs = get_option( self::LOG_OPTION_KEY, array() );

		if ( empty( $logs ) ) {
			return '<p>' . esc_html__( 'No log entries yet.', 'ainp-ai-native-publisher' ) . '</p>';
		}

		// Map types to translated labels and icons.
		$log_type_map = array(
			'info'    => array(
				'label' => __( 'Info', 'ainp-ai-native-publisher' ),
				'icon'  => 'dashicons-info-outline',
			),
			'success' => array(
				'label' => __( 'Success', 'ainp-ai-native-publisher' ),
				'icon'  => 'dashicons-yes-alt',
			),
			'warning' => array(
				'label' => __( 'Warning', 'ainp-ai-native-publisher' ),
				'icon'  => 'dashicons-warning',
			),
			'error'   => array(
				'label' => __( 'Error', 'ainp-ai-native-publisher' ),
				'icon'  => 'dashicons-dismiss',
			),
		);

		// Map contexts to translated labels and icons.
		$log_context_map = array(
			'user' => array(
				'label' => __( 'Manual', 'ainp-ai-native-publisher' ),
				'icon'  => 'dashicons-admin-users',
			),
			'cron' => array(
				'label' => __( 'System', 'ainp-ai-native-publisher' ),
				'icon'  => 'dashicons-admin-generic',
			),
		);

		$html = '<ul class="ainp-log-list">';
		foreach ( $logs as $entry ) {
			$type         = isset( $entry['type'] ) && array_key_exists( $entry['type'], $log_type_map ) ? $entry['type'] : 'info';
			$log_info     = $log_type_map[ $type ];
			$timestamp    = $entry['timestamp'] ?? time();
			$details      = $entry['details'] ?? '';
			$context      = isset( $entry['context'] ) && array_key_exists( $entry['context'], $log_context_map ) ? $entry['context'] : 'user';
			$context_info = $log_context_map[ $context ];
			$metadata     = $entry['metadata'] ?? array();
			$message      = $entry['message'] ?? '';

			// Link message to post if post_id exists.
			if ( ! empty( $metadata['post_id'] ) ) {
				$edit_link = get_edit_post_link( absint( $metadata['post_id'] ) );
				if ( $edit_link ) {
					$message = '<a href="' . esc_url( $edit_link ) . '" target="_blank">' . wp_kses_post( $message ) . '</a>';
				} else {
					$message = wp_kses_post( $message );
				}
			} else {
				$message = wp_kses_post( $message );
			}

			$html .= '<li class="log-entry log-' . esc_attr( $type ) . '">';
			$html .= '<div class="log-main-line">';
			// Timestamp
			$html .= '<span class="log-timestamp">[' . esc_html( wp_date( get_option( 'date_format' ) . ' ' . get_option( 'time_format' ), $timestamp ) ) . ']</span> ';
			// Context Marker
			$html .= '<span class="log-context-marker log-context-' . esc_attr( $context ) . '"><span class="dashicons ' . esc_attr( $context_info['icon'] ) . '"></span> ' . esc_html( $context_info['label'] ) . '</span> ';
			// Type Badge
			$html .= '<span class="log-type-badge log-type-' . esc_attr( $type ) . '"><span class="dashicons ' . esc_attr( $log_info['icon'] ) . '"></span> ' . esc_html( $log_info['label'] ) . '</span> ';
			// Message
			$html .= '<span class="log-message">' . $message . '</span>';
			$html .= '</div>'; // .log-main-line
			// Details
			if ( ! empty( $details ) ) {
				$html .= '<div class="log-details">' . esc_html( $details ) . '</div>';
			}
			$html .= '</li>'; // .log-entry
		}
		$html .= '</ul>'; // .ainp-log-list
		return $html;
	}

	/**
	 * Updates the processing status stored in WordPress options.
	 * Calculates progress and clears manual context transient when finished.
	 *
	 * @param string $status  Status ('idle', 'processing', 'processing_images', 'completed').
	 * @param int    $current Items processed so far.
	 * @param int    $total   Total items in the batch.
	 * @param string $message Descriptive status message.
	 */
	public function update_processing_status( $status, $current, $total, $message ) {
		$progress = 0;
		if ( $total > 0 ) {
			$progress = round( ( $current / $total ) * 100 );
		} elseif ( 'completed' === $status || ( 'idle' === $status && 0 === $current && 0 === $total ) ) {
			$progress = 100;
		}

		$status_data = array(
			'status'      => sanitize_key( $status ),
			'progress'    => absint( $progress ),
			'current'     => absint( $current ),
			'total'       => absint( $total ),
			'message'     => wp_kses_post( $message ),
			'last_update' => time(),
		);
		// Status is small, autoload is fine/preferred for dashboard widget checks
		update_option( self::STATUS_OPTION_KEY, $status_data );

		// Clear manual context transient when processing ends.
		if ( in_array( $status, array( 'completed', 'idle' ), true ) ) {
			delete_transient( 'ainp_manual_context' );
		}
	}

	/**
	 * Retrieves the current processing status from WordPress options.
	 *
	 * @return array Processing status data or defaults.
	 */
	public function get_processing_status() {
		return get_option(
			self::STATUS_OPTION_KEY,
			array(
				'status'      => 'idle',
				'progress'    => 0,
				'current'     => 0,
				'total'       => 0,
				'message'     => __( 'System is idle.', 'ainp-ai-native-publisher' ),
				'last_update' => time(),
			)
		);
	}

	/**
	 * Checks if a cancellation request flag is set.
	 *
	 * @return bool True if cancellation requested, false otherwise.
	 */
	public function is_cancellation_requested() {
		return (bool) get_option( self::CANCEL_FLAG_OPTION, false );
	}
}