<?php
/**
 * Sources & General Settings Tab View
 *
 * Renders the settings page for managing content sources (RSS feeds)
 * and configuring general plugin behavior like caching, timeouts, and default post settings.
 *
 * @package           AINP_AI_Native_Publisher
 * @author            AI News Publisher
 * @copyright         2025, AI News Publisher
 * @license           GPL-2.0+
 */

// Exit if accessed directly.
if ( ! defined( 'ABSPATH' ) ) {
	exit;
}

// Ensure the sources_config option exists and is an array to prevent errors.
$ainp_sources_config = $this->plugin->options['sources_config'] ?? array();
if ( ! is_array( $ainp_sources_config ) ) {
	$ainp_sources_config = array();
}

// Get the count for the JavaScript index.
$ainp_source_count = count( $ainp_sources_config );

// Get current target language
$ainp_current_target_language = $this->plugin->options['target_language'] ?? 'Portuguese (Brazil)';
$ainp_available_languages = array(
	'Portuguese (Brazil)',
	'Portuguese (Portugal)',
	'English',
	'Spanish',
	'French',
	'German',
	'Italian',
	'Dutch',
	'Russian',
	'Chinese (Simplified)',
	'Japanese',
	'Korean',
	'Hindi',
	'Arabic',
	'Turkish',
);
?>

<form method="post" action="">
	<?php wp_nonce_field( 'ainp_settings_nonce' ); // Security nonce ?>
	<input type="hidden" name="current_page_slug" value="ainp-sources">

	<h2><?php esc_html_e( 'News Sources', 'ainp-ai-native-publisher' ); ?></h2>
	<p><?php esc_html_e( 'Add and configure your content sources.', 'ainp-ai-native-publisher' ); ?></p>

	<table id="ainp-sources-table" class="wp-list-table widefat striped" style="width: 100%; border-collapse: collapse; margin-bottom: 20px;">
		<thead>
			<tr>
				<th scope="col" style="text-align: left; padding: 12px 15px; background-color: #f7f7f7; font-weight: 600;"><?php esc_html_e( 'Source URL', 'ainp-ai-native-publisher' ); ?></th>
				<th scope="col" style="text-align: left; padding: 12px 15px; background-color: #f7f7f7; font-weight: 600;"><?php esc_html_e( 'Source Type', 'ainp-ai-native-publisher' ); ?></th>
				<th scope="col" style="text-align: left; padding: 12px 15px; background-color: #f7f7f7; font-weight: 600;"><?php esc_html_e( 'Actions', 'ainp-ai-native-publisher' ); ?></th>
			</tr>
		</thead>
		<tbody id="ainp-sources-tbody">
			<?php foreach ( $ainp_sources_config as $ainp_index => $ainp_source ) : ?>
				<?php
				// Ensure array keys exist for safety
				$ainp_source_url = $ainp_source['url'] ?? '';
				$ainp_source_type = $ainp_source['type'] ?? 'rss';
				?>
				<tr class="source-row">
					<td style="padding: 12px 15px; vertical-align: top;">
						<label for="source-url-<?php echo esc_attr( $ainp_index ); ?>" class="screen-reader-text"><?php esc_html_e( 'Source URL', 'ainp-ai-native-publisher' ); ?></label>
						<input type="url" id="source-url-<?php echo esc_attr( $ainp_index ); ?>" name="sources[<?php echo esc_attr( $ainp_index ); ?>][url]" value="<?php echo esc_attr( $ainp_source_url ); ?>" class="large-text" required placeholder="<?php esc_attr_e( 'e.g., https://example.com/feed.xml', 'ainp-ai-native-publisher' ); ?>" style="width: 100%; max-width: 450px;">
					</td>
					<td style="padding: 12px 15px; vertical-align: top;">
						<label for="source-type-<?php echo esc_attr( $ainp_index ); ?>" class="screen-reader-text"><?php esc_html_e( 'Source Type', 'ainp-ai-native-publisher' ); ?></label>
						<select id="source-type-<?php echo esc_attr( $ainp_index ); ?>" name="sources[<?php echo esc_attr( $ainp_index ); ?>][type]" class="source-type-select" style="width: 100%; max-width: 450px;">
							<option value="rss" <?php selected( $ainp_source_type, 'rss' ); ?>><?php esc_html_e( 'RSS/Atom Feed', 'ainp-ai-native-publisher' ); ?></option>
						</select>
					</td>
					<td style="padding: 12px 15px; vertical-align: top;"><button type="button" class="button button-secondary remove-source" style="margin-top: 5px;"><?php esc_html_e( 'Remove', 'ainp-ai-native-publisher' ); ?></button></td>
				</tr>
			<?php endforeach; ?>
		</tbody>
	</table>

	<p>
		<button type="button" 
			id="ainp-add-source-btn" 
			class="button button-secondary"
			data-next-index="<?php echo esc_attr( $ainp_source_count ); ?>"
			data-i18n-add-btn-text="<?php esc_attr_e( 'Add New Source', 'ainp-ai-native-publisher' ); ?>">
			<span class="dashicons dashicons-plus"></span> <?php esc_html_e( 'Add New Source', 'ainp-ai-native-publisher' ); ?>
		</button>
	</p>

	<p class="submit">
		<input type="submit" name="ainp_save_sources_settings" class="button button-primary" value="<?php esc_attr_e( 'Save Sources', 'ainp-ai-native-publisher' ); ?>">
	</p>
</form>

<hr>

<form method="post" action="">
	<?php wp_nonce_field( 'ainp_settings_nonce' ); // Security nonce ?>
	<input type="hidden" name="current_page_slug" value="ainp-sources">

	<h2><?php esc_html_e( 'General Settings', 'ainp-ai-native-publisher' ); ?></h2>
	
	<table class="form-table">
		<tbody>
			<tr>
				<th scope="row"><label for="ainp_target_language"><?php esc_html_e( 'Target Language', 'ainp-ai-native-publisher' ); ?></label></th>
				<td>
					<select name="ainp_target_language" id="ainp_target_language">
						<?php foreach ( $ainp_available_languages as $ainp_lang ) : ?>
							<option value="<?php echo esc_attr( $ainp_lang ); ?>" <?php selected( $ainp_current_target_language, $ainp_lang ); ?>>
								<?php echo esc_html( $ainp_lang ); ?>
							</option>
						<?php endforeach; ?>
					</select>
					<p class="description"><?php esc_html_e( 'Select the language in which the final news content (title, body, tags) should be generated.', 'ainp-ai-native-publisher' ); ?></p>
				</td>
			</tr>

			<tr>
				<th scope="row"><label for="ainp_default_post_status"><?php esc_html_e( 'Default Final Post Status', 'ainp-ai-native-publisher' ); ?></label></th>
				<td>
					<select name="ainp_default_post_status" id="ainp_default_post_status">
						<option value="publish" <?php selected( $this->plugin->options['default_post_status'] ?? 'draft', 'publish' ); ?>><?php esc_html_e( 'Published', 'ainp-ai-native-publisher' ); ?></option>
						<option value="draft" <?php selected( $this->plugin->options['default_post_status'] ?? 'draft', 'draft' ); ?>><?php esc_html_e( 'Draft', 'ainp-ai-native-publisher' ); ?></option>
						<option value="pending" <?php selected( $this->plugin->options['default_post_status'] ?? 'draft', 'pending' ); ?>><?php esc_html_e( 'Pending Review', 'ainp-ai-native-publisher' ); ?></option>
					</select>
					<p class="description"><?php esc_html_e( 'The final status posts should have after all processing (including image generation) is complete.', 'ainp-ai-native-publisher' ); ?></p>
				</td>
			</tr>
			<tr>
				<th scope="row"><label for="ainp_default_author_id"><?php esc_html_e( 'Default Post Author', 'ainp-ai-native-publisher' ); ?></label></th>
				<td>
					<?php
					wp_dropdown_users(
						array(
							'name'             => 'ainp_default_author_id',
							'id'               => 'ainp_default_author_id',
							'selected'         => $this->plugin->options['default_author_id'] ?? 1,
							'show_option_none' => false,
							'capability'       => array( 'edit_posts' ),
						)
					);
					?>
					<p class="description"><?php esc_html_e( 'Select the user to be assigned as the author for posts created by automated processes.', 'ainp-ai-native-publisher' ); ?></p>
				</td>
			</tr>
		</tbody>
	</table>

	<input type="hidden" name="ainp_enable_feed_cache" value="1">
	<input type="hidden" name="ainp_feed_cache_duration" value="300">
	<input type="hidden" name="ainp_item_processing_timeout" value="180">
	<input type="hidden" name="ainp_respect_interval" value="0">

	<h3><?php esc_html_e( 'System Performance (Active)', 'ainp-ai-native-publisher' ); ?></h3>
	<p><?php esc_html_e( 'These settings are optimized for the best balance of speed and stability in the Free version.', 'ainp-ai-native-publisher' ); ?></p>

	<div class="ainp-status-panel">
		<table class="form-table" role="presentation">
			<tbody>
				<tr>
					<th scope="row"><?php esc_html_e( 'RSS Feed Cache', 'ainp-ai-native-publisher' ); ?></th>
					<td>
						<p>
							<span class="dashicons dashicons-yes-alt" style="color: #46b450;"></span>
							<strong><?php esc_html_e( 'Active', 'ainp-ai-native-publisher' ); ?></strong>
						</p>
						<p class="description"><?php esc_html_e( 'Feed content is cached to ensure fast consecutive fetches.', 'ainp-ai-native-publisher' ); ?></p>
					</td>
				</tr>
				<tr>
					<th scope="row"><?php esc_html_e( 'Cache Duration', 'ainp-ai-native-publisher' ); ?></th>
					<td>
						<p><strong><?php esc_html_e( '300 seconds (5 minutes)', 'ainp-ai-native-publisher' ); ?></strong></p>
					</td>
				</tr>
				<tr>
					<th scope="row"><?php esc_html_e( 'Item Processing Timeout', 'ainp-ai-native-publisher' ); ?></th>
					<td>
						<p><strong><?php esc_html_e( '180 seconds', 'ainp-ai-native-publisher' ); ?></strong></p>
					</td>
				</tr>
			</tbody>
		</table>
	</div>

	<div class="ainp-pro-box" style="margin-top: 30px; padding: 20px; background: #f0f6fc; border: 1px solid #cce5ff; border-radius: 5px;">
		<h4><?php esc_html_e( 'Need Advanced Scrapers & Controls?', 'ainp-ai-native-publisher' ); ?></h4>
		<p><?php esc_html_e( 'Upgrade to Pro to fetch content from any website (Web Scraper), configure custom cache times, set specific timeouts, and manage domain cooldowns.', 'ainp-ai-native-publisher' ); ?></p>
		<p>
			<a href="https://ainewspublisher.shop/" target="_blank" class="button button-primary"><?php esc_html_e( 'Check Pro Features', 'ainp-ai-native-publisher' ); ?></a>
		</p>
	</div>

	<p class="submit">
		<input type="submit" name="ainp_save_general_settings" class="button button-primary" value="<?php esc_attr_e( 'Save General Settings', 'ainp-ai-native-publisher' ); ?>">
	</p>
</form>