<?php
if ( ! defined( 'ABSPATH' ) ) exit; // Exit if accessed directly

if( !class_exists('Ainisa_Chatbot_Settings') ) {
    /**
     * Class for Ainisa Chatbot Integration settings
     * This class helps to create sections with fields for Ainisa chatbot integration
     * And makes validations during post request
     */
    class Ainisa_Chatbot_Settings {
        /**
         * Plugin options
         * @var false|mixed|null
         */
        public static $options;

        public function __construct()
        {
            self::$options = get_option('ainisa_chatbot_options');
            add_action('admin_init', [$this, 'adminInit']);
        }

        /**
         * Initialize settings with fields
         * @return void
         */
        public function adminInit()
        {
            register_setting(
                'ainisa_chatbot_group',
                'ainisa_chatbot_options',
                [$this, 'aiNisaSettingsValidation']
            );

            add_settings_section(
                'ainisa_chatbot_main_section',
                esc_html__( 'Integration Settings', 'ainisa-chatbot' ),
                [$this, 'AiNisaIntegrationInformationCallback'],
                'ainisa_chatbot_page1'
            );

            add_settings_field(
                'ainisa_integration_uuid',
                esc_html__( 'Integration UUID', 'ainisa-chatbot' ),
                [$this, 'AiNisaIntegrationUuidCallback'],
                'ainisa_chatbot_page1',
                'ainisa_chatbot_main_section',
                [
                    'label_for' => 'ainisa_integration_uuid'
                ]
            );

            add_settings_field(
                'ainisa_api_key',
                esc_html__( 'API Key', 'ainisa-chatbot' ),
                [$this, 'AiNisaApiKeyCallback'],
                'ainisa_chatbot_page1',
                'ainisa_chatbot_main_section',
                [
                    'label_for' => 'ainisa_api_key'
                ]
            );

            add_settings_field(
                'ainisa_secret_key',
                esc_html__( 'Secret Key (Optional)', 'ainisa-chatbot' ),
                [$this, 'AiNisaSecretKeyCallback'],
                'ainisa_chatbot_page1',
                'ainisa_chatbot_main_section',
                [
                    'label_for' => 'ainisa_secret_key'
                ]
            );
        }

        public function AiNisaIntegrationInformationCallback()
        {
            ?>
            <p><?php echo esc_html__('Enter your AiNisa chatbot integration credentials below. The chatbot widget will automatically appear on your website once configured.', 'ainisa-chatbot') ?></p>
            <?php
        }

        public function AiNisaIntegrationUuidCallback()
        {
            ?>
            <input type="text"
                   name="ainisa_chatbot_options[ainisa_integration_uuid]"
                   id="ainisa_integration_uuid"
                   class="regular-text"
                   value="<?php echo isset(self::$options['ainisa_integration_uuid']) ? esc_attr(self::$options['ainisa_integration_uuid']) : '' ?>"
                   placeholder="Enter your integration UUID"
            >
            <p class="description"><?php echo esc_html__('Your unique integration UUID from AiNisa', 'ainisa-chatbot') ?></p>
            <?php
        }

        public function AiNisaApiKeyCallback()
        {
            ?>
            <input type="text"
                   name="ainisa_chatbot_options[ainisa_api_key]"
                   id="ainisa_api_key"
                   class="regular-text"
                   value="<?php echo isset(self::$options['ainisa_api_key']) ? esc_attr(self::$options['ainisa_api_key']) : '' ?>"
                   placeholder="Enter your API key"
            >
            <p class="description"><?php echo esc_html__('Your API key from AiNisa', 'ainisa-chatbot') ?></p>
            <?php
        }

        public function AiNisaSecretKeyCallback()
        {
            ?>
            <input type="password"
                   name="ainisa_chatbot_options[ainisa_secret_key]"
                   id="ainisa_secret_key"
                   class="regular-text"
                   value="<?php echo isset(self::$options['ainisa_secret_key']) ? esc_attr(self::$options['ainisa_secret_key']) : '' ?>"
                   placeholder="Enter your secret key"
            >
            <p class="description"><?php echo esc_html__('Optional: Provide secret key to enable logged-in user data collection for personalized chat experience', 'ainisa-chatbot') ?></p>
            <?php
        }

        /**
         * Do validation for AiNisa integration settings
         * @param $input
         * @return array
         */
        public function aiNisaSettingsValidation($input)
        {
            $new_input = [];

            if (isset($input['ainisa_integration_uuid'])) {
                $new_input['ainisa_integration_uuid'] = sanitize_text_field($input['ainisa_integration_uuid']);
            }

            if (isset($input['ainisa_api_key'])) {
                $new_input['ainisa_api_key'] = sanitize_text_field($input['ainisa_api_key']);
            }

            // Secret key is optional
            if (isset($input['ainisa_secret_key']) && !empty($input['ainisa_secret_key'])) {
                $new_input['ainisa_secret_key'] = sanitize_text_field($input['ainisa_secret_key']);
            }

            if (empty($new_input['ainisa_integration_uuid'])) {
                add_settings_error(
                    'ainisa_chatbot_options',
                    'ainisa_integration_uuid',
                    esc_html__('Integration UUID is required', 'ainisa-chatbot'),
                    'error'
                );
            }

            if (empty($new_input['ainisa_api_key'])) {
                add_settings_error(
                    'ainisa_chatbot_options',
                    'ainisa_api_key',
                    esc_html__('API Key is required', 'ainisa-chatbot'),
                    'error'
                );
            }

            return $new_input;
        }
    }
}
