<?php
if ( ! defined( 'ABSPATH' ) ) exit; // Exit if accessed directly

/**
 * Plugin name: Ainisa Chatbot
 * Plugin Uri: https://ainisa.com
 * Description: Integrate Ainisa AI-powered chatbot into your WordPress website for automated customer support, sales and engagement
 * Version: 1.0
 * Requires at least: 6.0
 * Author: Javid Karimov
 * License: GPL v2 or later
 * License URI: https://www.gnu.org/licenses/gpl-2.0.html
 * Text Domain: ainisa-chatbot
 * Domain Path: /languages
 */

class Ainisa_Chatbot {
    public function __construct()
    {
        $this->defineConstants();

        add_action('admin_menu', [$this, 'addMenu']);

        require_once AINISA_CHATBOT_PATH.'class.ainisa-chatbot-settings.php';
        $aiNisaChatbotSettings = new Ainisa_Chatbot_Settings();

        add_action('wp_footer', [$this, 'injectWidgetCode'], 1);
        add_action('wp_footer', [$this, 'injectUserDataScript'], 5);
        add_action('wp_footer', [$this, 'injectScriptCode'], 10);
    }

    public function defineConstants()
    {
        define( 'AINISA_CHATBOT_PATH', plugin_dir_path(__FILE__) );
        define( 'AINISA_CHATBOT_URL', plugin_dir_url(__FILE__) );
        define( 'AINISA_CHATBOT_VERSION', '1.0.0' );
    }

    public static function activate()
    {
        update_option('rewrite_rules', '');
    }

    public static function deactivate()
    {
        flush_rewrite_rules();
    }

    public static function uninstall()
    {
        delete_option('ainisa_chatbot_options');
    }

    public function addMenu()
    {
        add_menu_page(
            esc_html__('AiNisa Chatbot Settings', 'ainisa-chatbot' ),
            esc_html__( 'AiNisa Chatbot', 'ainisa-chatbot' ),
            'manage_options',
            'ainisa-chatbot-admin',
            [$this, 'AiNisaChatbotSettingsPage'],
            'dashicons-format-chat',
            10
        );

        add_submenu_page(
            'ainisa-chatbot-admin',
            esc_html__( 'How to Get Credentials', 'ainisa-chatbot' ),
            esc_html__( 'Help', 'ainisa-chatbot' ),
            'manage_options',
            'ainisa-chatbot-help',
            [$this, 'AiNisaChatbotHelpPage']
        );
    }

    public function AiNisaChatbotSettingsPage()
    {
        if( ! current_user_can( 'manage_options' ) ) {
            return;
        }

        $number_of_errors = count(get_settings_errors());
        $errors = get_settings_errors();
        $has_errors = !(($number_of_errors == 1 && $errors[0]['type'] === 'success'));
        // phpcs:ignore WordPress.Security.NonceVerification.Recommended -- Checking WordPress's own settings-updated parameter after Settings API redirect
        if( isset( $_GET['settings-updated'] ) ) {
            if($has_errors) {
                settings_errors('ainisa_chatbot_options');
            } else {
                add_settings_error( 'ainisa_chatbot_options', 'ainisa_chatbot_option_message', esc_html__( 'Settings saved !', 'ainisa-chatbot' ), 'success' );
                settings_errors('ainisa_chatbot_options');
            }
            set_transient( 'ainisa_chatbot_options', get_settings_errors(), 30 ); // 30 seconds.

        }

        require_once AINISA_CHATBOT_PATH.'views/settings_page.php';
    }

    public function AiNisaChatbotHelpPage()
    {
        if( ! current_user_can( 'manage_options' ) ) {
            return;
        }

        require_once AINISA_CHATBOT_PATH.'views/help_page.php';
    }

    /**
     * Inject user data script (optional - only if secret key is set and user is logged in)
     */
    public function injectUserDataScript()
    {
        $options = get_option('ainisa_chatbot_options');

        // Only inject if secret key is provided and user is logged in
        if (empty($options['ainisa_secret_key']) || !is_user_logged_in()) {
            return;
        }

        $current_user = wp_get_current_user();
        $user_id = (string)$current_user->ID;
        $secret_key = $options['ainisa_secret_key'];

        // Generate user hash using HMAC SHA256
        $user_hash = hash_hmac('sha256', $user_id, $secret_key);

        // Get user data
        $firstname = get_user_meta($current_user->ID, 'first_name', true);
        $lastname = get_user_meta($current_user->ID, 'last_name', true);
        $email = $current_user->user_email;

        // Build inline script
        $inline_script = sprintf(
            "const userId = '%s';
            const userFirstname = '%s';
            const userLastname = '%s';
            const userEmail = '%s';
            const userHash = '%s';
            window.ainisaUser = {
                user_id: userId,
                user_hash: userHash,
                firstname: userFirstname,
                lastname: userLastname,
                email: userEmail
            };",
            esc_js($user_id),
            esc_js($firstname),
            esc_js($lastname),
            esc_js($email),
            esc_js($user_hash)
        );

        // Add inline script before the main widget script
        wp_add_inline_script('ainisa-chatbot-widget', $inline_script, 'before');
    }

    /**
     * Inject main widget script
     */
    public function injectScriptCode()
    {
        $options = get_option('ainisa_chatbot_options');
        if (empty($options['ainisa_integration_uuid']) || empty($options['ainisa_api_key'])) {
            return;
        }

        wp_enqueue_script('ainisa-chatbot-widget', 'https://chatbot.ainisa.com/widget.js', array(), AINISA_CHATBOT_VERSION, true);
    }

    /**
     * Inject widget code in footer
     */
    public function injectWidgetCode()
    {
        $options = get_option('ainisa_chatbot_options');
        if (empty($options['ainisa_integration_uuid']) || empty($options['ainisa_api_key'])) {
            return;
        }
        ?>
        <!-- Ainisa Chatbot -->
        <div id="ainisa_chatbot" data-btn-bg="#a34af0" data-btn-color="#ffffff" data-btn-margin="20" data-btn-bottom="20">
            <iframe allow="microphone; clipboard-write" src="https://chatbot.ainisa.com/widget/<?php echo esc_attr($options['ainisa_integration_uuid']); ?>?publicKey=<?php echo esc_attr($options['ainisa_api_key']); ?>&lang=en"></iframe>
        </div>
        <!-- Ainisa Chatbot -->
        <?php
    }

}

if( class_exists( 'Ainisa_Chatbot' ) ) {
    register_activation_hook( __FILE__, ['Ainisa_Chatbot', 'activate'] );
    register_deactivation_hook( __FILE__, ['Ainisa_Chatbot', 'deactivate'] );
    register_uninstall_hook( __FILE__, ['Ainisa_Chatbot', 'uninstall'] );

    $ainisa_chatbot = new Ainisa_Chatbot();
}