<?php
if (!defined('ABSPATH')) exit;

function aigude_tools_render_list_page(): void {
    if (!current_user_can('upload_files')) {
        wp_die(esc_html__('You do not have permission to access this page.', 'aigude-tools'));
    }

    // Bootstrap prompt templates: ensure array + built-in default exists and is marked as such.
    $default_tpl_id = get_option('aigude_prompt_default_id', '');
    $templates      = get_option('aigude_prompt_templates', []);
    if (!is_array($templates)) {
        $templates = [];
    }
    $builtin_tpl = AIGUDE_Tools_Plugin::get_builtin_prompt_template();
    $builtin_id  = $builtin_tpl['id'] ?? AIGUDE_Tools_Plugin::BUILTIN_PROMPT_ID;
    $has_builtin = false;
    $templates_changed = false;
    foreach ($templates as &$tpl) {
        if (!isset($tpl['builtin'])) {
            $tpl['builtin'] = '0';
            $templates_changed = true;
        }
        if (($tpl['id'] ?? '') === $builtin_id) {
            if ($tpl['builtin'] !== '1') {
                $tpl['builtin'] = '1';
                $templates_changed = true;
            }
            $has_builtin = true;
        }
    }
    unset($tpl);
    if (!$has_builtin) {
        $templates[] = $builtin_tpl;
        $templates_changed = true;
    }
    if ($templates_changed) {
        update_option('aigude_prompt_templates', $templates);
    }
    // Helper to fetch a template by ID.
    $find_tpl = static function (array $items, string $id): ?array {
        foreach ($items as $tpl) {
            if (($tpl['id'] ?? '') === $id) {
                return $tpl;
            }
        }
        return null;
    };
    if ($default_tpl_id === '' || $find_tpl($templates, $default_tpl_id) === null) {
        $default_tpl_id = $builtin_id;
        update_option('aigude_prompt_default_id', $default_tpl_id);
    }
    $default_tpl = $find_tpl($templates, $default_tpl_id);
    if ($default_tpl === null) {
        $default_tpl = $builtin_tpl;
    }

    // Build edit links keyed by template ID (use original indexes for action=edit).
    $prompts_page_url = admin_url('admin.php?page=aigude-tools-prompts');
    $tpl_nonce        = wp_create_nonce('aigude_tpl_action');
    $edit_links_by_id = [];
    foreach ($templates as $idx => $tpl) {
        $tid = $tpl['id'] ?? '';
        if ($tid === '') continue;
        $edit_links_by_id[$tid] = add_query_arg(
                [
                        'action'          => 'edit',
                        'tpl_index'       => $idx,
                        'aigude_tpl_nonce'=> $tpl_nonce,
                ],
                $prompts_page_url
        );
    }

    // Sort templates for display: provider → language → title.
    $templates_sorted = $templates;
    usort($templates_sorted, static function ($a, $b) {
        $pa = strtolower($a['target_provider'] ?? '');
        $pb = strtolower($b['target_provider'] ?? '');
        if ($pa !== $pb) {
            return strcmp($pa, $pb);
        }
        // Use language label for user-friendly ordering (fallback to code).
        $la_label = strtolower(AIGUDE_Tools_Plugin::describe_target_language_choice($a['target_provider'] ?? '', $a['target_lang'] ?? '')['label'] ?? ($a['target_lang'] ?? ''));
        $lb_label = strtolower(AIGUDE_Tools_Plugin::describe_target_language_choice($b['target_provider'] ?? '', $b['target_lang'] ?? '')['label'] ?? ($b['target_lang'] ?? ''));
        if ($la_label !== $lb_label) {
            return strcmp($la_label, $lb_label);
        }
        $ta = strtolower($a['title'] ?? '');
        $tb = strtolower($b['title'] ?? '');
        return strcmp($ta, $tb);
    });
    $templates = $templates_sorted;

    // Group templates by provider for clearer separation in selects.
    $grouped_templates = [];
    foreach ($templates as $tpl) {
        $prov  = strtolower($tpl['target_provider'] ?? '');
        $label = $prov !== '' ? AIGUDE_Tools_Plugin::get_translation_provider_label($prov) : __('(No provider)', 'aigude-tools');
        if (!isset($grouped_templates[$prov])) {
            $grouped_templates[$prov] = ['label' => $label, 'items' => []];
        }
        $tid = $tpl['id'] ?? '';
        if ($tid !== '' && isset($edit_links_by_id[$tid])) {
            $tpl['edit_url'] = $edit_links_by_id[$tid];
        }
        $grouped_templates[$prov]['items'][] = $tpl;
    }

    // Nonce for harmless list filters (search/pagination/per-page); scoped to this admin view.
    $filters_nonce_action = 'aigude_tools_list_filters';
    $filters_nonce_value  = wp_create_nonce( $filters_nonce_action );

    // Default filter values
    $paged    = 1;
    $per_page = 20;
    $search   = '';

    // Only honor incoming filters if a filter was submitted and nonce validates.
    $has_filters = isset( $_GET['paged'] ) || isset( $_GET['per_page'] ) || isset( $_GET['s'] );
    if ( $has_filters ) {
        // Dies on failure; prevents bypass via crafted URLs.
        check_admin_referer( $filters_nonce_action );

        $paged = isset( $_GET['paged'] ) ? max( 1, absint( wp_unslash( $_GET['paged'] ) ) ) : 1;

        $per_page   = isset( $_GET['per_page'] ) ? absint( wp_unslash( $_GET['per_page'] ) ) : 20;
        $allowed_pp = [ 20, 50, 100 ];
        if ( ! in_array( $per_page, $allowed_pp, true ) ) {
            $per_page = 20;
        }

        $search = isset( $_GET['s'] ) ? sanitize_text_field( wp_unslash( $_GET['s'] ) ) : '';
    }


    $q = new WP_Query([
            'post_type'      => 'attachment',
            'post_status'    => 'inherit',
            'post_mime_type' => AIGUDE_Tools_Plugin::get_image_mime_types(),
            'orderby'        => [ 'date' => 'DESC', 'ID' => 'DESC' ],
            'order'          => 'DESC',
            'posts_per_page' => $per_page,
            'paged'          => $paged,
            's'              => $search,
            'ai_tools_list'  => 1,
    ]);

    $page_total  = (int) $q->post_count;
    $base        = add_query_arg( '_wpnonce', $filters_nonce_value, remove_query_arg('paged') );
    $pagination  = paginate_links([
        'base'      => $base . '%_%',
        'format'    => (strpos($base, '?') !== false ? '&' : '?') . 'paged=%#%',
        'total'     => max(1, (int) $q->max_num_pages),
        'current'   => $paged,
        'prev_text' => '«',
        'next_text' => '»',
        'type'      => 'plain',
    ]);

    // Resolve initial prompt text from default template, fallback to built-in prompt
    $initial_prompt_text = $default_tpl['prompt'] ?? ($builtin_tpl['prompt'] ?? '');

    ?>

    <div class="wrap ai-alttext-wrap">
        <h2 style="margin:0 0 10px;"><?php esc_html_e('Alt Text Generator - List view', 'aigude-tools'); ?></h2>
        <?php if ( method_exists('AIGUDE_Tools_Plugin','debug_enabled') && AIGUDE_Tools_Plugin::debug_enabled() && function_exists('wp_debug_log') ) {
            $debug_provider = $default_tpl['target_provider'] ?? '';
            $debug_lang     = $default_tpl['target_lang'] ?? '';
            wp_debug_log('[AiGude Tools] Rendering List view. default_provider=' . $debug_provider . ', default_lang=' . $debug_lang);
        } ?>

        <section class="bulk-panel">

            <div class="tablenav top" style="margin-bottom:12px;">

                <!-- Search box  -->
                <div class="alignleft">
                    <form method="get" style="display:inline-block;margin-right:12px;">
                        <input type="hidden" name="page" value="<?php echo esc_attr(AIGUDE_Tools_Plugin::MENU_SLUG); ?>" />
                        <input type="hidden" name="_wpnonce" value="<?php echo esc_attr( $filters_nonce_value ); ?>" />
                        <input type="hidden" name="per_page" value="<?php echo (int) $per_page; ?>" />
                        <p class="search-box" style="margin:0;">
                            <label class="screen-reader-text" for="aigude-tools-search">
                                <?php esc_html_e('Search images', 'aigude-tools'); ?>
                            </label>
                            <input
                                type="search"
                                id="aigude-tools-search"
                                name="s"
                                value="<?php echo esc_attr($search); ?>"
                                placeholder="<?php esc_attr_e('Search filename, title or alt-text…', 'aigude-tools'); ?>"
                                style="width:250px;"
                            />
                            <button class="button"><?php esc_html_e('Search', 'aigude-tools'); ?></button>
                        </p>
                    </form>
                </div>

                <div class="tablenav-pages">
                    <?php echo $pagination ? wp_kses_post( $pagination ) : '&nbsp;'; ?>
                </div>

                <div style="clear:both"></div>
            </div>

            <div style="display:flex;align-items:center;gap:10px;flex-wrap:wrap;margin-bottom:12px;">

                <!--  Pagination per page  -->
                <form method="get" style="display:inline-block;">
                        <input type="hidden" name="page" value="<?php echo esc_attr(AIGUDE_Tools_Plugin::MENU_SLUG); ?>" />
                        <input type="hidden" name="_wpnonce" value="<?php echo esc_attr( $filters_nonce_value ); ?>" />
                    <?php if ($search !== ''): ?>
                        <input type="hidden" name="s" value="<?php echo esc_attr($search); ?>" />
                    <?php endif; ?>
                    <label for="ai-per-page"><strong><?php esc_html_e('Per Page', 'aigude-tools'); ?></strong></label>
                    <select id="ai-per-page" name="per_page" onchange="this.form.submit()">
                        <?php foreach ([20,50,100] as $n): ?>
                            <option value="<?php echo esc_attr($n); ?>" <?php selected($n, $per_page); ?>><?php echo esc_html($n); ?></option>
                        <?php endforeach; ?>
                    </select>
                </form>
                <!-- Checkboxes -->
                <label style="margin:0;">
                    <input type="checkbox" id="skip-existing" checked> <?php esc_html_e('Skip existing', 'aigude-tools'); ?>
                </label>

                <label style="margin:0;">
                    <input type="checkbox" id="select-all-cb"> <?php esc_html_e('Select all (this page)', 'aigude-tools'); ?>
                </label>

                <label style="margin:0;">
                    <input type="checkbox" id="select-all-allpages"> <?php esc_html_e('Select all (across pages)', 'aigude-tools'); ?>
                </label>

                <small id="allpages-hint" style="display:none;opacity:.8;">
                    <?php esc_html_e('Will process', 'aigude-tools'); ?>
                    <strong><span id="allpages-count">0</span></strong>
                    <?php esc_html_e('images.', 'aigude-tools'); ?>
                </small>
            </div>

            <div style="display:flex;align-items:center;gap:10px;flex-wrap:wrap;margin-bottom:12px;">

                <!-- Prompt select -->
                <label for="global-prompt"><strong><?php esc_html_e('Prompt', 'aigude-tools'); ?></strong></label>
                <select id="global-prompt" class="aitools-select">
                    <?php foreach ( $grouped_templates as $group ) : ?>
                        <optgroup label="<?php echo esc_attr( $group['label'] ); ?>">
                            <?php foreach ( $group['items'] as $tpl ) :
                                $tid   = $tpl['id'] ?? '';
                                $value = $tpl['prompt'] ?? '';
                                $title = $tpl['title'] ?? '';
                                $target_info = AIGUDE_Tools_Plugin::describe_target_language_choice(
                                        $tpl['target_provider'] ?? '',
                                        $tpl['target_lang'] ?? ''
                                );
                                $target_display = $target_info['display'] ?? '';
                                $option_label = $title;
                                if ($target_display) {
                                    $option_label .= sprintf(' (%s)', $target_display);
                                }
                                ?>
                                <option
                                        value="<?php echo esc_attr( $value ); ?>"
                                        data-prompt-lang="<?php echo esc_attr( $tpl['prompt_lang'] ?? 'auto' ); ?>"
                                        data-src-lang="<?php echo esc_attr( $tpl['src_lang'] ?? 'auto' ); ?>"
                                        data-tpl-id="<?php echo esc_attr( $tid ); ?>"
                                        data-title="<?php echo esc_attr( $title ); ?>"
                                        data-target-provider="<?php echo esc_attr( $target_info['provider'] ?? '' ); ?>"
                                        data-target-provider-label="<?php echo esc_attr( $target_info['provider_label'] ?? '' ); ?>"
                                        data-target-lang="<?php echo esc_attr( $target_info['code'] ?? '' ); ?>"
                                        data-target-lang-label="<?php echo esc_attr( $target_info['label'] ?? '' ); ?>"
                                        data-edit-url="<?php echo esc_url( $tpl['edit_url'] ?? '' ); ?>"
                                        <?php selected( $tid, $default_tpl_id ); ?>
                                >
                                    <?php echo esc_html( $option_label ); ?><?php echo $tid === $default_tpl_id ? ' ★' : ''; ?>
                                </option>
                            <?php endforeach; ?>
                        </optgroup>
                    <?php endforeach; ?>
                </select>
                <button type="button" class="button ai-prompt-details-btn" id="global-prompt-details-btn" aria-label="<?php esc_attr_e('View prompt details', 'aigude-tools'); ?>">&#9432;</button>
            </div>

            <div style="margin-top:15px;display:flex;gap:10px;flex-wrap:wrap;">
                <?php
                // translators: %s: number of images (the %s is replaced dynamically in JS for the data attribute).
                $btn_default_text = __( 'Generate and save alternative text for %s Images', 'aigude-tools' );
                ?>
                <button id="bulk-generate"
                        class="button button-primary aitools-button"
                        data-default-text="<?php echo esc_attr( $btn_default_text ); ?>"
                        data-loading-text="<?php echo esc_attr( __( 'Generating...', 'aigude-tools' ) ); ?>"
                        data-count="<?php echo esc_attr( $page_total ); ?>">

                    <?php
                    printf(
                            // translators: %s: number of images on the current page.
                            esc_html__( 'Generate and save alternative text for %s images', 'aigude-tools' ),
                            esc_html( number_format_i18n( absint( $page_total ) ) )
                    );
                    ?>

                </button>
                <div class="credit-info-wrapper" id="credit-summary-wrapper">
                    <div id="credit-summary" class="text-muted"></div>
                </div>
            </div>

            <div id="bulk-progress" class="progress-container">
                <div id="bulk-progress-bar" class="progress-bar"></div>
            </div>
        </section>

        <?php foreach ($q->posts as $img): ?>
            <?php
            $id         = (int) $img->ID;
            $thumb_src  = wp_get_attachment_image_src($id, 'full');
            $thumb      = is_array($thumb_src) ? $thumb_src[0] : '';
            $saved_alt  = (string) get_post_meta($id, '_wp_attachment_image_alt', true);
            ?>

            <article class="ai-card"
                     data-id="<?php echo esc_attr($id); ?>"
                     data-saved-alt="<?php echo esc_attr($saved_alt); ?>">
                <header>
                    <input type="checkbox" class="ai-select" />
                    <a class="ai-media-link"
                       href="<?php echo esc_url(add_query_arg(['item'=>$id], admin_url('upload.php'))); ?>"
                       target="_blank" rel="noopener noreferrer"
                       title="<?php echo esc_attr__('Open in Media Library', 'aigude-tools'); ?>">
                        <?php if ($thumb): ?>
                            <img src="<?php echo esc_url($thumb); ?>" alt="" class="thumb-image" />
                        <?php endif; ?>
                    </a>
                    <h2 style="font-size:18px;margin:0;">
                        <?php
                        printf(
                        /* translators: %s: the file title (post_title) of the image. */
                                esc_html__( 'Generate File Metadata "%s"', 'aigude-tools' ),
                                esc_html( $img->post_title )
                        );
                        ?>
                    </h2>
                </header>

                <div class="suggestion-block" style="<?php echo $saved_alt ? 'display:inline-block;' : 'display:none;'; ?>">
                    <span class="dashicons dashicons-clock"></span>
                    <span class="suggestion-text" style="flex:1;"><?php echo esc_html($saved_alt); ?></span>
                </div>

                <section class="prompt-block" style="margin-bottom:20px;">
                    <!-- Prompt select card  -->
                    <label style="font-weight:600;"><?php esc_html_e('Prompt', 'aigude-tools'); ?></label>
                    <select class="ai-prompt-select">
                        <?php foreach ( $grouped_templates as $group ) : ?>
                            <optgroup label="<?php echo esc_attr( $group['label'] ); ?>">
                                <?php foreach ( $group['items'] as $tpl ) :
                                    $tid   = $tpl['id'] ?? '';
                                    $value = $tpl['prompt'] ?? '';
                                    $title = $tpl['title'] ?? '';
                                    $target_info = AIGUDE_Tools_Plugin::describe_target_language_choice(
                                            $tpl['target_provider'] ?? '',
                                            $tpl['target_lang'] ?? ''
                                    );
                                    $target_display = $target_info['display'] ?? '';
                                    $option_label = $title;
                                    if ($target_display) {
                                        $option_label .= sprintf(' (%s)', $target_display);
                                    }
                                    ?>
                                    <option
                                            value="<?php echo esc_attr( $value ); ?>"
                                            data-prompt-lang="<?php echo esc_attr( $tpl['prompt_lang'] ?? 'auto' ); ?>"
                                            data-src-lang="<?php echo esc_attr( $tpl['src_lang'] ?? 'auto' ); ?>"
                                            data-tpl-id="<?php echo esc_attr( $tid ); ?>"
                                            data-title="<?php echo esc_attr( $title ); ?>"
                                            data-target-provider="<?php echo esc_attr( $target_info['provider'] ?? '' ); ?>"
                                            data-target-provider-label="<?php echo esc_attr( $target_info['provider_label'] ?? '' ); ?>"
                                            data-target-lang="<?php echo esc_attr( $target_info['code'] ?? '' ); ?>"
                                            data-target-lang-label="<?php echo esc_attr( $target_info['label'] ?? '' ); ?>"
                                            data-edit-url="<?php echo esc_url( $tpl['edit_url'] ?? '' ); ?>"
                                    <?php selected( $tid, $default_tpl_id ); ?>
                            >
                                        <?php echo esc_html( $option_label ); ?><?php echo $tid === $default_tpl_id ? ' ★' : ''; ?>
                                    </option>
                                <?php endforeach; ?>
                            </optgroup>
                        <?php endforeach; ?>
                    </select>

                    <textarea rows="2" class="ai-prompt-input" placeholder="<?php echo esc_attr__('Custom prompt…', 'aigude-tools'); ?>"><?php
                        echo esc_textarea($initial_prompt_text);
                        ?></textarea>

                    <div style="margin-top:10px;display:flex;gap:10px;">
                        <button class="button generate-single"
                                data-default-text="<?php echo esc_attr__( 'Generate', 'aigude-tools' ); ?>"
                                data-loading-text="<?php echo esc_attr__( 'Generating', 'aigude-tools' ); ?>">
                            <?php esc_html_e( 'Generate', 'aigude-tools' ); ?> &nbsp;
                            <span class="ai-credit-badge">
                                <?php echo esc_html( AIGUDE_Tools_Plugin::PER_IMAGE_CREDITS ); ?>
                                <?php esc_html_e( 'Credits', 'aigude-tools' ); ?>
                            </span>
                        </button>
                        <button class="button continue-single" style="<?php echo $saved_alt ? 'display:inline-block;' : 'display:none;'; ?>padding:8px 16px;">
                            <?php esc_html_e('Continue with the current alternative text', 'aigude-tools'); ?>
                        </button>
                        <div class="credit-info-wrapper credits-used-box" style="display:none;">
                            <div class="credits-used text-muted"></div>
                        </div>
                    </div>
                </section>

                <!-- alternative-block -->
                <section class="alternative-block" style="display:none;">
                    <label style="font-weight:600;"><?php esc_html_e('Alternative Text', 'aigude-tools'); ?></label>
                    <textarea class="alt-text-input" rows="2"></textarea>
                    <div style="margin-top:10px;display:flex;gap:10px;">
                        <button class="button save-single"><?php esc_html_e('Save', 'aigude-tools'); ?></button>
                    </div>
                </section>
            </article>
        <?php endforeach; ?>

        <div class="tablenav bottom" style="margin-top:8px;">
            <div class="tablenav-pages">
                <?php echo $pagination ? wp_kses_post( $pagination ) : '&nbsp;'; ?>
            </div>
            <div style="clear:both"></div>
        </div>
    </div>

    <?php
    wp_reset_postdata();
}
