<?php
if (!defined('ABSPATH')) exit;

function aigude_tools_render_grid_page(): void {
    if (!current_user_can('upload_files')) {
        wp_die(esc_html__('You do not have permission to access this page.', 'aigude-tools'));
    }

    // Bootstrap prompt templates: ensure array + built-in default exists and is marked as such.
    $default_tpl_id = get_option('aigude_prompt_default_id', '');
    $templates      = get_option('aigude_prompt_templates', []);
    if (!is_array($templates)) {
        $templates = [];
    }
    $builtin_tpl = AIGUDE_Tools_Plugin::get_builtin_prompt_template();
    $builtin_id  = $builtin_tpl['id'] ?? AIGUDE_Tools_Plugin::BUILTIN_PROMPT_ID;
    $has_builtin = false;
    $templates_changed = false;
    foreach ($templates as &$tpl) {
        if (!isset($tpl['builtin'])) {
            $tpl['builtin'] = '0';
            $templates_changed = true;
        }
        if (($tpl['id'] ?? '') === $builtin_id) {
            if ($tpl['builtin'] !== '1') {
                $tpl['builtin'] = '1';
                $templates_changed = true;
            }
            $has_builtin = true;
        }
    }
    unset($tpl);
    if (!$has_builtin) {
        $templates[] = $builtin_tpl;
        $templates_changed = true;
    }
    if ($templates_changed) {
        update_option('aigude_prompt_templates', $templates);
    }
    // Helper to fetch a template by ID.
    $find_tpl = static function (array $items, string $id): ?array {
        foreach ($items as $tpl) {
            if (($tpl['id'] ?? '') === $id) {
                return $tpl;
            }
        }
        return null;
    };
    if ($default_tpl_id === '' || $find_tpl($templates, $default_tpl_id) === null) {
        $default_tpl_id = $builtin_id;
        update_option('aigude_prompt_default_id', $default_tpl_id);
    }
    $default_tpl = $find_tpl($templates, $default_tpl_id);
    if ($default_tpl === null) {
        $default_tpl = $builtin_tpl;
    }

    // Build edit links keyed by template ID (use original indexes for action=edit).
    $prompts_page_url = admin_url('admin.php?page=aigude-tools-prompts');
    $tpl_nonce        = wp_create_nonce('aigude_tpl_action');
    $edit_links_by_id = [];
    foreach ($templates as $idx => $tpl) {
        $tid = $tpl['id'] ?? '';
        if ($tid === '') continue;
        $edit_links_by_id[$tid] = add_query_arg(
                [
                        'action'          => 'edit',
                        'tpl_index'       => $idx,
                        'aigude_tpl_nonce'=> $tpl_nonce,
                ],
                $prompts_page_url
        );
    }

    // Sort templates for display: provider → language → title.
    $templates_sorted = $templates;
    usort($templates_sorted, static function ($a, $b) {
        $pa = strtolower($a['target_provider'] ?? '');
        $pb = strtolower($b['target_provider'] ?? '');
        if ($pa !== $pb) {
            return strcmp($pa, $pb);
        }
        // Use language label for user-friendly ordering (fallback to code).
        $la_label = strtolower(AIGUDE_Tools_Plugin::describe_target_language_choice($a['target_provider'] ?? '', $a['target_lang'] ?? '')['label'] ?? ($a['target_lang'] ?? ''));
        $lb_label = strtolower(AIGUDE_Tools_Plugin::describe_target_language_choice($b['target_provider'] ?? '', $b['target_lang'] ?? '')['label'] ?? ($b['target_lang'] ?? ''));
        if ($la_label !== $lb_label) {
            return strcmp($la_label, $lb_label);
        }
        $ta = strtolower($a['title'] ?? '');
        $tb = strtolower($b['title'] ?? '');
        return strcmp($ta, $tb);
    });
    $templates = $templates_sorted;

    // Group templates by provider for clearer separation in selects.
    $grouped_templates = [];
    foreach ($templates as $tpl) {
        $prov  = strtolower($tpl['target_provider'] ?? '');
        $label = $prov !== '' ? AIGUDE_Tools_Plugin::get_translation_provider_label($prov) : __('(No provider)', 'aigude-tools');
        if (!isset($grouped_templates[$prov])) {
            $grouped_templates[$prov] = ['label' => $label, 'items' => []];
        }
        $tid = $tpl['id'] ?? '';
        if ($tid !== '' && isset($edit_links_by_id[$tid])) {
            $tpl['edit_url'] = $edit_links_by_id[$tid];
        }
        $grouped_templates[$prov]['items'][] = $tpl;
    }

    ?>
    <div class="wrap ai-alttext-wrap">
        <h2 style="margin:0 0 10px;"><?php esc_html_e('Alt Text Generator - Grid view', 'aigude-tools'); ?></h2>
        <?php if ( method_exists('AIGUDE_Tools_Plugin','debug_enabled') && AIGUDE_Tools_Plugin::debug_enabled() && function_exists('wp_debug_log') ) {
            $debug_provider = $default_tpl['target_provider'] ?? '';
            $debug_lang     = $default_tpl['target_lang'] ?? '';
            wp_debug_log('[AiGude Tools] Rendering Grid view. default_provider=' . $debug_provider . ', default_lang=' . $debug_lang);
        } ?>

        <section class="bulk-panel" style="background:#ffe3ff;padding:20px;margin-bottom:30px;border-radius:4px;">
            <div style="display:flex;align-items:center;gap:10px;flex-wrap:wrap;">
                <label for="global-prompt"><strong><?php esc_html_e('Prompt', 'aigude-tools'); ?></strong></label>

                <select id="global-prompt" class="aitools-select">
                    <?php foreach ( $grouped_templates as $group ) : ?>
                        <optgroup label="<?php echo esc_attr( $group['label'] ); ?>">
                            <?php foreach ( $group['items'] as $tpl ) :
                                $tid   = $tpl['id'] ?? '';
                                $value = $tpl['prompt'] ?? '';
                                $title = $tpl['title'] ?? '';
                                $target_info = AIGUDE_Tools_Plugin::describe_target_language_choice(
                                        $tpl['target_provider'] ?? '',
                                        $tpl['target_lang'] ?? ''
                                );
                                $target_display = $target_info['display'] ?? '';
                                $option_label = $title;
                                if ($target_display) {
                                    $option_label .= sprintf(' (%s)', $target_display);
                                }
                                ?>
                                <option
                                        value="<?php echo esc_attr( $value ); ?>"
                                        data-prompt-lang="<?php echo esc_attr( $tpl['prompt_lang'] ?? 'auto' ); ?>"
                                        data-src-lang="<?php echo esc_attr( $tpl['src_lang'] ?? 'auto' ); ?>"
                                        data-tpl-id="<?php echo esc_attr( $tid ); ?>"
                                        data-title="<?php echo esc_attr( $title ); ?>"
                                        data-target-provider="<?php echo esc_attr( $target_info['provider'] ?? '' ); ?>"
                                        data-target-provider-label="<?php echo esc_attr( $target_info['provider_label'] ?? '' ); ?>"
                                        data-target-lang="<?php echo esc_attr( $target_info['code'] ?? '' ); ?>"
                                        data-target-lang-label="<?php echo esc_attr( $target_info['label'] ?? '' ); ?>"
                                        data-edit-url="<?php echo esc_url( $tpl['edit_url'] ?? '' ); ?>"
                                <?php selected( $tid, $default_tpl_id ); ?>
                        >
                                    <?php echo esc_html( $option_label ); ?>
                                    <?php echo $tid === $default_tpl_id ? ' ★' : ''; ?>
                                </option>
                            <?php endforeach; ?>
                        </optgroup>
                    <?php endforeach; ?>
                </select>
                <button type="button" class="button ai-prompt-details-btn" id="global-prompt-details-btn" aria-label="<?php esc_attr_e('View prompt details', 'aigude-tools'); ?>">&#9432;</button>

            </div>

            <div style="margin-top:15px;display:flex;gap:10px;flex-wrap:wrap;">
                <button id="media-select" class="button button-primary aitools-button">
                    <?php esc_html_e('Select images from Media Library', 'aigude-tools'); ?>
                </button>
                <button id="media-generate" class="button button-primary" disabled>
                    <?php esc_html_e('Generate alt text for selected', 'aigude-tools'); ?>
                </button>
                <div class="credit-info-wrapper" id="credit-summary-wrapper">
                    <div id="credit-summary" class="text-muted"></div>
                </div>
            </div>

            <div id="bulk-progress" class="progress-container"><div id="bulk-progress-bar" class="progress-bar"></div></div>
            <div id="media-selected-grid"></div>
        </section>
    </div>
    <?php
}
