<?php if ( ! defined( 'ABSPATH' ) ) exit;

/**
 * Callback for the "Templates" submenu.
 */
function aigude_prompt_templates_page() {
    if ( ! current_user_can( 'manage_options' ) ) {
        wp_die( esc_html__( 'Insufficient permissions', 'aigude-tools' ), 403 );
    }

    // Load current list
    $templates = get_option('aigude_prompt_templates', []);

    // Ensure every template has a stable ID (for defaults)
    $changed = false;
    foreach ($templates as &$tpl) {
        if (empty($tpl['id'])) {
            // Prefer core UUID if available; fallback to uniqid
            $tpl['id'] = function_exists('wp_generate_uuid4') ? wp_generate_uuid4() : ('tpl_' . uniqid('', true));
            $changed = true;
        }
    }
    unset($tpl);
    if ($changed) {
        update_option('aigude_prompt_templates', $templates);
    }
    $default_id = get_option('aigude_prompt_default_id', '');

    // Set default action
    if (isset($_GET['action'], $_GET['tpl_index'])
            && $_GET['action'] === 'set_default'
            && check_admin_referer('aigude_tpl_action','aigude_tpl_nonce')) {

        $idx = intval($_GET['tpl_index']);
        if (isset($templates[$idx]['id'])) {
            update_option('aigude_prompt_default_id', $templates[$idx]['id']);
            $default_id = $templates[$idx]['id'];
            echo '<div class="updated"><p>' . esc_html__('Default template updated.', 'aigude-tools') . '</p></div>';
        }
    }

    // Deletion
    if ( isset($_GET['action'], $_GET['tpl_index'])
            && $_GET['action']==='delete'
            && check_admin_referer('aigude_tpl_action','aigude_tpl_nonce') ) {
        $idx = intval($_GET['tpl_index']);
        if ( isset($templates[$idx]) ) {

            $deleted_id = $templates[$idx]['id'] ?? '';
            unset($templates[$idx]);
            $templates = array_values($templates);
            update_option('aigude_prompt_templates', $templates);

            // If the deleted one was default, clear default
            if ($deleted_id && get_option('aigude_prompt_default_id') === $deleted_id) {
                delete_option('aigude_prompt_default_id');
            }
            echo '<div class="updated"><p>' . esc_html__('Template deleted.', 'aigude-tools') . '</p></div>';
        }
    }

    // Load for editing
    $edit_index  = null;
    $edit_title  = '';
    $edit_prompt = '';
    if ( isset($_GET['action'], $_GET['tpl_index'])
            && $_GET['action']==='edit'
            && check_admin_referer('aigude_tpl_action','aigude_tpl_nonce') ) {
        $idx = intval($_GET['tpl_index']);
        if ( isset($templates[$idx]) ) {
            $edit_index  = $idx;
            $edit_title  = $templates[$idx]['title'];
            $edit_prompt = $templates[$idx]['prompt'];
        }
    }

    // Save (new or update)
    if (isset($_POST['ai_tpl_submit']) && check_admin_referer('aigude_tpl_action','aigude_tpl_nonce')) {
        $title   = sanitize_text_field(wp_unslash($_POST['ai_tpl_title']  ?? ''));
        $prompt  = sanitize_textarea_field(wp_unslash($_POST['ai_tpl_prompt'] ?? ''));

        $src_lang_placeholders = sanitize_text_field(wp_unslash($_POST['ai_tpl_src_lang'] ?? 'auto'));
        $prompt_lang           = sanitize_text_field(wp_unslash($_POST['ai_tpl_prompt_lang'] ?? 'auto'));

        // Keep ID if editing; otherwise make one
        $tpl_id = '';
        if (!empty($_POST['ai_tpl_id'])) {
            $tpl_id = sanitize_text_field(wp_unslash($_POST['ai_tpl_id']));
        } else {
            $tpl_id = function_exists('wp_generate_uuid4') ? wp_generate_uuid4() : ('tpl_' . uniqid('', true));
        }

        $data = [
                'id'           => $tpl_id,
                'title'        => $title,
                'prompt'       => $prompt,
                'src_lang'     => $src_lang_placeholders,
                'prompt_lang'  => $prompt_lang,
        ];

        $make_default = !empty($_POST['ai_tpl_is_default']);

        if ( $title && $prompt ) {
            if ( isset($_POST['ai_tpl_index']) && $_POST['ai_tpl_index'] !== '' ) {
                $idx = (int) $_POST['ai_tpl_index'];
                $templates[$idx] = $data;
                printf(
                        '<div class="updated"><p>%s</p></div>',
                        esc_html__( 'Template updated.', 'aigude-tools' )
                );
            } else {
                $templates[] = $data;
                printf(
                        '<div class="updated"><p>%s</p></div>',
                        esc_html__( 'Template saved.', 'aigude-tools' )
                );
            }

            update_option('aigude_prompt_templates', $templates);
            // Track recents for prompt and placeholder language selections
            if (class_exists('AIGUDE_Tools_Plugin')) {
                AIGUDE_Tools_Plugin::push_recent_lang('prompt', $prompt_lang);
                AIGUDE_Tools_Plugin::push_recent_lang('placeholder', $src_lang_placeholders);
            }
            $edit_index = null;
            $edit_title = '';
            $edit_prompt = '';
        }

        if ($make_default) {
            update_option('aigude_prompt_default_id', $tpl_id);
            $default_id = $tpl_id;
        }
    }

    // View
    ?>
    <div class="wrap">
        <h1><?php esc_html_e( 'Prompt Templates', 'aigude-tools' ); ?></h1>

        <h2><?php esc_html_e( 'Existing Templates', 'aigude-tools' ); ?></h2>
        <table class="widefat">
            <thead>
            <tr>
                <th><?php esc_html_e('Title', 'aigude-tools'); ?></th>
                <th><?php esc_html_e('Prompt', 'aigude-tools'); ?></th>
                <th><?php esc_html_e('Default', 'aigude-tools'); ?></th>
                <th><?php esc_html_e('Actions', 'aigude-tools'); ?></th>
            </tr>
            </thead>
            <tbody>
            <?php if ($templates) : foreach ($templates as $i => $tpl) :
                $is_default = !empty($tpl['id']) && $tpl['id'] === $default_id;
                $base = menu_page_url('aigude-tools-templates', false);
                $edit_url = wp_nonce_url(add_query_arg(['action'=>'edit','tpl_index'=>$i], $base), 'aigude_tpl_action', 'aigude_tpl_nonce');
                $del_url  = wp_nonce_url(add_query_arg(['action'=>'delete','tpl_index'=>$i], $base), 'aigude_tpl_action', 'aigude_tpl_nonce');
                $def_url  = wp_nonce_url(add_query_arg(['action'=>'set_default','tpl_index'=>$i], $base), 'aigude_tpl_action', 'aigude_tpl_nonce');
                ?>
                <tr>
                    <td><?php echo esc_html($tpl['title']); ?></td>
                    <td><?php echo esc_html($tpl['prompt']); ?></td>
                    <td>
                        <?php if ($is_default): ?>
                            <span class="dashicons dashicons-star-filled" title="<?php esc_attr_e('Default', 'aigude-tools'); ?>"></span>
                        <?php else: ?>
                            <a href="<?php echo esc_url($def_url); ?>"><?php esc_html_e('Make default', 'aigude-tools'); ?></a>
                        <?php endif; ?>
                    </td>
                    <td>
                        <a href="<?php echo esc_url($edit_url); ?>"><?php esc_html_e('Edit', 'aigude-tools'); ?></a> |
                        <a href="<?php echo esc_url($del_url); ?>" onclick="return confirm('<?php esc_attr_e('Really delete?', 'aigude-tools'); ?>');"><?php esc_html_e('Delete', 'aigude-tools'); ?></a>
                    </td>
                </tr>
            <?php endforeach; else: ?>
                <tr><td colspan="4"><?php esc_html_e('No templates added.', 'aigude-tools'); ?></td></tr>
            <?php endif; ?>
            </tbody>
        </table>

        <h2>
            <?php echo $edit_index!==null ? esc_html__( 'Edit Template', 'aigude-tools' ) : esc_html__( 'Add New Template', 'aigude-tools' ); ?>
            <span class="dashicons dashicons-info"
                  style="vertical-align:middle; cursor:help;"
                  title="<?php esc_attr_e( 'Prompts can be written in any language, but they work best when you define both the Prompt Language and the Placeholders Language.', 'aigude-tools' ); ?>">
            </span>
        </h2>

        <form method="post">
            <?php if ($edit_index!==null): ?>
                <input type="hidden" name="ai_tpl_index" value="<?php echo esc_attr($edit_index); ?>">
                <input type="hidden" name="ai_tpl_id" value="<?php echo esc_attr($templates[$edit_index]['id'] ?? ''); ?>">
            <?php endif; ?>

            <?php wp_nonce_field('aigude_tpl_action','aigude_tpl_nonce'); ?>
            <table class="form-table">
                <tr>
                    <th>
                        <label for="ai_tpl_is_default">
                            <?php esc_html_e( 'Default', 'aigude-tools' ); ?>
                        </label>
                    </th>
                    <td>
                        <?php
                        $editing_id = $edit_index !== null ? ( $templates[ $edit_index ]['id'] ?? '' ) : '';
                        ?>
                        <label>
                            <input type="checkbox"
                                   name="ai_tpl_is_default"
                                   id="ai_tpl_is_default"
                                    <?php checked( $editing_id, $default_id ); ?>>
                            <?php esc_html_e( 'Make this the default template', 'aigude-tools' ); ?>
                        </label>
                    </td>
                </tr>

                <tr>
                    <th><label for="ai_tpl_title"><?php esc_html_e( 'Title', 'aigude-tools' ); ?></label></th>
                    <td><input name="ai_tpl_title" id="ai_tpl_title" type="text" class="regular-text"
                               value="<?php echo esc_attr($edit_title); ?>"></td>
                </tr>
                <tr>
                    <th><label for="ai_tpl_prompt"><?php esc_html_e( 'Prompt', 'aigude-tools' ); ?></label></th>
                    <td>
                        <textarea name="ai_tpl_prompt" id="ai_tpl_prompt" class="large-text" rows="5"><?php echo esc_textarea($edit_prompt); ?></textarea>
                    </td>
                </tr>

                <tr>
                    <th><label for="ai_tpl_prompt_lang"><?php esc_html_e('Prompt Language', 'aigude-tools'); ?></label></th>
                    <td>
                        <select name="ai_tpl_prompt_lang" id="ai_tpl_prompt_lang">
                            <?php
                            $current = $edit_index !== null ? ($templates[$edit_index]['prompt_lang'] ?? 'auto') : 'auto';
                            // Auto-detect option
                            printf('<option value="auto"%s>%s</option>', selected($current, 'auto', false), esc_html__('Auto-detect', 'aigude-tools'));

                            $all = class_exists('AIGUDE_Tools_Plugin') ? AIGUDE_Tools_Plugin::get_deepl_languages() : [];
                            $rec = class_exists('AIGUDE_Tools_Plugin') ? AIGUDE_Tools_Plugin::get_recent_langs('prompt') : [];
                            // Render recents first (if any)
                            if (!empty($rec)) {
                                echo '<optgroup label="' . esc_attr__('Recent', 'aigude-tools') . '">';
                                foreach ($rec as $code) {
                                    if (!isset($all[$code])) continue;
                                    printf('<option value="%s"%s>%s</option>', esc_attr($code), selected($current, $code, false), esc_html($all[$code]));
                                }
                                echo '</optgroup>';
                            }
                            // All languages
                            echo '<optgroup label="' . esc_attr__('All languages', 'aigude-tools') . '">';
                            foreach ($all as $code => $label) {
                                printf('<option value="%s"%s>%s</option>', esc_attr($code), selected($current, $code, false), esc_html($label));
                            }
                            echo '</optgroup>';
                            ?>
                        </select>
                    </td>
                </tr>

                <tr>
                    <th><label for="ai_tpl_src_lang"><?php esc_html_e('Placeholders Language', 'aigude-tools'); ?></label></th>
                    <td>
                        <select name="ai_tpl_src_lang" id="ai_tpl_src_lang">
                            <?php
                            $current = $edit_index !== null ? ($templates[$edit_index]['src_lang'] ?? 'auto') : 'auto';
                            // Auto-detect
                            printf('<option value="auto"%s>%s</option>', selected($current, 'auto', false), esc_html__('Auto-detect', 'aigude-tools'));

                            $all = class_exists('AIGUDE_Tools_Plugin') ? AIGUDE_Tools_Plugin::get_deepl_languages() : [];
                            $rec = class_exists('AIGUDE_Tools_Plugin') ? AIGUDE_Tools_Plugin::get_recent_langs('placeholder') : [];
                            if (!empty($rec)) {
                                echo '<optgroup label="' . esc_attr__('Recent', 'aigude-tools') . '">';
                                foreach ($rec as $code) {
                                    if (!isset($all[$code])) continue;
                                    printf('<option value="%s"%s>%s</option>', esc_attr($code), selected($current, $code, false), esc_html($all[$code]));
                                }
                                echo '</optgroup>';
                            }
                            echo '<optgroup label="' . esc_attr__('All languages', 'aigude-tools') . '">';
                            foreach ($all as $code => $label) {
                                printf('<option value="%s"%s>%s</option>', esc_attr($code), selected($current, $code, false), esc_html($label));
                            }
                            echo '</optgroup>';
                            ?>
                        </select>
                        <p class="description" style="margin-top:6px;">
                            <?php esc_html_e( 'Available placeholders:', 'aigude-tools' ); ?>
                            <code>%filename%</code>, <code>%filename_no_ext%</code>, <code>%title%</code>, <code>%current-alt%</code>,
                            <code>%caption%</code>, <code>%description%</code>, <code>%mime%</code>, <code>%width%</code>, <code>%height%</code>.
                            <br>
                            <?php /* translators: %filename_no_ext% is the filename without extension. Example shows automatic quoting of text placeholders. */ ?>
                            <?php esc_html_e( 'Text placeholders are automatically quoted (e.g. %filename_no_ext% → "car-photo-123").', 'aigude-tools' ); ?><br>
                            <?php /* translators: %width% and %height% are numeric image dimensions; numeric placeholders are not quoted. */?>
                            <?php esc_html_e( 'Numeric placeholders like %width% and %height% are not quoted (e.g. → 1920).', 'aigude-tools' ); ?><br>
                            <?php esc_html_e( 'Modifiers: add "|q" to force quotes, or "|raw" to remove quotes.', 'aigude-tools' ); ?><br>
                            <?php esc_html_e( 'Unknown placeholders are left unchanged. Empty values become blank.', 'aigude-tools' ); ?><br>
                            <strong><?php esc_html_e( 'Example:', 'aigude-tools' ); ?></strong>
                            <?php
                            // Build the localized example line with ordered placeholders.
                            // translators: 1: %filename_no_ext|raw% token, 2: %width% token, 3: %height% token. Keep the tokens exactly as shown (including % signs).
                            $example_tpl = esc_html__( 'Describe %1$s (%2$ sx%3$s)', 'aigude-tools' );

                            // Echo the code example with the literal placeholder tokens.
                            echo '<code>' .
                                    esc_html(
                                            sprintf( $example_tpl, '%filename_no_ext|raw%', '%width%', '%height%' )
                                    ) .
                                    '</code>';
                            ?>
                            → <?php esc_html_e( 'Describe car-photo-123 (1920x1080)', 'aigude-tools' ); ?>
                        </p>


                    </td>
                </tr>
            </table>
            <p>
                <button name="ai_tpl_submit" class="button button-primary">
                    <?php echo $edit_index!==null? esc_html__( 'Update', 'aigude-tools' ) : esc_html__( 'Save Template', 'aigude-tools' ); ?>
                </button>
                <?php if ( $edit_index!==null ): ?>
                    <a href="<?php echo esc_url(menu_page_url('aigude-tools-templates', false)); ?>" class="button"><?php esc_html_e( 'Cancel', 'aigude-tools' ); ?></a>
                <?php endif; ?>
            </p>
        </form>
    </div>
    <?php
}
