<?php
if (! defined('ABSPATH')) {
    exit;
}

/**
 * Allowed server “types” for the dropdown.
 */
function aigude_get_allowed_server_types()
{
    return array(
            'AiGude',
    );
}

/**
 * Displays the server settings page and processes Add/Edit/Delete/Make default.
 */
function aigude_server_settings_page()
{
    // Capability fence
    if (! current_user_can('manage_options')) {
        wp_die(esc_html__('Insufficient permissions.', 'aigude-tools'), 403);
    }

    // Constants / keys
    $option_key        = 'aigude_alt_servers';
    $nonce_action      = 'aigude_server_save';
    $nonce_name        = 'aigude_server_nonce';
    // Translation provider settings are deprecated; keep key placeholders for back-compat.
    $provider_option_key   = 'aigude_translation_provider';
    $provider_nonce_action = 'aigude_translation_provider_save';
    $provider_nonce_name   = 'aigude_translation_provider_nonce';
    // Nonces for GET-driven views/actions
    $nonce_edit_action = 'aigude_server_edit';
    $nonce_edit_name   = 'nonce_edit';
    $nonce_add_action  = 'aigude_server_add';
    $nonce_add_name    = 'nonce_add';

    $messages_server   = 'aigude_server_messages';
    $messages_provider = 'aigude_provider_messages';

    // Load saved servers (array of assoc arrays)
    $servers = get_option($option_key, array());
    if (! is_array($servers)) {
        $servers = array();
    }
    $allowed_server_types = aigude_get_allowed_server_types();
    $default_server_type = $allowed_server_types[0] ?? 'AiGude';

    // Request params
    $action = isset($_REQUEST['action']) ? sanitize_key(wp_unslash($_REQUEST['action'])) : '';
    // Do not trust index globally; read only within nonce-verified branches or view rendering.
    $index  = -1;

    // --- Helper: enforce a single default -----------------------------------
    $set_single_default = static function (array &$list, int $default_idx) {
        foreach ($list as $i => $srv) {
            $list[ $i ]['is_default'] = ($i === $default_idx) ? 1 : 0;
        }
    };

    // --- Handle “make_default” from overview --------------------------------
    if ('make_default' === $action) {
        // Verify nonce sent via query arg; dies on failure.
        check_admin_referer('aigude_server_make_default', 'nonce_set_default');
        $index = isset($_GET['index']) ? (int) intval(wp_unslash($_GET['index'])) : -1;
        if ($index >= 0 && isset($servers[ $index ])) {
            $set_single_default($servers, $index);
            update_option($option_key, $servers);
            add_settings_error($messages_server, 'made_default', __('Default server updated.', 'aigude-tools'), 'updated');
            $action = '';
            $index  = -1;
        } else {
            add_settings_error($messages_server, 'err', __('Invalid index while setting default.', 'aigude-tools'), 'error');
        }
    }

    // --- Handle Add/Edit submit ---------------------------------------------
    if (isset($_POST['submit_ai_server'])) {

        // Verify form nonce; dies on failure.
        check_admin_referer($nonce_action, $nonce_name);

        // Capability fence
        if (! current_user_can('manage_options')) {
            wp_die(esc_html__('Insufficient permissions.', 'aigude-tools'), 403);
        }

        // Inputs
        $subaction   = isset($_POST['ai_action']) ? sanitize_key(wp_unslash($_POST['ai_action'])) : 'add';
        $server_type = isset($_POST['server']) ? sanitize_text_field(wp_unslash($_POST['server'])) : '';
        $name        = isset($_POST['name']) ? sanitize_text_field(wp_unslash($_POST['name'])) : '';
        $api_key     = isset($_POST['api_key']) ? sanitize_text_field(wp_unslash($_POST['api_key'])) : '';
        $enabled     = ! empty($_POST['enabled']) ? 1 : 0;
        $is_default  = ! empty($_POST['is_default']) ? 1 : 0;

        // Validate
        $errors = array();
        if ('' === $name) {
            $errors[] = __('Name cannot be empty.', 'aigude-tools');
        }
        if ('' === $api_key) {
            $errors[] = __('API Key cannot be empty.', 'aigude-tools');
        }
        if (! in_array($server_type, aigude_get_allowed_server_types(), true)) {
            $errors[] = __('Invalid server type.', 'aigude-tools');
        }

        if (empty($errors)) {
            if ('edit' === $subaction) {
                $edit_index = isset($_POST['edit_index']) ? (int) intval(wp_unslash($_POST['edit_index'])) : -1;
                if (isset($servers[ $edit_index ])) {
                    $servers[ $edit_index ] = array(
                            'server'     => $server_type,
                            'name'       => $name,
                            'api_key'    => $api_key,
                            'enabled'    => $enabled,
                            'is_default' => $is_default ? 1 : (int) ($servers[ $edit_index ]['is_default'] ?? 0),
                    );
                    if ($is_default) {
                        $set_single_default($servers, $edit_index);
                    }
                    add_settings_error($messages_server, 'updated', __('Server successfully updated.', 'aigude-tools'), 'updated');
                } else {
                    add_settings_error($messages_server, 'err', __('Invalid index while editing.', 'aigude-tools'), 'error');
                }
            } else {
                // Add
                $servers[] = array(
                        'server'     => $server_type,
                        'name'       => $name,
                        'api_key'    => $api_key,
                        'enabled'    => $enabled,
                        'is_default' => $is_default ? 1 : 0,
                );
                if ($is_default) {
                    $set_single_default($servers, count($servers) - 1);
                }
                add_settings_error($messages_server, 'added', __('New server added.', 'aigude-tools'), 'updated');
            }

            update_option($option_key, $servers);
            $action = '';
            $index  = -1;

        } else {
            foreach ($errors as $err) {
                add_settings_error($messages_server, 'err', $err, 'error');
            }
        }
    }

    // --- Handle Delete from overview ----------------------------------------
    if ('delete' === $action) {
        // Verify nonce sent via query arg; dies on failure.
        check_admin_referer('aigude_server_delete', 'nonce_delete');
        $index = isset($_GET['index']) ? (int) intval(wp_unslash($_GET['index'])) : -1;
        if ($index >= 0 && isset($servers[ $index ])) {
            unset($servers[ $index ]);
            $servers = array_values($servers);

            // If no servers left, nothing to normalize. If there are, ensure at most one default remains.
            $found_default = false;
            foreach ($servers as $i => $srv) {
                if (! empty($srv['is_default'])) {
                    if ($found_default) {
                        $servers[ $i ]['is_default'] = 0;
                    }
                    $found_default = true;
                }
            }

            update_option($option_key, $servers);
            add_settings_error($messages_server, 'deleted', __('Server deleted.', 'aigude-tools'), 'updated');
            $action = '';
            $index  = -1;
        } else {
            add_settings_error($messages_server, 'err', __('Invalid index for delete.', 'aigude-tools'), 'error');
        }
    }

    // Translation provider selection is deprecated; no-op for submitted forms.
    if (isset($_POST['submit_translation_provider'])) {
        check_admin_referer($provider_nonce_action, $provider_nonce_name);
        add_settings_error($messages_provider, 'provider_deprecated', __('Translation provider settings are no longer used.', 'aigude-tools'), 'updated');
    }

    // --- View ----------------------------------------------------------------
    $active_tab = 'connections';
    $page_url = menu_page_url('aigude-tools-settings', false);
    ?>
    <div class="wrap aigude-settings-wrap" data-active-tab="<?php echo esc_attr($active_tab); ?>">
        <h1><?php esc_html_e('Settings', 'aigude-tools'); ?></h1>
        <style>
            .aigude-settings-card {
                background:#fff;
                border:1px solid #dcdcde;
                box-shadow:0 1px 2px rgba(0,0,0,0.04);
                padding:24px;
                border-radius:6px;
                margin-bottom:32px;
            }
            .aigude-settings-card h2 {
                margin-top:0;
            }
            .aigude-section-header {
                display:flex;
                flex-wrap:wrap;
                justify-content:space-between;
                gap:16px;
                align-items:center;
            }
            .aigude-section-header .description {
                margin:0;
            }
            .aigude-provider-lang-badge {
                display:inline-flex;
                align-items:center;
                gap:6px;
                padding:4px 8px;
                border-radius:4px;
                font-weight:600;
                font-size:13px;
            }
            .aigude-provider-lang-badge.supported {
                background:#edf7f0;
                color:#14532d;
            }
            .aigude-provider-lang-badge.missing {
                background:#fef2f2;
                color:#7f1d1d;
            }
            .aigude-language-details {
                margin-top:12px;
            }
            .aigude-language-details summary {
                cursor:pointer;
                display:flex;
                align-items:center;
                gap:8px;
                padding:8px 10px;
                border:1px solid #ccd0d4;
                border-radius:4px;
                background:#f6f7f7;
                font-weight:600;
            }
            .aigude-language-details summary:focus {
                outline:2px solid #2271b1;
                outline-offset:1px;
            }
            .aigude-language-details summary .dashicons {
                transition:transform 0.2s ease;
            }
            .aigude-language-details[open] summary .dashicons {
                transform:rotate(90deg);
            }
            .aigude-language-details-hint {
                margin-left:auto;
                font-size:12px;
                font-weight:400;
                color:#50575e;
            }
            .aigude-language-select-wrapper select {
                width:100%;
                max-width:100%;
            }
            .aigude-language-status {
                display:block;
                min-height:16px;
                margin-top:6px;
                font-size:12px;
                color:#50575e;
            }
            .aigude-tab-panel {
                display:none;
            }
            .aigude-tab-panel.active {
                display:block;
            }
        </style>
        <h2 class="nav-tab-wrapper aigude-settings-tabs">
            <a href="<?php echo esc_url(add_query_arg('tab', 'connections', $page_url)); ?>" class="nav-tab nav-tab-active">
                <?php esc_html_e('API Connections', 'aigude-tools'); ?>
            </a>
        </h2>

        <div class="aigude-tab-panel <?php echo 'connections' === $active_tab ? 'active' : ''; ?>" data-tab="connections">
            <section id="aigude-connections" class="aigude-settings-card">
                <?php if ('add' !== $action && 'edit' !== $action) : ?>
                    <div class="aigude-section-header">
                        <p class="description" style="margin:0;">
                            <?php esc_html_e("Don't have an API key?", 'aigude-tools'); ?>
                            <a href="https://aigude.io/" class="button button-secondary" target="_blank" rel="noopener noreferrer">
                                <?php esc_html_e('Get API key at AiGude.io', 'aigude-tools'); ?>
                            </a>
                        </p>
                    </div>
                <?php endif; ?>
                <?php settings_errors($messages_server); ?>

                <?php
                if ('edit' === $action) {
                    check_admin_referer($nonce_edit_action, $nonce_edit_name);
                    $index = isset($_GET['index']) ? (int) intval(wp_unslash($_GET['index'])) : -1;
                }
                ?>
                <?php if ('edit' === $action && $index >= 0 && isset($servers[ $index ])) :
                    $srv = $servers[ $index ];
                    ?>
                    <h2><?php esc_html_e('Edit Connection', 'aigude-tools'); ?></h2>
                    <form method="post" novalidate>
                        <?php wp_nonce_field($nonce_action, $nonce_name); ?>
                        <input type="hidden" name="submit_ai_server" value="1">
                        <input type="hidden" name="ai_action" value="edit">
                        <input type="hidden" name="edit_index" value="<?php echo esc_attr($index); ?>">
                        <input type="hidden" name="tab" value="connections">

                        <table class="form-table" role="presentation">
                            <input type="hidden" id="server" name="server" value="<?php echo esc_attr($srv['server'] ?? $default_server_type); ?>">

                            <tr>
                                <th scope="row"><label for="name"><?php esc_html_e('Name', 'aigude-tools'); ?></label></th>
                                <td><input type="text" id="name" name="name" value="<?php echo esc_attr($srv['name']); ?>" class="regular-text"></td>
                            </tr>

                            <tr>
                                <th scope="row"><label for="api_key"><?php esc_html_e('API Key', 'aigude-tools'); ?></label></th>
                                <td>
                                    <div class="ai-key-field" style="display:flex;gap:8px;align-items:center;">
                                        <input
                                                type="password"
                                                id="api_key"
                                                name="api_key"
                                                value="<?php echo isset($srv) ? esc_attr($srv['api_key']) : ''; ?>"
                                                class="regular-text"
                                                autocomplete="off"
                                        >
                                        <button type="button"
                                                class="button button-secondary api-key-visibility"
                                                data-target="#api_key"
                                                aria-pressed="false"
                                                aria-controls="api_key">
                                            <?php esc_html_e('Show', 'aigude-tools'); ?>
                                        </button>
                                        <button type="button"
                                                class="button button-secondary api-key-copy"
                                                data-target="#api_key">
                                            <?php esc_html_e('Copy', 'aigude-tools'); ?>
                                        </button>
                                    </div>
                                </td>
                            </tr>

                            <tr>
                                <th scope="row"><?php esc_html_e('Enabled', 'aigude-tools'); ?></th>
                                <td><label><input type="checkbox" name="enabled" <?php checked(1, ! empty($srv['enabled'])); ?>> <?php esc_html_e('Activate', 'aigude-tools'); ?></label></td>
                            </tr>

                            <tr>
                                <th scope="row"><?php esc_html_e('Default', 'aigude-tools'); ?></th>
                                <td><label><input type="checkbox" name="is_default" <?php checked(1, ! empty($srv['is_default'])); ?>> <?php esc_html_e('Make this the default server', 'aigude-tools'); ?></label></td>
                            </tr>
                        </table>

                        <?php submit_button(__('Save', 'aigude-tools')); ?>
                        <a class="button" href="<?php echo esc_url(add_query_arg('tab', 'connections', $page_url)); ?>"><?php esc_html_e('Cancel', 'aigude-tools'); ?></a>
                    </form>

                <?php else : ?>

                    <?php if ('add' !== $action) : ?>
                        <p>
                            <a href="<?php echo esc_url(wp_nonce_url(add_query_arg(array( 'action' => 'add', 'tab' => 'connections' ), $page_url), $nonce_add_action, $nonce_add_name)); ?>" class="button button-primary">
                                <?php esc_html_e('Add New', 'aigude-tools'); ?>
                            </a>
                        </p>
                    <?php endif; ?>

                    <?php if ('add' === $action) : ?>
                        <?php
                        check_admin_referer($nonce_add_action, $nonce_add_name);
                        ?>
                        <h2><?php esc_html_e('Add Connection', 'aigude-tools'); ?></h2>
                        <form method="post" novalidate>
                            <?php wp_nonce_field($nonce_action, $nonce_name); ?>
                            <input type="hidden" name="submit_ai_server" value="1">
                            <input type="hidden" name="ai_action" value="add">
                            <input type="hidden" name="tab" value="connections">

                            <table class="form-table" role="presentation">
                                <input type="hidden" id="server" name="server" value="<?php echo esc_attr($default_server_type); ?>">

                                <tr>
                                    <th scope="row"><label for="name"><?php esc_html_e('Name', 'aigude-tools'); ?></label></th>
                                    <td><input type="text" id="name" name="name" value="" class="regular-text"></td>
                                </tr>

                                <tr>
                                    <th scope="row"><label for="api_key"><?php esc_html_e('API Key', 'aigude-tools'); ?></label></th>
                                    <td><input type="password" id="api_key" name="api_key" value="" class="regular-text"></td>
                                </tr>

                                <tr>
                                    <th scope="row"><?php esc_html_e('Enabled', 'aigude-tools'); ?></th>
                                    <td><label><input type="checkbox" name="enabled" checked> <?php esc_html_e('Activate', 'aigude-tools'); ?></label></td>
                                </tr>

                                <tr>
                                    <th scope="row"><?php esc_html_e('Default', 'aigude-tools'); ?></th>
                                    <td><label><input type="checkbox" name="is_default"> <?php esc_html_e('Make this the default server', 'aigude-tools'); ?></label></td>
                                </tr>
                            </table>

                            <?php submit_button(__('Add', 'aigude-tools')); ?>
                            <a class="button" href="<?php echo esc_url(add_query_arg('tab', 'connections', $page_url)); ?>"><?php esc_html_e('Cancel', 'aigude-tools'); ?></a>
                        </form>

                    <?php else : // Overview table?>
                        <?php if (empty($servers)) : ?>
                            <p><?php echo esc_html__('No servers configured yet.', 'aigude-tools'); ?></p>
                        <?php else : ?>
                            <table class="widefat fixed striped">
                                <thead>
                                <tr>
                                    <th><?php esc_html_e('Name', 'aigude-tools'); ?></th>
                                    <th><?php esc_html_e('API Key', 'aigude-tools'); ?></th>
                                    <th><?php esc_html_e('Enabled', 'aigude-tools'); ?></th>
                                    <th><?php esc_html_e('Default', 'aigude-tools'); ?></th>
                                    <th><?php esc_html_e('Remaining credits', 'aigude-tools'); ?></th>
                                    <th><?php esc_html_e('Actions', 'aigude-tools'); ?></th>
                                </tr>
                                </thead>
                                <tbody>
                                <?php foreach ($servers as $i => $srv) :
                                    $masked = str_repeat('*', max(4, strlen((string) ($srv['api_key'] ?? ''))));
                                    $enabled_icon = ! empty($srv['enabled']) ? '<span style="color:green;">✔</span>' : '<span style="color:red;">✖</span>';
                                    $is_default   = ! empty($srv['is_default']);
                                    $make_def_url = wp_nonce_url(
                                        add_query_arg(array( 'action' => 'make_default', 'index' => $i, 'tab' => 'connections' ), $page_url),
                                        'aigude_server_make_default',
                                        'nonce_set_default'
                                    );
                                    $del_url = wp_nonce_url(
                                        add_query_arg(array( 'action' => 'delete', 'index' => $i, 'tab' => 'connections' ), $page_url),
                                        'aigude_server_delete',
                                        'nonce_delete'
                                    );
                                    $edit_url = wp_nonce_url(
                                        add_query_arg(array( 'action' => 'edit', 'index' => $i, 'tab' => 'connections' ), $page_url),
                                        $nonce_edit_action,
                                        $nonce_edit_name
                                    );
                                    ?>
                                    <tr>
                                        <td><?php echo esc_html($srv['name']); ?></td>

                                        <td>
                                            <span class="api-key-mask" data-full="<?php echo esc_attr($srv['api_key']); ?>">
                                                <?php echo esc_html($masked); ?>
                                            </span>
                                            <?php if (! empty($srv['api_key'])) : ?>
                                                <a href="#" class="toggle-api-key"><?php esc_html_e('Show', 'aigude-tools'); ?></a>
                                            <?php endif; ?>
                                        </td>

                                        <td><?php echo wp_kses_post($enabled_icon); ?></td>
                                        <td>
                                            <?php if ($is_default) : ?>
                                                <strong>★ <?php esc_html_e('Default', 'aigude-tools'); ?></strong>
                                            <?php else : ?>
                                                <a href="<?php echo esc_url($make_def_url); ?>">
                                                    <?php esc_html_e('Make default', 'aigude-tools'); ?>
                                                </a>
                                            <?php endif; ?>
                                        </td>
                                        <td>
                                            <?php if (! empty($srv['enabled']) && ! empty($srv['api_key'])) : ?>
                                                <span class="ai-server-credits" data-index="<?php echo esc_attr($i); ?>">–</span>
                                            <?php else : ?>
                                                <span class="ai-server-credits" data-index="<?php echo esc_attr($i); ?>">
                                                    <?php esc_html_e('Disabled', 'aigude-tools'); ?>
                                                </span>
                                            <?php endif; ?>
                                        </td>
                                        <td>
                                            <a href="<?php echo esc_url($edit_url); ?>"><?php esc_html_e('Edit', 'aigude-tools'); ?></a> |
                                            <a href="<?php echo esc_url($del_url); ?>"
                                               onclick="return confirm('<?php echo esc_attr__('Do you really want to delete this server?', 'aigude-tools'); ?>');">
                                                <?php esc_html_e('Delete', 'aigude-tools'); ?>
                                            </a>
                                        </td>
                                    </tr>
                                <?php endforeach; ?>
                                </tbody>
                            </table>
                        <?php endif; ?>
                    <?php endif; ?>
                <?php endif; ?>
            </section>
        </div>

        <?php
        $provider_meta_all = class_exists('AIGUDE_Tools_Plugin') ? AIGUDE_Tools_Plugin::get_translation_providers_metadata() : [];
        $eu_only_providers = class_exists('AIGUDE_Tools_Plugin') ? AIGUDE_Tools_Plugin::eu_only_providers_enabled() : false;
        $provider_meta = class_exists('AIGUDE_Tools_Plugin') ? AIGUDE_Tools_Plugin::filter_providers_by_region($provider_meta_all, $eu_only_providers) : [];
        $current_provider = class_exists('AIGUDE_Tools_Plugin') ? AIGUDE_Tools_Plugin::get_translation_provider() : 'deepl';
        $saved_target_language = get_option('aigude_target_language', 'default');
        $language_nonce = wp_create_nonce(AIGUDE_Tools_Plugin::NONCE_ACTION);
        /* translators: %s = human-readable language label, e.g. "German (Germany)". */
        $language_summary_tpl_user    = __('Current default: %s', 'aigude-tools');
        /* translators: %s = human-readable language label that is no longer supported. */
        $language_summary_tpl_missing = __('Current default (%s) is unavailable. Pick another language.', 'aigude-tools');
        /* translators: %s = site language label, e.g. "English (US)". */
        $language_summary_tpl_site    = __('Following site language (%s).', 'aigude-tools');
        ?>

        <div class="aigude-tab-panel <?php echo 'providers' === $active_tab ? 'active' : ''; ?>" data-tab="providers">
            <section id="aigude-providers" class="aigude-settings-card">
                <div class="aigude-section-header">
                    <p class="description" style="margin:0;">
                        <?php esc_html_e('Select the translation provider for AI-generated alt texts. The provider determines the available target languages.', 'aigude-tools'); ?>
                    </p>
                </div>
                <?php settings_errors($messages_provider); ?>

                <form method="post" class="aigude-provider-form" style="max-width:500px;margin-bottom:24px;">
                    <?php wp_nonce_field($provider_nonce_action, $provider_nonce_name); ?>
                    <input type="hidden" name="submit_translation_provider" value="1">
                    <input type="hidden" name="tab" value="providers">
                    <label for="translation-provider-select" class="screen-reader-text">
                        <?php esc_html_e('Translation provider', 'aigude-tools'); ?>
                    </label>
                    <select id="translation-provider-select" name="translation_provider" class="regular-text" style="min-width:220px;">
                        <?php foreach ((array) $provider_meta as $slug => $info) :
                            $label = class_exists('AIGUDE_Tools_Plugin') ? AIGUDE_Tools_Plugin::get_translation_provider_label($slug) : ucfirst($slug);
                            ?>
                            <option value="<?php echo esc_attr($slug); ?>" <?php selected($current_provider, $slug); ?>>
                                <?php echo esc_html($label); ?>
                            </option>
                        <?php endforeach; ?>
                    </select>
                    <label style="display:flex;gap:8px;margin:8px 0 0;align-items:center;">
                        <input type="checkbox"
                               name="translation_provider_eu_only"
                               id="translation_provider_eu_only"
                               value="1"
                            <?php checked($eu_only_providers); ?>>
                        <?php esc_html_e('Show only EU-based translation providers', 'aigude-tools'); ?>
                    </label>
                </form>
                <script type="text/javascript">
                    document.addEventListener('DOMContentLoaded', function () {
                        var form   = document.querySelector('.aigude-provider-form');
                        var toggle = document.getElementById('translation_provider_eu_only');
                        var select = document.getElementById('translation-provider-select');

                        if (!form) {
                            return;
                        }

                        var submitProviderForm = function () {
                            if (typeof form.requestSubmit === 'function') {
                                form.requestSubmit();
                            } else {
                                form.submit();
                            }
                        };

                        if (toggle) {
                            toggle.addEventListener('change', submitProviderForm);
                        }

                        if (select) {
                            select.addEventListener('change', submitProviderForm);
                        }
                    });
                </script>

                <?php if (! empty($provider_meta)) : ?>
                    <div class="aigude-provider-cards" style="display:grid;grid-template-columns:repeat(auto-fit,minmax(260px,1fr));gap:16px;">
                        <?php foreach ($provider_meta as $slug => $info) :
                            $label = class_exists('AIGUDE_Tools_Plugin') ? AIGUDE_Tools_Plugin::get_translation_provider_label($slug) : ucfirst($slug);
                            $languages = class_exists('AIGUDE_Tools_Plugin') ? AIGUDE_Tools_Plugin::get_translation_languages($slug) : [];
                            $notes = isset($info['notes']) ? (string) $info['notes'] : '';
                            $site_lang = class_exists('AIGUDE_Tools_Plugin') ? AIGUDE_Tools_Plugin::describe_site_language($slug) : ['label' => strtoupper(substr(get_locale(), 0, 2)), 'supported' => true, 'code' => '', 'source' => 'site'];
                            $preference = null;
                            if ($slug === $current_provider && class_exists('AIGUDE_Tools_Plugin')) {
                                $preference = AIGUDE_Tools_Plugin::describe_language_preference($slug);
                            }
                            $site_supported = ! empty($site_lang['supported']);
                            $badge_class = $site_supported ? 'supported' : 'missing';
                            $badge_icon  = $site_supported ? 'dashicons-yes-alt' : 'dashicons-warning';
                            $select_id   = 'aigude-provider-lang-' . sanitize_html_class($slug);
                            $selected_lang = ($slug === $current_provider) ? $saved_target_language : 'default';
                            if ($selected_lang !== 'default' && ! isset($languages[$selected_lang])) {
                                $selected_lang = 'default';
                            }
                            $recent_langs_provider = class_exists('AIGUDE_Tools_Plugin') ? AIGUDE_Tools_Plugin::get_recent_langs('target', $slug) : [];
                            $recent_langs_provider = array_values(array_filter((array) $recent_langs_provider, static function ($code) use ($languages) {
                                return is_string($code) && isset($languages[$code]);
                            }));
                            $fallback_code = '';
                            if (!$site_supported) {
                                if (isset($languages['EN'])) {
                                    $fallback_code = 'EN';
                                } else {
                                    $resolved = AIGUDE_Tools_Plugin::resolve_target_lang_code('default', $slug);
                                    if ($resolved !== '' && isset($languages[$resolved])) {
                                        $fallback_code = $resolved;
                                    }
                                }
                            }
                            if ($selected_lang === 'default' && empty($recent_langs_provider)) {
                                $site_code = isset($site_lang['code']) ? (string) $site_lang['code'] : '';
                                if ($site_supported && $site_code !== '' && isset($languages[$site_code])) {
                                    $selected_lang = $site_code;
                                } elseif ($fallback_code !== '' && isset($languages[$fallback_code])) {
                                    $selected_lang = $fallback_code;
                                } elseif (!empty($languages)) {
                                    $first_code = array_key_first($languages);
                                    if ($first_code) {
                                        $selected_lang = $first_code;
                                    }
                                }
                            }
                            $summary_text = '';
                            if ($slug === $current_provider) {
                                if ($preference && $preference['source'] === 'user' && ! empty($preference['supported'])) {
                                    $summary_text = sprintf($language_summary_tpl_user, $preference['label']);
                                } elseif ($preference && $preference['source'] === 'user') {
                                    $summary_text = sprintf($language_summary_tpl_missing, $preference['label']);
                                } else {
                                    $summary_text = sprintf($language_summary_tpl_site, $site_lang['label']);
                                }
                            }
                            ?>
                            <div class="postbox" <?php if ($slug === $current_provider) : ?>style="border-color:#2271b1;"<?php endif; ?>>
                                <div class="postbox-header" style="display:flex;justify-content:space-between;align-items:center;">
                                    <h3 class="hndle" style="margin:0;"><?php echo esc_html($label); ?></h3>
                                    <?php if ($slug === $current_provider) : ?>
                                        <span class="dashicons dashicons-yes" aria-hidden="true" title="<?php esc_attr_e('Active provider', 'aigude-tools'); ?>"></span>
                                    <?php endif; ?>
                                </div>
                                <div class="inside">
                                    <p class="aigude-provider-lang-badge <?php echo esc_attr($badge_class); ?>">
                                        <span class="dashicons <?php echo esc_attr($badge_icon); ?>"></span>
                                        <?php if (! empty($site_lang['supported'])) : ?>
                                            <?php
                                            printf(
                                                /* translators: %s = site language label, e.g. "English (US)". */
                                                esc_html__('%s is supported for this site.', 'aigude-tools'),
                                                esc_html($site_lang['label'])
                                            );
                                            ?>
                                        <?php else : ?>
                                            <?php
                                            printf(
                                                /* translators: %s = site language label, e.g. "English (US)". */
                                                esc_html__('%s is not available for this provider.', 'aigude-tools'),
                                                esc_html($site_lang['label'])
                                            );
                                            ?>
                                        <?php endif; ?>
                                    </p>
                                    <p>
                                        <strong>
                                            <?php
                                            printf(
                                                /* translators: %d = number of languages supported by the provider. */
                                                esc_html__('%d supported languages', 'aigude-tools'),
                                                count($languages)
                                            );
                                            ?>
                                        </strong>
                                    </p>
                                    <?php if ($slug === $current_provider) : ?>
                                        <p
                                                class="description aigude-language-summary"
                                                data-current-tpl="<?php echo esc_attr($language_summary_tpl_user); ?>"
                                                data-missing-tpl="<?php echo esc_attr($language_summary_tpl_missing); ?>"
                                                data-site-tpl="<?php echo esc_attr($language_summary_tpl_site); ?>"
                                                data-site-label="<?php echo esc_attr($site_lang['label']); ?>"
                                        >
                                            <?php echo esc_html($summary_text); ?>
                                        </p>
                                    <?php endif; ?>
                            <details class="aigude-language-details">
                                <summary class="aigude-language-details-toggle">
                                    <span class="dashicons dashicons-list-view" aria-hidden="true"></span>
                                    <span><?php esc_html_e('Language details', 'aigude-tools'); ?></span>
                                    <span class="aigude-language-details-hint">
                                        <?php esc_html_e('Click to view the full list', 'aigude-tools'); ?>
                                    </span>
                                </summary>
                                <div class="aigude-language-select-wrapper">
                                            <label for="<?php echo esc_attr($select_id); ?>" style="font-weight:600;">
                                                <?php esc_html_e('Default alt text language', 'aigude-tools'); ?>
                                            </label>
                                            <select
                                                    id="<?php echo esc_attr($select_id); ?>"
                                                    class="aigude-language-select"
                                                    data-provider="<?php echo esc_attr($slug); ?>"
                                                    size="8"
                                                    <?php disabled($slug !== $current_provider); ?>
                                            >
                                                <?php if ($site_supported) : ?>
                                                    <option value="default" <?php selected($selected_lang, 'default'); ?>>
                                                        <?php
                                                        /* translators: %s = site language label, e.g. "English (US)". */
                                                        printf(esc_html__('System (%s)', 'aigude-tools'), esc_html($site_lang['label']));
                                                        ?>
                                                    </option>
                                                <?php endif; ?>
                                                <?php if (!empty($recent_langs_provider)) : ?>
                                                    <optgroup label="<?php esc_attr_e('Recent', 'aigude-tools'); ?>">
                                                        <?php foreach ($recent_langs_provider as $lang_code) : ?>
                                                            <option value="<?php echo esc_attr($lang_code); ?>" <?php selected($selected_lang, $lang_code); ?>>
                                                                <?php echo esc_html(sprintf('%s (%s)', $languages[$lang_code], $lang_code)); ?>
                                                            </option>
                                                        <?php endforeach; ?>
                                                    </optgroup>
                                                <?php endif; ?>
                                                <optgroup label="<?php esc_attr_e('All languages', 'aigude-tools'); ?>">
                                                    <?php foreach ($languages as $lang_code => $lang_label) : ?>
                                                        <option value="<?php echo esc_attr($lang_code); ?>" <?php selected($selected_lang, $lang_code); ?>>
                                                            <?php echo esc_html(sprintf('%s (%s)', $lang_label, $lang_code)); ?>
                                                        </option>
                                                    <?php endforeach; ?>
                                                </optgroup>
                                            </select>
                                            <span class="aigude-language-status" aria-live="polite"></span>
                                            <?php if ($slug !== $current_provider) : ?>
                                                <p class="description"><?php esc_html_e('Switch to this provider to edit the default language.', 'aigude-tools'); ?></p>
                                            <?php elseif (empty($site_lang['supported'])) : ?>
                                                <p class="description"><?php esc_html_e('Your site language is unavailable; "System" will fall back to the closest supported code.', 'aigude-tools'); ?></p>
                                            <?php endif; ?>
                                        </div>
                                    </details>
                                </div>
                            </div>
                        <?php endforeach; ?>
                    </div>
                <?php else : ?>
                    <p class="description"><?php esc_html_e('No translation provider metadata available. Add a server with a valid API key to load providers.', 'aigude-tools'); ?></p>
                <?php endif; ?>
            </section>
        </div>
        <script type="text/javascript">
            jQuery(function ($) {
                var ajaxUrl = window.ajaxurl || '<?php echo esc_js(admin_url('admin-ajax.php')); ?>';
                var recentLabel = '<?php echo esc_js(__('Recent', 'aigude-tools')); ?>';
                var updateLanguageSummary = function ($select) {
                    var $summary = $select.closest('.inside').find('.aigude-language-summary');
                    if (!$summary.length) {
                        return;
                    }
                    var currentTpl = $summary.data('currentTpl') || '';
                    var siteTpl = $summary.data('siteTpl') || '';
                    var siteLabel = $summary.data('siteLabel') || '';
                    var value = $select.val();
                    var label = $.trim($select.find('option:selected').text());
                    var text = '';

                    if (value === 'default') {
                        text = siteTpl ? siteTpl.replace('%s', siteLabel || label) : label;
                    } else if (currentTpl) {
                        text = currentTpl.replace('%s', label);
                    } else {
                        text = label;
                    }

                    $summary.text(text);
                };

                $('.aigude-language-select').on('change', function () {
                    var $select = $(this);
                    if ($select.is(':disabled')) {
                        return;
                    }
                    var $status = $select.closest('.aigude-language-select-wrapper').find('.aigude-language-status');
                    $status.text('<?php echo esc_js(__('Saving...', 'aigude-tools')); ?>');
                    var providerSlug = $select.data('provider') || '<?php echo esc_js($current_provider); ?>';
                    var selectedValue = $select.val();
                    var selectedText = $.trim($select.find('option:selected').text());

                    var updateRecents = function () {
                        if (!selectedValue || selectedValue === 'default') {
                            return;
                        }
                        var $recentGroup = $select.find('optgroup[label="' + recentLabel + '"]');
                        if (!$recentGroup.length) {
                            var $firstGroup = $select.find('optgroup').first();
                            $recentGroup = $('<optgroup>').attr('label', recentLabel);
                            if ($firstGroup.length) {
                                $recentGroup.insertBefore($firstGroup);
                            } else {
                                $select.append($recentGroup);
                            }
                        }
                        $recentGroup.find('option[value="' + selectedValue + '"]').remove();
                        if (selectedText) {
                            $recentGroup.prepend(
                                $('<option>').val(selectedValue).text(selectedText)
                            );
                        }
                        var opts = $recentGroup.find('option');
                        if (opts.length > 5) {
                            opts.slice(5).remove();
                        }
                    };

                    $.post(ajaxUrl, {
                        action: 'aigude_save_language',
                        lang: selectedValue,
                        provider: providerSlug,
                        _ajax_nonce: '<?php echo esc_js($language_nonce); ?>'
                    }).done(function (res) {
                        if (res && res.success) {
                            $status.text('<?php echo esc_js(__('Language saved.', 'aigude-tools')); ?>');
                            updateLanguageSummary($select);
                            updateRecents();
                        } else {
                            $status.text('<?php echo esc_js(__('Could not save language.', 'aigude-tools')); ?>');
                        }
                    }).fail(function () {
                        $status.text('<?php echo esc_js(__('Could not save language.', 'aigude-tools')); ?>');
                    }).always(function () {
                        setTimeout(function () {
                            $status.text('');
                        }, 4000);
                    });
                });
            });
        </script>
    </div>
    <?php
}
