<?php
if (! defined('ABSPATH')) {
    exit;
}

/**
 * Allowed server “types” for the dropdown.
 */
function aigude_get_allowed_server_types()
{
    return array(
            'AiGude',
    );
}

/**
 * Displays the server settings page and processes Add/Edit/Delete/Make default.
 */
function aigude_server_settings_page()
{
    // Capability fence
    if (! current_user_can('manage_options')) {
        wp_die(esc_html__('Insufficient permissions.', 'aigude-tools'), 403);
    }

    // Constants / keys
    $option_key        = 'aigude_alt_servers';
    $nonce_action      = 'aigude_server_save';
    $nonce_name        = 'aigude_server_nonce';
    // Nonces for GET-driven views/actions
    $nonce_edit_action = 'aigude_server_edit';
    $nonce_edit_name   = 'nonce_edit';
    $nonce_add_action  = 'aigude_server_add';
    $nonce_add_name    = 'nonce_add';

    // Load saved servers (array of assoc arrays)
    $servers = get_option($option_key, array());
    if (! is_array($servers)) {
        $servers = array();
    }

    // Request params
    $action = isset($_REQUEST['action']) ? sanitize_key(wp_unslash($_REQUEST['action'])) : '';
    // Do not trust index globally; read only within nonce-verified branches or view rendering.
    $index  = -1;

    // --- Helper: enforce a single default -----------------------------------
    $set_single_default = static function (array &$list, int $default_idx) {
        foreach ($list as $i => $srv) {
            $list[ $i ]['is_default'] = ($i === $default_idx) ? 1 : 0;
        }
    };

    // --- Handle “make_default” from overview --------------------------------
    if ('make_default' === $action) {
        // Verify nonce sent via query arg; dies on failure.
        check_admin_referer('aigude_server_make_default', 'nonce_set_default');
        $index = isset($_GET['index']) ? (int) intval(wp_unslash($_GET['index'])) : -1;
        if ($index >= 0 && isset($servers[ $index ])) {
            $set_single_default($servers, $index);
            update_option($option_key, $servers);
            add_settings_error('ai_alt_messages', 'made_default', __('Default server updated.', 'aigude-tools'), 'updated');
            $action = '';
            $index  = -1;
        } else {
            add_settings_error('ai_alt_messages', 'err', __('Invalid index while setting default.', 'aigude-tools'), 'error');
        }
    }

    // --- Handle Add/Edit submit ---------------------------------------------
    if (isset($_POST['submit_ai_server'])) {

        // Verify form nonce; dies on failure.
        check_admin_referer($nonce_action, $nonce_name);

        // Capability fence
        if (! current_user_can('manage_options')) {
            wp_die(esc_html__('Insufficient permissions.', 'aigude-tools'), 403);
        }

        // Inputs
        $subaction   = isset($_POST['ai_action']) ? sanitize_key(wp_unslash($_POST['ai_action'])) : 'add';
        $server_type = isset($_POST['server']) ? sanitize_text_field(wp_unslash($_POST['server'])) : '';
        $name        = isset($_POST['name']) ? sanitize_text_field(wp_unslash($_POST['name'])) : '';
        $api_key     = isset($_POST['api_key']) ? sanitize_text_field(wp_unslash($_POST['api_key'])) : '';
        $enabled     = ! empty($_POST['enabled']) ? 1 : 0;
        $is_default  = ! empty($_POST['is_default']) ? 1 : 0;

        // Validate
        $errors = array();
        if ('' === $name) {
            $errors[] = __('Name cannot be empty.', 'aigude-tools');
        }
        if ('' === $api_key) {
            $errors[] = __('API Key cannot be empty.', 'aigude-tools');
        }
        if (! in_array($server_type, aigude_get_allowed_server_types(), true)) {
            $errors[] = __('Invalid server type.', 'aigude-tools');
        }

        if (empty($errors)) {
            if ('edit' === $subaction) {
                $edit_index = isset($_POST['edit_index']) ? (int) intval(wp_unslash($_POST['edit_index'])) : -1;
                if (isset($servers[ $edit_index ])) {
                    $servers[ $edit_index ] = array(
                            'server'     => $server_type,
                            'name'       => $name,
                            'api_key'    => $api_key,
                            'enabled'    => $enabled,
                            'is_default' => $is_default ? 1 : (int) ($servers[ $edit_index ]['is_default'] ?? 0),
                    );
                    if ($is_default) {
                        $set_single_default($servers, $edit_index);
                    }
                    add_settings_error('ai_alt_messages', 'updated', __('Server successfully updated.', 'aigude-tools'), 'updated');
                } else {
                    add_settings_error('ai_alt_messages', 'err', __('Invalid index while editing.', 'aigude-tools'), 'error');
                }
            } else {
                // Add
                $servers[] = array(
                        'server'     => $server_type,
                        'name'       => $name,
                        'api_key'    => $api_key,
                        'enabled'    => $enabled,
                        'is_default' => $is_default ? 1 : 0,
                );
                if ($is_default) {
                    $set_single_default($servers, count($servers) - 1);
                }
                add_settings_error('ai_alt_messages', 'added', __('New server added.', 'aigude-tools'), 'updated');
            }

            update_option($option_key, $servers);
            $action = '';
            $index  = -1;

        } else {
            foreach ($errors as $err) {
                add_settings_error('ai_alt_messages', 'err', $err, 'error');
            }
        }
    }

    // --- Handle Delete from overview ----------------------------------------
    if ('delete' === $action) {
        // Verify nonce sent via query arg; dies on failure.
        check_admin_referer('aigude_server_delete', 'nonce_delete');
        $index = isset($_GET['index']) ? (int) intval(wp_unslash($_GET['index'])) : -1;
        if ($index >= 0 && isset($servers[ $index ])) {
            unset($servers[ $index ]);
            $servers = array_values($servers);

            // If no servers left, nothing to normalize. If there are, ensure at most one default remains.
            $found_default = false;
            foreach ($servers as $i => $srv) {
                if (! empty($srv['is_default'])) {
                    if ($found_default) {
                        $servers[ $i ]['is_default'] = 0;
                    }
                    $found_default = true;
                }
            }

            update_option($option_key, $servers);
            add_settings_error('ai_alt_messages', 'deleted', __('Server deleted.', 'aigude-tools'), 'updated');
            $action = '';
            $index  = -1;
        } else {
            add_settings_error('ai_alt_messages', 'err', __('Invalid index for delete.', 'aigude-tools'), 'error');
        }
    }

    // --- View ----------------------------------------------------------------
    ?>
    <div class="wrap">
        <h1><?php esc_html_e('Settings', 'aigude-tools'); ?></h1>
        <?php settings_errors('ai_alt_messages'); ?>

        <p class="description" style="margin-top:8px;">
            <?php esc_html_e("Don't have an API key?", 'aigude-tools'); ?>
            <a href="https://aigude.io/" class="button button-secondary" target="_blank" rel="noopener noreferrer">
                <?php esc_html_e('Get API key at AiGude.io', 'aigude-tools'); ?>
            </a>
        </p>

        <?php
        // For rendering the edit form (no state change), verify nonce and read index now.
        if ('edit' === $action) {
            // Verify nonce carried on the Edit link
            check_admin_referer($nonce_edit_action, $nonce_edit_name);
            $index = isset($_GET['index']) ? (int) intval(wp_unslash($_GET['index'])) : -1;
        }
    ?>
        <?php if ('edit' === $action && $index >= 0 && isset($servers[ $index ])) :
            $srv = $servers[ $index ];
            ?>
            <form method="post" novalidate>
                <?php wp_nonce_field($nonce_action, $nonce_name); ?>
                <input type="hidden" name="submit_ai_server" value="1">
                <input type="hidden" name="ai_action" value="edit">
                <input type="hidden" name="edit_index" value="<?php echo esc_attr($index); ?>">

                <table class="form-table" role="presentation">
                    <tr>
                        <th scope="row"><label for="server"><?php esc_html_e('Server', 'aigude-tools'); ?></label></th>
                        <td>
                            <select id="server" name="server">
                                <?php foreach (aigude_get_allowed_server_types() as $type) : ?>
                                    <option value="<?php echo esc_attr($type); ?>" <?php selected($srv['server'], $type); ?>>
                                        <?php echo esc_html($type); ?>
                                    </option>
                                <?php endforeach; ?>
                            </select>
                        </td>
                    </tr>

                    <tr>
                        <th scope="row"><label for="name"><?php esc_html_e('Name', 'aigude-tools'); ?></label></th>
                        <td><input type="text" id="name" name="name" value="<?php echo esc_attr($srv['name']); ?>" class="regular-text"></td>
                    </tr>

                    <tr>
                        <th scope="row"><label for="api_key"><?php esc_html_e('API Key', 'aigude-tools'); ?></label></th>
                        <td>
                            <div class="ai-key-field" style="display:flex;gap:8px;align-items:center;">
                                <input
                                        type="password"
                                        id="api_key"
                                        name="api_key"
                                        value="<?php echo isset($srv) ? esc_attr($srv['api_key']) : ''; ?>"
                                        class="regular-text"
                                        autocomplete="off"
                                >
                                <button type="button"
                                        class="button button-secondary api-key-visibility"
                                        data-target="#api_key"
                                        aria-pressed="false"
                                        aria-controls="api_key">
                                    <?php esc_html_e('Show', 'aigude-tools'); ?>
                                </button>
                                <button type="button"
                                        class="button button-secondary api-key-copy"
                                        data-target="#api_key">
                                    <?php esc_html_e('Copy', 'aigude-tools'); ?>
                                </button>
                            </div>
                        </td>
                    </tr>

                    <tr>
                        <th scope="row"><?php esc_html_e('Enabled', 'aigude-tools'); ?></th>
                        <td><label><input type="checkbox" name="enabled" <?php checked(1, ! empty($srv['enabled'])); ?>> <?php esc_html_e('Activate', 'aigude-tools'); ?></label></td>
                    </tr>

                    <tr>
                        <th scope="row"><?php esc_html_e('Default', 'aigude-tools'); ?></th>
                        <td><label><input type="checkbox" name="is_default" <?php checked(1, ! empty($srv['is_default'])); ?>> <?php esc_html_e('Make this the default server', 'aigude-tools'); ?></label></td>
                    </tr>
                </table>

                <?php submit_button(__('Save', 'aigude-tools')); ?>
                <a class="button" href="<?php echo esc_url(menu_page_url('aigude-tools-settings', false)); ?>"><?php esc_html_e('Cancel', 'aigude-tools'); ?></a>
            </form>

        <?php else : ?>

            <p>
                <a href="<?php echo esc_url(wp_nonce_url(add_query_arg(array( 'action' => 'add' )), $nonce_add_action, $nonce_add_name)); ?>" class="button button-primary">
                    <?php esc_html_e('Add New Server', 'aigude-tools'); ?>
                </a>
            </p>

            <?php if ('add' === $action) : ?>
                <?php
                // Verify nonce carried on the Add link before showing the form
                check_admin_referer($nonce_add_action, $nonce_add_name);
                ?>
                <form method="post" novalidate>
                    <?php wp_nonce_field($nonce_action, $nonce_name); ?>
                    <input type="hidden" name="submit_ai_server" value="1">
                    <input type="hidden" name="ai_action" value="add">

                    <table class="form-table" role="presentation">
                        <tr>
                            <th scope="row"><label for="server"><?php esc_html_e('Server', 'aigude-tools'); ?></label></th>
                            <td>
                                <select id="server" name="server">
                                <?php foreach (aigude_get_allowed_server_types() as $type) : ?>
                                        <option value="<?php echo esc_attr($type); ?>"><?php echo esc_html($type); ?></option>
                                    <?php endforeach; ?>
                                </select>
                            </td>
                        </tr>

                        <tr>
                            <th scope="row"><label for="name"><?php esc_html_e('Name', 'aigude-tools'); ?></label></th>
                            <td><input type="text" id="name" name="name" value="" class="regular-text"></td>
                        </tr>

                        <tr>
                            <th scope="row"><label for="api_key"><?php esc_html_e('API Key', 'aigude-tools'); ?></label></th>
                            <td><input type="password" id="api_key" name="api_key" value="" class="regular-text"></td>
                        </tr>

                        <tr>
                            <th scope="row"><?php esc_html_e('Enabled', 'aigude-tools'); ?></th>
                            <td><label><input type="checkbox" name="enabled" checked> <?php esc_html_e('Activate', 'aigude-tools'); ?></label></td>
                        </tr>

                        <tr>
                            <th scope="row"><?php esc_html_e('Default', 'aigude-tools'); ?></th>
                            <td><label><input type="checkbox" name="is_default"> <?php esc_html_e('Make this the default server', 'aigude-tools'); ?></label></td>
                        </tr>
                    </table>

                    <?php submit_button(__('Add', 'aigude-tools')); ?>
                    <a class="button" href="<?php echo esc_url(menu_page_url('aigude-tools-settings', false)); ?>"><?php esc_html_e('Cancel', 'aigude-tools'); ?></a>
                </form>

            <?php else : // Overview table?>
                <?php if (empty($servers)) : ?>
                    <p><?php echo esc_html__('No servers configured yet.', 'aigude-tools'); ?></p>
                <?php else : ?>
                    <table class="widefat fixed striped">
                        <thead>
                        <tr>
                            <th><?php esc_html_e('Server', 'aigude-tools'); ?></th>
                            <th><?php esc_html_e('Name', 'aigude-tools'); ?></th>
                            <th><?php esc_html_e('API Key', 'aigude-tools'); ?></th>
                            <th><?php esc_html_e('Enabled', 'aigude-tools'); ?></th>
                            <th><?php esc_html_e('Default', 'aigude-tools'); ?></th>
                            <th><?php esc_html_e('Remaining credits', 'aigude-tools'); ?></th>
                            <th><?php esc_html_e('Actions', 'aigude-tools'); ?></th>
                        </tr>
                        </thead>
                        <tbody>
                        <?php foreach ($servers as $i => $srv) :
                            $masked = str_repeat('*', max(4, strlen((string) ($srv['api_key'] ?? ''))));
                            $enabled_icon = ! empty($srv['enabled']) ? '<span style="color:green;">✔</span>' : '<span style="color:red;">✖</span>';
                            $is_default   = ! empty($srv['is_default']);
                            $make_def_url = wp_nonce_url(
                                add_query_arg(array( 'action' => 'make_default', 'index' => $i )),
                                'aigude_server_make_default',
                                'nonce_set_default'
                            );
                            $del_url = wp_nonce_url(
                                add_query_arg(array( 'action' => 'delete', 'index' => $i )),
                                'aigude_server_delete',
                                'nonce_delete'
                            );
                            $edit_url = wp_nonce_url(
                                add_query_arg(array( 'action' => 'edit', 'index' => $i )),
                                $nonce_edit_action,
                                $nonce_edit_name
                            );
                            ?>
                            <tr>
                                <td><?php echo esc_html($srv['server']); ?></td>
                                <td><?php echo esc_html($srv['name']); ?></td>

                                <td>
                                    <span class="api-key-mask" data-full="<?php echo esc_attr($srv['api_key']); ?>">
                                        <?php echo esc_html($masked); ?>
                                    </span>
                                    <?php if (! empty($srv['api_key'])) : ?>
                                        <a href="#" class="toggle-api-key"><?php esc_html_e('Show', 'aigude-tools'); ?></a>
                                    <?php endif; ?>
                                </td>

                                <td><?php echo wp_kses_post($enabled_icon); ?></td>
                                <td>
                                    <?php if ($is_default) : ?>
                                        <strong>★ <?php esc_html_e('Default', 'aigude-tools'); ?></strong>
                                    <?php else : ?>
                                        <a href="<?php echo esc_url($make_def_url); ?>">
                                            <?php esc_html_e('Make default', 'aigude-tools'); ?>
                                        </a>
                                    <?php endif; ?>
                                </td>
                                <td>
                                    <?php if (! empty($srv['enabled']) && ! empty($srv['api_key'])) : ?>
                                        <span class="ai-server-credits" data-index="<?php echo esc_attr($i); ?>">–</span>
                                    <?php else : ?>
                                        <span class="ai-server-credits" data-index="<?php echo esc_attr($i); ?>">
                                            <?php esc_html_e('Disabled', 'aigude-tools'); ?>
                                        </span>
                                    <?php endif; ?>
                                </td>
                                <td>
                                    <a href="<?php echo esc_url($edit_url); ?>"><?php esc_html_e('Edit', 'aigude-tools'); ?></a> |
                                    <a href="<?php echo esc_url($del_url); ?>"
                                       onclick="return confirm('<?php echo esc_attr__('Do you really want to delete this server?', 'aigude-tools'); ?>');">
                                        <?php esc_html_e('Delete', 'aigude-tools'); ?>
                                    </a>
                                </td>
                            </tr>
                        <?php endforeach; ?>
                        </tbody>
                    </table>
                <?php endif; ?>
            <?php endif; ?>
        <?php endif; ?>
    </div>
    <?php
}
