<?php if ( ! defined( 'ABSPATH' ) ) exit;

/**
 * Callback for the "Prompts" submenu.
 */
function aigude_prompt_templates_page() {
    if ( ! current_user_can( 'manage_options' ) ) {
        wp_die( esc_html__( 'Insufficient permissions', 'aigude-tools' ), 403 );
    }

    $nonce_action = 'aigude_tpl_action';
    $nonce_name   = 'aigude_tpl_nonce';
    $page_url     = menu_page_url( 'aigude-tools-prompts', false );

    $action    = isset( $_REQUEST['action'] ) ? sanitize_key( wp_unslash( $_REQUEST['action'] ) ) : '';
    $tpl_index = isset( $_REQUEST['tpl_index'] ) ? (int) sanitize_text_field( wp_unslash( $_REQUEST['tpl_index'] ) ) : -1;
    $dup_index = isset( $_REQUEST['dup_index'] ) ? (int) sanitize_text_field( wp_unslash( $_REQUEST['dup_index'] ) ) : -1;

    $templates = get_option( 'aigude_prompt_templates', [] );
    if ( ! is_array( $templates ) ) {
        $templates = [];
    }

    $notices    = [];
    $add_notice = static function ( $message, $class = 'updated' ) use ( &$notices ) {
        $notices[] = [
                'class'   => $class,
                'message' => $message,
        ];
    };
    $builtin_tpl = AIGUDE_Tools_Plugin::get_builtin_prompt_template();
    $builtin_id  = $builtin_tpl['id'] ?? AIGUDE_Tools_Plugin::BUILTIN_PROMPT_ID;
    $find_tpl_index = static function ( array $list, string $id ): int {
        foreach ( $list as $i => $tpl ) {
            if ( isset( $tpl['id'] ) && $tpl['id'] === $id ) {
                return (int) $i;
            }
        }
        return -1;
    };

    // Provider choices (all + EU only)
    $provider_meta_all = AIGUDE_Tools_Plugin::get_translation_providers_metadata();
    $provider_meta_eu  = AIGUDE_Tools_Plugin::filter_providers_by_region( $provider_meta_all, true );

    $provider_choices_all = [];
    foreach ( $provider_meta_all as $slug => $info ) {
        $languages = AIGUDE_Tools_Plugin::get_translation_languages( $slug );
        $provider_choices_all[ $slug ] = [
                'label'         => AIGUDE_Tools_Plugin::get_translation_provider_label( $slug ),
                'languages'     => $languages,
                'site_language' => AIGUDE_Tools_Plugin::describe_site_language( $slug ),
        ];
    }

    $provider_choices_eu = [];
    foreach ( $provider_meta_eu as $slug => $info ) {
        $languages = AIGUDE_Tools_Plugin::get_translation_languages( $slug );
        $provider_choices_eu[ $slug ] = [
                'label'         => AIGUDE_Tools_Plugin::get_translation_provider_label( $slug ),
                'languages'     => $languages,
                'site_language' => AIGUDE_Tools_Plugin::describe_site_language( $slug ),
        ];
    }

    // Ensure every template has a stable ID (for defaults)
    $changed = false;
    $has_builtin = false;
    foreach ( $templates as &$tpl ) {
        if ( empty( $tpl['id'] ) ) {
            $tpl['id'] = function_exists( 'wp_generate_uuid4' ) ? wp_generate_uuid4() : ( 'tpl_' . uniqid( '', true ) );
            $changed   = true;
        }
        if ( $tpl['id'] === $builtin_id ) {
            $has_builtin     = true;
            $tpl['builtin']  = '1';
        } elseif ( ! isset( $tpl['builtin'] ) || '0' !== $tpl['builtin'] ) {
            $tpl['builtin'] = '0';
            $changed        = true;
        }
        if ( ! isset( $tpl['eu_only_providers'] ) || ! in_array( $tpl['eu_only_providers'], array( '0', '1' ), true ) ) {
            $tpl['eu_only_providers'] = '0';
            $changed                  = true;
        }
        if ( ! isset( $tpl['recent_target_langs'] ) || ! is_array( $tpl['recent_target_langs'] ) ) {
            $tpl['recent_target_langs'] = [];
            $changed                    = true;
        }
    }
    unset( $tpl );
    if ( ! $has_builtin ) {
        $templates[] = $builtin_tpl;
        $changed     = true;
        $add_notice( esc_html__( 'Default prompt added.', 'aigude-tools' ) );
    }
    if ( $changed ) {
        update_option( 'aigude_prompt_templates', $templates );
    }
    $default_id = get_option( 'aigude_prompt_default_id', '' );
    if ( '' === $default_id || $find_tpl_index( $templates, $default_id ) < 0 ) {
        $default_id = $builtin_id;
        update_option( 'aigude_prompt_default_id', $default_id );
    }

    // Duplication (immediate copy)
    if ( 'duplicate' === $action && check_admin_referer( $nonce_action, $nonce_name ) ) {
        if ( $dup_index >= 0 && isset( $templates[ $dup_index ] ) ) {
            $src = $templates[ $dup_index ];
            $new = $src;
            $new['id'] = function_exists( 'wp_generate_uuid4' ) ? wp_generate_uuid4() : ( 'tpl_' . uniqid( '', true ) );
            $new['title'] = trim( ( $src['title'] ?? '' ) . ' (Copy)' );
            $new['builtin'] = '0';
            $templates[] = $new;
            update_option( 'aigude_prompt_templates', $templates );
            $add_notice( esc_html__( 'Prompt duplicated.', 'aigude-tools' ) );
        } else {
            $add_notice( esc_html__( 'Prompt not found.', 'aigude-tools' ), 'error' );
        }
        $action = '';
    }

    // Restore built-in defaults
    if ( 'restore_builtin' === $action && check_admin_referer( $nonce_action, $nonce_name ) ) {
        $idx = $find_tpl_index( $templates, $builtin_id );
        if ( $idx >= 0 ) {
            $templates[ $idx ] = $builtin_tpl;
        } else {
            $templates[] = $builtin_tpl;
        }
        update_option( 'aigude_prompt_templates', $templates );
        if ( '' === $default_id || $default_id === $builtin_id ) {
            $default_id = $builtin_id;
            update_option( 'aigude_prompt_default_id', $builtin_id );
        }
        $add_notice( esc_html__( 'Default prompt restored.', 'aigude-tools' ) );
        $action = '';
    }

    // Set default action
    if ( 'set_default' === $action && check_admin_referer( $nonce_action, $nonce_name ) ) {
        if ( $tpl_index >= 0 && isset( $templates[ $tpl_index ]['id'] ) ) {
            update_option( 'aigude_prompt_default_id', $templates[ $tpl_index ]['id'] );
            $default_id = $templates[ $tpl_index ]['id'];
            $add_notice( esc_html__( 'Default prompt updated.', 'aigude-tools' ) );
        } else {
            $add_notice( esc_html__( 'Prompt not found.', 'aigude-tools' ), 'error' );
        }
        $action = '';
    }

    // Deletion
    if ( 'delete' === $action && check_admin_referer( $nonce_action, $nonce_name ) ) {
        if ( $tpl_index >= 0 && isset( $templates[ $tpl_index ] ) ) {
            $deleted_id = $templates[ $tpl_index ]['id'] ?? '';
            if ( $deleted_id === $builtin_id ) {
                $add_notice( esc_html__( 'The default prompt cannot be deleted.', 'aigude-tools' ), 'error' );
            } else {
                unset( $templates[ $tpl_index ] );
                $templates = array_values( $templates );
                update_option( 'aigude_prompt_templates', $templates );

                // If the deleted one was default, clear default
                if ( $deleted_id && get_option( 'aigude_prompt_default_id' ) === $deleted_id ) {
                    delete_option( 'aigude_prompt_default_id' );
                    $default_id = '';
                }
                $add_notice( esc_html__( 'Prompt deleted.', 'aigude-tools' ) );
            }
        } else {
            $add_notice( esc_html__( 'Prompt not found.', 'aigude-tools' ), 'error' );
        }
        $action = '';
    }

    // Decide which view to show (list vs add/edit)
    $view_action = '';
    if ( in_array( $action, array( 'add', 'edit', 'dupedit' ), true ) ) {
        check_admin_referer( $nonce_action, $nonce_name );
        if ( 'edit' === $action && ( $tpl_index < 0 || ! isset( $templates[ $tpl_index ] ) ) ) {
            $add_notice( esc_html__( 'Prompt not found.', 'aigude-tools' ), 'error' );
        } else {
            $view_action = $action;
        }
    }
    if ( '' === $default_id || $find_tpl_index( $templates, $default_id ) < 0 ) {
        $default_id = $builtin_id;
        update_option( 'aigude_prompt_default_id', $default_id );
    }

    // Form state baseline
    $form_state = array(
            'index'               => null,
            'id'                  => '',
            'title'               => '',
            'prompt'              => '',
            'src_lang'            => 'auto',
            'prompt_lang'         => 'auto',
            'target_provider'     => 'deepl',
            'target_lang'         => '',
            'eu_only'             => '0',
            'recent_target_langs' => array(),
    );

    if ( in_array( $view_action, array( 'edit', 'dupedit' ), true ) && isset( $templates[ $tpl_index ] ) ) {
        $form_state = array_merge(
                $form_state,
                array(
                        'index'               => $tpl_index,
                        'id'                  => $templates[ $tpl_index ]['id'] ?? '',
                        'title'               => $templates[ $tpl_index ]['title'] ?? '',
                        'prompt'              => $templates[ $tpl_index ]['prompt'] ?? '',
                        'src_lang'            => $templates[ $tpl_index ]['src_lang'] ?? 'auto',
                        'prompt_lang'         => $templates[ $tpl_index ]['prompt_lang'] ?? 'auto',
                        'target_provider'     => $templates[ $tpl_index ]['target_provider'] ?? 'deepl',
                        'target_lang'         => $templates[ $tpl_index ]['target_lang'] ?? '',
                        'eu_only'             => isset( $templates[ $tpl_index ]['eu_only_providers'] ) && '1' === $templates[ $tpl_index ]['eu_only_providers'] ? '1' : '0',
                        'recent_target_langs' => isset( $templates[ $tpl_index ]['recent_target_langs'] ) && is_array( $templates[ $tpl_index ]['recent_target_langs'] ) ? $templates[ $tpl_index ]['recent_target_langs'] : array(),
                )
        );
    }

    // Save (new or update)
    if ( isset( $_POST['ai_tpl_submit'] ) ) {
        check_admin_referer( $nonce_action, $nonce_name );

        $title   = sanitize_text_field( wp_unslash( $_POST['ai_tpl_title'] ?? '' ) );
        $prompt  = sanitize_textarea_field( wp_unslash( $_POST['ai_tpl_prompt'] ?? '' ) );

        $src_lang_input        = 'auto';
        $prompt_lang_input     = 'auto';
        $target_provider_input = sanitize_text_field( wp_unslash( $_POST['ai_tpl_target_provider'] ?? 'deepl' ) );
        $target_lang_input     = sanitize_text_field( wp_unslash( $_POST['ai_tpl_target_lang'] ?? '' ) );
        $eu_only_input         = ! empty( $_POST['ai_tpl_eu_only'] ) ? '1' : '0';

        $active_provider_choices = '1' === $eu_only_input ? $provider_choices_eu : $provider_choices_all;

        $provider = '';
        $target_lang = '';
        if ( '' !== $target_provider_input ) {
            $normalized_provider = AIGUDE_Translation_Service::normalize_translation_provider( $target_provider_input );
            if ( isset( $active_provider_choices[ $normalized_provider ] ) ) {
                $provider = $normalized_provider;
                $languages = $active_provider_choices[ $normalized_provider ]['languages'] ?? array();
                $normalized_lang = AIGUDE_Translation_Service::normalize_provider_lang_code_generic( $target_lang_input, $provider, 'target' );
                if ( '' !== $normalized_lang && isset( $languages[ $normalized_lang ] ) ) {
                    $target_lang = $normalized_lang;
                }
            }
        }

        $prompt_lang = 'auto';
        $src_lang_placeholders = 'auto';
        if ( $provider && isset( $active_provider_choices[ $provider ] ) ) {
            $prompt_lang = 'auto' === $prompt_lang_input ? 'auto' : AIGUDE_Translation_Service::normalize_provider_lang_code_generic( $prompt_lang_input, $provider, 'source' );
            $src_lang_placeholders = 'auto' === $src_lang_input ? 'auto' : AIGUDE_Translation_Service::normalize_provider_lang_code_generic( $src_lang_input, $provider, 'source' );
            if ( '' === $prompt_lang ) {
                $prompt_lang = 'auto';
            }
            if ( '' === $src_lang_placeholders ) {
                $src_lang_placeholders = 'auto';
            }
        }

        if ( '' === $provider ) {
            $provider = isset( $active_provider_choices['deepl'] ) ? 'deepl' : (string) array_key_first( $active_provider_choices );
        }
        if ( '' === $target_lang && $provider && isset( $active_provider_choices[ $provider ]['languages'] ) ) {
            $site_info = $active_provider_choices[ $provider ]['site_language'] ?? array();
            $prov_langs = $active_provider_choices[ $provider ]['languages'];
            if ( ! empty( $site_info['supported'] ) && ! empty( $site_info['code'] ) && isset( $prov_langs[ $site_info['code'] ] ) ) {
                $target_lang = $site_info['code'];
            } elseif ( isset( $prov_langs['EN'] ) ) {
                $target_lang = 'EN';
            } elseif ( ! empty( $prov_langs ) ) {
                $target_lang = (string) array_key_first( $prov_langs );
            }
        }

        // Keep ID if editing; otherwise make one
        $tpl_id = '';
        if ( ! empty( $_POST['ai_tpl_id'] ) ) {
            $tpl_id = sanitize_text_field( wp_unslash( $_POST['ai_tpl_id'] ) );
        } else {
            $tpl_id = function_exists( 'wp_generate_uuid4' ) ? wp_generate_uuid4() : ( 'tpl_' . uniqid( '', true ) );
        }

        // Prepare per-prompt recents (persist last 5)
        $existing_recents = array();
        if ( ! empty( $_POST['ai_tpl_index'] ) ) {
            $existing_index = (int) $_POST['ai_tpl_index'];
            if ( isset( $templates[ $existing_index ]['recent_target_langs'] ) && is_array( $templates[ $existing_index ]['recent_target_langs'] ) ) {
                $existing_recents = $templates[ $existing_index ]['recent_target_langs'];
            }
        }
        $recent_target_langs_map = array();
        if ( is_array( $existing_recents ) ) {
            foreach ( $existing_recents as $provKey => $list ) {
                if ( ! is_array( $list ) ) {
                    continue;
                }
                $recent_target_langs_map[ $provKey ] = array_slice( array_values( array_unique( array_filter( $list, 'is_string' ) ) ), 0, 5 );
            }
        }
        if ( $provider && $target_lang ) {
            $provKey      = $provider;
            $valid_langs  = $active_provider_choices[ $provider ]['languages'] ?? array();
            $current_list = isset( $recent_target_langs_map[ $provKey ] ) && is_array( $recent_target_langs_map[ $provKey ] )
                    ? $recent_target_langs_map[ $provKey ]
                    : array();
            $current_list = array_values( array_unique( array_merge( array( $target_lang ), $current_list ) ) );
            $current_list = array_slice( $current_list, 0, 5 );
            $current_list = array_values( array_filter( $current_list, static function ( $code ) use ( $valid_langs ) {
                return isset( $valid_langs[ $code ] );
            } ) );
            $recent_target_langs_map[ $provKey ] = $current_list;
        }

        $data = [
                'id'                  => $tpl_id,
                'title'               => $title,
                'prompt'              => $prompt,
                'src_lang'            => $src_lang_placeholders,
                'prompt_lang'         => $prompt_lang,
                'target_provider'     => $provider,
                'target_lang'         => $target_lang,
                'eu_only_providers'   => $eu_only_input,
                'recent_target_langs' => $recent_target_langs_map,
        ];
        $data['builtin'] = ( $tpl_id === $builtin_id ) ? '1' : '0';

        $make_default = ! empty( $_POST['ai_tpl_is_default'] );
        $errors       = [];
        if ( '' === $title ) {
            $errors[] = __( 'Please enter a title.', 'aigude-tools' );
        }
        if ( '' === $prompt ) {
            $errors[] = __( 'Please enter a prompt.', 'aigude-tools' );
        }
        if ( '' === $provider ) {
            $errors[] = __( 'Please select a translation provider.', 'aigude-tools' );
        }
        if ( '' === $target_lang ) {
            $errors[] = __( 'Please select a target language.', 'aigude-tools' );
        }

        if ( ! empty( $errors ) ) {
            foreach ( $errors as $msg ) {
                $add_notice( $msg, 'error' );
            }
            // Keep user input visible after validation errors.
            $form_state = array(
                    'index'               => isset( $_POST['ai_tpl_index'] ) && '' !== $_POST['ai_tpl_index'] ? (int) $_POST['ai_tpl_index'] : null,
                    'id'                  => $tpl_id,
                    'title'               => $title,
                    'prompt'              => $prompt,
                    'src_lang'            => $src_lang_placeholders,
                    'prompt_lang'         => $prompt_lang,
                    'target_provider'     => $provider,
                    'target_lang'         => $target_lang,
                    'eu_only'             => $eu_only_input,
                    'recent_target_langs' => $recent_target_langs_map,
            );
            $view_action = ( isset( $_POST['ai_tpl_index'] ) && '' !== $_POST['ai_tpl_index'] ) ? 'edit' : 'add';
        } else {
            if ( isset( $_POST['ai_tpl_index'] ) && '' !== $_POST['ai_tpl_index'] ) {
                $idx = (int) $_POST['ai_tpl_index'];
                if ( isset( $templates[ $idx ] ) ) {
                    $templates[ $idx ] = $data;
                    $add_notice( esc_html__( 'Prompt updated.', 'aigude-tools' ) );
                } else {
                    $add_notice( esc_html__( 'Prompt not found.', 'aigude-tools' ), 'error' );
                }
            } else {
                $templates[] = $data;
                $add_notice( esc_html__( 'Prompt saved.', 'aigude-tools' ) );
            }

            update_option( 'aigude_prompt_templates', $templates );
            // Track recents for prompt and placeholder language selections
            if ( $make_default && empty( $errors ) ) {
                update_option( 'aigude_prompt_default_id', $tpl_id );
                $default_id = $tpl_id;
            }

            $view_action = '';
            $action      = '';
            $tpl_index   = -1;
        }
    }

    // Prepare JSON for dynamic selects
    $provider_choices_all_json = wp_json_encode( $provider_choices_all );
    $provider_choices_eu_json  = wp_json_encode( $provider_choices_eu );

    $add_url = wp_nonce_url( add_query_arg( array( 'action' => 'add' ), $page_url ), $nonce_action, $nonce_name );

    // Sorting for list view
    $orderby = isset( $_GET['orderby'] ) ? sanitize_key( wp_unslash( $_GET['orderby'] ) ) : '';
    $order   = isset( $_GET['order'] ) ? strtolower( sanitize_text_field( wp_unslash( $_GET['order'] ) ) ) : 'asc';
    if ( ! in_array( $order, array( 'asc', 'desc' ), true ) ) {
        $order = 'asc';
    }
    $sort_fields = array( 'title', 'prompt', 'target' );
    if ( '' !== $orderby && in_array( $orderby, $sort_fields, true ) && $templates ) {
        $templates = array_values( $templates );
        usort(
                $templates,
                static function ( $a, $b ) use ( $orderby, $order ) {
                    $direction = ( 'desc' === $order ) ? -1 : 1;
                    if ( 'title' === $orderby ) {
                        $va = strtolower( $a['title'] ?? '' );
                        $vb = strtolower( $b['title'] ?? '' );
                    } elseif ( 'prompt' === $orderby ) {
                        $va = strtolower( $a['prompt'] ?? '' );
                        $vb = strtolower( $b['prompt'] ?? '' );
                    } else {
                        $pa = strtolower( $a['target_provider'] ?? '' );
                        $pb = strtolower( $b['target_provider'] ?? '' );
                        $la = strtolower( $a['target_lang'] ?? '' );
                        $lb = strtolower( $b['target_lang'] ?? '' );
                        $va = $pa . '|' . $la;
                        $vb = $pb . '|' . $lb;
                    }
                    return $direction * strcmp( $va, $vb );
                }
        );
    } else {
        $orderby = '';
    }

    ?>
    <div class="wrap">
        <?php if ( '' === $view_action ) : ?>
            <h1 class="wp-heading-inline"><?php esc_html_e( 'Prompts', 'aigude-tools' ); ?></h1>
            <?php foreach ( $notices as $notice ) : ?>
                <div class="<?php echo esc_attr( $notice['class'] ); ?>"><p><?php echo esc_html( $notice['message'] ); ?></p></div>
            <?php endforeach; ?>
            <hr class="wp-header-end" />
            <p style="margin-top:10px;">
                <a href="<?php echo esc_url( $add_url ); ?>" class="button button-primary"><?php esc_html_e( 'Add New', 'aigude-tools' ); ?></a>
            </p>

            <table class="widefat">
                <thead>
                <tr>
                    <?php
                    $cols = array(
                            'title'  => __( 'Title', 'aigude-tools' ),
                            'prompt' => __( 'Prompt', 'aigude-tools' ),
                            'target' => __( 'Target language', 'aigude-tools' ),
                    );
                    foreach ( $cols as $key => $label ) {
                        $is_active = ( $orderby === $key );
                        $next_order = ( $is_active && 'asc' === $order ) ? 'desc' : 'asc';
                        $arrow = '';
                        if ( $is_active ) {
                            $arrow = 'asc' === $order ? ' ▲' : ' ▼';
                        }
                        $link = esc_url(
                                add_query_arg(
                                        array(
                                                'orderby' => $key,
                                                'order'   => $next_order,
                                        ),
                                        $page_url
                                )
                        );
                        printf(
                                '<th><a href="%s">%s%s</a></th>',
                                esc_url( $link ),
                                esc_html( $label ),
                                esc_html( $arrow )
                        );
                    }
                    ?>
                    <th><?php esc_html_e( 'Default', 'aigude-tools' ); ?></th>
                    <th><?php esc_html_e( 'Actions', 'aigude-tools' ); ?></th>
                </tr>
                </thead>
                <tbody>
                <?php if ( $templates ) : foreach ( $templates as $i => $tpl ) :
                    $is_default = ! empty( $tpl['id'] ) && $tpl['id'] === $default_id;
                    $is_builtin = ! empty( $tpl['builtin'] ) || ( isset( $tpl['id'] ) && $tpl['id'] === $builtin_id );
                    $base       = $page_url;
                    $edit_url   = wp_nonce_url( add_query_arg( ['action'=>'edit','tpl_index'=>$i], $base ), $nonce_action, $nonce_name );
                    $dup_url    = wp_nonce_url( add_query_arg( ['action'=>'duplicate','dup_index'=>$i], $base ), $nonce_action, $nonce_name );
                    $del_url    = wp_nonce_url( add_query_arg( ['action'=>'delete','tpl_index'=>$i], $base ), $nonce_action, $nonce_name );
                    $def_url    = wp_nonce_url( add_query_arg( ['action'=>'set_default','tpl_index'=>$i], $base ), $nonce_action, $nonce_name );
                    $restore_url= wp_nonce_url( add_query_arg( array( 'action' => 'restore_builtin' ), $base ), $nonce_action, $nonce_name );
                    $target_info = AIGUDE_Tools_Plugin::describe_target_language_choice( $tpl['target_provider'] ?? '', $tpl['target_lang'] ?? '' );
                    ?>
                    <tr>
                        <td><?php echo esc_html( $tpl['title'] ); ?></td>
                        <td><?php echo esc_html( $tpl['prompt'] ); ?></td>
                        <td>
                            <?php if ( ! empty( $target_info['code'] ) ) : ?>
                                <span><?php echo esc_html( $target_info['display'] ?: $target_info['label'] ); ?></span>
                            <?php else : ?>
                                <span class="description"><?php esc_html_e( 'Not set', 'aigude-tools' ); ?></span>
                            <?php endif; ?>
                        </td>
                        <td>
                            <?php if ( $is_default ) : ?>
                                <span class="dashicons dashicons-star-filled" title="<?php esc_attr_e( 'Default', 'aigude-tools' ); ?>"></span>
                            <?php else : ?>
                                <a href="<?php echo esc_url( $def_url ); ?>"><?php esc_html_e( 'Make default', 'aigude-tools' ); ?></a>
                            <?php endif; ?>
                        </td>
                        <td>
                            <?php
                            $actions = array(
                                    sprintf( '<a href="%s">%s</a>', esc_url( $edit_url ), esc_html__( 'Edit', 'aigude-tools' ) ),
                                    sprintf( '<a href="%s">%s</a>', esc_url( $dup_url ), esc_html__( 'Duplicate', 'aigude-tools' ) ),
                            );
                            if ( $is_builtin ) {
                                $actions[] = sprintf( '<a href="%s">%s</a>', esc_url( $restore_url ), esc_html__( 'Restore defaults', 'aigude-tools' ) );
                            } else {
                                $actions[] = sprintf(
                                        '<a href="%s" onclick="return confirm(\'%s\');">%s</a>',
                                        esc_url( $del_url ),
                                        esc_attr__( 'Really delete?', 'aigude-tools' ),
                                        esc_html__( 'Delete', 'aigude-tools' )
                                );
                            }
                            echo implode( ' | ', array_map( 'wp_kses_post', $actions ) );
                            ?>
                        </td>
                    </tr>
                <?php endforeach; else : ?>
                    <tr><td colspan="5"><?php esc_html_e( 'No prompts added.', 'aigude-tools' ); ?></td></tr>
                <?php endif; ?>
                </tbody>
            </table>

        <?php else : ?>
            <?php
            $form_index           = $form_state['index'];
            $form_id              = $form_state['id'];
            $form_title           = $form_state['title'];
            $form_prompt          = $form_state['prompt'];
            $form_src_lang        = $form_state['src_lang'];
            $form_prompt_lang     = $form_state['prompt_lang'];
            $form_target_provider = $form_state['target_provider'];
            $form_target_lang     = $form_state['target_lang'];
            $form_eu_only         = $form_state['eu_only'];
            $form_recent_langs    = $form_state['recent_target_langs'];
            $current_provider_choices = ( '1' === $form_eu_only ) ? $provider_choices_eu : $provider_choices_all;
            if ( empty( $form_target_provider ) || ! isset( $current_provider_choices[ $form_target_provider ] ) ) {
                $form_target_provider = isset( $current_provider_choices['deepl'] ) ? 'deepl' : (string) array_key_first( $current_provider_choices );
            }
            ?>
            <h1 class="wp-heading-inline">
                <?php echo 'edit' === $view_action ? esc_html__( 'Edit Prompt', 'aigude-tools' ) : esc_html__( 'Add New Prompt', 'aigude-tools' ); ?>
            </h1>
            <?php foreach ( $notices as $notice ) : ?>
                <div class="<?php echo esc_attr( $notice['class'] ); ?>"><p><?php echo esc_html( $notice['message'] ); ?></p></div>
            <?php endforeach; ?>
            <hr class="wp-header-end">

            <form method="post">
                <?php if ( null !== $form_index && 'dupedit' !== $view_action ) : ?>
                    <input type="hidden" name="ai_tpl_index" value="<?php echo esc_attr( $form_index ); ?>">
                    <input type="hidden" name="ai_tpl_id" value="<?php echo esc_attr( $form_id ); ?>">
                <?php endif; ?>

                <?php wp_nonce_field( $nonce_action, $nonce_name ); ?>
                <table class="form-table">
                    <tr>
                        <th>
                            <label for="ai_tpl_is_default">
                                <?php esc_html_e( 'Default', 'aigude-tools' ); ?>
                            </label>
                        </th>
                        <td>
                            <label>
                                <input type="checkbox"
                                       name="ai_tpl_is_default"
                                       id="ai_tpl_is_default"
                                        <?php checked( $form_id, $default_id ); ?>>
                                <?php esc_html_e( 'Make this the default prompt', 'aigude-tools' ); ?>
                            </label>
                        </td>
                    </tr>

                    <tr>
                        <th><label for="ai_tpl_title"><?php esc_html_e( 'Title', 'aigude-tools' ); ?></label></th>
                        <td><input name="ai_tpl_title" id="ai_tpl_title" type="text" class="regular-text"
                                   value="<?php echo esc_attr( $form_title ); ?>"></td>
                    </tr>
                    <tr>
                        <th><label for="ai_tpl_prompt"><?php esc_html_e( 'Prompt', 'aigude-tools' ); ?></label></th>
                        <td>
                            <textarea name="ai_tpl_prompt" id="ai_tpl_prompt" class="large-text" rows="5"><?php echo esc_textarea( $form_prompt ); ?></textarea>
                            <p class="description" style="margin-top:6px;">
                                <?php esc_html_e( 'You can write the prompt in any language supported by the provider you select for the Target Alt Text.', 'aigude-tools' ); ?>
                            </p>
                            <details class="description" style="margin-top:6px;">
                                <summary style="cursor:pointer;"><?php esc_html_e( 'Available placeholders', 'aigude-tools' ); ?></summary>
                                <div style="margin-top:6px;">
                                    <p style="margin:0 0 6px;">
                                        <code>%filename%</code>, <code>%filename_no_ext%</code>, <code>%title%</code>, <code>%current_alt%</code>,
                                        <code>%caption%</code>, <code>%description%</code>, <code>%mime%</code>, <code>%width%</code>, <code>%height%</code>.
                                    </p>
                                    <p style="margin:0 0 6px;">
                                        <?php /* translators: %filename_no_ext% is the filename without extension. Example shows automatic quoting of text placeholders. */ ?>
                                        <?php esc_html_e( 'Text placeholders are automatically quoted (e.g. %filename_no_ext% → "car-photo-123").', 'aigude-tools' ); ?><br>
                                        <?php /* translators: %width% and %height% are numeric image dimensions; numeric placeholders are not quoted. */?>
                                        <?php esc_html_e( 'Numeric placeholders like %width% and %height% are not quoted (e.g. → 1920).', 'aigude-tools' ); ?><br>
                                        <?php esc_html_e( 'Modifiers: |q (force quotes), |raw (no quotes), |trim, |lower, |upper, |ucfirst, |translatable (force translate), |untranslatable (no translate).', 'aigude-tools' ); ?><br>
                                        <?php esc_html_e( 'Unknown placeholders are left unchanged. Empty values become blank.', 'aigude-tools' ); ?>
                                    </p>
                                    <p style="margin:0;">
                                        <strong><?php esc_html_e( 'Examples:', 'aigude-tools' ); ?></strong> <code>%title|trim|ucfirst%</code>, <code>%filename_no_ext|lower%</code>, <code>%current_alt|raw%</code>, <code>%title|translatable%</code>, <code>%current_alt|untranslatable%</code>
                                    </p>
                                </div>
                            </details>
                        </td>
                    </tr>

                    <?php
                    $current_provider_langs = ( $form_target_provider && isset( $current_provider_choices[ $form_target_provider ]['languages'] ) )
                        ? $current_provider_choices[ $form_target_provider ]['languages']
                        : [];
                    $current_site_lang = ( $form_target_provider && isset( $current_provider_choices[ $form_target_provider ]['site_language'] ) )
                        ? $current_provider_choices[ $form_target_provider ]['site_language']
                        : null;
                    $prompt_selected_lang = $form_target_lang;
                    $site_supported       = $current_site_lang && ! empty( $current_site_lang['supported'] );
                    if ( '' === $prompt_selected_lang ) {
                        if ( $site_supported && ! empty( $current_site_lang['code'] ) && isset( $current_provider_langs[ $current_site_lang['code'] ] ) ) {
                            $prompt_selected_lang = $current_site_lang['code'];
                        } elseif ( isset( $current_provider_langs['EN'] ) ) {
                            $prompt_selected_lang = 'EN';
                        } elseif ( ! empty( $current_provider_langs ) ) {
                            $first_code           = array_key_first( $current_provider_langs );
                            $prompt_selected_lang = $first_code ?: '';
                        }
                    }
                    $recents_map            = is_array( $form_recent_langs ) ? $form_recent_langs : [];
                    $filtered_recent_langs  = [];
                    $recent_source          = [];
                    if ( $form_target_provider && isset( $recents_map[ $form_target_provider ] ) && is_array( $recents_map[ $form_target_provider ] ) ) {
                        $recent_source = $recents_map[ $form_target_provider ];
                    } elseif ( isset( $recents_map['_default'] ) && is_array( $recents_map['_default'] ) ) {
                        $recent_source = $recents_map['_default'];
                    }
                    foreach ( $recent_source as $code ) {
                        $code = (string) $code;
                        if ( isset( $current_provider_langs[ $code ] ) ) {
                            $filtered_recent_langs[] = $code;
                        }
                        if ( count( $filtered_recent_langs ) >= 5 ) {
                            break;
                        }
                    }
                    ?>
                    <tr>
                        <th><label for="ai_tpl_target_lang"><?php esc_html_e( 'Target Alt Text language', 'aigude-tools' ); ?></label></th>
                        <td>
                            <div style="display:flex;gap:12px;flex-wrap:wrap;">
                                <div>
                                    <label for="ai_tpl_target_provider" style="display:block;font-weight:600;">
                                        <?php esc_html_e( 'Provider', 'aigude-tools' ); ?>
                                    </label>
                                    <select name="ai_tpl_target_provider"
                                            id="ai_tpl_target_provider"
                                            data-providers-all="<?php echo esc_attr( $provider_choices_all_json ?: '{}' ); ?>"
                                            data-providers-eu="<?php echo esc_attr( $provider_choices_eu_json ?: '{}' ); ?>"
                                            data-initial-eu="<?php echo esc_attr( $form_eu_only ); ?>">
                                        <?php foreach ( $current_provider_choices as $slug => $info ) : ?>
                                            <option value="<?php echo esc_attr( $slug ); ?>" <?php selected( $slug, $form_target_provider ); ?>>
                                                <?php echo esc_html( $info['label'] ?? ucfirst( $slug ) ); ?>
                                            </option>
                                        <?php endforeach; ?>
                                    </select>
                                    <label style="display:flex;gap:8px;margin:8px 0 0;align-items:center;">
                                        <input type="checkbox"
                                               name="ai_tpl_eu_only"
                                               id="ai_tpl_eu_only"
                                               value="1"
                                            <?php checked( $form_eu_only, '1' ); ?>>
                                        <?php esc_html_e( 'Show only EU-based translation providers', 'aigude-tools' ); ?>
                                    </label>
                                </div>

                                <div>
                                    <label for="ai_tpl_target_lang" style="display:block;font-weight:600;">
                                        <?php esc_html_e( 'Language', 'aigude-tools' ); ?>
                                    </label>
                                    <select name="ai_tpl_target_lang"
                                            id="ai_tpl_target_lang"
                                            data-selected="<?php echo esc_attr( $prompt_selected_lang ); ?>"
                                            data-placeholder="<?php esc_attr_e( 'Select a provider to choose a language', 'aigude-tools' ); ?>"
                                            data-empty-text="<?php esc_attr_e( 'No languages available', 'aigude-tools' ); ?>"
                                            data-recent-langs="<?php echo esc_attr( wp_json_encode( $recents_map ) ); ?>"
                                            >
                                        <?php
                                        if ( $form_target_provider && ! empty( $current_provider_langs ) ) {
                                            if ( $site_supported && ! empty( $current_site_lang['code'] ) && isset( $current_provider_langs[ $current_site_lang['code'] ] ) ) {
                                                printf(
                                                    '<option value="%s"%s>%s</option>',
                                                    esc_attr( $current_site_lang['code'] ),
                                                    selected( $current_site_lang['code'], $prompt_selected_lang, false ),
                                                    esc_html(
                                                        sprintf(
                                                            /* translators: %s = site language label, e.g. "English (US)". */
                                                            __( 'System (%s)', 'aigude-tools' ),
                                                            $current_site_lang['label'] ?? $current_site_lang['code']
                                                        )
                                                    )
                                                );
                                            }
                                            if ( ! empty( $filtered_recent_langs ) ) {
                                                echo '<optgroup label="' . esc_attr__( 'Recent', 'aigude-tools' ) . '">';
                                                foreach ( $filtered_recent_langs as $code ) {
                                                    printf(
                                                        '<option value="%s"%s>%s</option>',
                                                        esc_attr( $code ),
                                                        selected( $code, $prompt_selected_lang, false ),
                                                        esc_html( $current_provider_langs[ $code ] ?? $code )
                                                    );
                                                }
                                                echo '</optgroup>';
                                            }
                                            echo '<optgroup label="' . esc_attr__( 'All languages', 'aigude-tools' ) . '">';
                                            foreach ( $current_provider_langs as $code => $label ) {
                                                printf(
                                                    '<option value="%s"%s>%s</option>',
                                                    esc_attr( $code ),
                                                    selected( $code, $prompt_selected_lang, false ),
                                                    esc_html( $label )
                                                );
                                            }
                                            echo '</optgroup>';
                                        } else {
                                            printf(
                                                '<option value="">%s</option>',
                                                esc_html__( 'Select a provider to choose a language', 'aigude-tools' )
                                            );
                                        }
                                        ?>
                                    </select>
                                </div>
                            </div>
                            <p class="description" style="margin-top:6px;">
                                <?php esc_html_e( 'Pick a provider and language you always want the generated alt text to use, overriding the default selection in List/Grid views.', 'aigude-tools' ); ?>
                                <br>
                                <?php esc_html_e( 'When set, the List/Grid views lock the Alt Text Language selector to this provider/language.', 'aigude-tools' ); ?>
                            </p>
                        </td>
                    </tr>
                </table>
                <p>
                    <button name="ai_tpl_submit" class="button button-primary">
                        <?php echo null !== $form_index ? esc_html__( 'Update', 'aigude-tools' ) : esc_html__( 'Save Prompt', 'aigude-tools' ); ?>
                    </button>
                    <a href="<?php echo esc_url( $page_url ); ?>" class="button"><?php esc_html_e( 'Cancel', 'aigude-tools' ); ?></a>
                </p>
            </form>
            <?php if ( ! empty( $provider_choices_all ) ) : ?>
                <script type="text/javascript">
                    jQuery(function ($) {
                        const $provider   = $('#ai_tpl_target_provider');
                        const $targetLang = $('#ai_tpl_target_lang');
                        const $euToggle   = $('#ai_tpl_eu_only');

                        function parseData(val) {
                            if (typeof val === 'string') {
                                try { return JSON.parse(val); } catch (e) { return {}; }
                            }
                            if (typeof val === 'object' && val !== null) return val;
                            return {};
                        }

                        const providersAll = parseData($provider.data('providersAll') || {});
                        const providersEu  = parseData($provider.data('providersEu') || {});
                        const recentTargetMap = parseData($targetLang.data('recentLangs') || {});

                        const placeholder = $targetLang.data('placeholder') || '';
                        const emptyText   = $targetLang.data('emptyText') || '';
                        let initialTarget = $targetLang.data('selected') || '';
                        <?php /* translators: %s = site language label (e.g., "English (US)"). */ ?>
                        const systemTpl   = '<?php echo esc_js( __( 'System (%s)', 'aigude-tools' ) ); ?>';
                        const recentLabel = '<?php echo esc_js( __( 'Recent', 'aigude-tools' ) ); ?>';
                        const allLabel    = '<?php echo esc_js( __( 'All languages', 'aigude-tools' ) ); ?>';

                        function getActiveProviderMap() {
                            return $euToggle.is(':checked') ? providersEu : providersAll;
                        }

                        function pickProvider(provider, providerMap) {
                            if (provider && providerMap[provider]) return provider;
                            if (providerMap.deepl) return 'deepl';
                            const keys = Object.keys(providerMap);
                            return keys.length ? keys[0] : '';
                        }

                        function disableSelect($select, message) {
                            $select.empty()
                                .append($('<option>').val('').text(message || placeholder));
                            $select.prop('disabled', true);
                        }

                        function renderTargetSelect(provider, providerMap, preserveCurrent) {
                            const providerData = providerMap[provider] || {};
                            const languages = providerData.languages || {};
                            const siteLang = providerData.site_language || {};
                            const current = preserveCurrent ? $targetLang.val() : '';
                            let selected = initialTarget || current || '';

                            if (!provider || !languages || $.isEmptyObject(languages)) {
                                disableSelect($targetLang, placeholder);
                                return;
                            }

                            $targetLang.empty();

                            if (siteLang.code && languages[siteLang.code]) {
                                $targetLang.append(
                                    $('<option>')
                                        .val(siteLang.code)
                                        .text(systemTpl.replace('%s', siteLang.label || siteLang.code))
                                );
                            }

                            const recents = (recentTargetMap && recentTargetMap[provider] && Array.isArray(recentTargetMap[provider]))
                                ? recentTargetMap[provider]
                                : (recentTargetMap && Array.isArray(recentTargetMap._default) ? recentTargetMap._default : []);

                            const filteredRecents = (recents || []).filter(code => languages[code]);
                            if (filteredRecents.length) {
                                const $grp = $('<optgroup>').attr('label', recentLabel);
                                filteredRecents.forEach(code => {
                                    $grp.append($('<option>').val(code).text(languages[code] || code));
                                });
                                $targetLang.append($grp);
                            }

                            const $all = $('<optgroup>').attr('label', allLabel);
                            $.each(languages, function (code, label) {
                                $all.append($('<option>').val(code).text(label));
                            });
                            $targetLang.append($all);

                            if (selected && !languages[selected]) {
                                selected = '';
                            }
                            if (!selected) {
                                if (siteLang && siteLang.supported && languages[siteLang.code]) {
                                    selected = siteLang.code;
                                } else if (languages.EN) {
                                    selected = 'EN';
                                } else {
                                    selected = Object.keys(languages)[0] || '';
                                }
                            }

                            $targetLang.val(selected);
                            $targetLang.prop('disabled', false);
                            initialTarget = '';
                        }

                        function renderProviderOptions(preserveCurrentProvider) {
                            const providerMap = getActiveProviderMap();
                            const currentValue = preserveCurrentProvider ? $provider.val() : '';
                            const chosen = pickProvider(currentValue, providerMap);

                            $provider.empty();
                            $.each(providerMap, function (slug, info) {
                                const $opt = $('<option>').val(slug).text(info.label || slug);
                                if (slug === chosen) $opt.prop('selected', true);
                                $provider.append($opt);
                            });

                            renderTargetSelect(chosen, providerMap, preserveCurrentProvider);
                        }

                        $provider.on('change', function () {
                            renderProviderOptions(true);
                        });

                        $euToggle.on('change', function () {
                            renderProviderOptions(true);
                        });

                        // Initialize provider select based on saved state
                        renderProviderOptions(true);
                    });
                </script>
            <?php endif; ?>
        <?php endif; ?>
    </div>
    <?php
}
