// assets/js/list-actions.js
jQuery(function ($) {
    // Data passed from PHP (only data, no human strings)
    const D = window.aigudeToolsListData || {};
    const AJAX_URL   = D.ajaxUrl || window.ajaxurl;
    const NONCE      = D.nonce;
    const PER_CREDIT = D.perImageCredits || 3;

    // i18n strings (all translatable via GlotPress)
    const { __ } = wp.i18n;

    /* translators: %d = total number of credits used across the whole operation */
    const totalCreditsUsedTpl = __('Total credits used: %d', 'aigude-tools');

    /* translators: %d = number of credits used for the current image/batch */
    const creditsUsedTpl = __('Credits used: %d', 'aigude-tools');

    const STR = {
        generating:         __('Generating...', 'aigude-tools'), // be consistent with three dots vs ellipsis
        done:               __('Done', 'aigude-tools'),
        error:              __('Error', 'aigude-tools'),
        altTextGenerated:   __('Alt-Text generated', 'aigude-tools'),
        altTextSaved:       __('Alt-Text saved', 'aigude-tools'),
        errorGenerating:    __('Error generating alt text', 'aigude-tools'),
        errorSaving:        __('Error saving alt text', 'aigude-tools'),
        pleaseSelectImage:  __('Please select at least one image.', 'aigude-tools'),
        confirmOverwrite:   __('This will overwrite existing alt texts. Are you sure?', 'aigude-tools'),
        noAiTextYet:        __('No AI text generated yet.', 'aigude-tools'),
        totalCreditsUsedTpl,
        creditsUsedTpl,
        invalidApiKey:      __('Invalid or unauthorized API key.', 'aigude-tools'),
        nonceFailed:        __('Security check failed. Please reload the page.', 'aigude-tools'),
        creditsWord:        __('credits', 'aigude-tools'),
        lockedByPrompt:     __('Language locked by selected prompt.', 'aigude-tools'),
    };


    // --- Toast ---------------------------------------------------------------
    window.agToast = function (msg, type) {
        const $t = $(`<div class="ag-toast ag-toast-${type}">${msg}</div>`);
        $('body').append($t);
        $t.fadeIn(150).delay(1000).fadeOut(200, function () { $t.remove(); });
    };

    // --- State ---------------------------------------------------------------
    let allSelectedIds = null; // across-pages selection

    // --- Tiny helpers --------------------------------------------------------
    const $bulkBtn            = $('#bulk-generate');
    const $skipCb             = $('#skip-existing');
    const $selectAllPageCb    = $('#select-all-cb');
    const $selectAllAllPagesCb= $('#select-all-allpages');
    const $progressWrap       = $('#bulk-progress');
    const $progressBar        = $('#bulk-progress-bar');

    const currentSearchTerm = () => $('#aigude-tools-search').val() || '';
    const isSkipEnabled     = () => $skipCb.is(':checked');

    function updateButtonLabelText($btn, current, total) {
        const base = $btn.data('default-text'); // has a %s placeholder
        $btn.html(base.replace('%s', `${current} / ${total}`));
    }

    function cardHasSavedAlt($card) {
        const saved = ($card && $card.attr('data-saved-alt')) || '';
        return String(saved).trim() !== '';
    }

    function updateButtonLabels() {
        let count;
        if (Array.isArray(allSelectedIds) && allSelectedIds.length) {
            count = allSelectedIds.length;
        } else {
            const skip = isSkipEnabled();
            count = 0;
            $('.ai-card').each(function () {
                const $card = $(this);
                const hasSavedAlt = cardHasSavedAlt($card);
                if (!skip || (skip && !hasSavedAlt)) {
                    if ($card.find('.ai-select').is(':checked')) count++;
                }
            });
        }
        const credits = count * PER_CREDIT;
        const base = $bulkBtn.data('default-text').replace('%s', count);
        $bulkBtn.html(`${base} — ${credits} ${STR.creditsWord}`);
    }

    function freezePageCheckboxes(freeze) {
        const $cards = $('.ai-card');
        const $cbs   = $cards.find('.ai-select');
        if (freeze) {
            $cbs.prop({ checked: false, disabled: true });
            $cards.addClass('ai-frozen');
        } else {
            $cards.removeClass('ai-frozen');
            applySkipLogicToCards();
        }
    }

    function applySkipLogicToCards() {
        if (Array.isArray(allSelectedIds) && allSelectedIds.length) {
            $('.ai-card .ai-select').prop({ checked: false, disabled: true });
            $('.ai-card').addClass('ai-frozen');
            return;
        }
        const skip = isSkipEnabled();
        $('.ai-card').each(function () {
            const $card  = $(this);
            const hasSavedAlt = cardHasSavedAlt($card);
            const $cb    = $card.find('.ai-select');

            if (skip && hasSavedAlt) {
                $cb.prop({ checked: false, disabled: true });
                $card.removeClass('ai-frozen').addClass('ai-disabled');
            } else {
                $cb.prop({ disabled: false });
                $card.removeClass('ai-disabled ai-frozen');
            }
        });
    }

    function fetchAllMatchingIds() {
        return $.post(AJAX_URL, {
            action: 'aigude_list_ids',
            s: currentSearchTerm(),
            skipExisting: isSkipEnabled() ? 1 : 0,
            _ajax_nonce: NONCE,
        });
    }

    function setBtnLoading($btn) {
        $btn.prop('disabled', true).html(
            `<span style="background:#f9c8f2;padding:8px 16px;border-radius:4px;color:#333;">${STR.generating}</span>`
        );
    }

    function resetBtnToDefault($btn) {
        $btn.prop('disabled', false).html($btn.data('default-text'));
    }

    function showAlt($card, text) {
        const $sec = $card.find('.alternative-block');
        $sec.find('.alt-text-input').val(text);
        $sec.slideDown();
    }

    function getTargetInfoFromSelect($promptSelect) {
        const opt = $promptSelect.find('option:selected');
        const provider = (opt.data('target-provider') || 'deepl').toString();
        const providerLabel = (opt.data('target-provider-label') || provider).toString();
        const lang = (opt.data('target-lang') || '').toString();
        const langLabel = (opt.data('target-lang-label') || lang).toString();
        return { provider, providerLabel, lang, langLabel };
    }

    function applyTemplateTarget($promptSelect, $info) {
        if (!$promptSelect.length || !$info.length) return;
        const info = getTargetInfoFromSelect($promptSelect);
        const parts = [];
        if (info.providerLabel) parts.push(info.providerLabel);
        if (info.langLabel) parts.push(info.langLabel);
        $info.text(parts.join(' — '));
        $info.data('targetProvider', info.provider);
        $info.data('targetLang', info.lang);
        $info.data('targetLangLabel', info.langLabel);
        $info.data('targetProviderLabel', info.providerLabel);
    }

    // --- Single-card actions -------------------------------------------------
    $(document)
        .off('change.ai', '.ai-prompt-select')
        .on('change.ai', '.ai-prompt-select', function () {
            const tpl = $(this).val();
            $(this).closest('.prompt-block').find('.ai-prompt-input').val(tpl);
            const $card = $(this).closest('.ai-card');
            applyTemplateTarget($(this), $card.find('.ai-target-info'));
        });

    const $globalPrompt = $('#global-prompt');
    const $globalInfo = $('#global-target-info');
    $globalPrompt.off('change.aiPromptLock').on('change.aiPromptLock', function () {
        applyTemplateTarget($(this), $globalInfo);
    });
    applyTemplateTarget($globalPrompt, $globalInfo);
    $('.ai-card').each(function () {
        applyTemplateTarget($(this).find('.ai-prompt-select'), $(this).find('.ai-target-info'));
    });

    // --- Prompt details popover (global prompt only) ------------------------
    let $promptPopover = null;
    let popoverVisible = false;
    function ensurePromptPopover() {
        if ($promptPopover) return $promptPopover;
        $promptPopover = $('<div class="ai-prompt-popover" />').hide();
        $('body').append($promptPopover);
        if (!document.getElementById('ai-prompt-popover-style')) {
            const css = `
                .ai-prompt-popover { position:absolute; z-index:9999; max-width:360px; background:#fff; border:1px solid #ccc; box-shadow:0 4px 12px rgba(0,0,0,0.12); padding:12px; border-radius:6px; font-size:13px; line-height:1.4; color:#222; }
                .ai-prompt-popover .ai-pop-title { font-weight:700; margin:0 0 6px; }
                .ai-prompt-popover .ai-pop-meta { font-size:12px; color:#555; margin:0 0 8px; }
                .ai-prompt-popover .ai-pop-body { max-height:180px; overflow:auto; background:#f7f7f7; padding:8px; border-radius:4px; white-space:pre-wrap; word-break:break-word; }
                .ai-prompt-popover .ai-pop-actions { margin-top:8px; display:flex; gap:10px; align-items:center; }
                .ai-prompt-popover .ai-pop-close { position:absolute; top:6px; right:6px; background:none; border:none; color:#666; cursor:pointer; padding:0; font-size:14px; line-height:1; }
            `;
            $('head').append(`<style id="ai-prompt-popover-style">${css}</style>`);
        }
        return $promptPopover;
    }
    function hidePromptPopover() {
        if ($promptPopover) $promptPopover.hide();
        popoverVisible = false;
    }
    function renderPromptPopover($trigger) {
        const $opt = $globalPrompt.find('option:selected');
        if (!$opt.length) return hidePromptPopover();
        const promptText = $opt.val() || '';
        const title = $.trim($opt.data('title') || $opt.text() || '') || __('Prompt', 'aigude-tools');
        const provider = $opt.data('target-provider-label') || '';
        const lang = $opt.data('target-lang-label') || '';
        const editUrl = $opt.data('edit-url') || '';
        const metaParts = [];
        if (lang) metaParts.push(lang);
        if (provider) metaParts.push(provider);
        const meta = metaParts.join(' — ');
        const $pop = ensurePromptPopover();
        const bodyHtml = $('<div>').text(promptText).html(); // escape
        const editLink = editUrl ? `<a href="${editUrl}">${__('Edit in Prompts', 'aigude-tools')}</a>` : '';
        $pop.html(
            `<button type="button" class="ai-pop-close" aria-label="${__('Close', 'aigude-tools')}">×</button>` +
            `<div class="ai-pop-title">${title}</div>` +
            (meta ? `<div class="ai-pop-meta">${meta}</div>` : '') +
            `<div class="ai-pop-body">${bodyHtml}</div>` +
            `<div class="ai-pop-actions">${editLink}</div>`
        );
        const offset = $trigger.offset();
        const top = offset.top + $trigger.outerHeight() + 6;
        const left = offset.left;
        $pop.css({ top, left }).show();
        popoverVisible = true;
    }
    $(document).on('click', '#global-prompt-details-btn', function (e) {
        e.preventDefault();
        const $btn = $(this);
        if (popoverVisible) {
            hidePromptPopover();
        } else {
            renderPromptPopover($btn);
        }
    });
    $(document).on('click', '.ai-pop-close', function () {
        hidePromptPopover();
    });
    $(document).on('click', function (e) {
        if (!popoverVisible) return;
        const $t = $(e.target);
        if ($t.closest('.ai-prompt-popover').length || $t.is('#global-prompt-details-btn')) return;
        hidePromptPopover();
    });
    $globalPrompt.on('change.aiPromptPopover', function () {
        if (popoverVisible) {
            renderPromptPopover($('#global-prompt-details-btn'));
        }
    });

    $(document)
        .off('click.ai', '.generate-single')
        .on('click.ai', '.generate-single', function () {
            const $card = $(this).closest('.ai-card');
            const id    = $card.data('id');
            const $sel  = $card.find('.ai-prompt-select');

            const txt    = $.trim($card.find('.ai-prompt-input').val());
            const sel    = $sel.val();
            const prompt = txt !== '' ? txt : sel;

            const $btn      = $(this);
            const opt       = $sel.find('option:selected');
            const tplSrcLang= opt.data('src-lang') || 'auto';
            const promptLang= opt.data('prompt-lang') || 'auto';
            const targetInfo = getTargetInfoFromSelect($sel);
            const lang      = targetInfo.lang || 'EN';
            const langProvider = targetInfo.provider || 'deepl';

            setBtnLoading($btn);

            $.ajax({
                url: AJAX_URL,
                method: 'POST',
                dataType: 'json',
                timeout: 20000,
                    data: {
                        action: 'aigude_generate',
                        id,
                        prompt,
                        lang,
                        translation_provider: langProvider,
                        tpl_src_lang: tplSrcLang,
                        prompt_lang: promptLang,
                    _ajax_nonce: NONCE,
                }
            })
                .done(function (res) {
                    if (!res || !res.success) {
                        agToast((res && res.data && res.data.message) || STR.errorGenerating, 'error');
                        return;
                    }
                    showAlt($card, res.data.text);
                    agToast(STR.altTextGenerated, 'success');

                    const savedAltNow = String($card.data('saved-alt') || '').trim();
                    $card.find('.continue-single').toggle(!!savedAltNow);

                    if (res.data.creditsUsed) {
                        $card.find('.credits-used-box').fadeIn()
                            .find('.credits-used')
                            .text(STR.creditsUsedTpl.replace('%d', res.data.creditsUsed));
                    }
                })
                .fail(function (jqXHR) {
                    const msg =
                        jqXHR.responseJSON?.data?.message ||
                        (jqXHR.status === 401 || jqXHR.status === 403
                            ? STR.invalidApiKey
                            : STR.errorGenerating);
                    agToast(msg, 'error');
                })
                .always(function () { resetBtnToDefault($btn); });
        });

    $(document)
        .off('click.ai', '.continue-single')
        .on('click.ai', '.continue-single', function () {
            const $card = $(this).closest('.ai-card');
            const savedAlt = ($card.attr('data-saved-alt') || '').trim();
            showAlt($card, savedAlt);
        });

    $(document)
        .off('click.ai', '.save-single')
        .on('click.ai', '.save-single', function () {
            const $card = $(this).closest('.ai-card');
            const id  = $card.data('id');
            const alt = $card.find('.alt-text-input').val();

            $.ajax({
                url: AJAX_URL,
                method: 'POST',
                dataType: 'json',
                timeout: 20000,
                data: { action: 'aigude_apply', id, alt, _ajax_nonce: NONCE }
            })
                .done(function (res) {
                    if (!res || !res.success) {
                        agToast(STR.errorSaving, 'error');
                        return;
                    }
                    agToast(STR.altTextSaved, 'success');
                    $card.attr('data-saved-alt', alt);

                    const $suggBlock = $card.find('.suggestion-block');
                    const $suggText  = $card.find('.suggestion-text');
                    const cleanAlt   = (alt || '').trim();

                    if (cleanAlt) {
                        $suggText.text(cleanAlt);
                        $suggBlock.show();
                        $card.find('.continue-single').show();
                    } else {
                        $suggBlock.hide();
                        $card.find('.continue-single').hide();
                    }

                    if (isSkipEnabled()) {
                        $card.find('.ai-select').prop({ checked: false, disabled: true });
                        $card.addClass('ai-disabled');
                    }
                })
                .fail(function () { agToast(STR.errorSaving, 'error'); });
        });

    // --- Bulk generate -------------------------------------------------------
    $bulkBtn.off('click.ai').on('click.ai', async function () {
        const idQueue =
            Array.isArray(allSelectedIds) && allSelectedIds.length
                ? allSelectedIds.slice()
                : $('.ai-card').has('.ai-select:checked').map(function () {
                    return $(this).data('id');
                }).get();

        const total = idQueue.length;
        if (!total) { agToast(STR.pleaseSelectImage, 'error'); return; }
        if (!confirm(STR.confirmOverwrite)) return;

        const $btn = $(this);
        const skip = isSkipEnabled() ? 1 : 0;
        const $tplSel = $('#global-prompt');
        const prompt = $.trim($tplSel.val());
        const targetInfo = getTargetInfoFromSelect($tplSel);
        const lang = targetInfo.lang || 'EN';
        const langProvider = targetInfo.provider || 'deepl';

        const opt = $tplSel.find('option:selected');
        const tplSrcLang = opt.data('src-lang') || 'auto';
        const promptLang = opt.data('prompt-lang') || 'auto';

        let done = 0;
        let creditsUsedTotal = 0;
        let aborted = false;

        setBtnLoading($btn);
        $progressBar.css('width', '0%');
        $progressWrap.show();
        updateButtonLabelText($btn, done, total);

        const tick = () => {
            done++;
            updateButtonLabelText($btn, done, total);
            $progressBar.css('width', Math.round((done / total) * 100) + '%');
        };

        function parseError(jqXHR, res) {
            if (res && res.data) {
                return { message: res.data.message || STR.errorGenerating, fatal: res.data.code === 'invalid_api_key' };
            }
            if (jqXHR) {
                const isNonceFail = jqXHR.responseText === '-1' || jqXHR.responseText === '0';
                if (isNonceFail) return { message: STR.nonceFailed, fatal: true };
                if (jqXHR.status === 401 || jqXHR.status === 403) return { message: STR.invalidApiKey, fatal: true };
            }
            return { message: STR.errorGenerating, fatal: false };
        }

        try {
            for (const id of idQueue) {
                if (aborted) break;

                try {
                    const res = await $.ajax({
                        url: AJAX_URL,
                        method: 'POST',
                        dataType: 'json',
                        timeout: 30000,
                        data: {
                            action: 'aigude_generate_bulk',
                            ids: [id],
                            prompt,
                            lang,
                            translation_provider: langProvider,
                            skipExisting: skip,
                            tpl_src_lang: tplSrcLang,
                            prompt_lang: promptLang,
                            _ajax_nonce: NONCE,
                        },
                    });

                    if (res && res.success && res.data && res.data.results) {
                        const r = res.data.results[id] || {};

                        if (typeof res.data.creditsUsed === 'number') {
                            creditsUsedTotal += res.data.creditsUsed;
                            if (creditsUsedTotal > 0) {
                                $('#credit-summary').text(STR.totalCreditsUsedTpl.replace('%d', creditsUsedTotal));
                                $('#credit-summary-wrapper').fadeIn();
                            }
                        }

                        const $card = $(`.ai-card[data-id="${id}"]`);
                        if ($card.length) {
                            if (r.status === 'ok' && r.alt) {
                                $card.find('.continue-single').show();
                                showAlt($card, r.alt);
                                $card.find('.suggestion-text').text(r.alt);
                                $card.find('.suggestion-block').show();
                                $card.attr('data-saved-alt', r.alt);
                                if (skip) {
                                    $card.find('.ai-select').prop({ checked: false, disabled: true });
                                    $card.addClass('ai-disabled');
                                }
                            } else if (r.status === 'skipped' && skip) {
                                $card.find('.ai-select').prop({ checked: false, disabled: true });
                                $card.addClass('ai-disabled');
                            } else if (r.status === 'error') {
                                const { message, fatal } = parseError(null, { data: { message: r.message, code: r.code }});
                                agToast(message, 'error');
                                if (fatal) aborted = true;
                            }
                        }
                    } else {
                        const { message, fatal } = parseError(null, res);
                        agToast(message, 'error');
                        if (fatal) aborted = true;
                    }
                } catch (jqXHR) {
                    const { message, fatal } = parseError(jqXHR, jqXHR.responseJSON);
                    agToast(message, 'error');
                    if (fatal) aborted = true;
                }

                tick();
                if (aborted) break;
            }
        } finally {
            setTimeout(() => $progressWrap.fadeOut(200), 300);
            const base = $bulkBtn.data('default-text').replace('%s', total);
            $btn.prop('disabled', false).html(`${base} — ${total * PER_CREDIT} ${STR.creditsWord}`);
            if (aborted) {
                $('#credit-summary').text(STR.invalidApiKey);
                $('#credit-summary-wrapper').fadeIn();
            }
        }

        if (creditsUsedTotal > 0) {
            $('#credit-summary').text(STR.totalCreditsUsedTpl.replace('%d', creditsUsedTotal));
            $('#credit-summary-wrapper').fadeIn();
        }
    });

    // --- Select all (across pages) -------------------------------------------
    $(document)
        .off('change.ai', '#select-all-allpages')
        .on('change.ai', '#select-all-allpages', function () {
            const on = $(this).is(':checked');

            if (on) {
                $selectAllPageCb.prop('checked', false);
                freezePageCheckboxes(true);

                $('#allpages-hint').show().find('#allpages-count').text('…');
                fetchAllMatchingIds()
                    .done(function (res) {
                        if (res && res.success && Array.isArray(res.data.ids)) {
                            allSelectedIds = res.data.ids;
                            $('#allpages-count').text(res.data.count || allSelectedIds.length);
                        } else {
                            allSelectedIds = null;
                            $selectAllAllPagesCb.prop('checked', false);
                            $('#allpages-hint').hide();
                            freezePageCheckboxes(false);
                            agToast(STR.error, 'error');
                        }
                        updateButtonLabels();
                    })
                    .fail(function () {
                        allSelectedIds = null;
                        $selectAllAllPagesCb.prop('checked', false);
                        $('#allpages-hint').hide();
                        freezePageCheckboxes(false);
                        agToast(STR.error, 'error');
                        updateButtonLabels();
                    });
            } else {
                allSelectedIds = null;
                $('#allpages-hint').hide();
                freezePageCheckboxes(false);
                updateButtonLabels();
            }
        });

    // --- Select all (this page) ----------------------------------------------
    $selectAllPageCb.off('change.ai').on('change.ai', function () {
        const selectAll = this.checked;

        if (selectAll) {
            if ($selectAllAllPagesCb.is(':checked')) {
                $selectAllAllPagesCb.prop('checked', false);
            }
            allSelectedIds = null;
            $('#allpages-hint').hide();
            freezePageCheckboxes(false);
        }

        const skip = isSkipEnabled();
        const $cards = $('.ai-card:visible');

        $cards.each(function () {
            const $card     = $(this);
            const hasAlt    = cardHasSavedAlt($card);
            const cb        = $card.find('.ai-select')[0];

            cb.indeterminate = false;
            cb.disabled      = false;

            if (skip && hasAlt) {
                cb.checked = false;
                cb.disabled = true;
                $card.addClass('ai-disabled');
            } else {
                cb.checked = selectAll;
                $card.removeClass('ai-disabled');
            }
        });

        updateButtonLabels();
    });

    // Individual checkbox
    $(document)
        .off('change.ai', '.ai-select')
        .on('change.ai', '.ai-select', function () {
            if (!$(this).is(':checked')) {
                $selectAllPageCb.prop('checked', false);
            } else {
                const all = $('.ai-select').length;
                const checked = $('.ai-select:checked').length;
                if (all === checked) $selectAllPageCb.prop('checked', true);
            }
            updateButtonLabels();
        });

    // --- Skip Existing toggle -------------------------------------------------
    $skipCb.off('change.ai-skip').on('change.ai-skip', function () {
        const across = $selectAllAllPagesCb.is(':checked');

        if (across) {
            freezePageCheckboxes(true);
            $('#allpages-hint').show().find('#allpages-count').text('…');
            fetchAllMatchingIds()
                .done(function (res) {
                    if (res && res.success && Array.isArray(res.data.ids)) {
                        allSelectedIds = res.data.ids;
                        $('#allpages-count').text(res.data.count || allSelectedIds.length);
                    } else {
                        allSelectedIds = null;
                        $selectAllAllPagesCb.prop('checked', false);
                        $('#allpages-hint').hide();
                        freezePageCheckboxes(false);
                        agToast(STR.error, 'error');
                    }
                    updateButtonLabels();
                })
                .fail(function () {
                    allSelectedIds = null;
                    $selectAllAllPagesCb.prop('checked', false);
                    $('#allpages-hint').hide();
                    freezePageCheckboxes(false);
                    agToast(STR.error, 'error');
                    updateButtonLabels();
                });
        } else {
            applySkipLogicToCards();
            if ($selectAllPageCb.is(':checked')) {
                $selectAllPageCb.trigger('change');
            } else {
                updateButtonLabels();
            }
        }
    });

    // --- Initial pass ---------------------------------------------------------
    updateButtonLabels();
    applySkipLogicToCards();
});
