// assets/js/grid-actions.js
(function ($) {
    const { __ } = wp.i18n;

    // -----------------------------------------------------------------------
    // i18n strings (translated via wp.i18n + JSON catalogs)
    // -----------------------------------------------------------------------
    const STR = {
        selectTitle:        __('Select images', 'aigude-tools'),
        /* translators: Label on a button that selects all currently visible items in the media grid */
        selectAllVisible:   __('Select all (visible)', 'aigude-tools'),
        /* translators: Label on a button that selects all results across all pages in the media grid */
        selectAllResults:   __('Select all (all results)', 'aigude-tools'),
        loading:            __('Loading…', 'aigude-tools'),
        selectedAll:        __('Selected', 'aigude-tools'),
        error:              __('Error', 'aigude-tools'),
        skipExistingLabel:  __('Skip images that already have alt text', 'aigude-tools'),
        generating:         __('Generating...', 'aigude-tools'),
        done:               __('Done', 'aigude-tools'),
        confirmOverwrite:   __('This will overwrite existing alt texts. Are you sure?', 'aigude-tools'),
        invalidApiKey:      __('Invalid or unauthorized API key.', 'aigude-tools'),
        nonceFailed:        __('Security check failed. Please reload the page.', 'aigude-tools'),
        creditsWord:        __('credits', 'aigude-tools'),
        /* translators: %d = total number of credits used across the whole operation */
        totalCreditsUsedTpl: __('Total credits used: %d', 'aigude-tools'),
        lockedByPrompt:     __('Language locked by selected prompt.', 'aigude-tools'),
    };

    // -----------------------------------------------------------------------
    // Fast tooltip for selected thumbnails
    // -----------------------------------------------------------------------
    let hoverTip = null;
    function ensureHoverTip() {
        if (hoverTip) return hoverTip;
        hoverTip = $('<div class="ai-hover-tip" />')
            .css({
                position: 'fixed',
                padding: '4px 6px',
                background: '#1e1e1e',
                color: '#fff',
                'border-radius': '3px',
                'font-size': '11px',
                'line-height': '1.2',
                'box-shadow': '0 2px 8px rgba(0,0,0,0.18)',
                'z-index': 9999,
                'white-space': 'pre-line',
                'max-width': '260px',
                'word-wrap': 'break-word',
                'pointer-events': 'none',
                opacity: 0,
            })
            .appendTo('body');
        return hoverTip;
    }
    function showHoverTip(text, x, y) {
        const tip = ensureHoverTip();
        tip.text(text || '');
        tip.css({ left: x + 12, top: y + 12, opacity: 1 });
    }
    function hideHoverTip() {
        if (hoverTip) hoverTip.css('opacity', 0);
    }

    // -----------------------------------------------------------------------
    // Prompt details popover (global prompt only)
    // -----------------------------------------------------------------------
    let $promptPopover = null;
    let popoverVisible = false;
    function ensurePromptPopover() {
        if ($promptPopover) return $promptPopover;
        $promptPopover = $('<div class="ai-prompt-popover" />').hide();
        $('body').append($promptPopover);
        if (!document.getElementById('ai-prompt-popover-style')) {
            const css = `
                .ai-prompt-popover { position:absolute; z-index:9999; max-width:360px; background:#fff; border:1px solid #ccc; box-shadow:0 4px 12px rgba(0,0,0,0.12); padding:12px; border-radius:6px; font-size:13px; line-height:1.4; color:#222; }
                .ai-prompt-popover .ai-pop-title { font-weight:700; margin:0 0 6px; }
                .ai-prompt-popover .ai-pop-meta { font-size:12px; color:#555; margin:0 0 8px; }
                .ai-prompt-popover .ai-pop-body { max-height:180px; overflow:auto; background:#f7f7f7; padding:8px; border-radius:4px; white-space:pre-wrap; word-break:break-word; }
                .ai-prompt-popover .ai-pop-actions { margin-top:8px; display:flex; gap:10px; align-items:center; }
                .ai-prompt-popover .ai-pop-close { position:absolute; top:6px; right:6px; background:none; border:none; color:#666; cursor:pointer; padding:0; font-size:14px; line-height:1; }
            `;
            $('head').append(`<style id="ai-prompt-popover-style">${css}</style>`);
        }
        return $promptPopover;
    }
    function hidePromptPopover() {
        if ($promptPopover) $promptPopover.hide();
        popoverVisible = false;
    }
    function renderPromptPopover($trigger, $select) {
        const $opt = $select.find('option:selected');
        if (!$opt.length) return hidePromptPopover();
        const promptText = $opt.val() || '';
        const title = $.trim($opt.data('title') || $opt.text() || '') || __('Prompt', 'aigude-tools');
        const provider = $opt.data('target-provider-label') || '';
        const lang = $opt.data('target-lang-label') || '';
        const editUrl = $opt.data('edit-url') || '';
        const metaParts = [];
        if (lang) metaParts.push(lang);
        if (provider) metaParts.push(provider);
        const meta = metaParts.join(' — ');
        const $pop = ensurePromptPopover();
        const bodyHtml = $('<div>').text(promptText).html();
        const editLink = editUrl ? `<a href="${editUrl}">${__('Edit in Prompts', 'aigude-tools')}</a>` : '';
        $pop.html(
            `<button type="button" class="ai-pop-close" aria-label="${__('Close', 'aigude-tools')}">×</button>` +
            `<div class="ai-pop-title">${title}</div>` +
            (meta ? `<div class="ai-pop-meta">${meta}</div>` : '') +
            `<div class="ai-pop-body">${bodyHtml}</div>` +
            `<div class="ai-pop-actions">${editLink}</div>`
        );
        const offset = $trigger.offset();
        const top = offset.top + $trigger.outerHeight() + 6;
        const left = offset.left;
        $pop.css({ top, left }).show();
        popoverVisible = true;
    }
    $(document).on('click', '#global-prompt-details-btn', function (e) {
        e.preventDefault();
        const $btn = $(this);
        const $sel = $('#global-prompt');
        if (popoverVisible) {
            hidePromptPopover();
        } else {
            renderPromptPopover($btn, $sel);
        }
    });
    $(document).on('click', '.ai-pop-close', function () {
        hidePromptPopover();
    });
    $(document).on('click', function (e) {
        if (!popoverVisible) return;
        const $t = $(e.target);
        if ($t.closest('.ai-prompt-popover').length || $t.is('#global-prompt-details-btn')) return;
        hidePromptPopover();
    });
    $(document).on('change', '#global-prompt', function () {
        if (popoverVisible) {
            renderPromptPopover($('#global-prompt-details-btn'), $(this));
        }
    });

    // -----------------------------------------------------------------------
    // State & config (data only from PHP)
    // -----------------------------------------------------------------------
    let frameAll = null;
    let frameSkip = null;
    let selectedIds = [];

    const cfg = window.aigudeToolsGridData || {};
    const DEFAULT_SKIP = typeof cfg.defaultSkip === 'boolean' ? cfg.defaultSkip : true;

    // Cache progress elements
    const $bar  = $(cfg.selectors?.progressBar || '');
    const $wrap = $(cfg.selectors?.progressWrap || '');
    let PROG_TOTAL = 0;
    let PROG_DONE  = 0;

    // Notifications
    function toast(msg, type) {
        const el = $(`<div class="ag-toast ag-toast-${type || 'error'}">${msg}</div>`);
        $('body').append(el);
        el.fadeIn(150).delay(1200).fadeOut(200, () => el.remove());
    }

    function parseAjaxError(jqXHR, res) {
        // Nonce failures
        const text = jqXHR?.responseText;
        if (text === '-1' || text === '0') {
            return { message: STR.nonceFailed, fatal: true };
        }
        // JSON error from server
        if (res && res.data) {
            const code  = res.data.code || res.data.error || '';
            const msg   = res.data.message || res.data.error || STR.error;
            const fatal = code === 'invalid_api_key' || code === 'unauthorized' || code === 'forbidden';
            return { message: msg, fatal };
        }
        // HTTP status
        const s = jqXHR?.status;
        if (s === 401 || s === 403) {
            return { message: STR.invalidApiKey, fatal: true };
        }
        return { message: STR.error, fatal: false };
    }

    function getTargetInfoFromSelect($promptSelect) {
        const opt = $promptSelect.find('option:selected');
        const provider = (opt.data('target-provider') || 'deepl').toString();
        const providerLabel = (opt.data('target-provider-label') || provider).toString();
        const lang = (opt.data('target-lang') || '').toString();
        const langLabel = (opt.data('target-lang-label') || lang).toString();
        return { provider, providerLabel, lang, langLabel };
    }

    function applyTemplateTarget($promptSelect, $info) {
        if (!$promptSelect.length || !$info.length) return;
        const info = getTargetInfoFromSelect($promptSelect);
        const parts = [];
        if (info.providerLabel) parts.push(info.providerLabel);
        if (info.langLabel) parts.push(info.langLabel);
        $info.text(parts.join(' — '));
        $info.data('targetProvider', info.provider);
        $info.data('targetLang', info.lang);
        $info.data('targetLangLabel', info.langLabel);
        $info.data('targetProviderLabel', info.providerLabel);
    }

    // -----------------------------------------------------------------------
    // Progress Bar
    // -----------------------------------------------------------------------
    function progStart(total) {
        PROG_TOTAL = Math.max(1, total);
        PROG_DONE  = 0;
        if ($wrap.length) $wrap.show();
        progStep(0);
    }

    function progStep(delta) {
        PROG_DONE = Math.min(PROG_TOTAL, PROG_DONE + (delta || 0));
        const pct = (PROG_DONE / PROG_TOTAL) * 100;
        if ($bar.length) {
            $bar.css('width', pct + '%');
            $bar.attr({
                'aria-valuemin': 0,
                'aria-valuemax': PROG_TOTAL,
                'aria-valuenow':  PROG_DONE,
            });
        }
    }

    function progFinish() {
        if ($bar.length) $bar.css('width', '100%');
        setTimeout(() => { if ($wrap.length) $wrap.fadeOut(200); }, 500);
    }

    // -----------------------------------------------------------------------
    // Labels / button state
    // -----------------------------------------------------------------------
    function updateGenerateBtnLabel() {
        const $btn = $(cfg.selectors?.generateBtn || '');
        if (!$btn.length) return;

        const base = $btn.data('baseLabel') || $btn.text();
        if (!$btn.data('baseLabel')) $btn.data('baseLabel', base);

        const per   = cfg.perImageCredits || 3;
        const word  = STR.creditsWord;
        const count = Array.isArray(selectedIds) ? selectedIds.length : 0;
        const credit = count * per;

        $btn.text(`${base} ${count} — ${credit} ${word}`);
        $btn.prop('disabled', count === 0);
    }

    // -----------------------------------------------------------------------
    // Server toggle for "skip existing" (used by media modal query filter)
    // -----------------------------------------------------------------------
    const setSkipMode = (mode) =>
        $.post(cfg.ajaxUrl, {
            action: 'aigude_set_skip_mode',
            mode:   mode ? 1 : 0,
            _ajax_nonce: cfg.nonce,
        });

    // -----------------------------------------------------------------------
    // Mini-grid rendering (selected thumbnails)
    // -----------------------------------------------------------------------
    function niceFilenameLabel(name) {
        if (!name) return '';
        return name.replace(/-\d{2,5}x\d{2,5}$/, '').trim(); // strip trailing -WxH
    }

    function renderGrid(models) {
        const $grid = $(cfg.selectors?.miniGrid || '');
        if (!$grid.length) return;

        $grid.empty();
        const frag = document.createDocumentFragment();

        models.forEach((m) => {
            const id = m.id || (typeof m.get === 'function' ? m.get('id') : null);
            const attrs = m.attributes || (typeof m.toJSON === 'function' ? m.toJSON() : {});
            const sizes = attrs.sizes || {};
            const url = (sizes.thumbnail && sizes.thumbnail.url) || attrs.url;

            const raw  = attrs.originalFilename || attrs.filename || attrs.title || 'ID ' + id;
            const name = niceFilenameLabel(raw);

            const alt  = (attrs.alt || (m.get && m.get('alt')) || '').toString().trim();
            const tooltip = alt ? `${name}\nAlt: ${alt}` : name;

            const linkBase =
                (cfg.urls && cfg.urls.library) ||
                (wp.media.view.settings.adminUrl + 'upload.php');

            const a = document.createElement('a');
            a.href   = `${linkBase}?item=${id}`;
            a.target = '_blank';
            a.rel    = 'noopener noreferrer';
            a.className = 'ai-selected-thumb';
            a.dataset.id = id;
            a.title      = '';
            a.setAttribute('aria-label', tooltip);
            a.setAttribute('data-tooltip', tooltip);

            const img = document.createElement('img');
            img.src = url;
            img.alt = '';
            a.appendChild(img);

            const caption = document.createElement('span');
            caption.className = 'ai-caption';
            caption.textContent = name;
            a.appendChild(caption);

            frag.appendChild(a);
        });

        if ($grid[0]) $grid[0].appendChild(frag);
    }

    function updateMiniGridTooltips(results) {
        Object.entries(results || {}).forEach(([id, r]) => {
            const $a = $(`${cfg.selectors.miniGrid} .ai-selected-thumb[data-id="${id}"]`);
            if (!$a.length) return;

            const name = $a.find('.ai-caption').text().trim();
            const alt  =
                (r.status === 'ok' && r.alt) ? r.alt :
                    (r.status === 'skipped' && r.existing_alt) ? r.existing_alt :
                        '';

            if (alt) {
                const tooltip = `${name}\nAlt: ${alt}`;
                $a.attr({
                    title: '',
                    'aria-label': tooltip,
                    'data-tooltip': tooltip,
                });
                try {
                    const att = wp.media && wp.media.attachment ? wp.media.attachment(+id) : null;
                    if (att) att.set('alt', alt);
                } catch (e) { /* no-op */ }
            }
        });
    }

    // -----------------------------------------------------------------------
    // Media frame setup
    // -----------------------------------------------------------------------
    function bindSelect(frame) {
        frame.on('select', () => {
            const selection   = frame.state().get('selection');
            const attachments = selection.toArray();
            selectedIds = attachments.map((a) => a.id);
            window.selectedIds = selectedIds; // optional debug
            renderGrid(attachments);
            updateGenerateBtnLabel();
        });
    }

    function createFrame(initialSkip) {
        const frame = wp.media({
            title: STR.selectTitle,
            multiple: true,
            library: { type: 'image' },
        });

        bindSelect(frame);
        attachSelectAllButtons(frame);
        attachSkipTop(frame, initialSkip);
        return frame;
    }

    function attachSelectAllButtons(frame) {
        frame.on('open', () => {
            const $primary = frame.$el.find('.media-frame-toolbar .media-toolbar-primary');
            if ($primary.find('.ai-select-all-visible').length) return;

            const $btnVisible = $('<button type="button" class="button ai-select-all-visible"/>')
                .text(STR.selectAllVisible);

            const $btnResults = $('<button type="button" class="button ai-select-all-results" style="margin-left:6px;"/>')
                .text(STR.selectAllResults);

            $btnVisible.on('click', () => {
                const state     = frame.state();
                const library   = state.get('library');
                const selection = state.get('selection');
                selection.add(library.models);
            });

            $btnResults.on('click', async () => {
                const state     = frame.state();
                const library   = state.get('library');
                const selection = state.get('selection');
                const max       = cfg.selectAllMax || 2000;

                const props = library.props ? library.props.toJSON() : {};
                const all   = wp.media.query(props);

                $btnResults.prop('disabled', true).text(STR.loading);

                try {
                    while (all.hasMore() && all.length < max) {
                        await all.more();
                    }
                    selection.add(all.models);
                    $btnResults.text(STR.selectedAll).prop('disabled', false);
                } catch (e) {
                    console.error('Select-all results failed:', e);
                    $btnResults.text(STR.error).prop('disabled', false);
                }
            });

            $primary.prepend($btnResults).prepend($btnVisible);
        });
    }

    function attachSkipTop(frame, initialChecked) {
        const inject = () => {
            const $date = frame.$el
                .find('.attachments-browser .media-toolbar #media-attachment-date-filters')
                .first();
            if (!$date.length) return false;

            const $secondary = $date.closest('.media-toolbar-secondary');
            if (!$secondary.length) return false;

            if ($secondary.find('.ai-skip-existing-top').length) return true;

            const $skip = $(`
                <label class="ai-skip-existing-top" style="align-items:center;gap:6px;margin-left:10px;">
                  <input type="checkbox" class="ai-skip-existing-cb">
                  <span>${STR.skipExistingLabel}</span>
                </label>
            `);
            $skip.find('input').prop('checked', !!initialChecked);

            $date.after($skip);

            $skip.find('input').on('change', function () {
                const mode = this.checked ? 1 : 0;
                setSkipMode(mode).always(() => {
                    try {
                        const lib = frame.state().get('library');
                        lib.props.set('ai_refresh', Date.now());
                        lib.fetch();
                    } catch (e) {
                        const prev = frame.state().get('selection').toArray().map((m) => m.id);
                        frame.close();
                        const next = createFrame(!!mode);
                        next.on('open', () => {
                            const sel = next.state().get('selection');
                            prev.forEach((id) => sel.add(wp.media.attachment(id)));
                        });
                        next.on('close', () => setSkipMode(0));
                        if (mode) frameSkip = next;
                        else frameAll = next;
                        next.open();
                    }
                });
            });

            return true;
        };

        frame.on('open', () => { if (!inject()) setTimeout(inject, 0); });
        frame.on('content:render:browse', inject);
    }

    // -----------------------------------------------------------------------
    // AJAX helpers
    // -----------------------------------------------------------------------
    async function postChunk(ids, payload) {
        return $.ajax({
            url: cfg.ajaxUrl,
            method: 'POST',
            dataType: 'json',
            data: Object.assign({
                action: 'aigude_generate_bulk',
                ids: ids,
                _ajax_nonce: cfg.nonce,
            }, payload),
        });
    }

    function chunk(arr, size) {
        const n = Math.max(1, Number(size) || 1);
        const out = [];
        for (let i = 0; i < arr.length; i += n) out.push(arr.slice(i, i + n));
        return out;
    }

    // -----------------------------------------------------------------------
    // DOM ready
    // -----------------------------------------------------------------------
    $(function () {
        const $selectBtn   = $(cfg.selectors?.selectBtn || '');
        const $generateBtn = $(cfg.selectors?.generateBtn || '');

        updateGenerateBtnLabel();

        // Instant hover tooltip on selected thumbnails
        $(document)
            .off('mouseenter.aiTip mousemove.aiTip mouseleave.aiTip', '.ai-selected-thumb')
            .on('mouseenter.aiTip', '.ai-selected-thumb', function (e) {
                const txt = $(this).attr('data-tooltip') || $(this).attr('aria-label') || '';
                if (txt) showHoverTip(txt, e.clientX, e.clientY);
            })
            .on('mousemove.aiTip', '.ai-selected-thumb', function (e) {
                const txt = $(this).attr('data-tooltip') || $(this).attr('aria-label') || '';
                if (txt) showHoverTip(txt, e.clientX, e.clientY);
            })
            .on('mouseleave.aiTip', '.ai-selected-thumb', function () {
                hideHoverTip();
            });

        const $promptSel = $(cfg.selectors?.prompt || '');
        const $targetInfo = $('#grid-target-info');
        const applyLock = () => applyTemplateTarget($promptSel, $targetInfo);
        $promptSel.off('change.aiPromptLock').on('change.aiPromptLock', applyLock);
        applyLock();

        $(cfg.selectors?.skipExisting || '').on('change', function () {
            if (!this.checked) setSkipMode(0);
        });

        $selectBtn.on('click', function (e) {
            e.preventDefault();
            const skip = DEFAULT_SKIP;

            setSkipMode(skip ? 1 : 0).always(function () {
                if (skip) {
                    if (!frameSkip) {
                        frameSkip = createFrame(true);
                        frameSkip.on('close', () => setSkipMode(0));
                    }
                    frameSkip.open();
                } else {
                    if (!frameAll) {
                        frameAll = createFrame(false);
                        frameAll.on('close', () => setSkipMode(0));
                    }
                    frameAll.open();
                }
            });
        });

        $generateBtn.on('click', async function () {
            if (!selectedIds.length) return;

            if (!window.confirm(STR.confirmOverwrite)) return;

            const $btn = $(this);
            const defaultLabel = $btn.text();
            $btn.prop('disabled', true).text(STR.generating);

            const skipExisting = $(cfg.selectors?.skipExisting || '').is(':checked') ? 1 : 0;
            const prompt = $promptSel.val() || '';
            const targetInfo = getTargetInfoFromSelect($promptSel);
            const lang   = targetInfo.lang || 'EN';
            const langProvider = targetInfo.provider || 'deepl';

            const BATCH_SIZE = Math.max(1, Number(cfg.batchSize) || 12);
            const batches = chunk(selectedIds, BATCH_SIZE);

            let totalCredits = 0;
            let aborted = false;

            progStart(selectedIds.length);

            const opt         = $promptSel.find('option:selected');
            const tplSrcLang  = opt.data('src-lang') || 'auto';
            const promptLang  = opt.data('prompt-lang') || 'auto';

            try {
                for (const ids of batches) {
                    if (aborted) break;

                    try {
                        const res = await postChunk(ids, {
                            prompt, lang, skipExisting,
                            translation_provider: langProvider,
                            tpl_src_lang: tplSrcLang,
                            prompt_lang:  promptLang,
                        });

                        if (res && res.success && res.data) {
                            updateMiniGridTooltips(res.data.results);

                            if (typeof res.data.creditsUsed === 'number') {
                                totalCredits += res.data.creditsUsed;
                                if (totalCredits > 0) {
                                    $('#credit-summary')
                                        .text(STR.totalCreditsUsedTpl.replace('%d', totalCredits));
                                    $('#credit-summary-wrapper').fadeIn();
                                }
                            }

                            const processed = res.data.results ? Object.keys(res.data.results).length : ids.length;
                            progStep(processed);
                        } else {
                            const { message, fatal } = parseAjaxError(null, res);
                            toast(message, 'error');
                            progStep(ids.length);
                            if (fatal) { aborted = true; break; }
                        }
                    } catch (jqXHR) {
                        const res = jqXHR.responseJSON;
                        const { message, fatal } = parseAjaxError(jqXHR, res);
                        toast(message, 'error');
                        progStep(ids.length);
                        if (fatal) { aborted = true; break; }
                    }
                }
            } finally {
                progFinish();

                if (totalCredits > 0) {
                    $('#credit-summary')
                        .text(STR.totalCreditsUsedTpl.replace('%d', totalCredits));
                    $('#credit-summary-wrapper').fadeIn();
                }
                if (aborted) {
                    $('#credit-summary').text(STR.invalidApiKey);
                    $('#credit-summary-wrapper').fadeIn();
                }

                $btn.text(STR.done);
                setTimeout(() => { $btn.text(defaultLabel).prop('disabled', false); }, 1200);
            }
        });
    });
})(jQuery);
