<?php
/**
 * Plugin Name:       AiGude Tools
 * Plugin URI:        https://wordpress.org/plugins/aigude-tools/
 * Description:       Generate and manage image alt text with AI — supports bulk actions, custom multilingual prompts, and full Media Library integration.
 * Version:           2.4.0
 * Requires at least: 6.0
 * Requires PHP:      7.4
 * Author:            Pagemachine AG
 * Author URI:        https://pagemachine.de
 * Text Domain:       aigude-tools
 * Domain Path:       /languages
 * License:           GPL-2.0-or-later
 * License URI:       https://www.gnu.org/licenses/gpl-2.0.html
 */


if (!defined('ABSPATH')) {
    exit;
}

require_once plugin_dir_path(__FILE__) . 'includes/class-aigude-translation-service.php';
require_once plugin_dir_path(__FILE__) . 'includes/class-aigude-admin-ui.php';
require_once plugin_dir_path(__FILE__) . 'includes/class-aigude-media-controller.php';
require_once plugin_dir_path(__FILE__) . 'includes/class-aigude-media-query.php';
require_once plugin_dir_path(__FILE__) . 'includes/admin-prompts.php';
require_once plugin_dir_path(__FILE__) . 'includes/admin-settings.php';
require_once plugin_dir_path(__FILE__) . 'includes/list-view.php';
require_once plugin_dir_path(__FILE__) . 'includes/grid-view.php';


final class AIGUDE_Tools_Plugin {
    /*** Constants ***/
    const MENU_SLUG     = 'aigude-tools';
    const NONCE_ACTION  = 'aigude-tools';
    const PER_IMAGE_CREDITS = 3;
    const BUILTIN_PROMPT_ID = 'aigude-default-prompt';

    const API_URL_IMG2DESC = 'https://credits.aigude.io/img2desc_file';
    const API_URL_CREDITS  = 'https://credits.aigude.io/remaining_credits';
    const API_URL_TRANSLATE_PROVIDERS = AIGUDE_Translation_Service::API_URL_TRANSLATE_PROVIDERS;
    const DEFAULT_TRANSLATION_PROVIDER = AIGUDE_Translation_Service::DEFAULT_PROVIDER;

    /*** Configuration ***/
    private static array $IMAGE_MIME_TYPES = ['image/jpeg','image/png','image/bmp','image/tiff','image/webp'];

    /*** Modules ***/
    private static ?AIGUDE_Admin_UI $admin_ui = null;
    private static ?AIGUDE_Media_Controller $media_controller = null;
    private static ?AIGUDE_Media_Query $media_query = null;

    /**
     * Return image MIME types that should be considered when searching attachments.
     */
    public static function get_image_mime_types(): array {
        return self::$IMAGE_MIME_TYPES;
    }

    /**
     * Build an absolute path inside the plugin directory, optionally appending a relative segment.
     */
    public static function plugin_path(string $relative = ''): string {
        $base = plugin_dir_path(__FILE__);
        return $relative !== '' ? $base . ltrim($relative, '/\\') : $base;
    }

    /**
     * Build a plugin-relative URL, mirroring plugin_path() for enqueue/asset helpers.
     */
    public static function plugin_url(string $relative = ''): string {
        $base = plugin_dir_url(__FILE__);
        return $relative !== '' ? $base . ltrim($relative, '/\\') : $base;
    }

    /*** Translation helpers (proxied to AIGUDE_Translation_Service) ***/
    /**
     * Fetch the currently selected translation provider slug from settings.
     */
    public static function get_translation_provider(): string {
        $provider = AIGUDE_Translation_Service::get_translation_provider();
        return self::enforce_provider_policy($provider);
    }

    /**
     * Convert a provider slug into a human readable label.
     */
    public static function get_translation_provider_label(?string $provider = null): string {
        $resolved = $provider !== null ? $provider : self::get_translation_provider();
        return AIGUDE_Translation_Service::get_translation_provider_label($resolved);
    }

    /**
     * List all available translation providers.
     *
     * @return string[]
     */
    public static function get_available_translation_providers(): array {
        return AIGUDE_Translation_Service::get_available_translation_providers();
    }

    /**
     * Retrieve provider metadata, optionally bypassing caches.
     */
    public static function get_translation_providers_metadata(bool $force = false): array {
        return AIGUDE_Translation_Service::get_translation_providers_metadata($force);
    }

    /**
     * List supported translation languages for the active (or provided) provider.
     */
    public static function get_translation_languages(?string $provider = null): array {
        if ($provider === null) {
            $provider = self::get_translation_provider();
        }
        return AIGUDE_Translation_Service::get_translation_languages($provider);
    }

    /**
     * Expose the DeepL language list for UIs that need canonical codes.
     */
    public static function get_deepl_languages(): array {
        return AIGUDE_Translation_Service::get_deepl_languages();
    }

    public static function eu_only_providers_enabled(): bool {
        return get_option('aigude_eu_only_providers', '0') === '1';
    }

    public static function get_provider_region(string $provider): string {
        $slug = AIGUDE_Translation_Service::normalize_translation_provider($provider);
        $meta = self::get_translation_providers_metadata();

        if (isset($meta[$slug]) && is_array($meta[$slug]) && isset($meta[$slug]['region'])) {
            return self::normalize_provider_region((string) $meta[$slug]['region']);
        }

        return self::normalize_provider_region('global');
    }

    public static function provider_is_eu(?string $provider): bool {
        if (!$provider) {
            return false;
        }
        return self::get_provider_region($provider) === 'eu';
    }

    public static function get_default_eu_provider(): string {
        return 'deepl';
    }

    public static function filter_providers_by_region(array $meta, bool $euOnly): array {
        if (!$euOnly) {
            return $meta;
        }
        $filtered = [];
        foreach ($meta as $slug => $info) {
            $normalized = strtolower(trim((string) $slug));
            if (self::provider_is_eu($normalized)) {
                $filtered[$normalized] = $info;
            }
        }
        return $filtered;
    }

    private static function normalize_provider_region(?string $region): string {
        $normalized = strtolower(trim((string) $region));
        return $normalized === 'eu' ? 'eu' : 'global';
    }

    private static function enforce_provider_policy(string $provider): string {
        $normalized = strtolower(trim($provider));
        if (! self::eu_only_providers_enabled()) {
            return $normalized;
        }
        if (self::provider_is_eu($normalized)) {
            return $normalized;
        }

        $meta = self::get_translation_providers_metadata();
        $fallback = self::get_default_eu_provider();
        if (!isset($meta[$fallback])) {
            foreach ($meta as $slug => $_info) {
                if (self::provider_is_eu($slug)) {
                    $fallback = $slug;
                    break;
                }
            }
        }

        if (isset($meta[$fallback]) && self::provider_is_eu($fallback)) {
            update_option('aigude_translation_provider', $fallback);
            return $fallback;
        }

        return $normalized;
    }

    /**
     * Describe the site's locale in a way the Credits API understands.
     */
    public static function describe_site_language(?string $provider = null): array {
        return AIGUDE_Translation_Service::describe_site_language($provider);
    }

    /**
     * Return the built-in default prompt template (non-erasable).
     */
    public static function get_builtin_prompt_template(): array {
        $prompt_text = __('Describe the essential content of the picture briefly and concisely. Limit the text to a very short sentence', 'aigude-tools');
        $title       = __('Default Prompt', 'aigude-tools');

        $deepl_langs  = self::get_translation_languages('deepl');
        $google_langs = self::get_translation_languages('google');
        $site_lang    = self::describe_site_language('deepl');
        $site_code    = strtoupper($site_lang['code'] ?? '');

        $provider = 'deepl';
        $lang     = 'EN';

        if ($site_code !== '') {
            if (isset($deepl_langs[$site_code])) {
                $provider = 'deepl';
                $lang     = $site_code;
            } elseif (isset($google_langs[$site_code])) {
                $provider = 'google';
                $lang     = $site_code;
            }
        }

        $provider_langs = $provider === 'google' ? $google_langs : $deepl_langs;
        if ($lang === '' || !isset($provider_langs[$lang])) {
            if (isset($deepl_langs['EN'])) {
                $provider = 'deepl';
                $lang     = 'EN';
            } elseif (!empty($deepl_langs)) {
                $provider = 'deepl';
                $lang     = (string) array_key_first($deepl_langs);
            } elseif (!empty($google_langs)) {
                $provider = 'google';
                $lang     = (string) array_key_first($google_langs);
            }
        }

        $recents = [];
        if ($provider && $lang) {
            $recents[$provider] = [$lang];
        }

        return [
                'id'                  => self::BUILTIN_PROMPT_ID,
                'title'               => $title,
                'prompt'              => $prompt_text,
                'src_lang'            => 'auto',
                'prompt_lang'         => 'auto',
                'target_provider'     => $provider,
                'target_lang'         => $lang,
                'eu_only_providers'   => '0',
                'recent_target_langs' => $recents,
                'builtin'             => '1',
        ];
    }

    /**
     * Provide a human-friendly description of the user's preferred translation language.
     */
    public static function describe_language_preference(?string $provider = null): array {
        return AIGUDE_Translation_Service::describe_language_preference($provider);
    }

    /**
     * Normalize arbitrary language input into the provider's target language code.
     */
    public static function resolve_target_lang_code(string $code, ?string $provider = null): string {
        return AIGUDE_Translation_Service::resolve_target_lang_code($code, $provider);
    }

    /**
     * Describe a provider/language pair for display purposes.
     *
     * @return array{provider:string,provider_label:string,code:string,label:string,display:string,available:bool}
     */
    public static function describe_target_language_choice(?string $provider, ?string $code): array {
        $provider = is_string($provider)
            ? AIGUDE_Translation_Service::normalize_translation_provider($provider)
            : '';
        $meta = self::get_translation_providers_metadata();
        if ($provider === '' || !isset($meta[$provider])) {
            return [
                'provider'       => '',
                'provider_label' => '',
                'code'           => '',
                'label'          => '',
                'display'        => '',
                'available'      => false,
            ];
        }

        $languages = self::get_translation_languages($provider);
        $normalized = '';
        if (is_string($code) && $code !== '') {
            if ($provider === 'deepl') {
                $normalized = AIGUDE_Translation_Service::normalize_deepl_lang_code($code, 'target');
            } else {
                $normalized = AIGUDE_Translation_Service::normalize_provider_lang_code_generic($code, $provider, 'target');
            }
        }
        if ($normalized === '') {
            $normalized = is_string($code) ? strtoupper(trim($code)) : '';
        }

        $label = '';
        $available = false;
        if ($normalized !== '') {
            if (isset($languages[$normalized])) {
                $label = $languages[$normalized];
                $available = true;
            } else {
                $label = strtoupper($normalized);
            }
        }

        $provider_label = self::get_translation_provider_label($provider);
        $display = '';
        if ($label !== '' && $provider_label !== '') {
            $display = sprintf('%s — %s', $label, $provider_label);
        } else {
            $display = $label;
        }

        return [
            'provider'       => $provider,
            'provider_label' => $provider_label,
            'code'           => $normalized,
            'label'          => $label,
            'display'        => $display,
            'available'      => $available,
        ];
    }

    /** Provide the cached API key so the translation service can talk to AiGude. */
    public static function get_translation_api_key(): string {
        [$apiKey] = self::media_controller()->get_active_server_credentials();
        return $apiKey;
    }

    /**
     * Proxy to the translation service for DeepL-specific normalization.
     */
    private static function normalize_deepl_lang_code(string $code, string $context = 'target'): string {
        return AIGUDE_Translation_Service::normalize_deepl_lang_code($code, $context);
    }

    /**
     * Map a WordPress locale string into the format DeepL expects.
     */
    private static function map_wp_locale_to_deepl(string $locale): string {
        return AIGUDE_Translation_Service::map_wp_locale_to_deepl($locale);
    }

    /**
     * Map a WordPress locale to whatever representation a given provider uses.
     */
    private static function map_wp_locale_to_provider(string $locale, ?string $provider = null): string {
        return AIGUDE_Translation_Service::map_wp_locale_to_provider($locale, $provider);
    }

    /**
     * Normalize a provider-specific language code when we do not know the provider in advance.
     */
    private static function normalize_provider_lang_code_generic(string $code, string $provider, string $context = 'target'): string {
        return AIGUDE_Translation_Service::normalize_provider_lang_code_generic($code, $provider, $context);
    }

    /**
     * Resolve the Img2Desc endpoint, honoring overrides via constants/env/filters.
     */
    public static function get_img2desc_url(): string {
        if (defined('AIGUDE_IMG2DESC_URL') && is_string(AIGUDE_IMG2DESC_URL) && AIGUDE_IMG2DESC_URL !== '') {
            return trim(AIGUDE_IMG2DESC_URL);
        }

        $env = getenv('AIGUDE_IMG2DESC_URL');
        if (is_string($env) && trim($env) !== '') {
            return trim($env);
        }

        if (function_exists('apply_filters')) {
            $filtered = apply_filters('aigude_img2desc_url', self::API_URL_IMG2DESC);
            if (is_string($filtered) && trim($filtered) !== '') {
                return trim($filtered);
            }
        }

        return self::API_URL_IMG2DESC;
    }

    /**
     * Resolve the credits endpoint using the same override cascade as the Img2Desc URL.
     */
    public static function get_credits_url(): string {
        if (defined('AIGUDE_REMAINING_CREDITS_URL') && is_string(AIGUDE_REMAINING_CREDITS_URL) && AIGUDE_REMAINING_CREDITS_URL !== '') {
            return trim(AIGUDE_REMAINING_CREDITS_URL);
        }

        $env = getenv('AIGUDE_REMAINING_CREDITS_URL');
        if (is_string($env) && trim($env) !== '') {
            return trim($env);
        }

        if (function_exists('apply_filters')) {
            $filtered = apply_filters('aigude_remaining_credits_url', self::API_URL_CREDITS);
            if (is_string($filtered) && trim($filtered) !== '') {
                return trim($filtered);
            }
        }

        return self::API_URL_CREDITS;
    }

    /*** Bootstrap ***/
    /**
     * Wire up all plugin modules (UI, media, query integrations).
     */
    public static function init(): void {
        add_action('init', [__CLASS__, 'load_textdomain']);
        self::admin_ui()->register();
        self::media_controller()->register();
        self::media_query()->register();
    }

    /**
     * Load the plugin's bundled translations.
     */
    public static function load_textdomain(): void {
        $domain = 'aigude-tools';

        $locale = function_exists('determine_locale') ? determine_locale() : get_locale();

        // Load GlotPress/WordPress language packs first (wp-content/languages/plugins).
        if (defined('WP_LANG_DIR')) {
            $global_mofile = trailingslashit(WP_LANG_DIR) . 'plugins/' . $domain . '-' . $locale . '.mo';
            if (file_exists($global_mofile)) {
                load_textdomain($domain, $global_mofile);
            }
        }

        // Always allow the bundled translations to override the global ones.
        $mofile = plugin_dir_path(__FILE__) . 'languages/' . $domain . '-' . $locale . '.mo';
        if (file_exists($mofile)) {
            load_textdomain($domain, $mofile);
        }
    }

    /**
     * Get last used languages (max 5) for a given type: 'target' | 'prompt' | 'placeholder'.
     * Optionally scoped to a provider (recent per provider).
     */
    public static function get_recent_langs(string $type, ?string $provider = null): array {
        return self::media_query()->get_recent_langs($type, $provider);
    }

    /**
     * Push one language code into the recent list for a type (dedup, keep 5).
     * Optionally scope to a provider.
     */
    public static function push_recent_lang(string $type, string $code, ?string $provider = null): void {
        self::media_query()->push_recent_lang($type, $code, $provider);
    }

    /**
     * Lazily instantiate and return the admin UI helper.
     */
    private static function admin_ui(): AIGUDE_Admin_UI {
        if (!self::$admin_ui) {
            self::$admin_ui = new AIGUDE_Admin_UI();
        }
        return self::$admin_ui;
    }

    /**
     * Lazily instantiate and return the media controller helper.
     */
    private static function media_controller(): AIGUDE_Media_Controller {
        if (!self::$media_controller) {
            self::$media_controller = new AIGUDE_Media_Controller();
        }
        return self::$media_controller;
    }

    /**
     * Lazily instantiate and return the media/query helper.
     */
    private static function media_query(): AIGUDE_Media_Query {
        if (!self::$media_query) {
            self::$media_query = new AIGUDE_Media_Query();
        }
        return self::$media_query;
    }

    /**
     * Determine whether debug logging features should be enabled (WP_DEBUG, SCRIPT_DEBUG, or env flag).
     */
    public static function debug_enabled(): bool {
        // Standard WordPress flags
        $wp_debug    = defined('WP_DEBUG') && WP_DEBUG;
        $script_debug = defined('SCRIPT_DEBUG') && SCRIPT_DEBUG;

        // Environment variable (e.g., Docker compose WORDPRESS_DEBUG=1)
        $env = getenv('WORDPRESS_DEBUG');
        $env_debug = is_string($env) ? filter_var($env, FILTER_VALIDATE_BOOLEAN) : false;

        return ($wp_debug || $script_debug || $env_debug) === true;
    }

}

// Kick off
AIGUDE_Tools_Plugin::init();
