        class ChatWidget {
            constructor(accessToken, botName = 'Custom Bot Name', wsUrl, logoUrl = '') {
                this.accessToken = accessToken;
                this.botName = botName;
                this.ws = null;
                this.currentMessage = '';
                this.isTyping = false;
                this.wsUrl = wsUrl;
                this.logoUrl = logoUrl;
                this.lastActivity = Date.now();
                this.inactivityTimeout = 300000; // 5 minutes
                
                // Initialize and setup event listeners
                this.initialize();
                
                // Add WordPress-specific styling
                this.addWordPressStyles();
                
                // Debug initialization
                console.log('ChatWidget initialized with:', {
                    accessToken: !!this.accessToken,
                    botName: this.botName,
                    wsUrl: this.wsUrl,
                    logoUrl: this.logoUrl
                });
            }
            
            addWordPressStyles() {
                // Create a style element
                const style = document.createElement('style');
                style.textContent = `
                    /* WordPress specific overrides */
                    #chat-window {
                        box-sizing: border-box !important;
                        font-family: -apple-system, BlinkMacSystemFont, "Segoe UI", Roboto, Oxygen-Sans, Ubuntu, Cantarell, "Helvetica Neue", sans-serif !important;
                    }
                    
                    #chat-window * {
                        box-sizing: border-box !important;
                    }
                    
                    #chat-window button {
                        min-width: auto !important;
                        min-height: auto !important;
                        line-height: normal !important;
                    }
                    
                    #toggle-chat {
                        min-width: auto !important;
                        min-height: auto !important;
                        line-height: normal !important;
                    }
                    
                    @media (min-width: 768px) {
                        #chat-window {
                            height: 500px !important;
                            width: 350px !important;
                            position: fixed !important;
                            bottom: 70px !important;
                            right: 20px !important;
                            border-radius: 0.75rem !important;
                            overflow: hidden !important;
                        }
                    }
                `;
                document.head.appendChild(style);
            }

            initialize() {
                // Create widget container
                const container = document.createElement('div');
                container.className = 'fixed bottom-5 right-5 flex flex-col items-end z-[9999]';
                
                // Create chat window HTML
                container.innerHTML = `
                    <div id="chat-window" class="hidden transform transition-all duration-300 bg-white shadow-2xl flex flex-col
                        fixed inset-0 m-0 h-full w-full rounded-none z-[999]
                        md:fixed md:inset-auto md:right-5 md:bottom-20 md:h-[500px] md:w-[350px] md:rounded-xl md:m-0 md:mb-4">
                        <!-- Header -->
                        <div class="bg-blue-600 p-4 flex justify-between items-center shadow-sm
                            rounded-t-none
                            md:rounded-t-xl">
                            <div class="flex items-center space-x-3">
                                <div class="w-10 h-10 bg-white rounded-full flex items-center justify-center overflow-hidden">
                                    <img src="${this.logoUrl}" alt="${this.botName}" class="w-8 h-8 object-contain">
                                </div>
                                <h3 class="text-white font-semibold text-lg">${this.botName}</h3>
                            </div>
                            <button class="text-white/80 hover:text-white text-2xl transition-transform transform hover:scale-110 active:scale-95" id="close-chat">&times;</button>
                        </div>

                        <!-- Messages -->
                        <div class="flex-1 p-4 space-y-3 overflow-y-auto scroll-smooth" id="messages-container">
                            <!-- Welcome message -->
                            <div class="flex justify-center">
                                <div class="bg-gray-100 px-4 py-2 rounded-xl text-sm text-gray-600">
                                    ${this.botName} is here to help!
                                </div>
                            </div>
                        </div>

                        <!-- Input Area -->
                        <div class="p-3 bg-gray-50 border-t">
                            <div class="flex items-end space-x-2">
                                <textarea 
                                    id="message-input"
                                    rows="1"
                                    placeholder="Type your message..."
                                    class="flex-1 p-2.5 resize-none border rounded-xl focus:outline-none focus:ring-2 focus:ring-indigo-500 transition-all max-h-32 overflow-y-auto scroll-smooth"
                                    style="scrollbar-width: thin;"
                                ></textarea>
                                <button id="send-button"
                                    class="bg-indigo-600 text-white p-2.5 rounded-xl hover:bg-indigo-700 transition-colors disabled:opacity-50 disabled:cursor-not-allowed">
                                    <svg class="w-5 h-5" fill="none" stroke="currentColor" viewBox="0 0 24 24">
                                        <path stroke-linecap="round" stroke-linejoin="round" stroke-width="2" d="M13 5l7 7-7 7M5 5l7 7-7 7"></path>
                                    </svg>
                                </button>
                            </div>
                            <div class="text-xs text-gray-400 mt-1 px-1">
                                Press Enter to send, Shift+Enter for new line
                            </div>
                        </div>

                        <!-- Footer -->
                        <div class="p-2 text-center text-xs text-gray-400 border-t bg-gray-50 
                            rounded-b-none
                            md:rounded-b-xl">
                            Powered by ChatDocLM
                        </div>
                    </div>

                    <!-- Toggle Button -->
                    <button id="toggle-chat" class="bg-blue-600 text-white w-14 h-14 rounded-full flex items-center justify-center shadow-lg hover:shadow-xl hover:scale-105 active:scale-95 transition-all z-[1000]">
                        <svg class="w-6 h-6" fill="none" stroke="currentColor" viewBox="0 0 24 24">
                            <path stroke-linecap="round" stroke-linejoin="round" stroke-width="2" d="M8 12h.01M12 12h.01M16 12h.01M21 12c0 4.418-4.03 8-9 8a9.863 9.863 0 01-4.255-.949L3 20l1.395-3.72C3.512 15.042 3 13.574 3 12c0-4.418 4.03-8 9-8s9 3.582 9 8z"></path>
                        </svg>
                    </button>
                `;

                // Add to document
                document.body.appendChild(container);

                // Initialize elements
                this.chatWindow = document.getElementById('chat-window');
                this.messagesContainer = document.getElementById('messages-container');
                this.messageInput = document.getElementById('message-input');
                this.sendButton = document.getElementById('send-button');
                this.toggleButton = document.getElementById('toggle-chat');
                this.closeButton = document.getElementById('close-chat');

                // Setup event listeners
                this.setupEventListeners();
            }

            setupEventListeners() {
                // Chat toggle
                this.toggleButton.addEventListener('click', (e) => {
                    console.log('Toggle button clicked');
                    e.preventDefault();
                    e.stopPropagation();
                    this.toggleChat();
                });
                
                this.closeButton.addEventListener('click', (e) => {
                    e.preventDefault();
                    e.stopPropagation();
                    this.toggleChat();
                });

                // Message input handling
                this.messageInput.addEventListener('input', () => this.adjustTextareaHeight());
                this.messageInput.addEventListener('keydown', (e) => {
                    if (e.key === 'Enter' && !e.shiftKey) {
                        e.preventDefault();
                        this.sendMessage();
                    }
                });

                // Send button
                this.sendButton.addEventListener('click', () => this.sendMessage());

                // Activity tracking
                document.addEventListener('mousemove', () => this.resetInactivityTimer());
                document.addEventListener('keydown', () => this.resetInactivityTimer());
                this.messageInput.addEventListener('focus', () => this.resetInactivityTimer());

                // Disable send button when empty
                this.messageInput.addEventListener('input', () => {
                    this.sendButton.disabled = this.messageInput.value.trim() === '';
                });
                
                // Handle window resize for responsiveness
                window.addEventListener('resize', () => this.handleResize());
                
                // Initial responsive adjustment
                this.handleResize();
            }
            
            handleResize() {
                // Get window width
                const windowWidth = window.innerWidth;
                
                // Apply specific styles for mobile vs desktop
                if (windowWidth < 768) { // Mobile
                    if (!this.chatWindow.classList.contains('hidden')) {
                        document.body.style.overflow = 'hidden'; // Prevent scrolling when chat is open on mobile
                    }
                } else { // Desktop
                    document.body.style.overflow = ''; // Always allow scrolling on desktop
                }
            }

            adjustTextareaHeight() {
                this.messageInput.style.height = 'auto';
                this.messageInput.style.height = `${Math.min(this.messageInput.scrollHeight, 128)}px`;
            }

            resetInactivityTimer() {
                this.lastActivity = Date.now();
                if (this.chatWindow.classList.contains('hidden')) return;
                clearTimeout(this.inactivityTimer);
                this.inactivityTimer = setTimeout(() => {
                    if (!this.chatWindow.classList.contains('hidden')) {
                        this.toggleChat();
                    }
                }, this.inactivityTimeout);
            }

            connectWebSocket() {
                const wsUrl = `${this.wsUrl}/api/v1/chatbots/chat?access_token=${this.accessToken}`;
                this.ws = new WebSocket(wsUrl);

                this.ws.onopen = () => {
                    console.log('WebSocket connected');
                    this.currentMessage = '';  // Reset current message for new query
                    this.showTypingIndicator();
                    setTimeout(() => {
                        this.removeTypingIndicator();
                        this.addMessage(`Hello! I'm ${this.botName}. How can I help you today?`);
                    }, 1000);
                };
                this.messageInput.focus();

                this.ws.onmessage = (event) => {
                    const data = JSON.parse(event.data);
                    console.log('Received message:', data);
                    if (data.type === 'message') {
                        if (!this.currentMessage) {
                            this.showTypingIndicator();
                        }
                        this.currentMessage += data.content;
                        // Update the last message or add a new one
                        this.updateOrAddMessage(this.currentMessage);
                    } else if (data.type === 'end' || !data.content) {
                        this.removeTypingIndicator();
                        this.currentMessage = '';  // Clear current message after receiving 'end'
                    }
                };
                this.messageInput.focus();

                this.ws.onclose = () => {
                    console.log('WebSocket disconnected');
                    this.ws = null;  // Reset WebSocket connection
                };
            }
            
            updateOrAddMessage(content) {
                const lastMessage = this.messagesContainer.lastElementChild;
                if (lastMessage && !lastMessage.classList.contains('typing-indicator')) {
                    const messageContent = lastMessage.querySelector('div > div');
                    if (messageContent) {
                        messageContent.textContent = content;
                    }
                } else {
                    this.addMessage(content);
                }
                this.scrollToBottom();
            }


            sendMessage() {
                const message = this.messageInput.value.trim();
                if (!message) return;
                
                // Add user message to chat
                this.addMessage(message, true);
                
                // Clear input
                this.messageInput.value = '';
                this.adjustTextareaHeight();
                this.sendButton.disabled = true;
                
                // Show typing indicator
                this.showTypingIndicator();
                
                // Send message via WebSocket
                if (this.ws && this.ws.readyState === WebSocket.OPEN) {
                    this.ws.send(JSON.stringify({
                        type: 'message',
                        content: message
                    }));
                } else {
                    console.error('WebSocket is not connected');
                }
            }

            toggleChat() {
                const isHidden = this.chatWindow.classList.contains('hidden');
                
                if (isHidden) {
                    // Show chat window
                    this.chatWindow.classList.remove('hidden');
                    this.chatWindow.classList.add('flex');
                    
                    // Properly set z-index to ensure it's visible above other elements
                    this.chatWindow.style.zIndex = '9999';
                    
                    // Handle body overflow based on screen size
                    if (window.innerWidth < 768) {
                        document.body.style.overflow = 'hidden'; // Prevent background scrolling on mobile
                    }
                    
                    // Connect WebSocket if not already connected
                    if (!this.ws || this.ws.readyState !== WebSocket.OPEN) {
                        this.connectWebSocket();
                    }
                    
                    // Reset inactivity timer
                    this.resetInactivityTimer();
                    
                    // Focus input
                    setTimeout(() => {
                        this.messageInput.focus();
                    }, 300);
                } else {
                    // Hide chat window
                    this.chatWindow.classList.add('hidden');
                    this.chatWindow.classList.remove('flex');
                    
                    // Restore body overflow
                    document.body.style.overflow = '';
                }
            }

            createTypingIndicator() {
                const indicator = document.createElement('div');
                indicator.className = 'flex justify-start';
                indicator.innerHTML = `
                    <div class="flex items-center space-x-1 p-2 bg-gray-100 rounded-xl w-16">
                        <div class="w-1.5 h-1.5 bg-gray-400 rounded-full animate-pulse"></div>
                        <div class="w-1.5 h-1.5 bg-gray-400 rounded-full animate-pulse delay-150"></div>
                        <div class="w-1.5 h-1.5 bg-gray-400 rounded-full animate-pulse delay-300"></div>
                    </div>
                `;
                return indicator;
            }

            addMessage(content, isUser = false) {
                const messageDiv = document.createElement('div');
                messageDiv.className = `flex ${isUser ? 'justify-end' : 'justify-start'} animate-fade-in`;
                
                messageDiv.innerHTML = `
                    <div class="rounded-xl px-4 py-2 max-w-[85%] ${
                        isUser ? 
                            'bg-blue-600 text-white' : 
                            'bg-gray-100 text-gray-800 shadow-sm'
                    } transform transition-transform duration-200 ${
                        isUser ? 'hover:scale-[1.02]' : 'hover:scale-[1.01]'
                    }">
                        ${content.replace(/\n/g, '<br>')}
                    </div>
                `;
                
                // Remove typing indicator if exists
                const existingIndicator = this.messagesContainer.querySelector('.typing-indicator');
                if (existingIndicator) {
                    existingIndicator.remove();
                }

                this.messagesContainer.appendChild(messageDiv);
                this.scrollToBottom();
            }

            showTypingIndicator() {
                const indicator = this.createTypingIndicator();
                indicator.classList.add('typing-indicator');
                this.messagesContainer.appendChild(indicator);
                this.scrollToBottom();
            }

            removeTypingIndicator() {
                const indicator = this.messagesContainer.querySelector('.typing-indicator');
                if (indicator) {
                    indicator.remove();
                }
            }

            scrollToBottom() {
                this.messagesContainer.scrollTop = this.messagesContainer.scrollHeight;
            }
        }

        // Initialize ChatWidget when DOM is ready
        document.addEventListener('DOMContentLoaded', function() {
            window.ChatWidget = ChatWidget;
        });
    